<?php
/**
 * API لأرشفة الطلبات
 * يستخدم هذا الملف لأرشفة الطلبات في قاعدة البيانات
 * مع مراعاة صلاحيات المستخدم (متاح فقط للإدارة)
 */

// تفعيل عرض الأخطاء للتصحيح
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// بدء الجلسة إذا لم تكن قد بدأت بالفعل
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// تضمين ملف الاتصال بقاعدة البيانات
require_once '../config/db_connection.php';

// التحقق من تسجيل دخول المستخدم
if (!isset($_SESSION['user_id'])) {
    header('Content-Type: application/json');
    echo json_encode(['status' => 'error', 'message' => 'يجب تسجيل الدخول لأرشفة الطلبات']);
    exit;
}

// التحقق من صلاحيات المستخدم (متاح فقط للإدارة)
$userRole = isset($_SESSION['role']) ? $_SESSION['role'] : 'user';
if ($userRole !== 'admin' && $userRole !== 'manager') {
    header('Content-Type: application/json');
    echo json_encode(['status' => 'error', 'message' => 'ليس لديك صلاحية لأرشفة الطلبات']);
    exit;
}

// التحقق من أن الطلب هو طلب POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Content-Type: application/json');
    echo json_encode(['status' => 'error', 'message' => 'يجب استخدام طريقة POST']);
    exit;
}

// الحصول على البيانات المرسلة
$data = json_decode(file_get_contents('php://input'), true);

// التحقق من وجود معرف الطلب
if (!isset($data['order_id']) || empty($data['order_id'])) {
    header('Content-Type: application/json');
    echo json_encode(['status' => 'error', 'message' => 'معرف الطلب مطلوب']);
    exit;
}

$orderId = $data['order_id'];
$userId = $_SESSION['user_id'];
$archiveDate = date('Y-m-d');

try {
    // بدء المعاملة
    $conn->beginTransaction();

    // التحقق من وجود الطلب وأنه غير مؤرشف بالفعل
    $checkSql = "SELECT id, archived FROM orders WHERE id = ?";
    $checkStmt = $conn->prepare($checkSql);
    $checkStmt->execute([$orderId]);
    $order = $checkStmt->fetch(PDO::FETCH_ASSOC);

    if (!$order) {
        throw new Exception('الطلب غير موجود');
    }

    if ($order['archived'] == 1) {
        throw new Exception('الطلب مؤرشف بالفعل');
    }

    // تحديث حالة الطلب إلى مؤرشف
    $updateSql = "UPDATE orders SET archived = 1, archive_date = ? WHERE id = ?";
    $updateStmt = $conn->prepare($updateSql);
    $updateStmt->execute([$archiveDate, $orderId]);

    // تأكيد المعاملة
    $conn->commit();

    // إرجاع استجابة النجاح
    header('Content-Type: application/json');
    echo json_encode([
        'status' => 'success',
        'message' => 'تم أرشفة الطلب بنجاح',
        'order_id' => $orderId,
        'archive_date' => $archiveDate
    ]);

} catch (Exception $e) {
    // التراجع عن المعاملة في حالة حدوث خطأ
    $conn->rollBack();

    // إرجاع رسالة الخطأ
    header('Content-Type: application/json');
    echo json_encode(['status' => 'error', 'message' => 'فشل أرشفة الطلب: ' . $e->getMessage()]);
}
?>
