<?php
/**
 * ملف تصدير الطلبات إلى PDF
 * يقوم هذا الملف بإنشاء ملف PDF يحتوي على قائمة الطلبات حسب المعايير المحددة
 */

// تفعيل عرض الأخطاء للتصحيح
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// بدء الجلسة
session_start();

// التحقق من تسجيل دخول المستخدم
if (!isset($_SESSION['user_id'])) {
    header('Content-Type: application/json');
    echo json_encode([
        'status' => 'error',
        'message' => 'يجب تسجيل الدخول لتصدير الطلبات'
    ]);
    exit;
}

// تضمين ملف الاتصال بقاعدة البيانات
require_once '../config/db_connection.php';

// تضمين مكتبة TCPDF
require_once '../vendor/tcpdf/tcpdf.php';

// الحصول على معلمات التصفية
$dateFrom = isset($_GET['date_from']) ? $_GET['date_from'] : null;
$dateTo = isset($_GET['date_to']) ? $_GET['date_to'] : null;
$branchId = isset($_GET['branch_id']) ? $_GET['branch_id'] : null;
$status = isset($_GET['status']) ? $_GET['status'] : null;

// التحقق من دور المستخدم وتحديد ما إذا كان يجب تقييد الطلبات
$userRole = isset($_SESSION['role']) ? $_SESSION['role'] : 'user';
$userId = isset($_SESSION['user_id']) ? $_SESSION['user_id'] : null;
$userName = isset($_SESSION['full_name']) ? $_SESSION['full_name'] : 'مستخدم النظام';

try {
    // بناء استعلام SQL
    $sql = "
        SELECT
            o.id, o.order_number, o.client_name, b.name as branch,
            o.departure_area, o.delivery_area, o.goods_name,
            o.service_fees, o.amount_paid, o.remaining_amount,
            o.order_upload_date, o.scheduled_delivery_date,
            o.status, o.created_at, u.full_name as created_by
        FROM orders o
        LEFT JOIN branches b ON o.branch_id = b.id
        LEFT JOIN users u ON o.user_id = u.id
        WHERE o.archived = 0
    ";

    // إضافة شروط التصفية
    $params = [];

    // تقييد الطلبات للمستخدمين العاديين (غير المديرين)
    if ($userRole !== 'admin' && $userRole !== 'manager' && $userId) {
        $sql .= " AND o.user_id = ?";
        $params[] = $userId;
    }

    // إضافة فلتر التاريخ
    if ($dateFrom) {
        $sql .= " AND o.order_upload_date >= ?";
        $params[] = $dateFrom;
    }

    if ($dateTo) {
        $sql .= " AND o.order_upload_date <= ?";
        $params[] = $dateTo;
    }

    // إضافة فلتر الفرع
    if ($branchId) {
        $sql .= " AND o.branch_id = ?";
        $params[] = $branchId;
    }

    // إضافة فلتر الحالة
    if ($status) {
        $sql .= " AND o.status = ?";
        $params[] = $status;
    }

    // ترتيب النتائج حسب تاريخ الإنشاء (الأحدث أولاً)
    $sql .= " ORDER BY o.created_at DESC";

    // تنفيذ الاستعلام
    $stmt = $conn->prepare($sql);
    $stmt->execute($params);
    $orders = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // إنشاء كائن TCPDF
    $pdf = new TCPDF('L', 'mm', 'A4', true, 'UTF-8', false);

    // تعيين معلومات المستند
    $pdf->SetCreator('نظام إدارة الطلبات');
    $pdf->SetAuthor($userName);
    $pdf->SetTitle('قائمة الطلبات');
    $pdf->SetSubject('تقرير الطلبات');
    $pdf->SetKeywords('طلبات, تقرير, PDF');

    // تعيين الهوامش
    $pdf->SetMargins(10, 10, 10);
    $pdf->SetHeaderMargin(5);
    $pdf->SetFooterMargin(10);

    // تعطيل رأس وتذييل الصفحة الافتراضي
    $pdf->setPrintHeader(false);
    $pdf->setPrintFooter(true);

    // تعيين الخط الافتراضي
    $pdf->SetFont('aealarabiya', '', 12);

    // إضافة صفحة جديدة
    $pdf->AddPage();

    // إنشاء عنوان التقرير
    $title = 'قائمة الطلبات';
    if ($dateFrom && $dateTo) {
        $title .= ' من ' . $dateFrom . ' إلى ' . $dateTo;
    } elseif ($dateFrom) {
        $title .= ' من ' . $dateFrom;
    } elseif ($dateTo) {
        $title .= ' حتى ' . $dateTo;
    }

    // إضافة عنوان التقرير
    $pdf->SetFont('aealarabiya', 'B', 16);
    $pdf->Cell(0, 10, $title, 0, 1, 'C');
    $pdf->Ln(5);

    // إضافة معلومات إضافية
    $pdf->SetFont('aealarabiya', '', 10);
    $pdf->Cell(0, 5, 'تاريخ التقرير: ' . date('Y-m-d'), 0, 1, 'L');
    $pdf->Cell(0, 5, 'عدد الطلبات: ' . count($orders), 0, 1, 'L');
    if ($userRole !== 'admin' && $userRole !== 'manager') {
        $pdf->Cell(0, 5, 'المستخدم: ' . $userName, 0, 1, 'L');
    }
    $pdf->Ln(5);

    // تحديد عرض الأعمدة
    $colWidth = [20, 30, 40, 25, 25, 25, 25, 25, 25, 30];

    // إنشاء رأس الجدول
    $pdf->SetFont('aealarabiya', 'B', 10);
    $pdf->SetFillColor(230, 230, 230);
    $pdf->Cell($colWidth[0], 10, 'رقم الطلب', 1, 0, 'C', true);
    $pdf->Cell($colWidth[1], 10, 'اسم العميل', 1, 0, 'C', true);
    $pdf->Cell($colWidth[2], 10, 'الفرع', 1, 0, 'C', true);
    $pdf->Cell($colWidth[3], 10, 'منطقة المغادرة', 1, 0, 'C', true);
    $pdf->Cell($colWidth[4], 10, 'منطقة التسليم', 1, 0, 'C', true);
    $pdf->Cell($colWidth[5], 10, 'رسوم الخدمة', 1, 0, 'C', true);
    $pdf->Cell($colWidth[6], 10, 'المبلغ المدفوع', 1, 0, 'C', true);
    $pdf->Cell($colWidth[7], 10, 'المبلغ المتبقي', 1, 0, 'C', true);
    $pdf->Cell($colWidth[8], 10, 'تاريخ الطلب', 1, 0, 'C', true);
    $pdf->Cell($colWidth[9], 10, 'الحالة', 1, 1, 'C', true);

    // إضافة بيانات الطلبات
    $pdf->SetFont('aealarabiya', '', 9);
    $fill = false;
    foreach ($orders as $order) {
        // التحقق من أن الصف سيناسب الصفحة الحالية، وإلا إضافة صفحة جديدة
        if ($pdf->GetY() > 180) {
            $pdf->AddPage();
            
            // إعادة رسم رأس الجدول في الصفحة الجديدة
            $pdf->SetFont('aealarabiya', 'B', 10);
            $pdf->SetFillColor(230, 230, 230);
            $pdf->Cell($colWidth[0], 10, 'رقم الطلب', 1, 0, 'C', true);
            $pdf->Cell($colWidth[1], 10, 'اسم العميل', 1, 0, 'C', true);
            $pdf->Cell($colWidth[2], 10, 'الفرع', 1, 0, 'C', true);
            $pdf->Cell($colWidth[3], 10, 'منطقة المغادرة', 1, 0, 'C', true);
            $pdf->Cell($colWidth[4], 10, 'منطقة التسليم', 1, 0, 'C', true);
            $pdf->Cell($colWidth[5], 10, 'رسوم الخدمة', 1, 0, 'C', true);
            $pdf->Cell($colWidth[6], 10, 'المبلغ المدفوع', 1, 0, 'C', true);
            $pdf->Cell($colWidth[7], 10, 'المبلغ المتبقي', 1, 0, 'C', true);
            $pdf->Cell($colWidth[8], 10, 'تاريخ الطلب', 1, 0, 'C', true);
            $pdf->Cell($colWidth[9], 10, 'الحالة', 1, 1, 'C', true);
            $pdf->SetFont('aealarabiya', '', 9);
        }

        $pdf->Cell($colWidth[0], 10, $order['order_number'], 1, 0, 'C', $fill);
        $pdf->Cell($colWidth[1], 10, $order['client_name'], 1, 0, 'R', $fill);
        $pdf->Cell($colWidth[2], 10, $order['branch'], 1, 0, 'R', $fill);
        $pdf->Cell($colWidth[3], 10, $order['departure_area'], 1, 0, 'R', $fill);
        $pdf->Cell($colWidth[4], 10, $order['delivery_area'], 1, 0, 'R', $fill);
        $pdf->Cell($colWidth[5], 10, $order['service_fees'], 1, 0, 'C', $fill);
        $pdf->Cell($colWidth[6], 10, $order['amount_paid'], 1, 0, 'C', $fill);
        $pdf->Cell($colWidth[7], 10, $order['remaining_amount'], 1, 0, 'C', $fill);
        $pdf->Cell($colWidth[8], 10, $order['order_upload_date'], 1, 0, 'C', $fill);
        $pdf->Cell($colWidth[9], 10, $order['status'], 1, 1, 'C', $fill);
        
        // تبديل لون الخلفية للصف التالي
        $fill = !$fill;
    }

    // تعيين معلومات تذييل الصفحة
    $pdf->SetY(-15);
    $pdf->SetFont('aealarabiya', 'I', 8);
    $pdf->Cell(0, 10, 'الصفحة ' . $pdf->getAliasNumPage() . '/' . $pdf->getAliasNbPages(), 0, 0, 'C');

    // إخراج ملف PDF
    $fileName = 'orders_report_' . date('Y-m-d') . '.pdf';
    $pdf->Output($fileName, 'D');

} catch (PDOException $e) {
    // إرجاع رسالة الخطأ
    header('Content-Type: application/json');
    echo json_encode(['status' => 'error', 'message' => 'فشل تصدير الطلبات: ' . $e->getMessage()]);
}
?>
