<?php
/**
 * ملف جلب الطلبات الخاصة بالمستخدم الحالي
 * يستخدم هذا الملف لجلب الطلبات الخاصة بالمستخدم الحالي من قاعدة البيانات
 */

// تفعيل عرض الأخطاء للتصحيح
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// تسجيل الأخطاء في ملف السجل
error_log("بدء تنفيذ get_user_orders.php");

// بدء الجلسة
session_start();

// تضمين ملف الاتصال بقاعدة البيانات
require_once '../config/db_connection.php';

// تسجيل حالة الجلسة
error_log("حالة الجلسة: " . (isset($_SESSION['user_id']) ? "المستخدم مسجل الدخول" : "المستخدم غير مسجل الدخول"));

// التحقق من تسجيل دخول المستخدم
if (!isset($_SESSION['user_id'])) {
    error_log("خطأ: المستخدم غير مسجل الدخول");

    // للتصحيح: إرجاع طلبات افتراضية بدلاً من رفض الطلب
    header('Content-Type: application/json');
    echo json_encode([
        'status' => 'success',
        'orders' => [],
        'user_role' => 'guest',
        'debug_message' => 'المستخدم غير مسجل الدخول، تم إرجاع مصفوفة فارغة'
    ]);
    exit;
}

// الحصول على معرف المستخدم
$userId = $_SESSION['user_id'];
$userRole = $_SESSION['role'] ?? 'employee';

error_log("معرف المستخدم: $userId، دور المستخدم: $userRole");

try {
    // التحقق من وجود الجداول المطلوبة
    error_log("التحقق من وجود الجداول المطلوبة");

    $requiredTables = ['orders', 'branches', 'users'];
    $missingTables = [];

    foreach ($requiredTables as $table) {
        $stmt = $conn->query("SHOW TABLES LIKE '$table'");
        if ($stmt->rowCount() === 0) {
            $missingTables[] = $table;
        }
    }

    if (!empty($missingTables)) {
        error_log("الجداول المفقودة: " . implode(', ', $missingTables));
        throw new Exception("بعض الجداول المطلوبة غير موجودة: " . implode(', ', $missingTables));
    }

    // إعداد استعلام جلب الطلبات
    error_log("إعداد استعلام جلب الطلبات");

    $query = "
        SELECT
            o.id, o.order_number, o.client_name, o.service_type,
            o.branch_id, b.name as branch_name,
            o.departure_area, o.second_destination, o.delivery_area,
            o.goods_name, o.goods_type, o.country_of_origin,
            o.weight, o.service_fees, o.currency,
            o.payment_type, o.delivery_duration,
            o.amount_paid, o.remaining_amount,
            o.order_upload_date, o.scheduled_delivery_date,
            o.recipient_name, o.phone_number,
            o.delivery_company, o.employee_name,
            o.payment_status, o.status,
            o.created_at, o.updated_at,
            u.full_name as user_full_name
        FROM orders o
        LEFT JOIN branches b ON o.branch_id = b.id
        LEFT JOIN users u ON o.user_id = u.id
        WHERE o.archived = 0
    ";

    // إذا كان المستخدم موظفًا عاديًا، قم بجلب الطلبات الخاصة به فقط
    if ($userRole === 'employee') {
        $query .= " AND o.user_id = :user_id";
    }

    // ترتيب الطلبات حسب تاريخ الإنشاء (الأحدث أولاً)
    $query .= " ORDER BY o.created_at DESC";

    error_log("استعلام SQL: $query");

    // إعداد الاستعلام
    $stmt = $conn->prepare($query);

    // ربط المعلمات
    if ($userRole === 'employee') {
        $stmt->bindParam(':user_id', $userId);
        error_log("ربط معرف المستخدم: $userId");
    }

    // تنفيذ الاستعلام
    error_log("تنفيذ الاستعلام");
    $stmt->execute();

    // الحصول على النتائج
    error_log("الحصول على النتائج");
    $orders = $stmt->fetchAll(PDO::FETCH_ASSOC);
    error_log("عدد الطلبات المستردة: " . count($orders));

    // تحويل التواريخ إلى التنسيق المناسب
    foreach ($orders as &$order) {
        if (!empty($order['order_upload_date'])) {
            $order['order_upload_date_formatted'] = date('d/m/Y', strtotime($order['order_upload_date']));
        } else {
            $order['order_upload_date_formatted'] = '';
        }

        if (!empty($order['scheduled_delivery_date'])) {
            $order['scheduled_delivery_date_formatted'] = date('d/m/Y', strtotime($order['scheduled_delivery_date']));
        } else {
            $order['scheduled_delivery_date_formatted'] = '';
        }

        if (!empty($order['created_at'])) {
            $order['created_at_formatted'] = date('d/m/Y H:i', strtotime($order['created_at']));
        } else {
            $order['created_at_formatted'] = '';
        }
    }

    // إرجاع النتائج
    header('Content-Type: application/json');
    echo json_encode([
        'status' => 'success',
        'orders' => $orders,
        'user_role' => $userRole
    ]);

} catch (PDOException $e) {
    // تسجيل الخطأ
    error_log("خطأ PDO: " . $e->getMessage());

    // إرجاع رسالة الخطأ
    header('Content-Type: application/json');
    echo json_encode([
        'status' => 'error',
        'message' => 'حدث خطأ أثناء جلب الطلبات: ' . $e->getMessage(),
        'error_type' => 'database_error'
    ]);
} catch (Exception $e) {
    // تسجيل الخطأ
    error_log("خطأ عام: " . $e->getMessage());

    // إرجاع رسالة الخطأ
    header('Content-Type: application/json');
    echo json_encode([
        'status' => 'error',
        'message' => 'حدث خطأ أثناء جلب الطلبات: ' . $e->getMessage(),
        'error_type' => 'general_error'
    ]);
}
?>
