/**
 * ملف سكريبت إدارة الأرشيف
 * يستخدم هذا الملف لإدارة وظائف الأرشفة في صفحة الطلبات النشطة
 */

// عند تحميل الصفحة
document.addEventListener('DOMContentLoaded', function() {
  // إضافة أزرار الأرشفة إلى جدول الطلبات
  addArchiveButtons();
});

/**
 * إضافة أزرار الأرشفة إلى جدول الطلبات
 */
function addArchiveButtons() {
  // التحقق من وجود جدول الطلبات
  const ordersTable = document.getElementById('ordersTableBody');
  if (!ordersTable) return;

  // الحصول على جميع صفوف الجدول
  const rows = ordersTable.querySelectorAll('tr');
  
  // إضافة زر الأرشفة لكل صف
  rows.forEach(row => {
    // الحصول على خلية الإجراءات
    const actionsCell = row.querySelector('td:last-child');
    if (!actionsCell) return;
    
    // الحصول على معرف الطلب
    const orderId = row.getAttribute('data-order-id');
    if (!orderId) return;
    
    // إضافة زر الأرشفة
    const archiveButton = document.createElement('button');
    archiveButton.className = 'text-gray-600 hover:text-gray-900 mx-1';
    archiveButton.title = 'أرشفة';
    archiveButton.innerHTML = '<i class="fas fa-archive"></i>';
    archiveButton.onclick = function(e) {
      e.preventDefault();
      e.stopPropagation();
      archiveOrder(orderId);
    };
    
    // إضافة الزر إلى خلية الإجراءات
    actionsCell.appendChild(archiveButton);
  });
}

/**
 * أرشفة طلب
 * @param {number} orderId - معرف الطلب
 */
function archiveOrder(orderId) {
  // التأكيد قبل الأرشفة
  Swal.fire({
    title: 'تأكيد الأرشفة',
    text: 'هل أنت متأكد من رغبتك في أرشفة هذا الطلب؟',
    icon: 'question',
    showCancelButton: true,
    confirmButtonText: 'نعم، أرشفة',
    cancelButtonText: 'إلغاء',
    confirmButtonColor: '#3085d6',
    cancelButtonColor: '#d33'
  }).then((result) => {
    if (result.isConfirmed) {
      // إظهار مؤشر التحميل
      Swal.fire({
        title: 'جاري الأرشفة...',
        text: 'يرجى الانتظار',
        allowOutsideClick: false,
        didOpen: () => {
          Swal.showLoading();
        }
      });
      
      // إرسال طلب الأرشفة
      fetch('../api/archive_order.php', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({ order_id: orderId })
      })
      .then(response => response.json())
      .then(data => {
        if (data.status === 'success') {
          Swal.fire({
            icon: 'success',
            title: 'تم الأرشفة بنجاح',
            text: data.message,
            showConfirmButton: false,
            timer: 1500
          });
          
          // إزالة الصف من الجدول
          const row = document.querySelector(`tr[data-order-id="${orderId}"]`);
          if (row) {
            row.classList.add('fade-out');
            setTimeout(() => {
              row.remove();
              
              // التحقق من وجود صفوف في الجدول
              const remainingRows = document.querySelectorAll('#ordersTableBody tr');
              if (remainingRows.length === 0) {
                document.getElementById('ordersTableBody').innerHTML = `
                  <tr>
                    <td colspan="16" class="px-3 py-4 text-center text-gray-500">
                      <i class="fas fa-box ml-2"></i>
                      لا توجد طلبات نشطة حالياً
                    </td>
                  </tr>
                `;
              }
            }, 500);
          }
        } else {
          throw new Error(data.message || 'فشل في أرشفة الطلب');
        }
      })
      .catch(error => {
        console.error('خطأ:', error);
        Swal.fire({
          icon: 'error',
          title: 'خطأ',
          text: error.message
        });
      });
    }
  });
}

/**
 * إضافة زر الأرشفة الجماعية
 */
function addBulkArchiveButton() {
  // التحقق من وجود قسم أزرار الإجراءات الجماعية
  const bulkActionsSection = document.querySelector('.bulk-actions');
  if (!bulkActionsSection) return;
  
  // إنشاء زر الأرشفة الجماعية
  const bulkArchiveButton = document.createElement('button');
  bulkArchiveButton.className = 'btn btn-secondary ml-2';
  bulkArchiveButton.innerHTML = '<i class="fas fa-archive ml-1"></i> أرشفة المحدد';
  bulkArchiveButton.onclick = bulkArchiveOrders;
  
  // إضافة الزر إلى قسم الإجراءات الجماعية
  bulkActionsSection.appendChild(bulkArchiveButton);
}

/**
 * أرشفة الطلبات المحددة
 */
function bulkArchiveOrders() {
  // الحصول على الطلبات المحددة
  const selectedOrders = getSelectedOrders();
  
  if (selectedOrders.length === 0) {
    Swal.fire({
      icon: 'warning',
      title: 'تنبيه',
      text: 'يرجى تحديد طلب واحد على الأقل للأرشفة'
    });
    return;
  }
  
  // التأكيد قبل الأرشفة
  Swal.fire({
    title: 'تأكيد الأرشفة الجماعية',
    text: `هل أنت متأكد من رغبتك في أرشفة ${selectedOrders.length} طلب؟`,
    icon: 'question',
    showCancelButton: true,
    confirmButtonText: 'نعم، أرشفة',
    cancelButtonText: 'إلغاء',
    confirmButtonColor: '#3085d6',
    cancelButtonColor: '#d33'
  }).then((result) => {
    if (result.isConfirmed) {
      // إظهار مؤشر التحميل
      Swal.fire({
        title: 'جاري الأرشفة...',
        text: 'يرجى الانتظار',
        allowOutsideClick: false,
        didOpen: () => {
          Swal.showLoading();
        }
      });
      
      // أرشفة الطلبات واحداً تلو الآخر
      let successCount = 0;
      let failCount = 0;
      
      // استخدام وعود متسلسلة لأرشفة الطلبات
      selectedOrders.reduce((promise, orderId) => {
        return promise.then(() => {
          return fetch('../api/archive_order.php', {
            method: 'POST',
            headers: {
              'Content-Type': 'application/json'
            },
            body: JSON.stringify({ order_id: orderId })
          })
          .then(response => response.json())
          .then(data => {
            if (data.status === 'success') {
              successCount++;
              
              // إزالة الصف من الجدول
              const row = document.querySelector(`tr[data-order-id="${orderId}"]`);
              if (row) {
                row.classList.add('fade-out');
                setTimeout(() => {
                  row.remove();
                }, 500);
              }
            } else {
              failCount++;
              console.error(`فشل في أرشفة الطلب ${orderId}: ${data.message}`);
            }
          })
          .catch(error => {
            failCount++;
            console.error(`خطأ في أرشفة الطلب ${orderId}:`, error);
          });
        });
      }, Promise.resolve())
      .then(() => {
        // عرض نتيجة الأرشفة
        Swal.fire({
          icon: successCount > 0 ? 'success' : 'error',
          title: 'نتيجة الأرشفة',
          html: `
            <p>تم أرشفة ${successCount} طلب بنجاح</p>
            ${failCount > 0 ? `<p>فشل في أرشفة ${failCount} طلب</p>` : ''}
          `
        });
        
        // التحقق من وجود صفوف في الجدول
        const remainingRows = document.querySelectorAll('#ordersTableBody tr');
        if (remainingRows.length === 0) {
          document.getElementById('ordersTableBody').innerHTML = `
            <tr>
              <td colspan="16" class="px-3 py-4 text-center text-gray-500">
                <i class="fas fa-box ml-2"></i>
                لا توجد طلبات نشطة حالياً
              </td>
            </tr>
          `;
        }
      });
    }
  });
}

/**
 * الحصول على الطلبات المحددة
 * @returns {Array} مصفوفة تحتوي على معرفات الطلبات المحددة
 */
function getSelectedOrders() {
  const selectedRows = document.querySelectorAll('#ordersTableBody tr input[type="checkbox"]:checked');
  return Array.from(selectedRows).map(checkbox => {
    const row = checkbox.closest('tr');
    return row.getAttribute('data-order-id');
  });
}
