/**
 * ملف JavaScript لإدارة نموذج إنشاء الطلب
 * يتعامل مع إرسال البيانات إلى قاعدة البيانات عبر API
 */

document.addEventListener('DOMContentLoaded', function() {
  // تهيئة النموذج
  initForm();

  // إضافة مستمعي الأحداث
  const form = document.getElementById('shipmentForm');
  if (form) {
    // زر إرسال الطلب
    const submitBtn = document.getElementById('submitOrderBtn');
    if (submitBtn) {
      submitBtn.addEventListener('click', handleFormSubmit);
    }

    // زر إعادة تعيين النموذج
    const resetBtn = document.querySelector('button[type="reset"]');
    if (resetBtn) {
      resetBtn.addEventListener('click', resetForm);
    }

    // تحديث رقم الطلب عند تغيير الفرع
    const branchSelect = document.getElementById('branch');
    if (branchSelect) {
      branchSelect.addEventListener('change', updateOrderNumber);
    }

    // حساب المبلغ المتبقي عند تغيير المبلغ المدفوع أو رسوم الخدمة
    const serviceFees = document.getElementById('serviceFees');
    const amountPaid = document.getElementById('amountPaid');
    const remainingAmount = document.getElementById('remainingAmount');

    if (serviceFees && amountPaid && remainingAmount) {
      serviceFees.addEventListener('input', calculateRemainingAmount);
      amountPaid.addEventListener('input', calculateRemainingAmount);
    }

    // حساب الربح عند تغيير المبالغ
    const customerAmount = document.getElementById('amountAgreedWithCustomer');
    const agentAmount = document.getElementById('amountAgreedWithAgent');
    const otherExpenses = document.getElementById('otherExpenses');
    const profit = document.getElementById('profit');

    if (customerAmount && agentAmount && otherExpenses && profit) {
      customerAmount.addEventListener('input', calculateProfit);
      agentAmount.addEventListener('input', calculateProfit);
      otherExpenses.addEventListener('input', calculateProfit);
    }
  }
});

/**
 * تهيئة النموذج
 */
function initForm() {
  console.log('تهيئة نموذج إنشاء الطلب...');

  // تعيين تاريخ اليوم كتاريخ افتراضي
  const today = new Date().toISOString().split('T')[0];
  const uploadDateField = document.getElementById('orderUploadDate');
  if (uploadDateField) {
    uploadDateField.value = today;
  }

  // تحديث رقم الطلب
  updateOrderNumber();
}

/**
 * تحديث رقم الطلب بناءً على الفرع المحدد
 */
function updateOrderNumber() {
  const branchSelect = document.getElementById('branch');
  const orderNumberField = document.getElementById('orderNumber');

  if (!branchSelect || !orderNumberField) return;

  const selectedOption = branchSelect.options[branchSelect.selectedIndex];
  if (selectedOption && selectedOption.value) {
    // الحصول على رمز الفرع
    const branchCode = selectedOption.getAttribute('data-code');
    if (branchCode) {
      // طلب رقم طلب جديد من الخادم
      fetch(`api/generate_order_number.php?branch=${encodeURIComponent(selectedOption.value)}`)
        .then(response => response.json())
        .then(data => {
          if (data.status === 'success') {
            orderNumberField.value = data.orderNumber;

            // إضافة تأثير بصري لتنبيه المستخدم بتغيير رقم الطلب
            orderNumberField.classList.add('bg-indigo-50');
            setTimeout(() => {
              orderNumberField.classList.remove('bg-indigo-50');
            }, 1000);

            console.log(`تم إنشاء رقم طلب جديد: ${data.orderNumber} للفرع: ${selectedOption.value}`);
          } else {
            console.error('فشل في إنشاء رقم طلب جديد:', data.message);
            // استخدام طريقة احتياطية لإنشاء رقم الطلب
            const year = new Date().getFullYear().toString().substr(-2);
            const randomNum = Math.floor(Math.random() * 100);
            const fallbackOrderNumber = `${branchCode}${year}-${String(randomNum).padStart(2, '0')}`;
            orderNumberField.value = fallbackOrderNumber;
          }
        })
        .catch(error => {
          console.error('خطأ في طلب رقم الطلب:', error);
          // استخدام طريقة احتياطية لإنشاء رقم الطلب
          const year = new Date().getFullYear().toString().substr(-2);
          const randomNum = Math.floor(Math.random() * 100);
          const fallbackOrderNumber = `${branchCode}${year}-${String(randomNum).padStart(2, '0')}`;
          orderNumberField.value = fallbackOrderNumber;
        });
    }
  } else {
    orderNumberField.value = '';
  }
}

/**
 * حساب المبلغ المتبقي
 */
function calculateRemainingAmount() {
  const serviceFees = document.getElementById('serviceFees');
  const amountPaid = document.getElementById('amountPaid');
  const remainingAmount = document.getElementById('remainingAmount');

  if (!serviceFees || !amountPaid || !remainingAmount) return;

  const serviceFeesValue = parseFloat(serviceFees.value) || 0;
  const amountPaidValue = parseFloat(amountPaid.value) || 0;

  const remaining = serviceFeesValue - amountPaidValue;
  remainingAmount.value = remaining.toFixed(2);
}

/**
 * حساب الربح
 */
function calculateProfit() {
  const customerAmount = document.getElementById('amountAgreedWithCustomer');
  const agentAmount = document.getElementById('amountAgreedWithAgent');
  const otherExpenses = document.getElementById('otherExpenses');
  const profit = document.getElementById('profit');

  if (!customerAmount || !agentAmount || !otherExpenses || !profit) return;

  const customerAmountValue = parseFloat(customerAmount.value) || 0;
  const agentAmountValue = parseFloat(agentAmount.value) || 0;
  const otherExpensesValue = parseFloat(otherExpenses.value) || 0;

  const profitValue = customerAmountValue - agentAmountValue - otherExpensesValue;
  profit.value = profitValue.toFixed(2);
}

/**
 * معالجة إرسال النموذج
 */
function handleFormSubmit() {
  const form = document.getElementById('shipmentForm');

  if (!form) {
    console.error('لم يتم العثور على النموذج بمعرف shipmentForm');
    return;
  }

  // إزالة رسائل الخطأ السابقة
  const errorMessages = document.querySelectorAll('.error-message');
  errorMessages.forEach(el => el.remove());

  // التحقق من صحة النموذج
  if (!validateForm(form)) {
    return;
  }

  // جمع بيانات النموذج
  const formData = new FormData(form);

  // إضافة بيانات إضافية
  formData.append('order_number', document.getElementById('orderNumber').value);
  formData.append('payment_status', parseFloat(document.getElementById('remainingAmount').value) <= 0 ? 'مدفوع' : 'جزئي');
  formData.append('status', 'قيد المعالجة');

  // إضافة معرف الفرع
  const branchSelect = document.getElementById('branch');
  if (branchSelect && branchSelect.selectedIndex > 0) {
    const branchOption = branchSelect.options[branchSelect.selectedIndex];
    formData.append('branch_id', branchSelect.selectedIndex);
    formData.append('branch_code', branchOption.dataset.code);
  }

  // عرض مؤشر التحميل
  const submitBtn = document.getElementById('submitOrderBtn');
  if (submitBtn) {
    submitBtn.disabled = true;
    submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin ml-2"></i> <span>جاري الحفظ...</span>';
  }

  // طباعة البيانات للتصحيح
  console.log('بيانات النموذج المرسلة:');
  for (let pair of formData.entries()) {
    console.log(pair[0] + ': ' + pair[1]);
  }

  // إرسال البيانات إلى الخادم
  fetch('api/save_order.php', {
    method: 'POST',
    body: formData
  })
  .then(response => {
    // التحقق من حالة الاستجابة
    if (!response.ok) {
      console.error('خطأ في استجابة الخادم:', response.status, response.statusText);
      throw new Error(`خطأ في استجابة الخادم: ${response.status}`);
    }

    // محاولة تحليل الاستجابة كـ JSON
    return response.json().catch(err => {
      console.error('خطأ في تحليل استجابة JSON:', err);
      throw new Error('فشل في تحليل استجابة الخادم');
    });
  })
  .then(data => {
    console.log('استجابة الخادم:', data);

    if (submitBtn) {
      submitBtn.disabled = false;
      submitBtn.innerHTML = '<i class="fas fa-paper-plane ml-2"></i> <span>إرسال الطلب</span>';
    }

    if (data.status === 'success') {
      // عرض رسالة نجاح
      showSuccess(`تم إنشاء الطلب بنجاح! رقم الطلب: ${data.orderNumber || data.order_number}`);

      // إعادة توجيه المستخدم إلى صفحة إدارة الطلبات بعد 2 ثانية
      setTimeout(() => {
        window.location.href = 'orders.php';
      }, 2000);
    } else {
      // عرض رسائل الخطأ من الخادم
      if (data.errors) {
        displayFormErrors(data.errors);
      } else {
        throw new Error(data.message || 'حدث خطأ أثناء معالجة البيانات');
      }
    }
  })
  .catch(error => {
    console.error('خطأ في إرسال البيانات:', error);

    if (submitBtn) {
      submitBtn.disabled = false;
      submitBtn.innerHTML = '<i class="fas fa-paper-plane ml-2"></i> <span>إرسال الطلب</span>';
    }

    // عرض رسالة الخطأ للمستخدم
    showError(error.message || 'حدث خطأ أثناء إرسال البيانات. يرجى المحاولة مرة أخرى.');
  });
}

/**
 * التحقق من صحة النموذج
 * @param {HTMLFormElement} form - عنصر النموذج
 * @returns {boolean} - صحة النموذج
 */
function validateForm(form) {
  // إعادة تعيين تنسيق الحقول
  const formInputs = form.querySelectorAll('.form-input');
  formInputs.forEach(input => {
    input.classList.remove('border-red-500');
  });

  // التحقق من الحقول المطلوبة
  let isValid = true;
  const requiredFields = form.querySelectorAll('[required]');

  requiredFields.forEach(field => {
    if (!field.value.trim()) {
      field.classList.add('border-red-500');

      // إضافة رسالة خطأ تحت الحقل
      const fieldContainer = field.closest('div');
      const fieldError = document.createElement('p');
      fieldError.className = 'text-red-500 text-xs mt-1 error-message';
      fieldError.textContent = 'هذا الحقل مطلوب';
      fieldContainer.appendChild(fieldError);

      isValid = false;
    }
  });

  if (!isValid) {
    // عرض رسالة خطأ عامة
    showError('يرجى ملء جميع الحقول المطلوبة');
  }

  return isValid;
}

/**
 * عرض رسالة نجاح
 * @param {string} message - رسالة النجاح
 */
function showSuccess(message) {
  const form = document.getElementById('shipmentForm');

  const successDiv = document.createElement('div');
  successDiv.className = 'bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded relative mb-4 success-message';
  successDiv.innerHTML = `
      <strong class="font-bold">تم بنجاح!</strong>
      <span class="block sm:inline">${message}</span>
  `;

  form.prepend(successDiv);
  successDiv.scrollIntoView({ behavior: 'smooth', block: 'start' });
}

/**
 * عرض رسالة خطأ
 * @param {string} message - رسالة الخطأ
 */
function showError(message) {
  const form = document.getElementById('shipmentForm');

  const errorDiv = document.createElement('div');
  errorDiv.className = 'bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded relative mb-4 error-message';
  errorDiv.innerHTML = `
      <strong class="font-bold">خطأ!</strong>
      <span class="block sm:inline">${message}</span>
  `;

  form.prepend(errorDiv);
  errorDiv.scrollIntoView({ behavior: 'smooth', block: 'start' });
}

/**
 * عرض أخطاء النموذج
 * @param {Object} errors - كائن يحتوي على أخطاء الحقول
 */
function displayFormErrors(errors) {
  const form = document.getElementById('shipmentForm');

  // إنشاء رسالة خطأ عامة
  const errorSummary = document.createElement('div');
  errorSummary.className = 'bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded relative mb-4 error-message';
  errorSummary.innerHTML = '<strong class="font-bold">يرجى تصحيح الأخطاء التالية:</strong><ul class="mt-2 list-disc list-inside"></ul>';

  const errorList = errorSummary.querySelector('ul');

  // إضافة كل خطأ إلى القائمة وتمييز الحقل المقابل
  Object.keys(errors).forEach(fieldName => {
    const errorMessage = errors[fieldName];

    // إضافة الخطأ إلى قائمة الأخطاء
    const listItem = document.createElement('li');
    listItem.textContent = errorMessage;
    errorList.appendChild(listItem);

    // تمييز الحقل الذي يحتوي على خطأ
    // تحويل اسم الحقل من snake_case إلى camelCase
    const camelCaseFieldName = fieldName.replace(/_([a-z])/g, (match, letter) => letter.toUpperCase());

    const field = document.getElementById(camelCaseFieldName);
    if (field) {
      field.classList.add('border-red-500');

      // إضافة رسالة خطأ تحت الحقل
      const fieldContainer = field.closest('div');
      const fieldError = document.createElement('p');
      fieldError.className = 'text-red-500 text-xs mt-1 error-message';
      fieldError.textContent = errorMessage;
      fieldContainer.appendChild(fieldError);

      // إضافة مستمع حدث لإزالة تمييز الخطأ عند تغيير القيمة
      field.addEventListener('input', function() {
        this.classList.remove('border-red-500');
        const errorMsg = fieldContainer.querySelector('.error-message');
        if (errorMsg) {
          errorMsg.remove();
        }
      }, { once: true });
    }
  });

  // إضافة ملخص الأخطاء إلى أعلى النموذج
  form.prepend(errorSummary);

  // التمرير إلى أعلى النموذج لعرض الأخطاء
  errorSummary.scrollIntoView({ behavior: 'smooth', block: 'start' });
}

/**
 * إعادة تعيين النموذج
 */
function resetForm() {
  const form = document.getElementById('shipmentForm');

  if (form) {
    // إزالة رسائل الخطأ والنجاح
    const messages = document.querySelectorAll('.error-message, .success-message');
    messages.forEach(el => el.remove());

    // إعادة تعيين تنسيق الحقول
    const formInputs = form.querySelectorAll('.form-input');
    formInputs.forEach(input => {
      input.classList.remove('border-red-500');
    });

    // إعادة تعيين النموذج
    form.reset();
    initForm();
  } else {
    console.error('لم يتم العثور على النموذج بمعرف shipmentForm');
  }
}
