/**
 * ملف JavaScript الرئيسي للنظام
 * يحتوي على جميع وظائف النظام المشتركة
 */

// التأكد من تحميل المستند بالكامل
document.addEventListener('DOMContentLoaded', function() {
    // تفعيل التلميحات
    initTooltips();
    
    // تفعيل التنبيهات التلقائية الإغلاق
    initAutoCloseAlerts();
    
    // تفعيل تأكيد الحذف
    initDeleteConfirmation();
    
    // تفعيل تحقق النموذج
    initFormValidation();
    
    // تفعيل تنسيق التاريخ
    initDateFormatting();
    
    // تفعيل تنسيق الأرقام
    initNumberFormatting();
});

/**
 * تفعيل التلميحات
 */
function initTooltips() {
    // تحديد جميع العناصر التي تحتوي على خاصية data-bs-toggle="tooltip"
    var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
    
    // إنشاء كائنات التلميحات
    var tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
        return new bootstrap.Tooltip(tooltipTriggerEl);
    });
}

/**
 * تفعيل التنبيهات التلقائية الإغلاق
 */
function initAutoCloseAlerts() {
    // تحديد جميع التنبيهات
    var alerts = document.querySelectorAll('.alert:not(.alert-permanent)');
    
    // إغلاق التنبيهات تلقائيًا بعد 5 ثوانٍ
    alerts.forEach(function(alert) {
        setTimeout(function() {
            var bsAlert = new bootstrap.Alert(alert);
            bsAlert.close();
        }, 5000);
    });
}

/**
 * تفعيل تأكيد الحذف
 */
function initDeleteConfirmation() {
    // تحديد جميع أزرار الحذف
    var deleteButtons = document.querySelectorAll('.btn-delete, [data-action="delete"]');
    
    // إضافة مستمع حدث النقر لكل زر
    deleteButtons.forEach(function(button) {
        button.addEventListener('click', function(event) {
            // منع السلوك الافتراضي
            event.preventDefault();
            
            // الحصول على رسالة التأكيد المخصصة أو استخدام الرسالة الافتراضية
            var confirmMessage = button.getAttribute('data-confirm-message') || 'هل أنت متأكد من أنك تريد حذف هذا العنصر؟';
            
            // عرض مربع حوار التأكيد
            if (confirm(confirmMessage)) {
                // إذا تم التأكيد، اتبع الرابط أو أرسل النموذج
                if (button.tagName === 'A') {
                    window.location.href = button.getAttribute('href');
                } else if (button.form) {
                    button.form.submit();
                }
            }
        });
    });
}

/**
 * تفعيل تحقق النموذج
 */
function initFormValidation() {
    // تحديد جميع النماذج التي تحتوي على خاصية data-validate="true"
    var forms = document.querySelectorAll('form[data-validate="true"]');
    
    // إضافة مستمع حدث الإرسال لكل نموذج
    forms.forEach(function(form) {
        form.addEventListener('submit', function(event) {
            // التحقق من صحة النموذج
            if (!form.checkValidity()) {
                // منع إرسال النموذج إذا كان غير صالح
                event.preventDefault();
                event.stopPropagation();
            }
            
            // إضافة فئة was-validated لإظهار رسائل التحقق
            form.classList.add('was-validated');
        });
    });
}

/**
 * تفعيل تنسيق التاريخ
 */
function initDateFormatting() {
    // تحديد جميع حقول التاريخ
    var dateInputs = document.querySelectorAll('input[type="date"], input[data-type="date"]');
    
    // إضافة مستمع حدث التغيير لكل حقل
    dateInputs.forEach(function(input) {
        input.addEventListener('change', function() {
            // تنسيق التاريخ إذا كان هناك قيمة
            if (input.value) {
                // تحويل التاريخ إلى كائن Date
                var date = new Date(input.value);
                
                // تنسيق التاريخ بالتنسيق العربي (DD/MM/YYYY)
                var formattedDate = date.getDate().toString().padStart(2, '0') + '/' +
                                   (date.getMonth() + 1).toString().padStart(2, '0') + '/' +
                                   date.getFullYear();
                
                // عرض التاريخ المنسق في عنصر مرتبط إذا كان موجودًا
                var displayElement = document.getElementById(input.getAttribute('data-display-id'));
                if (displayElement) {
                    displayElement.textContent = formattedDate;
                }
            }
        });
    });
}

/**
 * تفعيل تنسيق الأرقام
 */
function initNumberFormatting() {
    // تحديد جميع حقول الأرقام
    var numberInputs = document.querySelectorAll('input[type="number"], input[data-type="number"]');
    
    // إضافة مستمع حدث التغيير لكل حقل
    numberInputs.forEach(function(input) {
        input.addEventListener('change', function() {
            // تنسيق الرقم إذا كان هناك قيمة
            if (input.value) {
                // تحويل القيمة إلى رقم
                var number = parseFloat(input.value);
                
                // تنسيق الرقم بالتنسيق العربي
                var formattedNumber = number.toLocaleString('ar-SA');
                
                // عرض الرقم المنسق في عنصر مرتبط إذا كان موجودًا
                var displayElement = document.getElementById(input.getAttribute('data-display-id'));
                if (displayElement) {
                    displayElement.textContent = formattedNumber;
                }
            }
        });
    });
}

/**
 * دالة لإرسال طلب AJAX
 * @param {string} url - عنوان URL للطلب
 * @param {string} method - طريقة الطلب (GET, POST, PUT, DELETE)
 * @param {object} data - البيانات المراد إرسالها
 * @param {function} successCallback - دالة رد الاتصال عند النجاح
 * @param {function} errorCallback - دالة رد الاتصال عند الخطأ
 */
function sendAjaxRequest(url, method, data, successCallback, errorCallback) {
    // إنشاء كائن XMLHttpRequest
    var xhr = new XMLHttpRequest();
    
    // تهيئة الطلب
    xhr.open(method, url, true);
    
    // تعيين رؤوس الطلب
    xhr.setRequestHeader('Content-Type', 'application/json');
    xhr.setRequestHeader('X-Requested-With', 'XMLHttpRequest');
    
    // إضافة رمز CSRF إذا كان موجودًا
    var csrfToken = document.querySelector('meta[name="csrf-token"]');
    if (csrfToken) {
        xhr.setRequestHeader('X-CSRF-TOKEN', csrfToken.getAttribute('content'));
    }
    
    // تعيين دالة رد الاتصال عند تغيير حالة الطلب
    xhr.onreadystatechange = function() {
        if (xhr.readyState === 4) {
            if (xhr.status >= 200 && xhr.status < 300) {
                // نجاح الطلب
                var response;
                try {
                    response = JSON.parse(xhr.responseText);
                } catch (e) {
                    response = xhr.responseText;
                }
                
                if (typeof successCallback === 'function') {
                    successCallback(response);
                }
            } else {
                // فشل الطلب
                var error;
                try {
                    error = JSON.parse(xhr.responseText);
                } catch (e) {
                    error = xhr.responseText;
                }
                
                if (typeof errorCallback === 'function') {
                    errorCallback(error);
                }
            }
        }
    };
    
    // إرسال الطلب
    xhr.send(JSON.stringify(data));
}

/**
 * دالة لعرض رسالة نجاح
 * @param {string} message - نص الرسالة
 * @param {string} container - معرف حاوية الرسالة
 */
function showSuccessMessage(message, container) {
    var alertContainer = document.getElementById(container || 'alert-container');
    if (alertContainer) {
        var alert = document.createElement('div');
        alert.className = 'alert alert-success alert-dismissible fade show';
        alert.role = 'alert';
        alert.innerHTML = message +
            '<button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>';
        
        alertContainer.appendChild(alert);
        
        // إغلاق التنبيه تلقائيًا بعد 5 ثوانٍ
        setTimeout(function() {
            var bsAlert = new bootstrap.Alert(alert);
            bsAlert.close();
        }, 5000);
    }
}

/**
 * دالة لعرض رسالة خطأ
 * @param {string} message - نص الرسالة
 * @param {string} container - معرف حاوية الرسالة
 */
function showErrorMessage(message, container) {
    var alertContainer = document.getElementById(container || 'alert-container');
    if (alertContainer) {
        var alert = document.createElement('div');
        alert.className = 'alert alert-danger alert-dismissible fade show';
        alert.role = 'alert';
        alert.innerHTML = message +
            '<button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>';
        
        alertContainer.appendChild(alert);
        
        // إغلاق التنبيه تلقائيًا بعد 5 ثوانٍ
        setTimeout(function() {
            var bsAlert = new bootstrap.Alert(alert);
            bsAlert.close();
        }, 5000);
    }
}
