/**
 * نظام التنقل SPA لنظام طلبات الشحن
 *
 * هذا الملف يدير التنقل بين الصفحات في تطبيق الصفحة الواحدة (SPA)
 * ويقوم بتحميل محتوى الصفحات بشكل ديناميكي دون إعادة تحميل الصفحة بالكامل.
 */

// تعريف الصفحات المتاحة ومساراتها
const pages = {
  'dashboard': 'pages/dashboard.html',
  'orders': 'pages/orders.html',
  'create-order': 'pages/create-order.html',
  'archive': 'pages/archive.html',
  'settings': 'pages/settings.html',
  'users': 'pages/users.html',
};

// الصفحة الافتراضية
const defaultPage = 'dashboard';

// عناصر DOM
const pageContent = document.getElementById('pageContent');
const navLinks = document.querySelectorAll('.nav-item');
const toggleSidebar = document.getElementById('toggleSidebar');
const sidebar = document.querySelector('.sidebar');
const mobileMenuButton = document.getElementById('mobileMenuButton');
const overlay = document.getElementById('overlay');
const mainContent = document.querySelector('.main-content');

// تخزين الصفحات المحملة مسبقاً
const pageCache = {};

// تخزين الوظائف المرتبطة بكل صفحة
const pageScripts = {
  'dashboard': initDashboard,
  'orders': initOrders,
  'create-order': initCreateOrder,
  'archive': initArchive,
  'settings': initSettings,
  'users': initUsers,
};

// دالة لتحميل محتوى الصفحة
async function loadPage(pageName) {
  // إظهار مؤشر التحميل
  pageContent.innerHTML = '<div class="flex justify-center items-center h-64"><div class="loading-spinner"></div></div>';
  pageContent.classList.remove('active');

  try {
    // التحقق من وجود الصفحة في التخزين المؤقت
    let content;
    if (pageCache[pageName]) {
      content = pageCache[pageName];
    } else {
      // تحميل محتوى الصفحة من الخادم
      const response = await fetch(pages[pageName]);
      if (!response.ok) {
        throw new Error(`فشل تحميل الصفحة: ${response.status} ${response.statusText}`);
      }
      content = await response.text();

      // تخزين المحتوى في التخزين المؤقت
      pageCache[pageName] = content;
    }

    // إضافة محتوى الصفحة إلى العنصر الرئيسي
    pageContent.innerHTML = content;

    // تنفيذ سكريبت الصفحة إذا كان موجوداً
    if (pageScripts[pageName]) {
      pageScripts[pageName]();
    }

    // تحديث العنوان
    document.title = `نظام طلبات الشحن | ${getPageTitle(pageName)}`;

    // تحديث الرابط النشط في القائمة الجانبية
    updateActiveLink(pageName);

    // إظهار المحتوى بتأثير انتقالي
    setTimeout(() => {
      pageContent.classList.add('active');
    }, 50);

  } catch (error) {
    console.error('خطأ في تحميل الصفحة:', error);
    pageContent.innerHTML = `
      <div class="bg-red-50 border border-red-200 text-red-800 rounded-lg p-4 my-4">
        <h3 class="text-lg font-bold mb-2">خطأ في تحميل الصفحة</h3>
        <p>${error.message}</p>
        <button class="mt-2 px-4 py-2 bg-red-600 text-white rounded hover:bg-red-700" onclick="loadPage('${defaultPage}')">
          العودة إلى الصفحة الرئيسية
        </button>
      </div>
    `;
  }
}

// دالة لتحديث الرابط النشط في القائمة الجانبية
function updateActiveLink(pageName) {
  navLinks.forEach(link => {
    const linkPage = link.getAttribute('data-page');
    if (linkPage === pageName) {
      link.classList.add('active');
    } else {
      link.classList.remove('active');
    }
  });
}

// دالة للحصول على عنوان الصفحة
function getPageTitle(pageName) {
  const titles = {
    'dashboard': 'لوحة التحكم',
    'orders': 'إدارة الطلبات',
    'create-order': 'إنشاء طلب',
    'archive': 'الأرشيف',
    'settings': 'إعدادات النظام',
    'users': 'المستخدمون',
  };

  return titles[pageName] || 'الصفحة غير موجودة';
}

// دالة للتنقل إلى صفحة معينة
function navigateTo(pageName) {
  if (pages[pageName]) {
    loadPage(pageName);
    window.location.hash = pageName;
  } else {
    console.error(`الصفحة "${pageName}" غير موجودة`);
    loadPage(defaultPage);
    window.location.hash = defaultPage;
  }
}

// دالة لمعالجة تغيير الهاش في عنوان URL
function handleHashChange() {
  const hash = window.location.hash.substring(1); // إزالة علامة #
  const pageName = hash || defaultPage;

  if (pages[pageName]) {
    loadPage(pageName);
  } else {
    console.error(`الصفحة "${pageName}" غير موجودة`);
    loadPage(defaultPage);
    window.location.hash = defaultPage;
  }
}

// إضافة مستمعي الأحداث
document.addEventListener('DOMContentLoaded', function() {
  // تحميل الصفحة الأولية
  handleHashChange();

  // إضافة مستمع لتغيير الهاش
  window.addEventListener('hashchange', handleHashChange);

  // إضافة مستمعي الأحداث لروابط التنقل
  navLinks.forEach(link => {
    link.addEventListener('click', function(e) {
      e.preventDefault();
      const pageName = this.getAttribute('data-page');
      navigateTo(pageName);

      // إغلاق القائمة الجانبية في الشاشات الصغيرة
      if (window.innerWidth < 768) {
        sidebar.classList.remove('sidebar-open');
        overlay.classList.remove('overlay-open');
      }
    });
  });

  // تبديل القائمة الجانبية
  toggleSidebar.addEventListener('click', () => {
    sidebar.classList.toggle('sidebar-collapsed');
    sidebar.classList.toggle('sidebar-expanded');
    mainContent.classList.toggle('mr-[80px]');
    mainContent.classList.toggle('mr-[280px]');

    // تغيير أيقونة الزر
    const icon = toggleSidebar.querySelector('i');
    if (icon.classList.contains('fa-chevron-right')) {
      icon.classList.remove('fa-chevron-right');
      icon.classList.add('fa-chevron-left');
    } else {
      icon.classList.remove('fa-chevron-left');
      icon.classList.add('fa-chevron-right');
    }
  });

  // فتح/إغلاق القائمة على الجوال
  mobileMenuButton.addEventListener('click', () => {
    sidebar.classList.add('sidebar-open');
    overlay.classList.add('overlay-open');
  });

  // إغلاق القائمة عند النقر على الخلفية
  overlay.addEventListener('click', () => {
    sidebar.classList.remove('sidebar-open');
    overlay.classList.remove('overlay-open');
  });
});

// دوال تهيئة الصفحات
function initDashboard() {
  console.log('تم تهيئة صفحة لوحة التحكم');
  // أي كود خاص بصفحة لوحة التحكم
}

function initOrders() {
  console.log('تم تهيئة صفحة إدارة الطلبات');

  // تحميل وعرض الطلبات
  loadOrders();

  // دالة لتحميل الطلبات من localStorage وعرضها
  function loadOrders() {
    const ordersTableBody = document.getElementById('ordersTableBody');
    if (!ordersTableBody) return;

    console.log("جاري تحميل الطلبات...");

    // الحصول على الطلبات من localStorage
    const ordersJson = localStorage.getItem('shippingOrders');
    console.log("بيانات الطلبات من التخزين المحلي:", ordersJson);

    const orders = JSON.parse(ordersJson || '[]');
    console.log("عدد الطلبات المستردة:", orders.length);

    // إضافة بيانات افتراضية إذا لم تكن هناك طلبات
    if (orders.length === 0) {
      console.log("لا توجد طلبات، سيتم إضافة بيانات افتراضية");
      // إضافة بيانات افتراضية
      const defaultOrders = [
        {
          id: 1,
          orderNumber: 'ARP0501',
          clientName: 'محمد يوسف',
          branch: 'فرع جدة',
          departureArea: 'الرياض',
          deliveryArea: 'الدمام',
          goodsName: 'أجهزة كهربائية',
          goodsType: 'كهربائيات',
          weight: '25',
          quantity: '10',
          unit: 'صندوق',
          recipientName: 'عبدالله علي',
          phoneNumber: '0551234567',
          deliveryDuration: '3-5 أيام',
          paymentType: 'كاش',
          paymentStatus: 'مدفوع',
          status: 'قيد المعالجة'
        }
      ];

      // حفظ البيانات الافتراضية
      localStorage.setItem('shippingOrders', JSON.stringify(defaultOrders));
      console.log("تم حفظ البيانات الافتراضية");

      // استخدام البيانات الافتراضية
      renderOrders(defaultOrders);
    } else {
      console.log("تم العثور على طلبات، جاري عرضها...");
      // عرض الطلبات المخزنة
      renderOrders(orders);
    }
  }
}

function initCreateOrder() {
  console.log('تم تهيئة صفحة إنشاء طلب');
  // أي كود خاص بصفحة إنشاء طلب
}

function initArchive() {
  console.log('تم تهيئة صفحة الأرشيف');
  // أي كود خاص بصفحة الأرشيف
}

function initSettings() {
  console.log('تم تهيئة صفحة الإعدادات');
  // أي كود خاص بصفحة الإعدادات
}

function initUsers() {
  console.log('تم تهيئة صفحة المستخدمين');
  // أي كود خاص بصفحة المستخدمين
}
