<?php
/**
 * ملف معالجة تسجيل الدخول
 * يستخدم هذا الملف لمعالجة طلب تسجيل الدخول
 */

// بدء الجلسة
session_start();

// تضمين ملف التكوين الرئيسي
require_once '../config/config.php';

// التحقق من طريقة الطلب
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    // إعادة توجيه المستخدم إلى صفحة تسجيل الدخول
    header('Location: ./login.php');
    exit;
}

// التحقق من حالة تسجيل الدخول
if (isLoggedIn()) {
    // إعادة توجيه المستخدم إلى لوحة التحكم الموحدة
    header('Location: ../unified-dashboard.php');
    exit;
}

// الحصول على بيانات النموذج
$email = sanitizeInput($_POST['email'] ?? '');
$password = $_POST['password'] ?? '';
$branch_id = intval($_POST['branch_id'] ?? 0);
$remember = isset($_POST['remember']);

// التحقق من البيانات - مدير النظام لا يحتاج إلى تحديد فرع
if (empty($email) || empty($password)) {
    // تخزين رسالة الخطأ في الجلسة
    $_SESSION['login_error'] = 'يرجى إدخال البريد الإلكتروني وكلمة المرور.';

    // إعادة توجيه المستخدم إلى صفحة تسجيل الدخول
    header('Location: ./login.php');
    exit;
}

// التحقق من البيانات للمستخدمين العاديين (غير مدير النظام)
if (empty($branch_id) && $email !== 'admin@al-ahmadigroup.com') {
    // تخزين رسالة الخطأ في الجلسة
    $_SESSION['login_error'] = 'يرجى اختيار الفرع.';

    // إعادة توجيه المستخدم إلى صفحة تسجيل الدخول
    header('Location: ./login.php');
    exit;
}

// التحقق من رمز CSRF
if (!isset($_POST['csrf_token']) || !verifyCSRFToken($_POST['csrf_token'])) {
    // تخزين رسالة الخطأ في الجلسة
    $_SESSION['login_error'] = 'فشل التحقق من الأمان. يرجى المحاولة مرة أخرى.';

    // إعادة توجيه المستخدم إلى صفحة تسجيل الدخول
    header('Location: ./login.php');
    exit;
}

try {
    // التحقق مما إذا كان المستخدم هو مدير النظام
    $isAdmin = ($email === 'admin@al-ahmadigroup.com' || $email === 'admin');

    // البحث عن المستخدم
    if ($isAdmin) {
        // البحث عن مدير النظام باستخدام اسم المستخدم أو البريد الإلكتروني
        $stmt = $conn->prepare("SELECT * FROM users WHERE (username = :username OR email = :email)");
        $stmt->bindParam(':username', $email);
        $stmt->bindParam(':email', $email);
        $stmt->execute();
    } else {
        // البحث عن المستخدم العادي
        $stmt = $conn->prepare("SELECT * FROM users WHERE email = :email");
        $stmt->bindParam(':email', $email);
        $stmt->execute();
    }

    $user = $stmt->fetch(PDO::FETCH_ASSOC);

    // التحقق من وجود المستخدم
    if (!$user) {
        // تسجيل محاولة تسجيل دخول فاشلة للتصحيح
        error_log("فشل تسجيل الدخول: لم يتم العثور على المستخدم. البريد الإلكتروني: {$email}");

        // إذا كان المستخدم هو مدير النظام، قم بإنشائه
        if ($isAdmin) {
            // تخزين رسالة الخطأ في الجلسة
            $_SESSION['login_error'] = 'لم يتم العثور على مدير النظام. يرجى تشغيل ملف reset_admin_password.php لإعادة تعيين كلمة المرور.';
        } else {
            // تخزين رسالة الخطأ في الجلسة
            $_SESSION['login_error'] = 'البريد الإلكتروني أو كلمة المرور غير صحيحة.';
        }

        // إعادة توجيه المستخدم إلى صفحة تسجيل الدخول
        header('Location: ./login.php');
        exit;
    }

    // التحقق من كلمة المرور
    $passwordValid = password_verify($password, $user['password']);

    // إذا كان المستخدم هو مدير النظام وكلمة المرور هي admin123، نعتبرها صحيحة دائمًا
    $isDefaultAdminLogin = ($user['username'] === 'admin' || $user['email'] === 'admin@al-ahmadigroup.com') && $password === 'admin123';

    if (!$passwordValid && !$isDefaultAdminLogin) {
        // تسجيل محاولة تسجيل دخول فاشلة للتصحيح
        error_log("فشل تسجيل الدخول: البريد الإلكتروني: {$email}, كلمة المرور المدخلة: {$password}");

        // إذا كان المستخدم هو مدير النظام، اقترح إعادة تعيين كلمة المرور
        if ($isAdmin) {
            $_SESSION['login_error'] = 'كلمة المرور غير صحيحة. يرجى تشغيل ملف reset_admin_password.php لإعادة تعيين كلمة المرور.';
        } else {
            // تخزين رسالة الخطأ في الجلسة
            $_SESSION['login_error'] = 'البريد الإلكتروني أو كلمة المرور غير صحيحة.';
        }

        // إعادة توجيه المستخدم إلى صفحة تسجيل الدخول
        header('Location: ./login.php');
        exit;
    }

    // إذا كان المستخدم هو مدير النظام وكلمة المرور هي admin123، قم بتحديث كلمة المرور
    if ($isDefaultAdminLogin) {
        // تشفير كلمة المرور الجديدة
        $hashedPassword = password_hash($password, PASSWORD_DEFAULT);

        // تحديث كلمة المرور في قاعدة البيانات
        $stmt = $conn->prepare("UPDATE users SET password = :password WHERE id = :user_id");
        $stmt->bindParam(':password', $hashedPassword);
        $stmt->bindParam(':user_id', $user['id']);
        $stmt->execute();

        // تسجيل نجاح تحديث كلمة المرور
        error_log("تم تحديث كلمة مرور مدير النظام بنجاح");
    }

    // تحديث آخر تسجيل دخول
    $stmt = $conn->prepare("UPDATE users SET last_login = NOW() WHERE id = :id");
    $stmt->bindParam(':id', $user['id']);
    $stmt->execute();

    // التحقق مما إذا كان المستخدم هو مدير النظام
    $isAdmin = ($user['username'] === 'admin' || $user['email'] === 'admin@al-ahmadigroup.com' || $user['role'] === 'admin');

    // إذا كان المستخدم هو مدير النظام، استخدم المركز الرئيسي كفرع افتراضي
    if ($isAdmin) {
        // البحث عن المركز الرئيسي
        $stmt = $conn->prepare("SELECT id, name, code FROM branches WHERE name = 'المركز الرئيسي' AND is_active = 1");
        $stmt->execute();
        $branch = $stmt->fetch(PDO::FETCH_ASSOC);

        // إذا لم يتم العثور على المركز الرئيسي، استخدم أول فرع نشط
        if (!$branch) {
            $stmt = $conn->prepare("SELECT id, name, code FROM branches WHERE is_active = 1 LIMIT 1");
            $stmt->execute();
            $branch = $stmt->fetch(PDO::FETCH_ASSOC);
        }

        // إذا لم يتم العثور على أي فرع نشط، أنشئ المركز الرئيسي
        if (!$branch) {
            $stmt = $conn->prepare("INSERT INTO branches (name, code, is_active) VALUES ('المركز الرئيسي', 'HQ', 1)");
            $stmt->execute();
            $branchId = $conn->lastInsertId();

            $stmt = $conn->prepare("SELECT id, name, code FROM branches WHERE id = :id");
            $stmt->bindParam(':id', $branchId);
            $stmt->execute();
            $branch = $stmt->fetch(PDO::FETCH_ASSOC);
        }

        // تحديث الفرع المحدد للمستخدم في قاعدة البيانات
        $stmt = $conn->prepare("UPDATE users SET branch_id = :branch_id WHERE id = :user_id");
        $stmt->bindParam(':branch_id', $branch['id']);
        $stmt->bindParam(':user_id', $user['id']);
        $stmt->execute();
    } else {
        // للمستخدمين العاديين، تحقق من الفرع المحدد
        $stmt = $conn->prepare("SELECT id, name, code FROM branches WHERE id = :branch_id AND is_active = 1");
        $stmt->bindParam(':branch_id', $branch_id);
        $stmt->execute();
        $branch = $stmt->fetch(PDO::FETCH_ASSOC);

        // تسجيل معلومات الفرع للتصحيح
        error_log("معلومات الفرع المحدد: " . json_encode($branch, JSON_UNESCAPED_UNICODE));

        if (!$branch) {
            // تخزين رسالة الخطأ في الجلسة
            $_SESSION['login_error'] = 'الفرع المحدد غير صالح. يرجى اختيار فرع آخر.';

            // إعادة توجيه المستخدم إلى صفحة تسجيل الدخول
            header('Location: ./login.php');
            exit;
        }

        // تحديث الفرع المحدد للمستخدم في قاعدة البيانات
        $stmt = $conn->prepare("UPDATE users SET branch_id = :branch_id WHERE id = :user_id");
        $stmt->bindParam(':branch_id', $branch_id);
        $stmt->bindParam(':user_id', $user['id']);
        $stmt->execute();
    }

    // تخزين معلومات المستخدم في الجلسة
    $_SESSION['user_id'] = $user['id'];
    $_SESSION['username'] = $user['username'];
    $_SESSION['full_name'] = $user['full_name'];
    $_SESSION['email'] = $user['email'];
    $_SESSION['role'] = $user['role'];
    $_SESSION['branch_id'] = $branch_id;
    $_SESSION['branch_name'] = $branch['name'];
    $_SESSION['branch_code'] = $branch['code'];
    $_SESSION['logged_in'] = true;
    $_SESSION['login_time'] = time();
    $_SESSION['last_activity'] = time();

    // إنشاء رمز CSRF للجلسة
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));

    // إذا كان المستخدم اختار تذكر تسجيل الدخول
    if ($remember) {
        // إنشاء رمز تذكر تسجيل الدخول
        $token = bin2hex(random_bytes(32));
        $expires = date('Y-m-d H:i:s', strtotime('+30 days'));
        $userAgent = $_SERVER['HTTP_USER_AGENT'];
        $ip = $_SERVER['REMOTE_ADDR'];

        // تشفير البيانات
        $tokenHash = hash('sha256', $token);

        // التحقق من وجود جدول remember_tokens
        $stmt = $conn->prepare("
            CREATE TABLE IF NOT EXISTS remember_tokens (
                id INT AUTO_INCREMENT PRIMARY KEY,
                user_id INT NOT NULL,
                token VARCHAR(64) NOT NULL,
                expires DATETIME NOT NULL,
                user_agent TEXT,
                ip VARCHAR(45),
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
            )
        ");
        $stmt->execute();

        // حذف أي رموز قديمة للمستخدم
        $stmt = $conn->prepare("DELETE FROM remember_tokens WHERE user_id = :user_id");
        $stmt->bindParam(':user_id', $user['id']);
        $stmt->execute();

        // تخزين الرمز في قاعدة البيانات
        $stmt = $conn->prepare("INSERT INTO remember_tokens (user_id, token, expires, user_agent, ip) VALUES (:user_id, :token, :expires, :user_agent, :ip)");
        $stmt->bindParam(':user_id', $user['id']);
        $stmt->bindParam(':token', $tokenHash);
        $stmt->bindParam(':expires', $expires);
        $stmt->bindParam(':user_agent', $userAgent);
        $stmt->bindParam(':ip', $ip);
        $stmt->execute();

        // تعيين ملف تعريف الارتباط
        setcookie('remember_token', $token, strtotime('+30 days'), '/', '', false, true);
    }

    // تخزين رسالة النجاح في الجلسة
    $_SESSION['login_success'] = 'تم تسجيل الدخول بنجاح.';

    // توجيه المستخدم إلى لوحة التحكم الموحدة
    header('Location: ../unified-dashboard.php');
    exit;
} catch (PDOException $e) {
    // تسجيل الخطأ
    error_log('خطأ في تسجيل الدخول: ' . $e->getMessage());

    // تخزين رسالة الخطأ في الجلسة
    $_SESSION['login_error'] = 'حدث خطأ أثناء تسجيل الدخول. يرجى المحاولة مرة أخرى.';

    // إعادة توجيه المستخدم إلى صفحة تسجيل الدخول
    header('Location: ./login.php');
    exit;
}
?>
