<?php
/**
 * ملف النسخ الاحتياطي لقاعدة البيانات
 * يستخدم هذا الملف لإنشاء نسخة احتياطية من قاعدة البيانات
 */

// تعيين وقت التنفيذ الأقصى
set_time_limit(300);

// تعيين ترميز الصفحة
header('Content-Type: text/html; charset=utf-8');

// تضمين ملف التكوين
require_once 'config/config.php';

// التحقق من تسجيل الدخول وصلاحيات المستخدم
if (!isLoggedIn() || $_SESSION['role'] !== 'admin') {
    header('Location: auth/login.php');
    exit;
}

// دالة لإنشاء نسخة احتياطية من قاعدة البيانات MySQL
function backupMySQLDatabase($host, $user, $pass, $dbname, $tables = '*') {
    try {
        // إنشاء اتصال PDO
        $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8mb4", $user, $pass);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        
        // الحصول على جميع الجداول
        if ($tables == '*') {
            $tables = [];
            $stmt = $pdo->query('SHOW TABLES');
            while ($row = $stmt->fetch(PDO::FETCH_NUM)) {
                $tables[] = $row[0];
            }
        } else {
            $tables = is_array($tables) ? $tables : explode(',', $tables);
        }
        
        // بدء ملف النسخ الاحتياطي
        $output = "-- نسخة احتياطية من قاعدة البيانات $dbname\n";
        $output .= "-- تاريخ الإنشاء: " . date('Y-m-d H:i:s') . "\n";
        $output .= "-- إصدار MySQL: " . $pdo->getAttribute(PDO::ATTR_SERVER_VERSION) . "\n";
        $output .= "-- إصدار PHP: " . PHP_VERSION . "\n\n";
        
        $output .= "SET NAMES utf8mb4;\n";
        $output .= "SET FOREIGN_KEY_CHECKS = 0;\n";
        $output .= "SET SQL_MODE = \"NO_AUTO_VALUE_ON_ZERO\";\n\n";
        
        // إنشاء نسخة احتياطية لكل جدول
        foreach ($tables as $table) {
            // الحصول على هيكل الجدول
            $stmt = $pdo->query("SHOW CREATE TABLE `$table`");
            $row = $stmt->fetch(PDO::FETCH_NUM);
            $output .= "-- هيكل جدول `$table`\n";
            $output .= "DROP TABLE IF EXISTS `$table`;\n";
            $output .= $row[1] . ";\n\n";
            
            // الحصول على بيانات الجدول
            $stmt = $pdo->query("SELECT * FROM `$table`");
            $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            if (count($rows) > 0) {
                $output .= "-- بيانات جدول `$table`\n";
                $output .= "INSERT INTO `$table` VALUES\n";
                
                $values = [];
                foreach ($rows as $row) {
                    $rowValues = [];
                    foreach ($row as $value) {
                        if ($value === null) {
                            $rowValues[] = 'NULL';
                        } else {
                            $rowValues[] = $pdo->quote($value);
                        }
                    }
                    $values[] = '(' . implode(', ', $rowValues) . ')';
                }
                
                $output .= implode(",\n", $values) . ";\n\n";
            }
        }
        
        $output .= "SET FOREIGN_KEY_CHECKS = 1;\n";
        
        // إنشاء مجلد النسخ الاحتياطي إذا لم يكن موجودًا
        $backupDir = __DIR__ . '/backups';
        if (!file_exists($backupDir)) {
            mkdir($backupDir, 0755, true);
        }
        
        // إنشاء اسم ملف النسخ الاحتياطي
        $backupFile = $backupDir . '/' . $dbname . '_' . date('Y-m-d_H-i-s') . '.sql';
        
        // حفظ ملف النسخ الاحتياطي
        file_put_contents($backupFile, $output);
        
        return [
            'success' => true,
            'message' => 'تم إنشاء النسخة الاحتياطية بنجاح',
            'file' => $backupFile
        ];
    } catch (PDOException $e) {
        return [
            'success' => false,
            'message' => 'فشل إنشاء النسخة الاحتياطية: ' . $e->getMessage()
        ];
    }
}

// معالجة طلب النسخ الاحتياطي
$result = null;
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['backup'])) {
    $result = backupMySQLDatabase(DB_HOST, DB_USER, DB_PASS, DB_NAME);
}

// الحصول على قائمة ملفات النسخ الاحتياطي
$backupFiles = [];
$backupDir = __DIR__ . '/backups';
if (file_exists($backupDir)) {
    $files = scandir($backupDir);
    foreach ($files as $file) {
        if ($file !== '.' && $file !== '..' && pathinfo($file, PATHINFO_EXTENSION) === 'sql') {
            $backupFiles[] = [
                'name' => $file,
                'path' => $backupDir . '/' . $file,
                'size' => filesize($backupDir . '/' . $file),
                'date' => date('Y-m-d H:i:s', filemtime($backupDir . '/' . $file))
            ];
        }
    }
    
    // ترتيب الملفات حسب التاريخ (الأحدث أولاً)
    usort($backupFiles, function($a, $b) {
        return strtotime($b['date']) - strtotime($a['date']);
    });
}

// تنزيل ملف النسخ الاحتياطي
if (isset($_GET['download']) && !empty($_GET['file'])) {
    $file = $_GET['file'];
    $filePath = $backupDir . '/' . $file;
    
    if (file_exists($filePath)) {
        header('Content-Description: File Transfer');
        header('Content-Type: application/octet-stream');
        header('Content-Disposition: attachment; filename="' . $file . '"');
        header('Expires: 0');
        header('Cache-Control: must-revalidate');
        header('Pragma: public');
        header('Content-Length: ' . filesize($filePath));
        readfile($filePath);
        exit;
    }
}

// حذف ملف النسخ الاحتياطي
if (isset($_GET['delete']) && !empty($_GET['file'])) {
    $file = $_GET['file'];
    $filePath = $backupDir . '/' . $file;
    
    if (file_exists($filePath)) {
        unlink($filePath);
        header('Location: backup_database.php');
        exit;
    }
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>النسخ الاحتياطي لقاعدة البيانات</title>
    <link rel="stylesheet" href="assets/css/bootstrap.rtl.min.css">
    <link rel="stylesheet" href="assets/css/fontawesome.min.css">
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <div class="container mt-5">
        <div class="card shadow">
            <div class="card-header bg-primary text-white">
                <h3 class="mb-0">النسخ الاحتياطي لقاعدة البيانات</h3>
            </div>
            <div class="card-body">
                <?php if ($result): ?>
                    <div class="alert alert-<?php echo $result['success'] ? 'success' : 'danger'; ?> alert-dismissible fade show" role="alert">
                        <?php echo $result['message']; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="إغلاق"></button>
                    </div>
                <?php endif; ?>
                
                <form method="post" class="mb-4">
                    <button type="submit" name="backup" class="btn btn-primary">
                        <i class="fas fa-database me-2"></i> إنشاء نسخة احتياطية جديدة
                    </button>
                    <a href="ASDRF43/unified-dashboard.php" class="btn btn-secondary">
                        <i class="fas fa-arrow-right me-2"></i> العودة إلى لوحة التحكم
                    </a>
                </form>
                
                <h4>النسخ الاحتياطية المتوفرة</h4>
                <?php if (empty($backupFiles)): ?>
                    <div class="alert alert-info">
                        لا توجد نسخ احتياطية متوفرة.
                    </div>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table table-striped table-hover">
                            <thead>
                                <tr>
                                    <th>اسم الملف</th>
                                    <th>الحجم</th>
                                    <th>تاريخ الإنشاء</th>
                                    <th>الإجراءات</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($backupFiles as $file): ?>
                                    <tr>
                                        <td><?php echo $file['name']; ?></td>
                                        <td><?php echo round($file['size'] / 1024, 2); ?> كيلوبايت</td>
                                        <td><?php echo $file['date']; ?></td>
                                        <td>
                                            <a href="backup_database.php?download=1&file=<?php echo urlencode($file['name']); ?>" class="btn btn-sm btn-success">
                                                <i class="fas fa-download"></i> تنزيل
                                            </a>
                                            <a href="backup_database.php?delete=1&file=<?php echo urlencode($file['name']); ?>" class="btn btn-sm btn-danger" onclick="return confirm('هل أنت متأكد من حذف هذا الملف؟')">
                                                <i class="fas fa-trash"></i> حذف
                                            </a>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <script src="assets/js/bootstrap.bundle.min.js"></script>
</body>
</html>
