<?php
/**
 * ملف التكوين الرئيسي
 * يستخدم هذا الملف لتضمين ملفات التكوين الأخرى وتعريف الدوال المساعدة
 */

// تضمين ملف الاتصال بقاعدة البيانات
require_once __DIR__ . '/../config/db_connection.php';

/**
 * دالة لتنظيف المدخلات
 * @param string $input - النص المدخل
 * @return string - النص بعد التنظيف
 */
function sanitizeInput($input) {
    // إزالة المسافات الزائدة من بداية ونهاية النص
    $input = trim($input);

    // إزالة الأكواد الضارة
    $input = htmlspecialchars($input, ENT_QUOTES, 'UTF-8');

    return $input;
}

// دالة لتحويل التاريخ من التنسيق العربي إلى تنسيق قاعدة البيانات
function convertDateToMysql($date) {
    if (empty($date)) return null;

    // تحويل التاريخ من التنسيق العربي (DD/MM/YYYY) إلى تنسيق MySQL (YYYY-MM-DD)
    $parts = explode('/', $date);
    if (count($parts) === 3) {
        return $parts[2] . '-' . $parts[1] . '-' . $parts[0];
    }

    return $date;
}

// دالة لتحويل التاريخ من تنسيق قاعدة البيانات إلى التنسيق العربي
function convertDateToArabic($date) {
    if (empty($date)) return '';

    // تحويل التاريخ من تنسيق MySQL (YYYY-MM-DD) إلى التنسيق العربي (DD/MM/YYYY)
    $parts = explode('-', $date);
    if (count($parts) === 3) {
        return $parts[2] . '/' . $parts[1] . '/' . $parts[0];
    }

    return $date;
}

// دالة للحصول على رقم طلب جديد (تستخدم الاتصال $pdo)
function getNewOrderNumber($pdo, $branchCode) {
    try {
        // البحث عن آخر رقم تسلسلي للفرع
        $stmt = $pdo->prepare("SELECT last_sequence_number FROM order_sequences WHERE branch_code = ?");
        $stmt->execute([$branchCode]);
        $result = $stmt->fetch();

        // الحصول على السنة الحالية (آخر رقمين)
        $year = date('y');

        if ($result) {
            $lastNumber = $result['last_sequence_number'];
            $newNumber = $lastNumber + 1;

            // تحديث الرقم التسلسلي في قاعدة البيانات
            $updateStmt = $pdo->prepare("UPDATE order_sequences SET last_sequence_number = ? WHERE branch_code = ?");
            $updateStmt->execute([$newNumber, $branchCode]);

            // تنسيق الرقم التسلسلي
            $sequenceNumber = str_pad($newNumber, 2, '0', STR_PAD_LEFT);

            // إنشاء رقم الطلب الكامل
            $newOrderNumber = $branchCode . $year . '-' . $sequenceNumber;

            return $newOrderNumber;
        } else {
            // إذا لم يتم العثور على رمز الفرع، قم بإنشاء سجل جديد
            $insertStmt = $pdo->prepare("INSERT INTO order_sequences (branch_code, last_sequence_number) VALUES (?, 1)");
            $insertStmt->execute([$branchCode]);

            return $branchCode . $year . '-01';
        }
    } catch (PDOException $e) {
        // في حالة حدوث خطأ
        return false;
    }
}
?>
