<?php
// بدء الجلسة
session_start();

// تضمين ملف التكوين
require_once 'config/config.php';

// التحقق من تسجيل الدخول
if (!isLoggedIn()) {
    header('Location: auth/login.php');
    exit;
}

/**
 * دالة لتحويل النص العربي إلى حروف لاتينية
 * @param string $text النص العربي
 * @return string النص بالحروف اللاتينية
 */
function transliterateArabic($text) {
    $arabic = [
        'ا', 'أ', 'إ', 'آ', 'ب', 'ت', 'ث', 'ج', 'ح', 'خ', 'د', 'ذ', 'ر', 'ز', 'س', 'ش', 'ص', 'ض', 'ط', 'ظ', 'ع', 'غ', 'ف', 'ق', 'ك', 'ل', 'م', 'ن', 'ه', 'و', 'ي', 'ة', 'ى', 'ء'
    ];
    $latin = [
        'a', 'a', 'e', 'a', 'b', 't', 'th', 'j', 'h', 'kh', 'd', 'th', 'r', 'z', 's', 'sh', 's', 'd', 't', 'z', 'a', 'gh', 'f', 'q', 'k', 'l', 'm', 'n', 'h', 'w', 'y', 'a', 'a', 'a'
    ];

    $text = str_replace($arabic, $latin, $text);
    // إزالة المسافات واستبدالها بشرطة سفلية
    $text = str_replace(' ', '_', $text);
    // إزالة أي أحرف غير أبجدية رقمية
    $text = preg_replace('/[^a-zA-Z0-9_]/', '', $text);

    return $text;
}

// الحصول على معلومات المستخدم الحالي
$stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
$stmt->execute([$_SESSION['user_id']]);
$user = $stmt->fetch();

// التحقق مما إذا كان المستخدم مديرًا
$isAdmin = ($user['role'] === 'admin');

// تعيين مسار الشعار
$company_logo = 'assets/images/backgrounds/background.png';

// معالجة إرسال نموذج إنشاء الطلب
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['submit_order'])) {
    // التحقق من رمز CSRF
    if (!isset($_POST['csrf_token']) || !verifyCSRFToken($_POST['csrf_token'])) {
        $error_message = 'خطأ في التحقق من الأمان. يرجى المحاولة مرة أخرى.';
    } else {
        // تنظيف وتحقق من البيانات المدخلة
        $orderNumber = sanitizeInput($_POST['orderNumber'] ?? '');
        $clientName = sanitizeInput($_POST['clientName'] ?? '');
        $serviceType = sanitizeInput($_POST['serviceType'] ?? '');
        $branchId = null;

        // الحصول على معرف الفرع من اسم الفرع
        $branchName = sanitizeInput($_POST['branch'] ?? '');
        if (!empty($branchName)) {
            $stmt = $pdo->prepare("SELECT id FROM branches WHERE name = ?");
            $stmt->execute([$branchName]);
            $branch = $stmt->fetch();
            if ($branch) {
                $branchId = $branch['id'];
            } else {
                // إنشاء فرع جديد إذا لم يكن موجودًا
                // إنشاء رمز للفرع من الاسم
                $baseCode = strtoupper(substr(transliterateArabic($branchName), 0, 3));
                $branchCode = $baseCode;

                // التحقق مما إذا كان الرمز موجودًا بالفعل
                $codeExists = true;
                $counter = 1;

                while ($codeExists && $counter < 100) {
                    // التحقق من وجود الرمز في قاعدة البيانات
                    $checkStmt = $pdo->prepare("SELECT id FROM branches WHERE code = ?");
                    $checkStmt->execute([$branchCode]);
                    $existingBranch = $checkStmt->fetch();

                    if (!$existingBranch) {
                        // الرمز غير موجود، يمكن استخدامه
                        $codeExists = false;
                    } else {
                        // الرمز موجود، إضافة رقم عشوائي
                        $branchCode = $baseCode . $counter;
                        $counter++;
                    }
                }

                // إدراج الفرع الجديد بالرمز الفريد
                $stmt = $pdo->prepare("INSERT INTO branches (name, code, is_active) VALUES (?, ?, 1)");
                $stmt->execute([$branchName, $branchCode]);
                $branchId = $pdo->lastInsertId();
            }
        }

        $departureArea = sanitizeInput($_POST['departureArea'] ?? '');
        $secondDestination = sanitizeInput($_POST['secondDestination'] ?? '');
        $deliveryArea = sanitizeInput($_POST['deliveryArea'] ?? '');
        $goodsName = sanitizeInput($_POST['goodsName'] ?? '');
        $goodsType = sanitizeInput($_POST['goodsType'] ?? '');
        $countryOfOrigin = sanitizeInput($_POST['countryOfOrigin'] ?? '');
        $weight = floatval($_POST['weight'] ?? 0);
        $deliveryDuration = intval($_POST['deliveryDuration'] ?? 0);
        $orderDate = sanitizeInput($_POST['orderDate'] ?? date('Y-m-d'));
        $expectedDeliveryDate = sanitizeInput($_POST['expectedDeliveryDate'] ?? '');
        $receiverName = sanitizeInput($_POST['receiverName'] ?? '');
        $phoneNumber = sanitizeInput($_POST['phoneNumber'] ?? '');
        $serviceFee = floatval($_POST['serviceFee'] ?? 0);
        $currency = sanitizeInput($_POST['currency'] ?? '');
        $clientAmount = floatval($_POST['clientAmount'] ?? 0);
        $agentAmount = floatval($_POST['agentAmount'] ?? 0);
        $paidAmount = floatval($_POST['paidAmount'] ?? 0);
        $remainingAmount = floatval($_POST['remainingAmount'] ?? 0);
        $otherExpenses = floatval($_POST['otherExpenses'] ?? 0);
        $profit = floatval($_POST['profit'] ?? 0);
        $responsibleEmployee = sanitizeInput($_POST['responsibleEmployee'] ?? '');
        $orderStatus = sanitizeInput($_POST['orderStatus'] ?? 'قيد المعالجة');

        // تحديد حالة الدفع
        $paymentStatus = 'غير مدفوع';
        if ($paidAmount >= $clientAmount) {
            $paymentStatus = 'مدفوع';
        } elseif ($paidAmount > 0) {
            $paymentStatus = 'جزئي';
        }

        // إنشاء رقم طلب إذا لم يتم إدخاله
        if (empty($orderNumber)) {
            // الحصول على رمز الفرع
            $branchCode = 'SHP';
            if (!empty($branchName)) {
                switch ($branchName) {
                    case 'عدن':
                        $branchCode = 'ADN';
                        break;
                    case 'صنعاء':
                        $branchCode = 'SNA';
                        break;
                    case 'المكلا':
                        $branchCode = 'MKL';
                        break;
                    case 'الصين':
                        $branchCode = 'CN';
                        break;
                    case 'شحن':
                        $branchCode = 'SH';
                        break;
                }
            }

            // تسجيل رمز الفرع للتصحيح
            error_log("رمز الفرع المستخدم: " . $branchCode);

            // الحصول على التاريخ الحالي
            $now = new DateTime();
            $year = date('y');
            $month = date('m');

            try {
                // تحسين الاتصال بـ API باستخدام المسار الكامل
                $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https://' : 'http://';
                $host = $_SERVER['HTTP_HOST'];
                $baseUrl = $protocol . $host;

                // استخدام المسار المطلق للـ API
                $apiUrl = $baseUrl . "/ASDRF43/get_next_sequence.php?branch_code={$branchCode}&reserve=true";

                // استخدام cURL للاتصال بالخادم بشكل أكثر موثوقية
                $ch = curl_init();
                curl_setopt($ch, CURLOPT_URL, $apiUrl);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_TIMEOUT, 15); // زيادة مهلة الانتظار إلى 15 ثوانٍ
                curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10); // زيادة مهلة الاتصال إلى 10 ثوانٍ
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false); // تعطيل التحقق من شهادة SSL في بيئة التطوير
                curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false); // تعطيل التحقق من اسم المضيف في بيئة التطوير

                $apiResponse = curl_exec($ch);
                $curlError = curl_error($ch);
                $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                curl_close($ch);

                // تسجيل معلومات الاتصال للتصحيح
                error_log("محاولة الاتصال بـ API: {$apiUrl}");
                error_log("رمز HTTP: {$httpCode}, خطأ cURL: {$curlError}");

                if ($apiResponse !== false && $httpCode == 200) {
                    $responseData = json_decode($apiResponse, true);

                    if (isset($responseData['status']) && $responseData['status'] === 'success') {
                        // استخدام الرقم التسلسلي من API
                        $sequenceNumber = $responseData['sequence_number'];
                        // تنسيق رقم الطلب (مثلاً: MKL25-01, SNA25-01, ...)
                        $orderNumber = $branchCode . '25-' . str_pad($sequenceNumber, 2, '0', STR_PAD_LEFT);
                        error_log("تم الحصول على رقم الطلب من API: " . $orderNumber);
                    } else {
                        throw new Exception("فشل في الحصول على رقم الطلب من API: " . ($responseData['message'] ?? 'خطأ غير معروف'));
                    }
                } else {
                    throw new Exception("فشل في الاتصال بـ API: " . ($curlError ?: 'خطأ غير معروف'));
                }
            } catch (Exception $e) {
                error_log("خطأ في الحصول على رقم الطلب من API: " . $e->getMessage());

                // استخدام الطريقة التقليدية كاحتياطي مع تحسينات
                try {
                    // بدء معاملة قاعدة البيانات لضمان عدم تداخل العمليات
                    $pdo->beginTransaction();

                    // الحصول على آخر رقم تسلسلي للفرع مع قفل الصف
                    $stmt = $pdo->prepare("SELECT last_sequence_number FROM order_sequences WHERE branch_code = ? FOR UPDATE");
                    $stmt->execute([$branchCode]);
                    $sequence = $stmt->fetch();

                    $sequenceNumber = 1;
                    if ($sequence) {
                        $sequenceNumber = $sequence['last_sequence_number'] + 1;
                        $stmt = $pdo->prepare("UPDATE order_sequences SET last_sequence_number = ? WHERE branch_code = ?");
                        $stmt->execute([$sequenceNumber, $branchCode]);
                    } else {
                        // إنشاء سجل جديد إذا لم يكن موجوداً
                        $stmt = $pdo->prepare("INSERT INTO order_sequences (branch_code, last_sequence_number) VALUES (?, ?)");
                        $stmt->execute([$branchCode, $sequenceNumber]);
                    }

                    // تأكيد المعاملة
                    $pdo->commit();

                    // تنسيق رقم الطلب (مثلاً: MKL25-01, SNA25-01, ...)
                    $orderNumber = $branchCode . '25-' . str_pad($sequenceNumber, 2, '0', STR_PAD_LEFT);

                    // إضافة رسالة للمستخدم
                    $_SESSION['warning_message'] = 'تم استخدام رقم طلب احتياطي بسبب مشكلة في الاتصال بالخادم. يمكنك المتابعة باستخدام هذا الرقم.';

                    // تسجيل استخدام رقم احتياطي في ملف السجل
                    $logMessage = date('Y-m-d H:i:s') . " - تم استخدام رقم طلب احتياطي: " . $orderNumber . " للفرع: " . $branchCode . "\n";
                    file_put_contents('logs/backup_order_numbers.log', $logMessage, FILE_APPEND);

                    error_log("تم استخدام رقم طلب احتياطي بنجاح: " . $orderNumber);
                } catch (PDOException $pdoEx) {
                    // في حالة فشل الطريقة الاحتياطية أيضًا، استخدم رقمًا عشوائيًا مع تسجيله
                    error_log("فشل في استخدام رقم طلب احتياطي: " . $pdoEx->getMessage());

                    // استخدام الوقت الحالي كجزء من الرقم العشوائي لتجنب التكرار
                    $timestamp = time();
                    $randomPart = ($timestamp % 100) + 1; // استخدام آخر رقمين من الطابع الزمني + 1

                    // تنسيق رقم الطلب مع إضافة حرف R للإشارة إلى أنه رقم عشوائي
                    $orderNumber = $branchCode . '25-' . str_pad($randomPart, 2, '0', STR_PAD_LEFT) . 'R';

                    // إضافة رسالة تحذير للمستخدم
                    $_SESSION['warning_message'] = 'تم إنشاء رقم طلب عشوائي بسبب مشكلة في النظام. يرجى التحقق من صحة الرقم قبل المتابعة.';

                    // تسجيل استخدام رقم عشوائي في ملف السجل
                    $logMessage = date('Y-m-d H:i:s') . " - تم إنشاء رقم طلب عشوائي: " . $orderNumber . " للفرع: " . $branchCode . " - " . $pdoEx->getMessage() . "\n";
                    file_put_contents('logs/random_order_numbers.log', $logMessage, FILE_APPEND);

                    error_log("تم إنشاء رقم طلب عشوائي: " . $orderNumber);

                    // إرسال إشعار للمسؤول
                    if (function_exists('sendAdminNotification')) {
                        sendAdminNotification('تم إنشاء رقم طلب عشوائي', 'تم إنشاء رقم طلب عشوائي: ' . $orderNumber . ' للفرع: ' . $branchCode . ' بسبب مشكلة في النظام.');
                    }
                }
            }
        }

        try {
            // إدراج الطلب في قاعدة البيانات
            $stmt = $pdo->prepare("
                INSERT INTO orders (
                    order_number, client_name, service_type, branch_id,
                    departure_area, second_destination, delivery_area, goods_name, goods_type,
                    country_of_origin, weight, delivery_duration,
                    order_upload_date, scheduled_delivery_date, recipient_name, phone_number,
                    service_fees, currency, amount_agreed_with_customer, amount_agreed_with_agent,
                    amount_paid, remaining_amount, other_expenses, profit,
                    employee_name, status, payment_status, user_id
                ) VALUES (
                    ?, ?, ?, ?,
                    ?, ?, ?, ?, ?,
                    ?, ?, ?,
                    ?, ?, ?, ?,
                    ?, ?, ?, ?,
                    ?, ?, ?, ?,
                    ?, ?, ?, ?
                )
            ");

            // تسجيل البيانات قبل الإدراج للتصحيح
            error_log("بيانات الطلب قبل الإدراج: " . json_encode([
                $orderNumber, $clientName, $serviceType, $branchId,
                $departureArea, $secondDestination, $deliveryArea, $goodsName, $goodsType,
                $countryOfOrigin, $weight, $deliveryDuration,
                $orderDate, $expectedDeliveryDate, $receiverName, $phoneNumber,
                $serviceFee, $currency, $clientAmount, $agentAmount,
                $paidAmount, $remainingAmount, $otherExpenses, $profit,
                $responsibleEmployee, $orderStatus, $paymentStatus, $_SESSION['user_id']
            ], JSON_UNESCAPED_UNICODE));

            $stmt->execute([
                $orderNumber, $clientName, $serviceType, $branchId,
                $departureArea, $secondDestination, $deliveryArea, $goodsName, $goodsType,
                $countryOfOrigin, $weight, $deliveryDuration,
                $orderDate, $expectedDeliveryDate, $receiverName, $phoneNumber,
                $serviceFee, $currency, $clientAmount, $agentAmount,
                $paidAmount, $remainingAmount, $otherExpenses, $profit,
                $responsibleEmployee, $orderStatus, $paymentStatus, $_SESSION['user_id']
            ]);

            // الحصول على معرف الطلب المدرج
            $newOrderId = $pdo->lastInsertId();
            error_log("تم إنشاء الطلب بنجاح برقم: " . $orderNumber . ", معرف الطلب: " . $newOrderId);

            // تحويل المستخدم إلى صفحة قائمة الطلبات مع رسالة نجاح
            $_SESSION['success_message'] = 'تم إنشاء الطلب بنجاح برقم: ' . $orderNumber;
            header('Location: orders.php');
            exit;

        } catch (PDOException $e) {
            $error_message = 'حدث خطأ أثناء إنشاء الطلب: ' . $e->getMessage();
        }
    }
}

// الحصول على إحصائيات الطلبات (للعرض في لوحة التحكم)
$totalOrders = 0;
$pendingOrders = 0;
$inTransitOrders = 0;
$deliveredOrders = 0;
$cancelledOrders = 0;

// استعلام لجلب إحصائيات الطلبات
$stmt = $pdo->query("SELECT
    COUNT(*) as total,
    SUM(CASE WHEN status = 'قيد المعالجة' THEN 1 ELSE 0 END) as pending,
    SUM(CASE WHEN status = 'قيد التوصيل' THEN 1 ELSE 0 END) as in_transit,
    SUM(CASE WHEN status = 'تم التسليم' THEN 1 ELSE 0 END) as delivered,
    SUM(CASE WHEN status = 'ملغى' THEN 1 ELSE 0 END) as cancelled
FROM orders");

if ($stmt) {
    $stats = $stmt->fetch();
    $totalOrders = $stats['total'] ?? 0;
    $pendingOrders = $stats['pending'] ?? 0;
    $inTransitOrders = $stats['in_transit'] ?? 0;
    $deliveredOrders = $stats['delivered'] ?? 0;
    $cancelledOrders = $stats['cancelled'] ?? 0;
}

// الحصول على إحصائيات المستخدمين والفروع (للمدير فقط)
$totalUsers = 0;
$totalBranches = 0;

if ($isAdmin) {
    $stmt = $pdo->query("SELECT COUNT(*) as total FROM users");
    if ($stmt) {
        $totalUsers = $stmt->fetchColumn();
    }

    $stmt = $pdo->query("SELECT COUNT(*) as total FROM branches");
    if ($stmt) {
        $totalBranches = $stmt->fetchColumn();
    }
}

// الحصول على آخر الطلبات
$recentOrders = [];
$stmt = $pdo->query("SELECT o.*, b.name as branch_name
                    FROM orders o
                    LEFT JOIN branches b ON o.branch_id = b.id
                    ORDER BY o.created_at DESC LIMIT 5");
if ($stmt) {
    $recentOrders = $stmt->fetchAll();
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">

<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title><?php echo $system_name; ?> | إنشاء طلب جديد</title>
  <script src="https://cdn.tailwindcss.com"></script>
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
  <link rel="stylesheet" href="assets/css/unified-design.css">
  <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
  <style>
    :root {
      --primary: #4f46e5;
      --primary-dark: #3730a3;
      --primary-light: #818cf8;
      --secondary: #f59e0b;
      --secondary-dark: #d97706;
      --secondary-light: #fbbf24;
      --info: #3b82f6;
      --success: #10b981;
      --warning: #f59e0b;
      --danger: #ef4444;
      --dark: #1e293b;
      --light: #f8fafc;
      --gray-100: #f1f5f9;
      --gray-200: #e2e8f0;
      --gray-300: #cbd5e1;
      --gray-400: #94a3b8;
      --gray-500: #64748b;
      --gray-600: #475569;
      --gray-700: #334155;
      --gray-800: #1e293b;
      --gray-900: #0f172a;
      --border-radius: 0.75rem;
      --shadow-sm: 0 1px 2px 0 rgba(0, 0, 0, 0.05);
      --shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
      --shadow-md: 0 10px 15px -3px rgba(0, 0, 0, 0.1), 0 4px 6px -2px rgba(0, 0, 0, 0.05);
      --shadow-lg: 0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04);
      --transition: all 0.3s ease;
      --sidebar-bg: #0f1e3d;
      --sidebar-gradient: linear-gradient(135deg, #0f1e3d 0%, #1a365d 100%);
    }

    @import url('https://fonts.googleapis.com/css2?family=Tajawal:wght@300;400;500;700;800&display=swap');

    body {
      font-family: 'Tajawal', sans-serif;
      background-color: #f1f5f9;
      color: var(--dark);
      overflow-x: hidden;
    }

    /* تصميم القائمة الجانبية */
    aside {
      background: var(--sidebar-gradient);
      box-shadow: var(--shadow-lg);
      transition: var(--transition);
      position: sticky;
      top: 0;
      height: 100vh;
      overflow-y: auto;
      border-left: 1px solid rgba(255, 255, 255, 0.1);
      z-index: 40;
    }

    /* تصميم شريط التمرير للقائمة الجانبية */
    aside::-webkit-scrollbar {
      width: 6px;
    }

    aside::-webkit-scrollbar-track {
      background: rgba(255, 255, 255, 0.05);
    }

    aside::-webkit-scrollbar-thumb {
      background-color: rgba(255, 255, 255, 0.2);
      border-radius: 3px;
    }

    aside::-webkit-scrollbar-thumb:hover {
      background-color: rgba(255, 255, 255, 0.3);
    }

    /* تصميم حاوية الشعار */
    aside .logo-container {
      background: white;
      border-radius: var(--border-radius);
      padding: 0.75rem;
      margin-bottom: 1.5rem;
      box-shadow: 0 4px 15px rgba(0, 0, 0, 0.2);
      border: 2px solid rgba(255, 255, 255, 0.8);
      position: relative;
      overflow: hidden;
      max-width: 75%;
      margin-left: auto;
      margin-right: auto;
      transform: translateY(0);
      transition: all 0.4s cubic-bezier(0.175, 0.885, 0.32, 1.275);
    }

    aside .logo-container:hover {
      transform: translateY(-5px);
      box-shadow: 0 8px 25px rgba(0, 0, 0, 0.3);
    }

    aside .logo-container::before {
      content: '';
      position: absolute;
      top: -50%;
      left: -50%;
      width: 200%;
      height: 200%;
      background: radial-gradient(circle, rgba(26, 54, 93, 0.1) 0%, transparent 70%);
      opacity: 0;
      transition: opacity 0.5s ease;
      z-index: 1;
    }

    aside .logo-container::after {
      content: '';
      position: absolute;
      bottom: -2px;
      left: 0;
      width: 100%;
      height: 6px;
      background: linear-gradient(90deg, #3182ce, #63b3ed);
      border-radius: 0 0 var(--border-radius) var(--border-radius);
      opacity: 0.8;
      transition: all 0.3s ease;
    }

    aside .logo-container:hover::before {
      opacity: 1;
    }

    aside .logo-container:hover::after {
      height: 8px;
      opacity: 1;
      box-shadow: 0 0 15px rgba(49, 130, 206, 0.6);
    }

    /* تأثير الشعار */
    .logo-reveal {
      animation: logoReveal 1.5s ease-out forwards;
      transition: all 0.5s cubic-bezier(0.175, 0.885, 0.32, 1.275);
      border-radius: 8px;
      position: relative;
      z-index: 2;
      max-height: 100px;
      width: auto;
    }

    .logo-reveal:hover {
      transform: scale(1.08) rotate(1deg);
      filter: brightness(1.2) contrast(1.1) drop-shadow(0 10px 20px rgba(0, 0, 0, 0.25));
    }

    @keyframes logoReveal {
      0% {
        opacity: 0;
        transform: scale(0.8);
      }
      50% {
        opacity: 0.5;
        transform: scale(1.05);
      }
      100% {
        opacity: 1;
        transform: scale(1);
      }
    }

    /* تأثير إضافي للشعار */
    aside .logo-container::before {
      content: '';
      position: absolute;
      top: 0;
      left: 0;
      width: 100%;
      height: 100%;
      background: linear-gradient(135deg, rgba(255,255,255,0.3) 0%, rgba(255,255,255,0) 50%, rgba(255,255,255,0.3) 100%);
      transform: translateX(-100%);
      transition: transform 1s ease;
    }

    aside .logo-container:hover::before {
      transform: translateX(100%);
    }

    /* تصميم عناصر القائمة */
    aside nav a {
      border-radius: var(--border-radius);
      transition: var(--transition);
      position: relative;
      overflow: hidden;
    }

    aside nav a:hover {
      background-color: rgba(255, 255, 255, 0.1);
    }

    aside nav a.active {
      background-color: rgba(255, 255, 255, 0.2);
      position: relative;
    }

    aside nav a.active::before {
      content: '';
      position: absolute;
      right: 0;
      top: 0;
      height: 100%;
      width: 4px;
      background-color: var(--secondary);
      border-radius: 0 4px 4px 0;
    }

    aside nav a i {
      transition: var(--transition);
    }

    aside nav a:hover i {
      transform: translateX(-5px);
    }

    /* تنسيق القائمة المنسدلة */
    .dropdown-menu {
      max-height: 0;
      overflow: hidden;
      transition: max-height 0.3s ease;
      margin-right: 2.5rem;
    }

    .dropdown-menu.active {
      max-height: 200px;
    }

    .dropdown-toggle {
      position: relative;
    }

    .dropdown-toggle::after {
      content: '\f107';
      font-family: 'Font Awesome 5 Free';
      font-weight: 900;
      position: absolute;
      left: 1rem;
      top: 50%;
      transform: translateY(-50%);
      transition: transform 0.3s ease;
    }

    .dropdown-toggle.active::after {
      transform: translateY(-50%) rotate(180deg);
    }

    /* تصميم النموذج */
    .form-input {
      width: 100%;
      padding: 0.75rem 1rem;
      border: 1.5px solid #e2e8f0;
      border-radius: 0.5rem;
      background-color: #fff;
      color: #1e293b;
      transition: all 0.3s ease;
      font-size: 0.95rem;
      box-shadow: 0 1px 2px rgba(0, 0, 0, 0.02);
    }

    .form-input:focus {
      outline: none;
      border-color: var(--primary-light);
      box-shadow: 0 0 0 3px rgba(79, 70, 229, 0.15);
    }

    .form-input::placeholder {
      color: #94a3b8;
      opacity: 0.7;
    }

    .form-input:hover {
      border-color: #cbd5e1;
      box-shadow: 0 2px 4px rgba(0, 0, 0, 0.05);
    }

    .form-input[readonly], .form-input[disabled] {
      background-color: #f8fafc;
      cursor: not-allowed;
    }

    /* تصميم أقسام النموذج */
    .form-section {
      border: 1px solid #e2e8f0;
      border-radius: 0.75rem;
      padding: 1.5rem;
      margin-bottom: 1.5rem;
      background-color: #fff;
      box-shadow: 0 4px 6px rgba(0, 0, 0, 0.05);
      transition: all 0.3s ease;
      position: relative;
      overflow: hidden;
    }

    .form-section::before {
      content: '';
      position: absolute;
      top: 0;
      right: 0;
      width: 6px;
      height: 0;
      background: linear-gradient(to bottom, var(--primary), var(--primary-light));
      border-radius: 0 0.75rem 0 0;
      transition: height 0.3s ease;
    }

    .form-section:hover {
      box-shadow: 0 8px 15px rgba(0, 0, 0, 0.08);
      border-color: #cbd5e1;
      transform: translateY(-2px);
    }

    .form-section:hover::before {
      height: 100%;
    }

    .form-section-title {
      font-size: 1.1rem;
      font-weight: 600;
      color: var(--primary-dark);
      margin-bottom: 1.25rem;
      padding-bottom: 0.75rem;
      border-bottom: 1px solid #e2e8f0;
      display: flex;
      align-items: center;
      position: relative;
    }

    .form-section-title::after {
      content: '';
      position: absolute;
      bottom: -1px;
      right: 0;
      width: 50px;
      height: 2px;
      background: linear-gradient(to left, var(--primary), transparent);
      border-radius: 2px;
      transition: width 0.3s ease;
    }

    .form-section:hover .form-section-title::after {
      width: 100px;
    }

    .form-section-title i {
      margin-left: 0.75rem;
      color: var(--primary);
      font-size: 1.2rem;
      background-color: rgba(79, 70, 229, 0.1);
      width: 32px;
      height: 32px;
      display: flex;
      align-items: center;
      justify-content: center;
      border-radius: 8px;
    }

    /* تصميم الأزرار */
    .btn {
      border-radius: var(--border-radius);
      padding: 0.75rem 1.5rem;
      font-weight: 500;
      transition: var(--transition);
      position: relative;
      overflow: hidden;
      display: inline-flex;
      align-items: center;
      justify-content: center;
      gap: 0.5rem;
    }

    .btn-primary {
      background: linear-gradient(135deg, var(--primary) 0%, var(--primary-dark) 100%);
      color: white;
      box-shadow: 0 4px 6px -1px rgba(79, 70, 229, 0.3);
    }

    .btn-primary:hover {
      transform: translateY(-2px);
      box-shadow: 0 10px 15px -3px rgba(79, 70, 229, 0.4);
    }

    .btn-primary:active {
      transform: translateY(0);
    }

    .btn-secondary {
      background: linear-gradient(135deg, var(--secondary) 0%, var(--secondary-dark) 100%);
      color: white;
      box-shadow: 0 4px 6px -1px rgba(245, 158, 11, 0.3);
    }

    .btn-secondary:hover {
      transform: translateY(-2px);
      box-shadow: 0 10px 15px -3px rgba(245, 158, 11, 0.4);
    }

    .btn-secondary:active {
      transform: translateY(0);
    }

    /* تأثيرات الأزرار */
    .btn::before {
      content: '';
      position: absolute;
      top: 0;
      left: 0;
      width: 0;
      height: 100%;
      background-color: rgba(255, 255, 255, 0.1);
      transition: width 0.3s ease;
      z-index: 0;
    }

    .btn:hover::before {
      width: 100%;
    }



    /* تصميم الشريط العلوي */
    .top-bar {
      display: flex;
      align-items: center;
      justify-content: space-between;
      padding: 0.75rem 1.5rem;
      background-color: white;
      border-bottom: 1px solid var(--gray-200);
      box-shadow: var(--shadow-sm);
      position: sticky;
      top: 0;
      z-index: 30;
    }

    /* تصميم ملف المستخدم */
    .user-profile {
      display: flex;
      align-items: center;
      gap: 0.75rem;
    }

    .user-avatar {
      width: 40px;
      height: 40px;
      border-radius: 50%;
      display: flex;
      align-items: center;
      justify-content: center;
      font-weight: bold;
      background: linear-gradient(135deg, var(--primary) 0%, var(--primary-dark) 100%);
      color: white;
      box-shadow: var(--shadow-sm);
      transition: var(--transition);
    }

    /* تصميم صفحة إنشاء الطلب */
    .animate-fadeIn {
      animation: fadeIn 0.5s ease-in-out;
    }

    @keyframes fadeIn {
      from {
        opacity: 0;
        transform: translateY(10px);
      }
      to {
        opacity: 1;
        transform: translateY(0);
      }
    }

    /* تأثير عنوان الصفحة */
    h1.text-2xl {
      position: relative;
      display: inline-block;
      transition: all 0.3s ease;
    }

    h1.text-2xl::after {
      content: '';
      position: absolute;
      bottom: -5px;
      right: 0;
      width: 0;
      height: 2px;
      background: linear-gradient(to left, var(--primary), transparent);
      transition: width 0.5s ease;
    }

    h1.text-2xl:hover::after {
      width: 100%;
    }

    /* تحسين التجاوب مع الشاشات المختلفة */
    @media (max-width: 768px) {
      aside {
        position: fixed;
        right: -280px;
        top: 0;
        height: 100vh;
        z-index: 50;
        width: 280px;
      }

      aside.active {
        right: 0;
      }

      .overlay {
        position: fixed;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background-color: rgba(0, 0, 0, 0.5);
        z-index: 40;
        opacity: 0;
        pointer-events: none;
        transition: var(--transition);
      }

      .overlay.active {
        opacity: 1;
        pointer-events: auto;
      }

      .main-content {
        margin-right: 0 !important;
        width: 100%;
      }

      .form-section {
        padding: 1rem;
      }

      aside .logo-container {
        max-width: 80%;
      }
    }

    /* تأثيرات إضافية للصورة الرمزية */
    aside .user-avatar {
      background: rgba(255, 255, 255, 0.1);
      border: 2px solid rgba(255, 255, 255, 0.2);
    }

    .user-avatar:hover {
      transform: scale(1.1);
      box-shadow: 0 0 15px rgba(79, 70, 229, 0.3);
    }
  </style>
</head>

<body>
  <!-- طبقة التراكب للأجهزة المحمولة -->
  <div id="overlay" class="overlay"></div>

  <div class="flex min-h-screen">
    <!-- القائمة الجانبية -->
    <aside class="w-64 hidden md:block">
      <div class="p-6">
        <div class="logo-container flex items-center justify-center mb-8 p-2">
          <img src="<?php echo $company_logo; ?>" alt="شعار النظام" class="h-40 w-auto object-contain logo-reveal">
        </div>

        <div class="user-info mb-8">
          <div class="flex items-center space-x-4 space-x-reverse mb-3">
            <div class="user-avatar">
              <?php echo mb_substr($user['full_name'], 0, 1, 'UTF-8'); ?>
            </div>
            <div>
              <h3 class="text-white font-semibold"><?php echo htmlspecialchars($user['full_name']); ?></h3>
              <p class="text-indigo-200 text-sm"><?php echo $isAdmin ? 'مدير النظام' : 'مستخدم'; ?></p>
            </div>
          </div>
          <div class="bg-white/10 h-0.5 rounded-full mb-3"></div>
        </div>

        <nav>
          <ul class="space-y-3">
            <!-- لوحة التحكم -->
            <li>
              <a href="unified-dashboard.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white">
                <i class="fas fa-tachometer-alt ml-3 text-lg"></i>
                <span>لوحة التحكم</span>
              </a>
            </li>

            <!-- إنشاء طلب جديد -->
            <li>
              <a href="create-order-unified.php" class="flex items-center p-3 rounded-lg text-white active">
                <i class="fas fa-plus-circle ml-3 text-lg"></i>
                <span>إنشاء طلب جديد</span>
              </a>
            </li>

            <!-- قائمة الطلبات -->
            <li>
              <a href="orders.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white">
                <i class="fas fa-box ml-3 text-lg"></i>
                <span>قائمة الطلبات</span>
              </a>
            </li>

            <?php if ($isAdmin): ?>
            <!-- قائمة التقارير المنسدلة -->
            <li>
              <a href="#" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white dropdown-toggle" id="reportsDropdown">
                <i class="fas fa-chart-bar ml-3 text-lg"></i>
                <span>التقارير</span>
              </a>
              <ul class="dropdown-menu" id="reportsMenu">
                <li>
                  <a href="weekly-reports.php" class="flex items-center p-2 rounded-lg text-white/80 hover:text-white">
                    <i class="fas fa-calendar-week ml-2 text-sm"></i>
                    <span>التقارير الأسبوعية</span>
                  </a>
                </li>
                <li>
                  <a href="monthly-reports.php" class="flex items-center p-2 rounded-lg text-white/80 hover:text-white">
                    <i class="fas fa-calendar-alt ml-2 text-sm"></i>
                    <span>التقارير الشهرية</span>
                  </a>
                </li>
              </ul>
            </li>

            <!-- إدارة المستخدمين -->
            <li>
              <a href="users.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white">
                <i class="fas fa-users ml-3 text-lg"></i>
                <span>إدارة المستخدمين</span>
              </a>
            </li>

            <!-- الفروع -->
            <li>
              <a href="branches.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white">
                <i class="fas fa-code-branch ml-3 text-lg"></i>
                <span>الفروع</span>
              </a>
            </li>

            <!-- إعدادات النظام -->
            <li>
              <a href="settings.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white">
                <i class="fas fa-cog ml-3 text-lg"></i>
                <span>إعدادات النظام</span>
              </a>
            </li>
            <?php endif; ?>
          </ul>

          <div class="pt-6 mt-6 border-t border-white/10">
            <a href="auth/logout.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white">
              <i class="fas fa-sign-out-alt ml-3 text-lg"></i>
              <span>تسجيل الخروج</span>
            </a>
          </div>
        </nav>
      </div>
    </aside>

    <!-- المحتوى الرئيسي -->
    <main class="flex-1">
      <!-- الشريط العلوي -->
      <header class="top-bar">
        <div class="flex items-center">
          <button id="toggleSidebar" class="md:hidden text-gray-600 hover:text-primary p-2 rounded-lg">
            <i class="fas fa-bars text-xl"></i>
          </button>

          <!-- شعار النظام في الشريط العلوي (يظهر فقط في الأجهزة المحمولة) -->
          <div class="flex items-center md:hidden ml-3">
            <img src="<?php echo $company_logo; ?>" alt="شعار النظام" class="h-16 w-auto object-contain logo-reveal">
          </div>

          <div class="relative ml-4 hidden md:block">
            <input type="text" placeholder="البحث..." class="bg-gray-100 rounded-full py-2 px-4 pl-10 w-64 focus:outline-none focus:ring-2 focus:ring-primary/30 focus:bg-white transition-all">
            <i class="fas fa-search absolute right-3 top-3 text-gray-400"></i>
          </div>
        </div>

        <!-- معلومات المستخدم والإشعارات -->
        <div class="flex items-center space-x-4 space-x-reverse">
          <?php if ($isAdmin): ?>
          <div class="relative">
            <button class="relative p-2 rounded-full hover:bg-gray-100 transition-all">
              <i class="fas fa-bell text-gray-600"></i>
              <span class="absolute -top-1 -right-1 bg-red-500 text-white text-xs rounded-full h-5 w-5 flex items-center justify-center animate-pulse">3</span>
            </button>
          </div>
          <?php endif; ?>

          <div class="relative">
            <button class="relative p-2 rounded-full hover:bg-gray-100 transition-all">
              <i class="fas fa-envelope text-gray-600"></i>
              <?php if ($isAdmin): ?>
              <span class="absolute -top-1 -right-1 bg-primary text-white text-xs rounded-full h-5 w-5 flex items-center justify-center">2</span>
              <?php endif; ?>
            </button>
          </div>

          <div class="border-r border-gray-300 h-8 mx-2"></div>

          <div class="user-profile">
            <div class="user-avatar">
              <?php echo mb_substr($user['full_name'], 0, 1, 'UTF-8'); ?>
            </div>
            <div>
              <p class="font-medium text-sm"><?php echo htmlspecialchars($user['full_name']); ?></p>
              <p class="text-xs text-gray-500"><?php echo $isAdmin ? 'مدير النظام' : 'مستخدم'; ?></p>
            </div>
          </div>
        </div>
      </header>

      <!-- محتوى الصفحة -->
      <div class="p-6">
        <!-- رأس الصفحة -->
        <div class="flex flex-col md:flex-row md:items-center md:justify-between mb-8">
          <div class="animate-fadeIn" style="animation-delay: 0.1s;">
            <h1 class="text-3xl font-bold text-gray-800 flex items-center">
              <span class="bg-gradient-to-r from-primary to-primary-light bg-clip-text text-transparent">
                إنشاء طلب جديد
              </span>
            </h1>
            <p class="text-gray-500 mt-2 flex items-center">
              <i class="fas fa-plus-circle ml-2 text-primary"></i>
              <span>قم بإنشاء طلب شحن جديد وتتبع حالته بسهولة</span>
            </p>
          </div>

          <div class="mt-4 md:mt-0 animate-fadeIn" style="animation-delay: 0.2s;">
            <a href="unified-dashboard.php" class="btn btn-primary">
              <i class="fas fa-arrow-right ml-2"></i>
              <span>العودة إلى لوحة التحكم</span>
            </a>
          </div>
        </div>

        <!-- عرض رسائل التنبيه -->
        <?php if (isset($_SESSION['warning_message'])): ?>
        <div class="bg-yellow-50 border border-yellow-200 rounded-lg p-4 mb-6 flex items-start">
            <div class="flex-shrink-0 mt-0.5">
                <i class="fas fa-exclamation-triangle text-yellow-500 text-xl"></i>
            </div>
            <div class="mr-3">
                <h3 class="text-sm font-medium text-yellow-800">تنبيه</h3>
                <div class="mt-1 text-sm text-yellow-700">
                    <?php echo $_SESSION['warning_message']; ?>
                </div>
                <?php if (strpos($_SESSION['warning_message'], 'خطأ في الاتصال بالخادم') !== false): ?>
                <div class="mt-2">
                    <button type="button" id="fixConnectionBtn" class="bg-blue-500 hover:bg-blue-600 text-white py-1 px-3 rounded-md text-xs flex items-center">
                        <i class="fas fa-wrench ml-1"></i>
                        <span>حل المشكلة</span>
                    </button>
                </div>
                <?php endif; ?>
            </div>
            <button type="button" class="mr-auto -mt-1 -my-1.5 bg-yellow-50 text-yellow-500 rounded-lg p-1.5 hover:bg-yellow-100 inline-flex h-8 w-8 items-center justify-center" onclick="this.parentElement.remove()">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <?php unset($_SESSION['warning_message']); endif; ?>

        <!-- نموذج إنشاء الطلب -->
        <div class="bg-white rounded-xl shadow-sm overflow-hidden animate-fadeIn">
          <div class="p-6 max-w-full overflow-x-auto">
            <form id="shipmentForm" method="post" action="" class="grid grid-cols-1 gap-6 w-full">
              <!-- المعلومات الأساسية -->
              <div class="form-section bg-white rounded-lg shadow-sm p-6 mb-6 border border-gray-200 hover:border-indigo-300 transition-all duration-300">
                <h3 class="form-section-title">
                  <i class="fas fa-info-circle text-indigo-500"></i>
                  <span>المعلومات الأساسية للطلب</span>
                </h3>
                <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
                  <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">رقم الطلب</label>
                    <div class="relative">
                      <input type="text" id="orderNumber" name="orderNumber" readonly class="form-input w-full bg-gray-50 text-indigo-700 font-medium" placeholder="سيتم إنشاؤه تلقائياً">
                      <i class="fas fa-hashtag absolute left-3 top-3 text-gray-400"></i>
                    </div>
                  </div>
                  <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">اسم العميل <span class="text-red-500">*</span></label>
                    <div class="relative">
                      <input type="text" id="clientName" name="clientName" required class="form-input w-full" placeholder="أدخل اسم العميل">
                      <i class="fas fa-user absolute left-3 top-3 text-gray-400"></i>
                    </div>
                  </div>
                  <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">نوع الخدمة <span class="text-red-500">*</span></label>
                    <div class="relative">
                      <input type="text" id="serviceType" name="serviceType" required class="form-input w-full" placeholder="أدخل نوع الخدمة">
                      <i class="fas fa-cog absolute left-3 top-3 text-gray-400"></i>
                    </div>
                  </div>
                </div>
              </div>

              <!-- معلومات البضاعة -->
              <div class="form-section bg-white rounded-lg shadow-sm p-6 mb-6 border border-gray-200 hover:border-indigo-300 transition-all duration-300">
                <h3 class="form-section-title">
                  <i class="fas fa-box text-amber-500"></i>
                  <span>معلومات البضاعة</span>
                </h3>
                <div class="grid grid-cols-1 md:grid-cols-4 gap-6">
                  <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">اسم البضاعة <span class="text-red-500">*</span></label>
                    <input type="text" id="goodsName" name="goodsName" required class="form-input w-full" placeholder="أدخل اسم البضاعة">
                  </div>
                  <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">نوع البضاعة <span class="text-red-500">*</span></label>
                    <input type="text" id="goodsType" name="goodsType" required class="form-input w-full" placeholder="أدخل نوع البضاعة">
                  </div>
                  <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">بلد المنشأ <span class="text-red-500">*</span></label>
                    <div class="relative">
                      <input type="text" id="countryOfOrigin" name="countryOfOrigin" required class="form-input w-full" placeholder="أدخل بلد المنشأ">
                      <i class="fas fa-globe absolute left-3 top-3 text-gray-400"></i>
                    </div>
                  </div>
                  <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">الوزن <span class="text-red-500">*</span></label>
                    <div class="relative">
                      <input type="number" id="weight" name="weight" step="0.1" required class="form-input w-full" placeholder="أدخل الوزن">
                      <i class="fas fa-weight absolute left-3 top-3 text-gray-400"></i>
                    </div>
                  </div>
                </div>
              </div>

              <!-- معلومات الفرع والمناطق -->
              <div class="form-section bg-white rounded-lg shadow-sm p-6 mb-6 border border-gray-200 hover:border-indigo-300 transition-all duration-300">
                <h3 class="form-section-title">
                  <i class="fas fa-map-marker-alt text-red-500"></i>
                  <span>معلومات الفرع والمناطق</span>
                </h3>
                <div class="grid grid-cols-1 md:grid-cols-4 gap-6">
                  <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">الفرع <span class="text-red-500">*</span></label>
                    <div class="relative">
                      <select id="branch" name="branch" required class="form-input w-full">
                        <option value="">اختر الفرع</option>
                        <option value="المكلا" data-code="MKL">المكلا (MKL25-01)</option>
                        <option value="صنعاء" data-code="SNA">صنعاء (SNA25-01)</option>
                        <option value="عدن" data-code="ADN">عدن (ADN25-01)</option>
                        <option value="شحن" data-code="SH">شحن (SH25-01)</option>
                        <option value="الصين" data-code="CN">الصين (CN25-01)</option>
                      </select>
                      <i class="fas fa-building absolute left-3 top-3 text-gray-400"></i>
                    </div>
                  </div>
                  <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">منطقة المغادرة <span class="text-red-500">*</span></label>
                    <input type="text" id="departureArea" name="departureArea" required class="form-input w-full" placeholder="أدخل منطقة المغادرة">
                  </div>
                  <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">الوجهة الثانية</label>
                    <input type="text" id="secondDestination" name="secondDestination" class="form-input w-full" placeholder="أدخل الوجهة الثانية">
                  </div>
                  <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">منطقة التسليم <span class="text-red-500">*</span></label>
                    <input type="text" id="deliveryArea" name="deliveryArea" required class="form-input w-full" placeholder="أدخل منطقة التسليم">
                  </div>
                </div>
              </div>

              <!-- معلومات المستلم والتواريخ -->
              <div class="form-section bg-white rounded-lg shadow-sm p-6 mb-6 border border-gray-200 hover:border-indigo-300 transition-all duration-300">
                <h3 class="form-section-title">
                  <i class="fas fa-user-clock text-blue-500"></i>
                  <span>معلومات المستلم والتواريخ</span>
                </h3>
                <div class="grid grid-cols-1 md:grid-cols-5 gap-6">
                  <div class="md:col-span-2">
                    <label class="block text-sm font-medium text-gray-700 mb-1">اسم المستلم <span class="text-red-500">*</span></label>
                    <div class="relative">
                      <input type="text" id="receiverName" name="receiverName" required class="form-input w-full" placeholder="أدخل اسم المستلم">
                      <i class="fas fa-user-check absolute left-3 top-3 text-gray-400"></i>
                    </div>
                  </div>
                  <div class="md:col-span-2">
                    <label class="block text-sm font-medium text-gray-700 mb-1">رقم الهاتف <span class="text-red-500">*</span></label>
                    <div class="relative">
                      <input type="tel" id="phoneNumber" name="phoneNumber" required class="form-input w-full" placeholder="أدخل رقم الهاتف">
                      <i class="fas fa-phone absolute left-3 top-3 text-gray-400"></i>
                    </div>
                  </div>
                  <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">مدة الوصول (بالأيام)</label>
                    <div class="relative">
                      <input type="number" id="deliveryDuration" name="deliveryDuration" min="1" class="form-input w-full" placeholder="مدة الوصول">
                      <i class="fas fa-hourglass-half absolute left-3 top-3 text-gray-400"></i>
                    </div>
                  </div>
                  <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">تاريخ رفع الطلب</label>
                    <div class="relative">
                      <input type="date" id="orderDate" name="orderDate" class="form-input w-full" value="<?php echo date('Y-m-d'); ?>">
                      <i class="fas fa-calendar-plus absolute left-3 top-3 text-gray-400"></i>
                    </div>
                  </div>
                  <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">تاريخ التسليم المقرر</label>
                    <div class="relative">
                      <input type="date" id="expectedDeliveryDate" name="expectedDeliveryDate" class="form-input w-full">
                      <i class="fas fa-calendar-check absolute left-3 top-3 text-gray-400"></i>
                    </div>
                  </div>
                </div>
              </div>

              <!-- المعلومات المالية -->
              <div class="form-section bg-white rounded-lg shadow-sm p-6 mb-6 border border-gray-200 hover:border-indigo-300 transition-all duration-300">
                <h3 class="form-section-title">
                  <i class="fas fa-money-bill-wave text-green-500"></i>
                  <span>المعلومات المالية</span>
                </h3>
                <div class="grid grid-cols-1 md:grid-cols-4 gap-6">
                  <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">رسوم الخدمة <span class="text-red-500">*</span></label>
                    <div class="relative">
                      <input type="number" id="serviceFee" name="serviceFee" required class="form-input w-full" placeholder="أدخل رسوم الخدمة" min="0" step="0.01" onchange="calculateProfit()">
                      <i class="fas fa-dollar-sign absolute left-3 top-3 text-gray-400"></i>
                    </div>
                  </div>
                  <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">العملة <span class="text-red-500">*</span></label>
                    <div class="relative">
                      <select id="currency" name="currency" required class="form-input w-full">
                        <option value="">اختر العملة</option>
                        <option value="USD">دولار أمريكي (USD)</option>
                        <option value="SAR">ريال سعودي (SAR)</option>
                        <option value="YER">ريال يمني (YER)</option>
                        <option value="AED">درهم إماراتي (AED)</option>
                        <option value="CNY">يوان صيني (CNY)</option>
                      </select>
                      <i class="fas fa-money-bill absolute left-3 top-3 text-gray-400"></i>
                    </div>
                  </div>
                  <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">المبلغ المتفق مع العميل <span class="text-red-500">*</span></label>
                    <div class="relative">
                      <input type="number" id="clientAmount" name="clientAmount" required class="form-input w-full" placeholder="المبلغ المتفق مع العميل" min="0" step="0.01" onchange="calculateProfit()">
                      <i class="fas fa-hand-holding-usd absolute left-3 top-3 text-gray-400"></i>
                    </div>
                  </div>
                  <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">المبلغ المتفق مع الوكيل <span class="text-red-500">*</span></label>
                    <div class="relative">
                      <input type="number" id="agentAmount" name="agentAmount" required class="form-input w-full" placeholder="المبلغ المتفق مع الوكيل" min="0" step="0.01" onchange="calculateProfit()">
                      <i class="fas fa-handshake absolute left-3 top-3 text-gray-400"></i>
                    </div>
                  </div>
                  <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">المدفوع <span class="text-red-500">*</span></label>
                    <div class="relative">
                      <input type="number" id="paidAmount" name="paidAmount" required class="form-input w-full" placeholder="المبلغ المدفوع" min="0" step="0.01" onchange="calculateRemaining()">
                      <i class="fas fa-credit-card absolute left-3 top-3 text-gray-400"></i>
                    </div>
                  </div>
                  <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">المتبقي</label>
                    <div class="relative">
                      <input type="number" id="remainingAmount" name="remainingAmount" readonly class="form-input w-full bg-gray-50" placeholder="المبلغ المتبقي">
                      <i class="fas fa-calculator absolute left-3 top-3 text-gray-400"></i>
                    </div>
                  </div>
                  <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">مصاريف أخرى</label>
                    <div class="relative">
                      <input type="number" id="otherExpenses" name="otherExpenses" class="form-input w-full" placeholder="مصاريف أخرى" min="0" step="0.01" onchange="calculateProfit()">
                      <i class="fas fa-receipt absolute left-3 top-3 text-gray-400"></i>
                    </div>
                  </div>
                  <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">الربح</label>
                    <div class="relative">
                      <input type="number" id="profit" name="profit" readonly class="form-input w-full bg-gray-50 text-green-600 font-bold" placeholder="الربح المتوقع">
                      <i class="fas fa-chart-line absolute left-3 top-3 text-gray-400"></i>
                    </div>
                  </div>
                </div>
              </div>

              <!-- معلومات الموظف وحالة الطلب -->
              <div class="form-section bg-white rounded-lg shadow-sm p-6 mb-6 border border-gray-200 hover:border-indigo-300 transition-all duration-300">
                <h3 class="form-section-title">
                  <i class="fas fa-user-shield text-purple-500"></i>
                  <span>معلومات الموظف وحالة الطلب</span>
                </h3>
                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                  <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">الموظف المسؤول <span class="text-red-500">*</span></label>
                    <div class="relative">
                      <input type="text" id="responsibleEmployee" name="responsibleEmployee" required class="form-input w-full" placeholder="أدخل اسم الموظف المسؤول">
                      <i class="fas fa-user-tie absolute left-3 top-3 text-gray-400"></i>
                    </div>
                  </div>
                  <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">حالة الطلب <span class="text-red-500">*</span></label>
                    <div class="relative">
                      <select id="orderStatus" name="orderStatus" required class="form-input w-full">
                        <option value="">اختر حالة الطلب</option>
                        <option value="قيد المعالجة" class="bg-blue-50 text-blue-700" selected>قيد المعالجة</option>
                        <option value="في الطريق" class="bg-yellow-50 text-yellow-700">في الطريق</option>
                        <option value="تم التسليم" class="bg-green-50 text-green-700">تم التسليم</option>
                        <option value="ملغى" class="bg-red-50 text-red-700">ملغى</option>
                      </select>
                      <i class="fas fa-tasks absolute left-3 top-3 text-gray-400"></i>
                    </div>
                  </div>
                </div>
              </div>

              <!-- أزرار الإجراءات -->
              <div class="flex flex-col md:flex-row gap-4 mt-6 sticky bottom-0 bg-white p-4 rounded-lg shadow-lg border border-gray-200 z-10">
                <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">
                <button type="submit" name="submit_order" class="btn btn-primary flex-1">
                  <i class="fas fa-paper-plane ml-2"></i>
                  <span>إرسال الطلب</span>
                </button>
                <button type="reset" class="btn btn-secondary flex-1">
                  <i class="fas fa-undo ml-2"></i>
                  <span>إعادة تعيين</span>
                </button>
              </div>
            </form>
          </div>
        </div>
      </div>
    </main>
  </div>

  <!-- تضمين مكتبة SweetAlert2 للتنبيهات الجميلة -->
  <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>

  <script>
    document.addEventListener('DOMContentLoaded', function() {
      // تهيئة القائمة الجانبية للأجهزة المحمولة
      const toggleSidebar = document.getElementById('toggleSidebar');
      const sidebar = document.querySelector('aside');
      const overlay = document.getElementById('overlay');

      if (toggleSidebar && sidebar && overlay) {
        toggleSidebar.addEventListener('click', function() {
          sidebar.classList.toggle('hidden');
          sidebar.classList.toggle('active');
          overlay.classList.toggle('hidden');
          overlay.classList.toggle('active');
          document.body.classList.toggle('overflow-hidden');
        });

        overlay.addEventListener('click', function() {
          sidebar.classList.add('hidden');
          sidebar.classList.remove('active');
          overlay.classList.add('hidden');
          overlay.classList.remove('active');
          document.body.classList.remove('overflow-hidden');
        });
      }

      // تفعيل القائمة المنسدلة
      const dropdownToggle = document.getElementById('reportsDropdown');
      const dropdownMenu = document.getElementById('reportsMenu');

      if (dropdownToggle && dropdownMenu) {
        dropdownToggle.addEventListener('click', function(e) {
          e.preventDefault();
          this.classList.toggle('active');
          dropdownMenu.classList.toggle('active');
        });
      }

      // إنشاء رقم طلب عند اختيار الفرع
      const branchSelect = document.getElementById('branch');
      if (branchSelect) {
        branchSelect.addEventListener('change', generateOrderNumber);
      }

      // تفعيل حساب تاريخ التسليم المتوقع عند تغيير مدة الوصول أو تاريخ رفع الطلب
      const deliveryDurationField = document.getElementById('deliveryDuration');
      const orderDateField = document.getElementById('orderDate');

      if (deliveryDurationField && orderDateField) {
        deliveryDurationField.addEventListener('change', calculateExpectedDeliveryDate);
        orderDateField.addEventListener('change', calculateExpectedDeliveryDate);
      }

      // تفعيل حساب الربح عند تغيير أي من الحقول المالية
      const financialFields = [
        'clientAmount', 'agentAmount', 'serviceFee', 'otherExpenses'
      ];

      financialFields.forEach(fieldId => {
        const field = document.getElementById(fieldId);
        if (field) {
          field.addEventListener('input', calculateProfit);
        }
      });

      // تفعيل حساب المبلغ المتبقي عند تغيير المبلغ المدفوع
      const paidAmountField = document.getElementById('paidAmount');
      if (paidAmountField) {
        paidAmountField.addEventListener('input', calculateRemaining);
      }

      // تغيير لون حقل حالة الطلب عند تغيير القيمة
      const orderStatusField = document.getElementById('orderStatus');
      if (orderStatusField) {
        // تطبيق اللون المناسب عند تغيير الحالة
        orderStatusField.addEventListener('change', function() {
          const selectedStatus = this.value;
          this.className = 'form-input w-full';

          // إضافة الألوان حسب الحالة المختارة
          if (selectedStatus === 'قيد المعالجة') {
            this.classList.add('bg-blue-50', 'text-blue-700', 'border-blue-300');
          } else if (selectedStatus === 'في الطريق') {
            this.classList.add('bg-yellow-50', 'text-yellow-700', 'border-yellow-300');
          } else if (selectedStatus === 'تم التسليم') {
            this.classList.add('bg-green-50', 'text-green-700', 'border-green-300');
          } else if (selectedStatus === 'ملغى') {
            this.classList.add('bg-red-50', 'text-red-700', 'border-red-300');
          }
        });

        // تطبيق اللون المناسب عند تحميل الصفحة
        orderStatusField.dispatchEvent(new Event('change'));
      }

      // إرسال النموذج
      const shipmentForm = document.getElementById('shipmentForm');
      if (shipmentForm) {
        shipmentForm.addEventListener('submit', function(event) {
          if (!validateForm()) {
            event.preventDefault();
          }
        });
      }
    });

    /**
     * دالة لإنشاء رقم طلب جديد
     * تنشئ رقم طلب بناءً على رمز الفرع المحدد
     * تستخدم API للحصول على الرقم التسلسلي التالي من قاعدة البيانات
     * في حالة فشل الاتصال بالخادم، تستخدم دالة احتياطية لإنشاء رقم طلب
     */
    function generateOrderNumber() {
      // دالة احتياطية لإنشاء رقم طلب في حالة فشل الاتصال بالخادم
      function generateFallbackOrderNumber(branchCode) {
        // الحصول على التاريخ الحالي
        const now = new Date();
        const year = now.getFullYear().toString().substr(-2);
        const month = (now.getMonth() + 1).toString().padStart(2, '0');
        // إنشاء رقم عشوائي من ثلاثة أرقام
        const randomNum = Math.floor(Math.random() * 999 + 1).toString().padStart(3, '0');
        // تنسيق رقم الطلب: رمز الفرع + السنة + الشهر + رقم عشوائي
        return `${branchCode}${year}${month}-${randomNum}`;
      }
      const branchSelect = document.getElementById('branch');
      const orderNumberField = document.getElementById('orderNumber');

      if (!branchSelect || !orderNumberField) return;

      const selectedOption = branchSelect.options[branchSelect.selectedIndex];
      if (!selectedOption || !selectedOption.value) return;

      // الحصول على رمز الفرع
      const branchCode = selectedOption.getAttribute('data-code');
      if (!branchCode) return;

      // إظهار مؤشر التحميل
      orderNumberField.value = 'جاري إنشاء رقم الطلب...';
      orderNumberField.classList.add('bg-indigo-50');

      // تسجيل معلومات الفرع المحدد في وحدة التحكم للتصحيح
      console.log('الفرع المحدد:', selectedOption.value);
      console.log('رمز الفرع:', branchCode);

      // استخدام طلب AJAX للحصول على رقم الطلب من الخادم
      // إضافة معلمة reserve=true لحجز الرقم التسلسلي مباشرة
      // استخدام الملف في المجلد الرئيسي
      console.log('جاري الاتصال بـ API:', `get_next_sequence.php?branch_code=${branchCode}&reserve=true`);

      fetch(`get_next_sequence.php?branch_code=${branchCode}&reserve=true`, {
        method: 'GET',
        headers: {
          'Cache-Control': 'no-cache, no-store, must-revalidate',
          'Pragma': 'no-cache',
          'Expires': '0'
        }
      })
        .then(response => {
          console.log('استجابة HTTP:', response.status, response.statusText);
          if (!response.ok) {
            throw new Error(`خطأ في الاستجابة: ${response.status} ${response.statusText}`);
          }
          return response.json();
        })
        .then(data => {
          // عرض معلومات التصحيح في وحدة التحكم
          console.log('استجابة API:', data);

          if (data.status === 'success') {
            // تعيين رقم الطلب من الاستجابة
            orderNumberField.value = data.order_number;

            // عرض معلومات إضافية في وحدة التحكم
            console.log('رقم الطلب:', data.order_number);
            console.log('رمز الفرع:', data.branch_code);
            console.log('رمز الفرع الأساسي:', data.branch_code_base);
            console.log('الرقم التسلسلي:', data.sequence_number);
            console.log('الرقم التسلسلي المنسق:', data.formatted_sequence);

            // إضافة تأثير بصري لتأكيد إنشاء الرقم
            orderNumberField.classList.remove('bg-indigo-50');
            orderNumberField.classList.add('bg-green-50', 'border-green-300');
            setTimeout(() => {
              orderNumberField.classList.remove('bg-green-50', 'border-green-300');
            }, 1500);
          } else {
            // عرض رسالة خطأ
            orderNumberField.value = 'خطأ في إنشاء رقم الطلب';
            orderNumberField.classList.remove('bg-indigo-50');
            orderNumberField.classList.add('bg-red-50', 'border-red-300');
            console.error('خطأ في إنشاء رقم الطلب:', data.message);

            // إظهار رسالة خطأ للمستخدم
            Swal.fire({
              title: 'خطأ في إنشاء رقم الطلب',
              text: data.message || 'حدث خطأ أثناء إنشاء رقم الطلب. يرجى المحاولة مرة أخرى.',
              icon: 'error',
              confirmButtonText: 'حسناً'
            });

            // إعادة المحاولة بعد 3 ثوانٍ
            setTimeout(() => {
              orderNumberField.classList.remove('bg-red-50', 'border-red-300');
              orderNumberField.value = '';
            }, 3000);
          }
        })
        .catch(error => {
          // عرض رسالة خطأ في حالة فشل الطلب
          console.error('خطأ في الاتصال بالخادم:', error.message);

          // عرض معلومات إضافية في وحدة التحكم
          console.error('رمز الفرع المرسل:', branchCode);
          console.error('URL الطلب:', `get_next_sequence.php?branch_code=${branchCode}&reserve=true`);

          // إظهار رسالة خطأ للمستخدم
          Swal.fire({
            title: 'خطأ في الاتصال بالخادم',
            text: 'تعذر الاتصال بالخادم لإنشاء رقم الطلب. سيتم استخدام رقم طلب احتياطي.',
            icon: 'warning',
            confirmButtonText: 'متابعة'
          });

          // استخدام رقم طلب بتنسيق صحيح
          const now = new Date();
          const year = '25'; // استخدام 25 بدلاً من السنة الحالية
          const sequenceNum = Math.floor(Math.random() * 99) + 1;
          const formattedSequence = sequenceNum.toString().padStart(2, '0');
          const fallbackOrderNumber = `${branchCode}${year}-${formattedSequence}`;

          console.log('استخدام رقم طلب احتياطي:', fallbackOrderNumber);
          orderNumberField.value = fallbackOrderNumber;
          orderNumberField.classList.remove('bg-indigo-50', 'bg-red-50', 'border-red-300');
          orderNumberField.classList.add('bg-yellow-50', 'border-yellow-300');

          // إضافة علامة R في نهاية الرقم للإشارة إلى أنه رقم احتياطي
          setTimeout(() => {
            orderNumberField.value = fallbackOrderNumber + 'R';
            orderNumberField.classList.remove('bg-yellow-50', 'border-yellow-300');
          }, 1500);

          // إزالة تنسيق الخطأ بعد 3 ثوانٍ
          setTimeout(() => {
            orderNumberField.classList.remove('bg-red-50', 'border-red-300');
            // لا نقوم بإعادة تعيين قيمة حقل رقم الطلب لأننا نستخدم الرقم الاحتياطي
          }, 3000);
        });
    }

    /**
     * دالة لحساب الربح تلقائياً
     * تحسب الربح بناءً على المبلغ المتفق مع العميل والمبلغ المتفق مع الوكيل ورسوم الخدمة والمصاريف الأخرى
     */
    function calculateProfit() {
      const clientAmount = parseFloat(document.getElementById('clientAmount').value) || 0;
      const agentAmount = parseFloat(document.getElementById('agentAmount').value) || 0;
      const serviceFee = parseFloat(document.getElementById('serviceFee').value) || 0;
      const otherExpenses = parseFloat(document.getElementById('otherExpenses').value) || 0;

      // حساب الربح: المبلغ المتفق مع العميل - المبلغ المتفق مع الوكيل - المصاريف الأخرى + رسوم الخدمة
      const profit = clientAmount - agentAmount - otherExpenses + serviceFee;

      // عرض الربح في الحقل المخصص
      const profitField = document.getElementById('profit');
      profitField.value = profit.toFixed(2);

      // تغيير لون الربح بناءً على قيمته (أخضر للربح، أحمر للخسارة)
      if (profit > 0) {
        profitField.classList.remove('text-red-600');
        profitField.classList.add('text-green-600');
      } else if (profit < 0) {
        profitField.classList.remove('text-green-600');
        profitField.classList.add('text-red-600');
      }

      // حساب المبلغ المتبقي أيضاً في حالة تم إدخال المبلغ المدفوع
      calculateRemaining();
    }

    /**
     * دالة لحساب المبلغ المتبقي
     * تحسب المبلغ المتبقي بناءً على المبلغ المتفق مع العميل والمبلغ المدفوع
     */
    function calculateRemaining() {
      const clientAmount = parseFloat(document.getElementById('clientAmount').value) || 0;
      const paidAmount = parseFloat(document.getElementById('paidAmount').value) || 0;

      // حساب المبلغ المتبقي
      const remainingAmount = clientAmount - paidAmount;

      // عرض المبلغ المتبقي في الحقل المخصص
      document.getElementById('remainingAmount').value = remainingAmount.toFixed(2);
    }

    /**
     * دالة لحساب تاريخ التسليم المتوقع
     * تحسب تاريخ التسليم المتوقع بناءً على تاريخ رفع الطلب ومدة الوصول
     */
    function calculateExpectedDeliveryDate() {
      const orderDateField = document.getElementById('orderDate');
      const deliveryDurationField = document.getElementById('deliveryDuration');
      const expectedDeliveryDateField = document.getElementById('expectedDeliveryDate');

      if (orderDateField.value && deliveryDurationField.value) {
        const orderDate = new Date(orderDateField.value);
        const deliveryDuration = parseInt(deliveryDurationField.value);

        // إضافة مدة الوصول بالأيام إلى تاريخ رفع الطلب
        orderDate.setDate(orderDate.getDate() + deliveryDuration);

        // تنسيق التاريخ بصيغة YYYY-MM-DD
        const year = orderDate.getFullYear();
        const month = String(orderDate.getMonth() + 1).padStart(2, '0');
        const day = String(orderDate.getDate()).padStart(2, '0');

        // عرض تاريخ التسليم المتوقع في الحقل المخصص
        expectedDeliveryDateField.value = `${year}-${month}-${day}`;
      }
    }

    /**
     * دالة للتحقق من صحة النموذج
     * تتحقق من إدخال جميع الحقول المطلوبة
     */
    function validateForm() {
      const requiredFields = document.querySelectorAll('input[required], select[required], textarea[required]');
      let isValid = true;

      requiredFields.forEach(field => {
        if (!field.value.trim()) {
          field.classList.add('border-red-500');
          if (!field.nextElementSibling || !field.nextElementSibling.classList.contains('error-message')) {
            const errorMsg = document.createElement('p');
            errorMsg.className = 'error-message text-red-500 text-xs mt-1';
            errorMsg.textContent = 'هذا الحقل مطلوب';
            field.parentNode.insertBefore(errorMsg, field.nextSibling);
          }
          isValid = false;
        } else {
          field.classList.remove('border-red-500');
          const errorMsg = field.nextElementSibling;
          if (errorMsg && errorMsg.classList.contains('error-message')) {
            errorMsg.remove();
          }
        }
      });

      if (!isValid) {
        Swal.fire({
          title: 'خطأ في النموذج',
          text: 'يرجى ملء جميع الحقول المطلوبة',
          icon: 'error',
          confirmButtonText: 'حسناً'
        });
      }

      return isValid;
    }

    // معالجة نقر زر "حل المشكلة"
    $(document).ready(function() {
      // معالجة نقر زر "حل المشكلة"
      $(document).on('click', '#fixConnectionBtn', function() {
        // إظهار مؤشر التحميل
        Swal.fire({
          title: 'جاري إصلاح المشكلة...',
          text: 'يرجى الانتظار قليلاً',
          allowOutsideClick: false,
          didOpen: () => {
            Swal.showLoading();
          }
        });

        // إرسال طلب AJAX لاختبار الاتصال بالخادم
        $.ajax({
          url: 'test_connection.php',
          type: 'GET',
          dataType: 'json',
          success: function(response) {
            if (response.status === 'success') {
              // إظهار رسالة نجاح
              Swal.fire({
                title: 'تم بنجاح!',
                text: 'تم إصلاح المشكلة بنجاح. يمكنك الآن إنشاء الطلب.',
                icon: 'success',
                confirmButtonText: 'حسناً'
              }).then(() => {
                // إعادة تحميل الصفحة
                location.reload();
              });
            } else {
              // إظهار رسالة خطأ
              Swal.fire({
                title: 'خطأ!',
                text: 'لم يتم إصلاح المشكلة. يرجى الاتصال بمسؤول النظام.',
                icon: 'error',
                confirmButtonText: 'حسناً'
              });
            }
          },
          error: function() {
            // إظهار رسالة خطأ
            Swal.fire({
              title: 'خطأ!',
              text: 'حدث خطأ أثناء محاولة إصلاح المشكلة. يرجى المحاولة مرة أخرى.',
              icon: 'error',
              confirmButtonText: 'حسناً'
            });
          }
        });
      });
    });
  </script>
</body>

</html>