<?php
// ملف لنقل الطلبات من localStorage إلى قاعدة البيانات
// يمكن استخدام هذا الملف في لوحة تحكم cPanel لنقل الطلبات

// تضمين ملف تكوين قاعدة البيانات
require_once '../config/db_config.php';

// إنشاء اتصال PDO
try {
    echo "<!DOCTYPE html>
    <html dir='rtl' lang='ar'>
    <head>
        <meta charset='UTF-8'>
        <meta name='viewport' content='width=device-width, initial-scale=1.0'>
        <title>نقل الطلبات من localStorage إلى قاعدة البيانات</title>
        <style>
            body {
                font-family: 'Tajawal', sans-serif;
                margin: 20px;
                line-height: 1.6;
            }
            h1, h2 {
                color: #0a2056;
            }
            .container {
                max-width: 1200px;
                margin: 0 auto;
                padding: 20px;
                background-color: #f8fafc;
                border-radius: 10px;
                box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
            }
            .success {
                color: green;
                background-color: #e6f7e6;
                padding: 10px;
                border-radius: 5px;
                margin: 10px 0;
            }
            .error {
                color: red;
                background-color: #fee2e2;
                padding: 10px;
                border-radius: 5px;
                margin: 10px 0;
            }
            table {
                width: 100%;
                border-collapse: collapse;
                margin: 20px 0;
            }
            th, td {
                border: 1px solid #ddd;
                padding: 8px;
                text-align: right;
            }
            th {
                background-color: #f2f2f2;
            }
            tr:nth-child(even) {
                background-color: #f9f9f9;
            }
            button, input[type='submit'] {
                background-color: #4f46e5;
                color: white;
                border: none;
                padding: 10px 15px;
                border-radius: 5px;
                cursor: pointer;
                margin: 5px;
            }
            button:hover, input[type='submit']:hover {
                background-color: #4338ca;
            }
            textarea {
                width: 100%;
                padding: 10px;
                border-radius: 5px;
                border: 1px solid #ddd;
                margin-bottom: 10px;
            }
        </style>
    </head>
    <body>
        <div class='container'>
            <h1>نقل الطلبات من localStorage إلى قاعدة البيانات</h1>";
    
    // التحقق من وجود جدول الطلبات
    $stmt = $pdo->query("SHOW TABLES LIKE 'orders'");
    $tableExists = $stmt->rowCount() > 0;
    
    if (!$tableExists) {
        echo "<div class='error'>جدول الطلبات غير موجود. يرجى إنشاء الجدول أولاً.</div>";
        echo "<p>يمكنك استخدام ملف create_orders_table.sql لإنشاء جدول الطلبات.</p>";
        echo "</div></body></html>";
        exit;
    }
    
    // نموذج لنقل الطلبات من localStorage
    echo "<h2>نقل الطلبات من localStorage</h2>
          <p>استخدم هذا النموذج لنقل الطلبات من localStorage إلى قاعدة البيانات.</p>
          <p>للحصول على بيانات الطلبات من localStorage، افتح وحدة تحكم المتصفح (F12) وقم بتنفيذ الأمر التالي:</p>
          <pre>localStorage.getItem('shippingOrders')</pre>
          <p>ثم انسخ النتيجة والصقها في المربع أدناه:</p>
          
          <form method='post' action=''>
              <textarea name='orders_data' rows='10' placeholder='ضع بيانات الطلبات من localStorage هنا (بتنسيق JSON)'></textarea><br>
              <input type='submit' name='migrate_orders' value='نقل الطلبات'>
          </form>";
    
    // معالجة نقل الطلبات من localStorage
    if (isset($_POST['migrate_orders']) && !empty($_POST['orders_data'])) {
        try {
            $ordersData = json_decode($_POST['orders_data'], true);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                throw new Exception("بيانات JSON غير صالحة: " . json_last_error_msg());
            }
            
            if (!is_array($ordersData)) {
                throw new Exception("البيانات المدخلة ليست مصفوفة من الطلبات.");
            }
            
            $successCount = 0;
            $errorCount = 0;
            $errors = [];
            
            // بدء المعاملة
            $pdo->beginTransaction();
            
            foreach ($ordersData as $orderData) {
                try {
                    // التحقق من وجود البيانات الأساسية
                    if (empty($orderData['orderNumber']) || empty($orderData['clientName'])) {
                        throw new Exception("بيانات الطلب غير مكتملة. يجب توفير رقم الطلب واسم العميل على الأقل.");
                    }
                    
                    // التحقق من وجود الطلب مسبقاً
                    $stmt = $pdo->prepare("SELECT id FROM orders WHERE order_number = ?");
                    $stmt->execute([$orderData['orderNumber']]);
                    if ($stmt->rowCount() > 0) {
                        throw new Exception("الطلب رقم {$orderData['orderNumber']} موجود مسبقاً في قاعدة البيانات.");
                    }
                    
                    // البحث عن معرف الفرع
                    $branchId = null;
                    if (!empty($orderData['branch'])) {
                        $stmt = $pdo->prepare("SELECT id FROM branches WHERE name = ?");
                        $stmt->execute([$orderData['branch']]);
                        $branch = $stmt->fetch();
                        if ($branch) {
                            $branchId = $branch['id'];
                        }
                    }
                    
                    // تحويل التواريخ إلى تنسيق قاعدة البيانات
                    $orderUploadDate = !empty($orderData['orderUploadDate']) ? date('Y-m-d', strtotime($orderData['orderUploadDate'])) : null;
                    $scheduledDeliveryDate = !empty($orderData['scheduledDeliveryDate']) ? date('Y-m-d', strtotime($orderData['scheduledDeliveryDate'])) : null;
                    
                    // إعداد استعلام الإدراج
                    $stmt = $pdo->prepare("
                        INSERT INTO orders (
                            order_number, client_name, service_type, branch_id, 
                            departure_area, second_destination, delivery_area, goods_name, goods_type, 
                            country_of_origin, weight, quantity, unit, service_fees, 
                            currency, payment_type, delivery_duration, amount_paid, remaining_amount, 
                            order_upload_date, scheduled_delivery_date, recipient_name, phone_number, delivery_company, 
                            employee_name, notes, amount_agreed_with_customer, amount_agreed_with_agent, other_expenses, 
                            profit, payment_status, status
                        ) VALUES (
                            :order_number, :client_name, :service_type, :branch_id, 
                            :departure_area, :second_destination, :delivery_area, :goods_name, :goods_type, 
                            :country_of_origin, :weight, :quantity, :unit, :service_fees, 
                            :currency, :payment_type, :delivery_duration, :amount_paid, :remaining_amount, 
                            :order_upload_date, :scheduled_delivery_date, :recipient_name, :phone_number, :delivery_company, 
                            :employee_name, :notes, :amount_agreed_with_customer, :amount_agreed_with_agent, :other_expenses, 
                            :profit, :payment_status, :status
                        )
                    ");
                    
                    // تنفيذ الاستعلام مع البيانات
                    $stmt->execute([
                        ':order_number' => $orderData['orderNumber'],
                        ':client_name' => $orderData['clientName'],
                        ':service_type' => $orderData['serviceType'] ?? null,
                        ':branch_id' => $branchId,
                        ':departure_area' => $orderData['departureArea'] ?? null,
                        ':second_destination' => $orderData['secondDestination'] ?? null,
                        ':delivery_area' => $orderData['deliveryArea'] ?? null,
                        ':goods_name' => $orderData['goodsName'] ?? null,
                        ':goods_type' => $orderData['goodsType'] ?? null,
                        ':country_of_origin' => $orderData['countryOfOrigin'] ?? null,
                        ':weight' => $orderData['weight'] ?? null,
                        ':quantity' => $orderData['quantity'] ?? null,
                        ':unit' => $orderData['unit'] ?? null,
                        ':service_fees' => $orderData['serviceFees'] ?? null,
                        ':currency' => $orderData['currency'] ?? null,
                        ':payment_type' => $orderData['paymentType'] ?? null,
                        ':delivery_duration' => $orderData['deliveryDuration'] ?? null,
                        ':amount_paid' => $orderData['amountPaid'] ?? null,
                        ':remaining_amount' => $orderData['remainingAmount'] ?? null,
                        ':order_upload_date' => $orderUploadDate,
                        ':scheduled_delivery_date' => $scheduledDeliveryDate,
                        ':recipient_name' => $orderData['recipientName'] ?? null,
                        ':phone_number' => $orderData['phoneNumber'] ?? null,
                        ':delivery_company' => $orderData['deliveryCompany'] ?? null,
                        ':employee_name' => $orderData['employeeName'] ?? null,
                        ':notes' => $orderData['notes'] ?? null,
                        ':amount_agreed_with_customer' => $orderData['amountAgreedWithCustomer'] ?? null,
                        ':amount_agreed_with_agent' => $orderData['amountAgreedWithAgent'] ?? null,
                        ':other_expenses' => $orderData['otherExpenses'] ?? null,
                        ':profit' => $orderData['profit'] ?? null,
                        ':payment_status' => $orderData['paymentStatus'] ?? 'غير مدفوع',
                        ':status' => $orderData['status'] ?? 'قيد المعالجة'
                    ]);
                    
                    $successCount++;
                } catch (Exception $e) {
                    $errorCount++;
                    $errors[] = "خطأ في الطلب رقم {$orderData['orderNumber']}: " . $e->getMessage();
                }
            }
            
            // إتمام المعاملة
            $pdo->commit();
            
            echo "<div class='success'>تم نقل $successCount طلب بنجاح!</div>";
            
            if ($errorCount > 0) {
                echo "<div class='error'>فشل نقل $errorCount طلب:</div>";
                echo "<ul>";
                foreach ($errors as $error) {
                    echo "<li>$error</li>";
                }
                echo "</ul>";
            }
            
        } catch (Exception $e) {
            // التراجع عن المعاملة في حالة حدوث خطأ
            $pdo->rollBack();
            echo "<div class='error'>خطأ: " . $e->getMessage() . "</div>";
        }
    }
    
    // عرض الطلبات الحالية في قاعدة البيانات
    $stmt = $pdo->query("
        SELECT o.id, o.order_number, o.client_name, b.name as branch_name, 
               o.departure_area, o.delivery_area, o.status, o.payment_status, 
               o.created_at 
        FROM orders o 
        LEFT JOIN branches b ON o.branch_id = b.id 
        ORDER BY o.created_at DESC 
        LIMIT 20
    ");
    $orders = $stmt->fetchAll();
    
    echo "<h2>الطلبات الحالية في قاعدة البيانات (آخر 20 طلب)</h2>";
    
    if (count($orders) > 0) {
        echo "<table>
                <tr>
                    <th>#</th>
                    <th>رقم الطلب</th>
                    <th>اسم العميل</th>
                    <th>الفرع</th>
                    <th>من</th>
                    <th>إلى</th>
                    <th>حالة الطلب</th>
                    <th>حالة الدفع</th>
                    <th>تاريخ الإنشاء</th>
                </tr>";
        
        foreach ($orders as $order) {
            echo "<tr>
                    <td>{$order['id']}</td>
                    <td>{$order['order_number']}</td>
                    <td>{$order['client_name']}</td>
                    <td>{$order['branch_name'] ?: 'غير محدد'}</td>
                    <td>{$order['departure_area']}</td>
                    <td>{$order['delivery_area']}</td>
                    <td>{$order['status']}</td>
                    <td>{$order['payment_status']}</td>
                    <td>{$order['created_at']}</td>
                </tr>";
        }
        
        echo "</table>";
    } else {
        echo "<p>لا توجد طلبات في قاعدة البيانات.</p>";
    }
    
    echo "</div>
    
    <script>
        // سكريبت للحصول على بيانات الطلبات من localStorage
        document.addEventListener('DOMContentLoaded', function() {
            const getOrdersButton = document.createElement('button');
            getOrdersButton.textContent = 'الحصول على بيانات الطلبات من localStorage';
            getOrdersButton.addEventListener('click', function() {
                const ordersData = localStorage.getItem('shippingOrders');
                if (ordersData) {
                    document.querySelector('textarea[name=\"orders_data\"]').value = ordersData;
                    alert('تم استرداد بيانات الطلبات بنجاح!');
                } else {
                    alert('لا توجد بيانات طلبات في localStorage.');
                }
            });
            
            const form = document.querySelector('form');
            form.insertBefore(getOrdersButton, form.firstChild);
        });
    </script>
    
    </body>
    </html>";
    
} catch (PDOException $e) {
    // في حالة فشل الاتصال
    die("<!DOCTYPE html>
    <html dir='rtl' lang='ar'>
    <head>
        <meta charset='UTF-8'>
        <meta name='viewport' content='width=device-width, initial-scale=1.0'>
        <title>خطأ في الاتصال بقاعدة البيانات</title>
        <style>
            body {
                font-family: 'Tajawal', sans-serif;
                margin: 20px;
                line-height: 1.6;
            }
            .error {
                color: red;
                background-color: #fee2e2;
                padding: 10px;
                border-radius: 5px;
                margin: 10px 0;
            }
        </style>
    </head>
    <body>
        <div class='error'>
            <h1>فشل الاتصال بقاعدة البيانات</h1>
            <p>" . $e->getMessage() . "</p>
        </div>
    </body>
    </html>");
}
?>
