-- ملف إنشاء جداول إعدادات النظام والصلاحيات
-- تاريخ الإنشاء: 2023-07-15

-- استخدام قاعدة البيانات
USE `shipping_db`;

-- التأكد من وجود جدول إعدادات النظام
CREATE TABLE IF NOT EXISTS `system_settings` (
  `id` INT AUTO_INCREMENT PRIMARY KEY,
  `setting_key` VARCHAR(50) NOT NULL UNIQUE COMMENT 'مفتاح الإعداد',
  `setting_value` TEXT DEFAULT NULL COMMENT 'قيمة الإعداد',
  `setting_group` VARCHAR(50) DEFAULT 'general' COMMENT 'مجموعة الإعداد',
  `description` VARCHAR(255) DEFAULT NULL COMMENT 'وصف الإعداد',
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP COMMENT 'تاريخ الإنشاء',
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP COMMENT 'تاريخ التحديث'
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- إنشاء جدول الأدوار
CREATE TABLE IF NOT EXISTS `roles` (
  `id` INT AUTO_INCREMENT PRIMARY KEY,
  `name` VARCHAR(50) NOT NULL UNIQUE COMMENT 'اسم الدور',
  `display_name` VARCHAR(100) NOT NULL COMMENT 'الاسم المعروض',
  `description` TEXT DEFAULT NULL COMMENT 'وصف الدور',
  `is_system` TINYINT(1) DEFAULT 0 COMMENT 'هل هو دور نظام',
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP COMMENT 'تاريخ الإنشاء',
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP COMMENT 'تاريخ التحديث'
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- إنشاء جدول الصلاحيات
CREATE TABLE IF NOT EXISTS `permissions` (
  `id` INT AUTO_INCREMENT PRIMARY KEY,
  `name` VARCHAR(50) NOT NULL UNIQUE COMMENT 'اسم الصلاحية',
  `display_name` VARCHAR(100) NOT NULL COMMENT 'الاسم المعروض',
  `description` TEXT DEFAULT NULL COMMENT 'وصف الصلاحية',
  `category` VARCHAR(50) DEFAULT 'general' COMMENT 'فئة الصلاحية',
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP COMMENT 'تاريخ الإنشاء',
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP COMMENT 'تاريخ التحديث'
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- إنشاء جدول العلاقة بين الأدوار والصلاحيات
CREATE TABLE IF NOT EXISTS `role_permissions` (
  `id` INT AUTO_INCREMENT PRIMARY KEY,
  `role_id` INT NOT NULL COMMENT 'معرف الدور',
  `permission_id` INT NOT NULL COMMENT 'معرف الصلاحية',
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP COMMENT 'تاريخ الإنشاء',
  UNIQUE KEY `role_permission_unique` (`role_id`, `permission_id`),
  FOREIGN KEY (`role_id`) REFERENCES `roles`(`id`) ON DELETE CASCADE,
  FOREIGN KEY (`permission_id`) REFERENCES `permissions`(`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- إنشاء جدول وحدات النظام
CREATE TABLE IF NOT EXISTS `system_modules` (
  `id` INT AUTO_INCREMENT PRIMARY KEY,
  `name` VARCHAR(50) NOT NULL UNIQUE COMMENT 'اسم الوحدة',
  `display_name` VARCHAR(100) NOT NULL COMMENT 'الاسم المعروض',
  `description` TEXT DEFAULT NULL COMMENT 'وصف الوحدة',
  `status` ENUM('enabled', 'disabled') DEFAULT 'enabled' COMMENT 'حالة الوحدة',
  `order` INT DEFAULT 0 COMMENT 'ترتيب الوحدة',
  `icon` VARCHAR(50) DEFAULT NULL COMMENT 'أيقونة الوحدة',
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP COMMENT 'تاريخ الإنشاء',
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP COMMENT 'تاريخ التحديث'
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- إدراج الأدوار الافتراضية
INSERT IGNORE INTO `roles` (`name`, `display_name`, `description`, `is_system`) VALUES
('admin', 'مدير النظام', 'يمتلك جميع الصلاحيات في النظام', 1),
('manager', 'مدير', 'يمتلك صلاحيات إدارية محدودة', 1),
('employee', 'موظف', 'يمتلك صلاحيات أساسية للعمل', 1),
('client', 'عميل', 'يمتلك صلاحيات محدودة للعملاء', 1);

-- إدراج الصلاحيات الافتراضية
INSERT IGNORE INTO `permissions` (`name`, `display_name`, `description`, `category`) VALUES
-- صلاحيات لوحة التحكم
('view_dashboard', 'عرض لوحة التحكم', 'السماح بعرض لوحة التحكم الرئيسية', 'dashboard'),

-- صلاحيات الطلبات
('create_orders', 'إنشاء طلبات', 'السماح بإنشاء طلبات جديدة', 'orders'),
('manage_orders', 'إدارة الطلبات', 'السماح بإدارة جميع الطلبات (تعديل، حذف، إلخ)', 'orders'),
('view_orders', 'عرض الطلبات', 'السماح بعرض الطلبات فقط', 'orders'),
('delete_orders', 'حذف الطلبات', 'السماح بحذف الطلبات', 'orders'),
('approve_orders', 'الموافقة على الطلبات', 'السماح بالموافقة على الطلبات', 'orders'),
('reject_orders', 'رفض الطلبات', 'السماح برفض الطلبات', 'orders'),

-- صلاحيات المستخدمين
('manage_users', 'إدارة المستخدمين', 'السماح بإدارة المستخدمين', 'users'),
('view_users', 'عرض المستخدمين', 'السماح بعرض المستخدمين', 'users'),

-- صلاحيات الفروع
('manage_branches', 'إدارة الفروع', 'السماح بإدارة الفروع', 'branches'),
('view_branches', 'عرض الفروع', 'السماح بعرض الفروع', 'branches'),

-- صلاحيات الإعدادات
('manage_settings', 'إدارة الإعدادات', 'السماح بتعديل إعدادات النظام', 'settings'),
('manage_roles', 'إدارة الأدوار', 'السماح بإدارة الأدوار والصلاحيات', 'settings'),
('manage_system', 'إدارة النظام', 'السماح بإدارة إعدادات النظام المتقدمة', 'settings'),

-- صلاحيات التقارير
('view_reports', 'عرض التقارير', 'السماح بعرض التقارير', 'reports'),
('view_statistics', 'عرض الإحصائيات', 'السماح بعرض الإحصائيات', 'reports'),

-- صلاحيات البيانات
('export_data', 'تصدير البيانات', 'السماح بتصدير البيانات', 'data'),
('import_data', 'استيراد البيانات', 'السماح باستيراد البيانات', 'data'),
('manage_backup', 'إدارة النسخ الاحتياطي', 'السماح بإدارة النسخ الاحتياطي', 'data'),
('view_logs', 'عرض السجلات', 'السماح بعرض سجلات النظام', 'data');

-- إدراج صلاحيات الدور: مدير النظام (admin)
INSERT IGNORE INTO `role_permissions` (`role_id`, `permission_id`)
SELECT r.id, p.id FROM `roles` r, `permissions` p
WHERE r.name = 'admin';

-- إدراج صلاحيات الدور: مدير (manager)
INSERT IGNORE INTO `role_permissions` (`role_id`, `permission_id`)
SELECT r.id, p.id FROM `roles` r, `permissions` p
WHERE r.name = 'manager' AND p.name IN (
  'view_dashboard', 'create_orders', 'manage_orders', 'view_orders', 'approve_orders', 'reject_orders',
  'view_users', 'view_branches', 'view_reports', 'view_statistics', 'export_data'
);

-- إدراج صلاحيات الدور: موظف (employee)
INSERT IGNORE INTO `role_permissions` (`role_id`, `permission_id`)
SELECT r.id, p.id FROM `roles` r, `permissions` p
WHERE r.name = 'employee' AND p.name IN (
  'view_dashboard', 'create_orders', 'view_orders', 'view_branches', 'view_reports', 'export_data'
);

-- إدراج صلاحيات الدور: عميل (client)
INSERT IGNORE INTO `role_permissions` (`role_id`, `permission_id`)
SELECT r.id, p.id FROM `roles` r, `permissions` p
WHERE r.name = 'client' AND p.name IN (
  'view_dashboard', 'create_orders', 'view_orders'
);

-- إدراج وحدات النظام الافتراضية
INSERT IGNORE INTO `system_modules` (`name`, `display_name`, `description`, `status`, `order`, `icon`) VALUES
('dashboard', 'لوحة التحكم', 'لوحة التحكم الرئيسية للنظام', 'enabled', 1, 'fa-tachometer-alt'),
('orders', 'الطلبات', 'إدارة طلبات الشحن', 'enabled', 2, 'fa-box'),
('users', 'المستخدمين', 'إدارة المستخدمين والصلاحيات', 'enabled', 3, 'fa-users'),
('branches', 'الفروع', 'إدارة فروع الشركة', 'enabled', 4, 'fa-code-branch'),
('reports', 'التقارير', 'تقارير وإحصائيات النظام', 'enabled', 5, 'fa-chart-bar'),
('settings', 'الإعدادات', 'إعدادات النظام المختلفة', 'enabled', 6, 'fa-cog');

-- إدراج الإعدادات الافتراضية
INSERT IGNORE INTO `system_settings` (`setting_key`, `setting_value`, `setting_group`, `description`) VALUES
-- الإعدادات العامة
('system_name', 'نظام طلبات الشحن', 'general', 'اسم النظام'),
('company_name', 'شركة الشحن', 'general', 'اسم الشركة'),
('admin_email', 'admin@example.com', 'general', 'البريد الإلكتروني للمدير'),
('support_email', 'support@example.com', 'general', 'البريد الإلكتروني للدعم'),
('phone', '0112345678', 'general', 'رقم الهاتف'),
('address', 'شارع الملك فهد، الرياض', 'general', 'العنوان'),
('currency', 'ريال سعودي', 'general', 'العملة'),
('timezone', 'Asia/Riyadh', 'general', 'المنطقة الزمنية'),
('date_format', 'd/m/Y', 'general', 'تنسيق التاريخ'),
('time_format', 'H:i', 'general', 'تنسيق الوقت'),

-- إعدادات المظهر
('theme', 'default', 'appearance', 'السمة'),
('primary_color', '#4F46E5', 'appearance', 'اللون الرئيسي'),
('secondary_color', '#6B7280', 'appearance', 'اللون الثانوي'),
('success_color', '#10B981', 'appearance', 'لون النجاح'),
('danger_color', '#EF4444', 'appearance', 'لون الخطر'),
('warning_color', '#F59E0B', 'appearance', 'لون التحذير'),
('info_color', '#3B82F6', 'appearance', 'لون المعلومات'),
('logo', 'assets/images/backgrounds/background.png', 'appearance', 'شعار النظام'),
('favicon', 'assets/images/favicon.ico', 'appearance', 'أيقونة الموقع'),
('rtl_support', '1', 'appearance', 'دعم الواجهة من اليمين إلى اليسار'),
('show_footer', '1', 'appearance', 'عرض تذييل الصفحة'),

-- إعدادات الإشعارات
('email_notifications', '1', 'notifications', 'إشعارات البريد الإلكتروني'),
('browser_notifications', '1', 'notifications', 'إشعارات المتصفح'),
('order_created', '1', 'notifications', 'إشعار عند إنشاء طلب'),
('order_updated', '1', 'notifications', 'إشعار عند تحديث طلب'),
('order_completed', '1', 'notifications', 'إشعار عند اكتمال طلب'),
('order_cancelled', '1', 'notifications', 'إشعار عند إلغاء طلب'),
('user_registered', '1', 'notifications', 'إشعار عند تسجيل مستخدم جديد'),
('password_reset', '1', 'notifications', 'إشعار عند إعادة تعيين كلمة المرور'),
('system_updates', '1', 'notifications', 'إشعار عند توفر تحديثات للنظام'),
('backup_completed', '1', 'notifications', 'إشعار عند اكتمال النسخ الاحتياطي'),

-- إعدادات النسخ الاحتياطي
('auto_backup', 'weekly', 'backup', 'النسخ الاحتياطي التلقائي'),
('backup_retention', '30', 'backup', 'مدة الاحتفاظ بالنسخ الاحتياطية');
