<?php
/**
 * ملف تصدير التقارير
 * يستخدم هذا الملف لتصدير التقارير بتنسيق PDF أو Excel
 *
 * @version 1.0
 * @author فريق التطوير
 */

// بدء الجلسة
session_start();

// تضمين ملف التكوين
require_once 'config/config.php';

// تفعيل عرض الأخطاء للتصحيح
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// التحقق من حالة تسجيل الدخول
if (!isset($_SESSION['logged_in']) || $_SESSION['logged_in'] !== true) {
    // إعادة توجيه المستخدم إلى صفحة تسجيل الدخول
    header('Location: auth/login.php');
    exit;
}

// الحصول على معلومات المستخدم
$user = [
    'id' => $_SESSION['user_id'] ?? 0,
    'username' => $_SESSION['username'] ?? '',
    'full_name' => $_SESSION['full_name'] ?? 'المستخدم',
    'email' => $_SESSION['email'] ?? '',
    'role' => $_SESSION['role'] ?? 'user',
    'branch_id' => $_SESSION['branch_id'] ?? null
];

// تحديد ما إذا كان المستخدم مديرًا
$isAdmin = (isset($_SESSION['role']) && ($_SESSION['role'] === 'admin' || $_SESSION['role'] === 'manager'));

// التحقق من صلاحيات المستخدم للوصول إلى صفحة التقارير
if (!$isAdmin) {
    // إعادة توجيه المستخدم إلى لوحة التحكم
    header('Location: unified-dashboard.php');
    exit;
}

// الحصول على نوع التصدير
$exportType = $_GET['type'] ?? 'pdf';

// الحصول على نوع التقرير
$reportType = $_GET['report'] ?? 'weekly';

// الحصول على تاريخ بداية الأسبوع
$weekStart = $_GET['week_start'] ?? date('Y-m-d', strtotime('monday this week'));
$weekEnd = date('Y-m-d', strtotime($weekStart . ' +6 days'));

// الحصول على الشهر والسنة
$month = $_GET['month'] ?? date('m');
$year = $_GET['year'] ?? date('Y');

// الحصول على معرف الفرع
$branchId = $_GET['branch'] ?? 'all';

// الحصول على حالة الطلب
$status = $_GET['status'] ?? '';

// تحديد اسم الملف المصدر
$filename = '';
if ($reportType === 'weekly') {
    $filename = 'التقرير_الأسبوعي_' . $weekStart . '_' . $weekEnd;
} elseif ($reportType === 'monthly') {
    $filename = 'التقرير_الشهري_' . $month . '_' . $year;
} else {
    $filename = 'تقرير_' . date('Y-m-d');
}

// تنظيف اسم الملف
$filename = str_replace([' ', ':', '/', '\\'], '_', $filename);

// تحديد ما إذا كان التصدير للطلبات فقط
$ordersOnly = isset($_GET['orders_only']) && $_GET['orders_only'] === 'true';

// تحديد نوع المحتوى ورأس الملف
if ($exportType === 'pdf') {
    // تصدير بتنسيق PDF

    // إذا كان التقرير أسبوعي أو شهري، نقوم بتصدير قائمة الطلبات فقط
    if ($reportType === 'weekly' || $reportType === 'monthly') {
        $ordersOnly = true;
    }

    // التحقق من وجود مكتبة TCPDF
    if (file_exists('libs/tcpdf/tcpdf.php')) {
        header('Content-Type: application/pdf');
        header('Content-Disposition: attachment; filename="' . $filename . '.pdf"');

        // استدعاء دالة تصدير PDF
        exportToPDF($reportType, $weekStart, $weekEnd, $month, $year, $branchId, $status, $user, $ordersOnly);
    } else {
        // استخدام HTML كبديل لـ PDF
        header('Content-Type: text/html; charset=utf-8');

        // استدعاء دالة تصدير HTML
        exportToHTML($reportType, $weekStart, $weekEnd, $month, $year, $branchId, $status, $user, $filename, $ordersOnly);
    }
} elseif ($exportType === 'excel') {
    // تصدير بتنسيق Excel
    header('Content-Type: application/vnd.ms-excel');
    header('Content-Disposition: attachment; filename="' . $filename . '.xls"');

    // استدعاء دالة تصدير Excel
    exportToExcel($reportType, $weekStart, $weekEnd, $month, $year, $branchId, $status, $user, $ordersOnly);
} elseif ($exportType === 'orders_list') {
    // تصدير قائمة الطلبات فقط بتنسيق HTML
    header('Content-Type: text/html; charset=utf-8');

    // استدعاء دالة تصدير قائمة الطلبات فقط
    exportOrdersList($reportType, $weekStart, $weekEnd, $month, $year, $branchId, $status, $user);
} else {
    // نوع تصدير غير معروف
    echo 'نوع تصدير غير معروف';
    exit;
}

/**
 * دالة تصدير قائمة الطلبات فقط
 *
 * @param string $reportType نوع التقرير
 * @param string $weekStart تاريخ بداية الأسبوع
 * @param string $weekEnd تاريخ نهاية الأسبوع
 * @param string $month الشهر
 * @param string $year السنة
 * @param string $branchId معرف الفرع
 * @param string $status حالة الطلب
 * @param array $user معلومات المستخدم
 */
function exportOrdersList($reportType, $weekStart, $weekEnd, $month, $year, $branchId, $status, $user) {
    global $conn, $company_name, $company_logo;

    // بدء إخراج HTML
    echo '<!DOCTYPE html>
    <html dir="rtl">
    <head>
        <meta charset="UTF-8">
        <title>قائمة الطلبات ' . ($reportType === 'weekly' ? 'الأسبوعية' : 'الشهرية') . '</title>
        <style>
            @import url("https://fonts.googleapis.com/css2?family=Tajawal:wght@400;500;700&display=swap");

            @page {
                size: A4;
                margin: 1cm;
            }

            body {
                font-family: "Tajawal", Arial, sans-serif;
                direction: rtl;
                text-align: right;
                margin: 0;
                padding: 0;
                background-color: #f8f9fa;
                color: #333;
                font-size: 12pt;
            }

            .container {
                width: 21cm;
                min-height: 29.7cm;
                margin: 0 auto;
                background-color: #fff;
                padding: 1.5cm;
                box-shadow: 0 0 20px rgba(0, 0, 0, 0.1);
                position: relative;
                box-sizing: border-box;
            }

            .watermark {
                position: absolute;
                top: 50%;
                left: 50%;
                transform: translate(-50%, -50%) rotate(-45deg);
                opacity: 0.05;
                z-index: 0;
                width: 80%;
                height: auto;
            }

            .header {
                margin-bottom: 30px;
                padding-bottom: 20px;
                border-bottom: 2px solid #3b82f6;
            }

            .header-flex {
                display: flex;
                align-items: center;
                margin-bottom: 20px;
            }

            .logo {
                max-width: 120px;
                margin-left: 20px;
            }

            .header-text {
                flex: 1;
            }

            .company-info {
                color: #666;
                font-size: 10pt;
                margin-top: 5px;
            }

            .document-number {
                display: flex;
                justify-content: space-between;
                background-color: #f8f9fa;
                padding: 10px 15px;
                border-radius: 5px;
                font-size: 10pt;
                margin-top: 10px;
            }

            h1 {
                font-size: 24px;
                color: #1e40af;
                margin-bottom: 10px;
            }

            h2 {
                font-size: 18px;
                color: #1e40af;
                margin-bottom: 20px;
            }

            table {
                width: 100%;
                border-collapse: collapse;
                margin-bottom: 30px;
                box-shadow: 0 2px 5px rgba(0, 0, 0, 0.05);
            }

            th {
                background-color: #3b82f6;
                color: white;
                font-weight: 500;
                text-align: right;
                padding: 12px 15px;
            }

            td {
                padding: 10px 15px;
                border-bottom: 1px solid #e5e7eb;
            }

            tr:nth-child(even) {
                background-color: #f9fafb;
            }

            tr:hover {
                background-color: #f0f9ff;
            }

            .footer {
                text-align: center;
                margin-top: 40px;
                padding-top: 20px;
                border-top: 1px solid #e5e7eb;
                color: #6b7280;
                font-size: 10pt;
                position: relative;
            }

            .footer-stamp {
                display: flex;
                justify-content: space-between;
                margin-top: 30px;
                margin-bottom: 20px;
            }

            .stamp-section {
                text-align: center;
                width: 30%;
            }

            .stamp-title {
                font-weight: bold;
                margin-bottom: 5px;
            }

            .stamp-box {
                border: 1px dashed #ccc;
                height: 100px;
                display: flex;
                align-items: center;
                justify-content: center;
                margin-top: 10px;
                color: #999;
            }

            .page-number {
                position: absolute;
                bottom: 10px;
                left: 0;
                right: 0;
                text-align: center;
                font-size: 9pt;
                color: #999;
            }

            .print-btn {
                display: inline-block;
                background-color: #3b82f6;
                color: white;
                padding: 10px 20px;
                border-radius: 5px;
                text-decoration: none;
                margin-top: 20px;
                cursor: pointer;
                font-weight: 500;
            }

            .print-btn:hover {
                background-color: #2563eb;
            }

            .action-buttons {
                display: flex;
                justify-content: center;
                gap: 10px;
                margin-top: 20px;
            }

            @media print {
                @page {
                    size: A4;
                    margin: 1cm;
                }

                body {
                    background-color: white;
                    padding: 0;
                    margin: 0;
                    font-size: 12pt;
                }

                .container {
                    box-shadow: none;
                    padding: 1cm;
                    width: 100%;
                    height: 100%;
                    margin: 0;
                }

                .action-buttons {
                    display: none;
                }

                table {
                    page-break-inside: avoid;
                    border-collapse: collapse;
                }

                th, td {
                    border: 1px solid #000;
                }

                tr {
                    page-break-inside: avoid;
                }

                .watermark {
                    display: block !important;
                }

                .footer {
                    margin-top: 2cm;
                }

                .stamp-box {
                    border: 1px solid #000;
                }
            }
        </style>
        <script>
            function printReport() {
                window.print();
            }
        </script>
    </head>
    <body>
        <div class="container">
            <img src="assets/images/Company logo.png" alt="شعار الشركة" class="watermark">
            <div class="header">
                <div class="header-flex">
                    <img src="assets/images/Company logo.png" alt="شعار الشركة" class="logo">
                    <div class="header-text">
                        <h1>مجموعة ابراهيم الاحمدي اليمنية</h1>
                        <h2>قائمة الطلبات ' . ($reportType === 'weekly' ? 'الأسبوعية: ' . formatDateArabic($weekStart) . ' - ' . formatDateArabic($weekEnd) : 'الشهرية: ' . getMonthNameArabic(date('m', strtotime($month . '-01'))) . ' ' . date('Y', strtotime($month . '-01'))) . '</h2>
                        <p class="company-info">العنوان: اليمن، حضرموت، المكلا | هاتف: 05318281 | البريد الإلكتروني: ibrahim@al-ahmadigroup.com</p>
                    </div>
                </div>
                <div class="document-number">
                    <p>رقم المستند: <strong>' . ($reportType === 'weekly' ? 'WR-' . date('Ymd', strtotime($weekStart)) : 'MR-' . date('Ym', strtotime($month . '-01'))) . '</strong></p>
                    <p>تاريخ الإصدار: <strong>' . date('Y-m-d') . '</strong></p>
                </div>
            </div>';

    if ($reportType === 'weekly') {
        // الحصول على الطلبات الأسبوعية
        $weeklyOrders = getWeeklyOrders($conn, $weekStart, $weekEnd, $user['id'], $user['role'], $branchId, $status);

        // عرض قائمة الطلبات
        if (!empty($weeklyOrders)) {
            echo '<table>
                <thead>
                    <tr>
                        <th>رقم الطلب</th>
                        <th>العميل</th>
                        <th>نوع الخدمة</th>
                        <th>الفرع</th>
                        <th>تاريخ الطلب</th>
                        <th>المبلغ</th>
                        <th>الحالة</th>
                    </tr>
                </thead>
                <tbody>';

            foreach ($weeklyOrders as $order) {
                echo '<tr>
                    <td>' . $order['order_number'] . '</td>
                    <td>' . $order['client_name'] . '</td>
                    <td>' . $order['service_type'] . '</td>
                    <td>' . ($order['branch'] ?? 'غير محدد') . '</td>
                    <td>' . formatDateArabic($order['order_date']) . '</td>
                    <td>' . number_format(($order['paid_amount'] ?? 0) + ($order['remaining_amount'] ?? 0), 2) . ' ' . ($order['currency'] ?? 'ريال') . '</td>
                    <td>' . $order['status'] . '</td>
                </tr>';
            }

            echo '</tbody>
            </table>';
        } else {
            echo '<div style="text-align: center; padding: 20px; background-color: #f9fafb; border-radius: 8px; margin-bottom: 30px;">
                <p style="font-size: 16px; color: #6b7280;">لا توجد طلبات في هذا الأسبوع</p>
            </div>';
        }
    } elseif ($reportType === 'monthly') {
        // الحصول على الطلبات الشهرية
        $monthStart = $month . '-01';
        $monthEnd = date('Y-m-t', strtotime($monthStart));
        $monthlyOrders = getMonthlyOrders($conn, $monthStart, $monthEnd, $user['id'], $user['role'], $branchId, $status);

        // تحديث عنوان المستند
        echo '<script>
            document.querySelector(".document-number p:first-child strong").textContent = "MR-' . date('Ym', strtotime($monthStart)) . '";
        </script>';

        // عرض قائمة الطلبات
        if (!empty($monthlyOrders)) {
            echo '<table>
                <thead>
                    <tr>
                        <th>رقم الطلب</th>
                        <th>العميل</th>
                        <th>نوع الخدمة</th>
                        <th>الفرع</th>
                        <th>تاريخ الطلب</th>
                        <th>المبلغ</th>
                        <th>الحالة</th>
                    </tr>
                </thead>
                <tbody>';

            foreach ($monthlyOrders as $order) {
                echo '<tr>
                    <td>' . $order['order_number'] . '</td>
                    <td>' . $order['customer_name'] . '</td>
                    <td>' . $order['service_type'] . '</td>
                    <td>' . ($order['branch'] ?? 'غير محدد') . '</td>
                    <td>' . formatDateArabic($order['order_date']) . '</td>
                    <td>' . number_format(($order['paid_amount'] ?? 0) + ($order['remaining_amount'] ?? 0), 2) . ' ' . ($order['currency'] ?? 'ريال') . '</td>
                    <td>' . $order['status'] . '</td>
                </tr>';
            }

            echo '</tbody>
            </table>';
        } else {
            echo '<div style="text-align: center; padding: 20px; background-color: #f9fafb; border-radius: 8px; margin-bottom: 30px;">
                <p style="font-size: 16px; color: #6b7280;">لا توجد طلبات في هذا الشهر</p>
            </div>';
        }
    }

    // إضافة تذييل الصفحة
    echo '<div class="footer">
        <div class="footer-stamp">
            <div class="stamp-section">
                <div class="stamp-title">إعداد</div>
                <div class="stamp-box">توقيع المسؤول</div>
            </div>
            <div class="stamp-section">
                <div class="stamp-title">اعتماد</div>
                <div class="stamp-box">ختم الشركة</div>
            </div>
            <div class="stamp-section">
                <div class="stamp-title">تصديق</div>
                <div class="stamp-box">توقيع المدير</div>
            </div>
        </div>
        <p>تم إنشاء هذا التقرير بواسطة: ' . $user['full_name'] . '</p>
        <p>تاريخ التقرير: ' . date('Y-m-d H:i:s') . '</p>
        <div class="page-number">صفحة 1 من 1</div>
        <div class="action-buttons">
            <button class="print-btn" onclick="printReport()">طباعة القائمة</button>
        </div>
    </div>
    </div>
    </body>
    </html>';
    exit;
}

/**
 * دالة تصدير التقرير بتنسيق HTML (بديل عن PDF)
 *
 * @param string $reportType نوع التقرير
 * @param string $weekStart تاريخ بداية الأسبوع
 * @param string $weekEnd تاريخ نهاية الأسبوع
 * @param string $month الشهر
 * @param string $year السنة
 * @param string $branchId معرف الفرع
 * @param string $status حالة الطلب
 * @param array $user معلومات المستخدم
 * @param string $filename اسم الملف
 * @param bool $ordersOnly ما إذا كان يجب تصدير قائمة الطلبات فقط
 */
function exportToHTML($reportType, $weekStart, $weekEnd, $month, $year, $branchId, $status, $user, $filename, $ordersOnly = false) {
    global $conn, $company_name, $company_logo;

    // إذا كان المطلوب هو قائمة الطلبات فقط، استخدم دالة exportOrdersList
    if ($ordersOnly) {
        exportOrdersList($reportType, $weekStart, $weekEnd, $month, $year, $branchId, $status, $user);
        return;
    }

    // بدء إخراج HTML
    echo '<!DOCTYPE html>
    <html dir="rtl">
    <head>
        <meta charset="UTF-8">
        <title>تقرير ' . ($reportType === 'weekly' ? 'أسبوعي' : 'شهري') . '</title>
        <style>
            @import url("https://fonts.googleapis.com/css2?family=Tajawal:wght@400;500;700&display=swap");

            @page {
                size: A4;
                margin: 1cm;
            }

            body {
                font-family: "Tajawal", Arial, sans-serif;
                direction: rtl;
                text-align: right;
                margin: 0;
                padding: 0;
                background-color: #f8f9fa;
                color: #333;
                font-size: 12pt;
            }

            .container {
                width: 21cm;
                min-height: 29.7cm;
                margin: 0 auto;
                background-color: #fff;
                padding: 1.5cm;
                box-shadow: 0 0 20px rgba(0, 0, 0, 0.1);
                position: relative;
                box-sizing: border-box;
            }

            .watermark {
                position: absolute;
                top: 50%;
                left: 50%;
                transform: translate(-50%, -50%) rotate(-45deg);
                opacity: 0.05;
                z-index: 0;
                width: 80%;
                height: auto;
            }

            .header {
                margin-bottom: 30px;
                padding-bottom: 20px;
                border-bottom: 2px solid #3b82f6;
            }

            .header-flex {
                display: flex;
                align-items: center;
                margin-bottom: 20px;
            }

            .logo {
                max-width: 120px;
                margin-left: 20px;
            }

            .header-text {
                flex: 1;
            }

            .company-info {
                color: #666;
                font-size: 10pt;
                margin-top: 5px;
            }

            .document-number {
                display: flex;
                justify-content: space-between;
                background-color: #f8f9fa;
                padding: 10px 15px;
                border-radius: 5px;
                font-size: 10pt;
                margin-top: 10px;
            }

            h1 {
                font-size: 28px;
                color: #1e40af;
                margin-bottom: 10px;
            }

            h2 {
                font-size: 22px;
                color: #1e40af;
                margin-bottom: 20px;
            }

            h3 {
                font-size: 18px;
                color: #1e40af;
                margin-top: 30px;
                margin-bottom: 15px;
                border-right: 4px solid #3b82f6;
                padding-right: 10px;
            }

            .stats-grid {
                display: grid;
                grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
                gap: 20px;
                margin-bottom: 30px;
            }

            .stat-card {
                background-color: #f0f9ff;
                border-radius: 8px;
                padding: 15px;
                box-shadow: 0 2px 5px rgba(0, 0, 0, 0.05);
                border-right: 3px solid #3b82f6;
            }

            .stat-title {
                font-size: 14px;
                color: #6b7280;
                margin-bottom: 5px;
            }

            .stat-value {
                font-size: 24px;
                font-weight: bold;
                color: #1e40af;
            }

            .stat-unit {
                font-size: 14px;
                color: #6b7280;
            }

            table {
                width: 100%;
                border-collapse: collapse;
                margin-bottom: 30px;
                box-shadow: 0 2px 5px rgba(0, 0, 0, 0.05);
            }

            th {
                background-color: #3b82f6;
                color: white;
                font-weight: 500;
                text-align: right;
                padding: 12px 15px;
            }

            td {
                padding: 10px 15px;
                border-bottom: 1px solid #e5e7eb;
            }

            tr:nth-child(even) {
                background-color: #f9fafb;
            }

            tr:hover {
                background-color: #f0f9ff;
            }

            .footer {
                text-align: center;
                margin-top: 40px;
                padding-top: 20px;
                border-top: 1px solid #e5e7eb;
                color: #6b7280;
                font-size: 14px;
            }

            .print-btn {
                display: inline-block;
                background-color: #3b82f6;
                color: white;
                padding: 10px 20px;
                border-radius: 5px;
                text-decoration: none;
                margin-top: 20px;
                cursor: pointer;
                font-weight: 500;
            }

            .print-btn:hover {
                background-color: #2563eb;
            }

            @media print {
                @page {
                    size: A4;
                    margin: 1cm;
                }

                body {
                    background-color: white;
                    padding: 0;
                    margin: 0;
                    font-size: 12pt;
                }

                .container {
                    box-shadow: none;
                    padding: 1cm;
                    width: 100%;
                    height: 100%;
                    margin: 0;
                }

                .action-buttons {
                    display: none;
                }

                table {
                    page-break-inside: avoid;
                    border-collapse: collapse;
                }

                th, td {
                    border: 1px solid #000;
                }

                tr {
                    page-break-inside: avoid;
                }

                .watermark {
                    display: block !important;
                }

                .footer {
                    margin-top: 2cm;
                }

                .stamp-box {
                    border: 1px solid #000;
                }
            }
        </style>
        <script>
            function printReport() {
                window.print();
            }

            function downloadAsPDF() {
                window.location.href = "' . $_SERVER['REQUEST_URI'] . '&download=pdf";
            }
        </script>
    </head>
    <body>
        <div class="container">
            <img src="assets/images/Company logo.png" alt="شعار الشركة" class="watermark">
            <div class="header">
                <div class="header-flex">
                    <img src="assets/images/Company logo.png" alt="شعار الشركة" class="logo">
                    <div class="header-text">
                        <h1>مجموعة ابراهيم الاحمدي اليمنية</h1>
                        <h2>' . ($reportType === 'weekly' ? 'التقرير الأسبوعي: ' . formatDateArabic($weekStart) . ' - ' . formatDateArabic($weekEnd) : 'التقرير الشهري: ' . $month . '/' . $year) . '</h2>
                        <p class="company-info">العنوان: اليمن، حضرموت، المكلا | هاتف: 05318281 | البريد الإلكتروني: ibrahim@al-ahmadigroup.com</p>
                    </div>
                </div>
                <div class="document-number">
                    <p>رقم المستند: <strong>WR-' . date('Ymd', strtotime($weekStart)) . '</strong></p>
                    <p>تاريخ الإصدار: <strong>' . date('Y-m-d') . '</strong></p>
                </div>
            </div>';

    if ($reportType === 'weekly') {
        // الحصول على بيانات التقرير الأسبوعي
        $weeklyStats = getWeeklyStats($conn, $weekStart, $weekEnd, $user['id'], $user['role']);
        $weeklyOrders = getWeeklyOrders($conn, $weekStart, $weekEnd, $user['id'], $user['role'], $branchId, $status);

        // عرض ملخص الإحصائيات
        echo '<h3>ملخص الإحصائيات</h3>
        <div class="stats-grid">
            <div class="stat-card">
                <div class="stat-title">إجمالي الطلبات</div>
                <div class="stat-value">' . $weeklyStats['total_orders'] . '</div>
                <div class="stat-unit">طلب</div>
            </div>

            <div class="stat-card">
                <div class="stat-title">إجمالي الإيرادات</div>
                <div class="stat-value">' . number_format($weeklyStats['total_revenue'], 0) . '</div>
                <div class="stat-unit">ريال</div>
            </div>

            <div class="stat-card">
                <div class="stat-title">متوسط قيمة الطلب</div>
                <div class="stat-value">' . number_format($weeklyStats['average_order_value'], 0) . '</div>
                <div class="stat-unit">ريال</div>
            </div>

            <div class="stat-card">
                <div class="stat-title">الطلبات المكتملة</div>
                <div class="stat-value">' . $weeklyStats['completed_orders'] . '</div>
                <div class="stat-unit">طلب</div>
            </div>

            <div class="stat-card">
                <div class="stat-title">الطلبات قيد المعالجة</div>
                <div class="stat-value">' . $weeklyStats['pending_orders'] . '</div>
                <div class="stat-unit">طلب</div>
            </div>

            <div class="stat-card">
                <div class="stat-title">نسبة إكمال الطلبات</div>
                <div class="stat-value">' . number_format($weeklyStats['total_orders'] > 0 ? ($weeklyStats['completed_orders'] / $weeklyStats['total_orders']) * 100 : 0, 1) . '</div>
                <div class="stat-unit">%</div>
            </div>
        </div>';

        // عرض قائمة الطلبات
        echo '<h3>قائمة الطلبات الأسبوعية</h3>';

        if (!empty($weeklyOrders)) {
            echo '<table>
                <thead>
                    <tr>
                        <th>رقم الطلب</th>
                        <th>العميل</th>
                        <th>نوع الخدمة</th>
                        <th>الفرع</th>
                        <th>تاريخ الطلب</th>
                        <th>المبلغ</th>
                        <th>الحالة</th>
                    </tr>
                </thead>
                <tbody>';

            foreach ($weeklyOrders as $order) {
                echo '<tr>
                    <td>' . $order['order_number'] . '</td>
                    <td>' . $order['client_name'] . '</td>
                    <td>' . $order['service_type'] . '</td>
                    <td>' . ($order['branch'] ?? 'غير محدد') . '</td>
                    <td>' . formatDateArabic($order['order_date']) . '</td>
                    <td>' . number_format(($order['paid_amount'] ?? 0) + ($order['remaining_amount'] ?? 0), 2) . ' ريال</td>
                    <td>' . $order['status'] . '</td>
                </tr>';
            }

            echo '</tbody>
            </table>';
        } else {
            echo '<div style="text-align: center; padding: 20px; background-color: #f9fafb; border-radius: 8px; margin-bottom: 30px;">
                <p style="font-size: 16px; color: #6b7280;">لا توجد طلبات في هذا الأسبوع</p>
            </div>';
        }
    } elseif ($reportType === 'monthly') {
        // هنا يمكن إضافة كود تصدير التقرير الشهري
    }

    // إضافة تذييل الصفحة
    echo '<div class="footer">
        <div class="footer-stamp">
            <div class="stamp-section">
                <div class="stamp-title">إعداد</div>
                <div class="stamp-box">توقيع المسؤول</div>
            </div>
            <div class="stamp-section">
                <div class="stamp-title">اعتماد</div>
                <div class="stamp-box">ختم الشركة</div>
            </div>
            <div class="stamp-section">
                <div class="stamp-title">تصديق</div>
                <div class="stamp-box">توقيع المدير</div>
            </div>
        </div>
        <p>تم إنشاء هذا التقرير بواسطة: ' . $user['full_name'] . '</p>
        <p>تاريخ التقرير: ' . date('Y-m-d H:i:s') . '</p>
        <div class="page-number">صفحة 1 من 1</div>
        <div class="action-buttons">
            <button class="print-btn" onclick="printReport()">طباعة التقرير</button>
        </div>
    </div>
    </div>
    </body>
    </html>';
    exit;
}

/**
 * دالة تصدير التقرير بتنسيق PDF
 *
 * @param string $reportType نوع التقرير
 * @param string $weekStart تاريخ بداية الأسبوع
 * @param string $weekEnd تاريخ نهاية الأسبوع
 * @param string $month الشهر
 * @param string $year السنة
 * @param string $branchId معرف الفرع
 * @param string $status حالة الطلب
 * @param array $user معلومات المستخدم
 * @param bool $ordersOnly ما إذا كان يجب تصدير قائمة الطلبات فقط
 */
function exportToPDF($reportType, $weekStart, $weekEnd, $month, $year, $branchId, $status, $user, $ordersOnly = false) {
    global $conn, $company_name, $company_logo;

    // إذا كان المطلوب هو قائمة الطلبات فقط وكانت مكتبة TCPDF غير موجودة، استخدم دالة exportOrdersList
    if ($ordersOnly && !file_exists('libs/tcpdf/tcpdf.php')) {
        exportOrdersList($reportType, $weekStart, $weekEnd, $month, $year, $branchId, $status, $user);
        return;
    }

    // إذا كان التقرير أسبوعي أو شهري، نقوم بتصدير قائمة الطلبات فقط
    if ($reportType === 'weekly' || $reportType === 'monthly') {
        $ordersOnly = true;
    }

    // التحقق من وجود مكتبة TCPDF
    if (file_exists('libs/tcpdf/tcpdf.php')) {
        require_once('libs/tcpdf/tcpdf.php');

        // إنشاء مستند PDF جديد
        $pdf = new TCPDF(PDF_PAGE_ORIENTATION, PDF_UNIT, PDF_PAGE_FORMAT, true, 'UTF-8', false);
    } else {
        // إذا لم تكن المكتبة موجودة، قم بإنشاء ملف PDF بسيط
        header('Content-Type: text/html; charset=utf-8');
        echo '<html dir="rtl">';
        echo '<head>';
        echo '<title>تقرير ' . ($reportType === 'weekly' ? 'أسبوعي' : 'شهري') . '</title>';
        echo '<style>';
        echo 'body { font-family: Arial, sans-serif; direction: rtl; text-align: right; }';
        echo 'table { border-collapse: collapse; width: 100%; margin-bottom: 20px; }';
        echo 'th, td { border: 1px solid #000; padding: 8px; text-align: right; }';
        echo 'th { background-color: #f2f2f2; }';
        echo '.center { text-align: center; }';
        echo '.right { text-align: right; }';
        echo '</style>';
        echo '</head>';
        echo '<body>';
        echo '<h1 class="center">التقرير ' . ($reportType === 'weekly' ? 'الأسبوعي' : 'الشهري') . '</h1>';
        echo '<h2 class="center">' . $company_name . '</h2>';
        echo '<p class="center">الفترة: ' . formatDateArabic($weekStart) . ' - ' . formatDateArabic($weekEnd) . '</p>';

        // الحصول على بيانات التقرير
        $stats = getWeeklyStats($conn, $weekStart, $weekEnd, $user['id'], $user['role']);
        $orders = getWeeklyOrders($conn, $weekStart, $weekEnd, $user['id'], $user['role'], $branchId, $status);

        // عرض ملخص الإحصائيات
        echo '<h3 class="right">ملخص الإحصائيات</h3>';
        echo '<table>';
        echo '<tr>';
        echo '<th>إجمالي الطلبات</th>';
        echo '<th>إجمالي الإيرادات</th>';
        echo '<th>الطلبات المكتملة</th>';
        echo '<th>الطلبات قيد المعالجة</th>';
        echo '<th>متوسط قيمة الطلب</th>';
        echo '<th>نسبة إكمال الطلبات</th>';
        echo '</tr>';
        echo '<tr>';
        echo '<td>' . $stats['total_orders'] . '</td>';
        echo '<td>' . number_format($stats['total_revenue'], 2) . ' ريال</td>';
        echo '<td>' . $stats['completed_orders'] . '</td>';
        echo '<td>' . $stats['pending_orders'] . '</td>';
        echo '<td>' . number_format($stats['average_order_value'], 2) . ' ريال</td>';
        echo '<td>' . number_format($stats['total_orders'] > 0 ? ($stats['completed_orders'] / $stats['total_orders']) * 100 : 0, 1) . '%</td>';
        echo '</tr>';
        echo '</table>';

        // عرض قائمة الطلبات
        echo '<h3 class="right">قائمة الطلبات</h3>';
        echo '<table>';
        echo '<tr>';
        echo '<th>رقم الطلب</th>';
        echo '<th>العميل</th>';
        echo '<th>نوع الخدمة</th>';
        echo '<th>الفرع</th>';
        echo '<th>تاريخ الطلب</th>';
        echo '<th>المبلغ</th>';
        echo '<th>الحالة</th>';
        echo '</tr>';

        foreach ($orders as $order) {
            echo '<tr>';
            echo '<td>' . $order['order_number'] . '</td>';
            echo '<td>' . $order['client_name'] . '</td>';
            echo '<td>' . $order['service_type'] . '</td>';
            echo '<td>' . ($order['branch'] ?? 'غير محدد') . '</td>';
            echo '<td>' . formatDateArabic($order['order_date']) . '</td>';
            echo '<td>' . number_format(($order['paid_amount'] ?? 0) + ($order['remaining_amount'] ?? 0), 2) . ' ريال</td>';
            echo '<td>' . $order['status'] . '</td>';
            echo '</tr>';
        }

        if (empty($orders)) {
            echo '<tr><td colspan="7" class="center">لا توجد طلبات في هذه الفترة</td></tr>';
        }

        echo '</table>';

        echo '<div class="center">';
        echo '<p>تم إنشاء هذا التقرير بواسطة: ' . $user['full_name'] . '</p>';
        echo '<p>تاريخ التقرير: ' . date('Y-m-d H:i:s') . '</p>';
        echo '</div>';

        echo '</body>';
        echo '</html>';
        exit;
    }

    // تعيين معلومات المستند
    $pdf->SetCreator('نظام طلبات الشحن');
    $pdf->SetAuthor($user['full_name']);
    $pdf->SetTitle('تقرير ' . ($reportType === 'weekly' ? 'أسبوعي' : 'شهري'));
    $pdf->SetSubject('تقرير ' . ($reportType === 'weekly' ? 'أسبوعي' : 'شهري'));
    $pdf->SetKeywords('تقرير, مجموعة ابراهيم الاحمدي اليمنية, طلبات الشحن');

    // تعيين الهوامش لورقة A4
    $pdf->SetMargins(15, 15, 15);
    $pdf->SetHeaderMargin(5);
    $pdf->SetFooterMargin(10);
    $pdf->SetAutoPageBreak(true, 15);

    // تعيين الخط الافتراضي
    $pdf->SetFont('aealarabiya', '', 12);

    // إضافة صفحة جديدة
    $pdf->AddPage();

    // إضافة محتوى التقرير
    if ($reportType === 'weekly') {
        // الحصول على بيانات التقرير الأسبوعي
        $weeklyOrders = getWeeklyOrders($conn, $weekStart, $weekEnd, $user['id'], $user['role'], $branchId, $status);

        // الحصول على إحصائيات الأسبوع فقط إذا لم يكن الخيار للطلبات فقط
        if (!$ordersOnly) {
            $weeklyStats = getWeeklyStats($conn, $weekStart, $weekEnd, $user['id'], $user['role']);
        }

        // استخدام شعار الشركة الجديد
        $company_logo_path = 'assets/images/Company logo.png';
        if (file_exists($company_logo_path)) {
            // إضافة شعار الشركة في الأعلى
            $pdf->Image($company_logo_path, 170, 15, 25, 0, '', '', '', false, 300, '', false, false, 0);
        } elseif (file_exists($company_logo)) {
            // استخدام الشعار القديم كاحتياطي
            $pdf->Image($company_logo, 170, 15, 25, 0, '', '', '', false, 300, '', false, false, 0);
        }

        // إضافة عنوان التقرير والمعلومات
        $pdf->SetTextColor(30, 64, 175); // لون أزرق غامق (#1e40af)
        $pdf->SetFont('aealarabiya', 'B', 18);
        $pdf->Cell(0, 10, 'مجموعة ابراهيم الاحمدي اليمنية', 0, 1, 'R');

        // تعديل العنوان حسب نوع التقرير (قائمة الطلبات فقط أو التقرير الكامل)
        $pdf->SetFont('aealarabiya', 'B', 16);
        if ($ordersOnly) {
            $pdf->Cell(0, 10, 'قائمة الطلبات الأسبوعية: ' . formatDateArabic($weekStart) . ' - ' . formatDateArabic($weekEnd), 0, 1, 'R');
        } else {
            $pdf->Cell(0, 10, 'التقرير الأسبوعي: ' . formatDateArabic($weekStart) . ' - ' . formatDateArabic($weekEnd), 0, 1, 'R');
        }

        // إضافة معلومات الشركة
        $pdf->SetTextColor(102, 102, 102); // لون رمادي (#666666)
        $pdf->SetFont('aealarabiya', '', 10);
        $pdf->Cell(0, 6, 'العنوان: اليمن، حضرموت، المكلا | هاتف: 05318281 | البريد الإلكتروني: ibrahim@al-ahmadigroup.com', 0, 1, 'R');

        // إضافة خط أفقي
        $pdf->SetDrawColor(59, 130, 246); // لون أزرق (#3b82f6)
        $pdf->SetLineWidth(0.5);
        $pdf->Line(15, $pdf->GetY() + 3, 195, $pdf->GetY() + 3);
        $pdf->Ln(8);

        // إضافة رقم المستند وتاريخ الإصدار
        $pdf->SetFillColor(248, 249, 250); // لون رمادي فاتح (#f8f9fa)
        $pdf->SetTextColor(0, 0, 0); // لون أسود
        $pdf->SetFont('aealarabiya', '', 10);

        // إضافة رقم المستند وتاريخ الإصدار
        $pdf->Cell(90, 10, 'رقم المستند: WR-' . date('Ymd', strtotime($weekStart)), 0, 0, 'R');
        $pdf->Cell(90, 10, 'تاريخ الإصدار: ' . date('Y-m-d'), 0, 1, 'L');

        // إضافة خط فاصل
        $pdf->Line(15, $pdf->GetY(), 195, $pdf->GetY());
        $pdf->Ln(5);

        // إضافة ملخص الإحصائيات فقط إذا لم يكن الخيار للطلبات فقط
        if (!$ordersOnly) {
            $pdf->SetTextColor(30, 64, 175); // لون أزرق غامق (#1e40af)
            $pdf->SetFont('aealarabiya', 'B', 14);
            $pdf->Cell(0, 10, 'ملخص الإحصائيات', 0, 1, 'R');

            // إنشاء جدول الإحصائيات بتنسيق أفضل
            $pdf->SetTextColor(0, 0, 0); // لون أسود
            $pdf->SetFont('aealarabiya', 'B', 11);

            // الصف الأول من الإحصائيات
            $pdf->SetFillColor(240, 249, 255); // لون أزرق فاتح جدًا (#f0f9ff)
            $pdf->Cell(58, 10, 'إجمالي الطلبات', 1, 0, 'C', true);
            $pdf->SetFont('aealarabiya', '', 11);
            $pdf->Cell(32, 10, $weeklyStats['total_orders'] . ' طلب', 1, 0, 'C');

            $pdf->SetFont('aealarabiya', 'B', 11);
            $pdf->Cell(58, 10, 'إجمالي الإيرادات', 1, 0, 'C', true);
            $pdf->SetFont('aealarabiya', '', 11);
            $pdf->Cell(32, 10, number_format($weeklyStats['total_revenue'], 0) . ' ريال', 1, 1, 'C');

            // الصف الثاني من الإحصائيات
            $pdf->SetFont('aealarabiya', 'B', 11);
            $pdf->Cell(58, 10, 'متوسط قيمة الطلب', 1, 0, 'C', true);
            $pdf->SetFont('aealarabiya', '', 11);
            $pdf->Cell(32, 10, number_format($weeklyStats['average_order_value'], 0) . ' ريال', 1, 0, 'C');

            $pdf->SetFont('aealarabiya', 'B', 11);
            $pdf->Cell(58, 10, 'الطلبات المكتملة', 1, 0, 'C', true);
            $pdf->SetFont('aealarabiya', '', 11);
            $pdf->Cell(32, 10, $weeklyStats['completed_orders'] . ' طلب', 1, 1, 'C');

            // الصف الثالث من الإحصائيات
            $pdf->SetFont('aealarabiya', 'B', 11);
            $pdf->Cell(58, 10, 'الطلبات قيد المعالجة', 1, 0, 'C', true);
            $pdf->SetFont('aealarabiya', '', 11);
            $pdf->Cell(32, 10, $weeklyStats['pending_orders'] . ' طلب', 1, 0, 'C');

            $pdf->SetFont('aealarabiya', 'B', 11);
            $pdf->Cell(58, 10, 'نسبة إكمال الطلبات', 1, 0, 'C', true);
            $pdf->SetFont('aealarabiya', '', 11);
            $pdf->Cell(32, 10, number_format($weeklyStats['total_orders'] > 0 ? ($weeklyStats['completed_orders'] / $weeklyStats['total_orders']) * 100 : 0, 1) . '%', 1, 1, 'C');

            $pdf->Ln(5);
        }

        // إضافة جدول الطلبات
        $pdf->SetFont('aealarabiya', 'B', 14);
        $pdf->Cell(0, 10, 'قائمة الطلبات الأسبوعية', 0, 1, 'R');

        // تعيين ألوان الجدول
        $headerBgColor = array(59, 130, 246); // لون خلفية الرأس (#3b82f6)
        $headerTextColor = array(255, 255, 255); // لون نص الرأس (أبيض)
        $rowBgColor1 = array(255, 255, 255); // لون الصف الأول (أبيض)
        $rowBgColor2 = array(249, 250, 251); // لون الصف الثاني (#f9fafb)
        $borderColor = array(229, 231, 235); // لون الحدود (#e5e7eb)
        $textColor = array(31, 41, 55); // لون النص (#1f2937)

        // إنشاء جدول الطلبات
        $pdf->SetFont('aealarabiya', 'B', 10);
        $pdf->SetFillColor($headerBgColor[0], $headerBgColor[1], $headerBgColor[2]);
        $pdf->SetTextColor($headerTextColor[0], $headerTextColor[1], $headerTextColor[2]);
        $pdf->SetDrawColor($borderColor[0], $borderColor[1], $borderColor[2]);
        $pdf->SetLineWidth(0.3);

        // تحديد عرض الأعمدة
        $colWidth = array(20, 30, 25, 25, 25, 25, 25);
        $totalWidth = array_sum($colWidth);

        // رؤوس الجدول
        $pdf->Cell($colWidth[0], 10, 'رقم الطلب', 1, 0, 'C', true);
        $pdf->Cell($colWidth[1], 10, 'العميل', 1, 0, 'C', true);
        $pdf->Cell($colWidth[2], 10, 'نوع الخدمة', 1, 0, 'C', true);
        $pdf->Cell($colWidth[3], 10, 'الفرع', 1, 0, 'C', true);
        $pdf->Cell($colWidth[4], 10, 'تاريخ الطلب', 1, 0, 'C', true);
        $pdf->Cell($colWidth[5], 10, 'المبلغ', 1, 0, 'C', true);
        $pdf->Cell($colWidth[6], 10, 'الحالة', 1, 1, 'C', true);

        // بيانات الجدول
        $pdf->SetFont('aealarabiya', '', 9);
        $pdf->SetTextColor($textColor[0], $textColor[1], $textColor[2]);

        $rowCount = 0;
        foreach ($weeklyOrders as $order) {
            // تبديل لون الخلفية بين الصفوف
            $fill = $rowCount % 2 === 0 ? false : true;
            if ($fill) {
                $pdf->SetFillColor($rowBgColor2[0], $rowBgColor2[1], $rowBgColor2[2]);
            } else {
                $pdf->SetFillColor($rowBgColor1[0], $rowBgColor1[1], $rowBgColor1[2]);
            }

            // تعيين لون النص حسب حالة الطلب
            $statusColor = $textColor;
            switch ($order['status']) {
                case 'تم التسليم':
                    $statusColor = array(22, 163, 74); // لون أخضر (#16a34a)
                    break;
                case 'قيد المعالجة':
                    $statusColor = array(234, 179, 8); // لون أصفر (#eab308)
                    break;
                case 'ملغى':
                    $statusColor = array(220, 38, 38); // لون أحمر (#dc2626)
                    break;
            }

            $pdf->Cell($colWidth[0], 8, $order['order_number'], 1, 0, 'C', $fill);
            $pdf->Cell($colWidth[1], 8, $order['client_name'], 1, 0, 'R', $fill);
            $pdf->Cell($colWidth[2], 8, $order['service_type'], 1, 0, 'R', $fill);
            $pdf->Cell($colWidth[3], 8, $order['branch'] ?? 'غير محدد', 1, 0, 'R', $fill);
            $pdf->Cell($colWidth[4], 8, formatDateArabic($order['order_date']), 1, 0, 'C', $fill);
            $pdf->Cell($colWidth[5], 8, number_format(($order['paid_amount'] ?? 0) + ($order['remaining_amount'] ?? 0), 0) . ' ريال', 1, 0, 'C', $fill);

            // تعيين لون النص لحالة الطلب
            $pdf->SetTextColor($statusColor[0], $statusColor[1], $statusColor[2]);
            $pdf->Cell($colWidth[6], 8, $order['status'], 1, 1, 'C', $fill);
            // استعادة لون النص الأصلي
            $pdf->SetTextColor($textColor[0], $textColor[1], $textColor[2]);

            $rowCount++;
        }

        if (empty($weeklyOrders)) {
            $pdf->SetFillColor($rowBgColor1[0], $rowBgColor1[1], $rowBgColor1[2]);
            $pdf->Cell($totalWidth, 10, 'لا توجد طلبات في هذا الأسبوع', 1, 1, 'C', true);
        }
    } elseif ($reportType === 'monthly') {
        // الحصول على بيانات التقرير الشهري
        $monthlyOrders = getMonthlyOrders($conn, $month . '-01', date('Y-m-t', strtotime($month . '-01')), $user['id'], $user['role'], $branchId, $status);

        // الحصول على إحصائيات الشهر فقط إذا لم يكن الخيار للطلبات فقط
        if (!$ordersOnly) {
            $monthlyStats = getMonthlyStats($conn, $month . '-01', date('Y-m-t', strtotime($month . '-01')), $user['id'], $user['role']);
        }

        // استخدام شعار الشركة الجديد
        $company_logo_path = 'assets/images/Company logo.png';
        if (file_exists($company_logo_path)) {
            // إضافة شعار الشركة في الأعلى
            $pdf->Image($company_logo_path, 170, 15, 25, 0, '', '', '', false, 300, '', false, false, 0);
        } elseif (file_exists($company_logo)) {
            // استخدام الشعار القديم كاحتياطي
            $pdf->Image($company_logo, 170, 15, 25, 0, '', '', '', false, 300, '', false, false, 0);
        }

        // إضافة عنوان التقرير والمعلومات
        $pdf->SetTextColor(30, 64, 175); // لون أزرق غامق (#1e40af)
        $pdf->SetFont('aealarabiya', 'B', 18);
        $pdf->Cell(0, 10, 'مجموعة ابراهيم الاحمدي اليمنية', 0, 1, 'R');

        // تعديل العنوان حسب نوع التقرير (قائمة الطلبات فقط أو التقرير الكامل)
        $pdf->SetFont('aealarabiya', 'B', 16);
        if ($ordersOnly) {
            $pdf->Cell(0, 10, 'قائمة الطلبات الشهرية: ' . getMonthNameArabic(date('m', strtotime($month . '-01'))) . ' ' . date('Y', strtotime($month . '-01')), 0, 1, 'R');
        } else {
            $pdf->Cell(0, 10, 'التقرير الشهري: ' . getMonthNameArabic(date('m', strtotime($month . '-01'))) . ' ' . date('Y', strtotime($month . '-01')), 0, 1, 'R');
        }

        // إضافة معلومات الشركة
        $pdf->SetTextColor(102, 102, 102); // لون رمادي (#666666)
        $pdf->SetFont('aealarabiya', '', 10);
        $pdf->Cell(0, 6, 'العنوان: اليمن، حضرموت، المكلا | هاتف: 05318281 | البريد الإلكتروني: ibrahim@al-ahmadigroup.com', 0, 1, 'R');

        // إضافة خط أفقي
        $pdf->SetDrawColor(59, 130, 246); // لون أزرق (#3b82f6)
        $pdf->SetLineWidth(0.5);
        $pdf->Line(15, $pdf->GetY() + 3, 195, $pdf->GetY() + 3);
        $pdf->Ln(8);

        // إضافة رقم المستند وتاريخ الإصدار
        $pdf->SetFillColor(248, 249, 250); // لون رمادي فاتح (#f8f9fa)
        $pdf->SetTextColor(0, 0, 0); // لون أسود
        $pdf->SetFont('aealarabiya', '', 10);

        // إضافة رقم المستند وتاريخ الإصدار
        $pdf->Cell(90, 10, 'رقم المستند: MR-' . date('Ym', strtotime($month . '-01')), 0, 0, 'R');
        $pdf->Cell(90, 10, 'تاريخ الإصدار: ' . date('Y-m-d'), 0, 1, 'L');

        // إضافة خط فاصل
        $pdf->Line(15, $pdf->GetY(), 195, $pdf->GetY());
        $pdf->Ln(5);

        // إضافة ملخص الإحصائيات فقط إذا لم يكن الخيار للطلبات فقط
        if (!$ordersOnly) {
            $pdf->SetTextColor(30, 64, 175); // لون أزرق غامق (#1e40af)
            $pdf->SetFont('aealarabiya', 'B', 14);
            $pdf->Cell(0, 10, 'ملخص الإحصائيات', 0, 1, 'R');

            // إنشاء جدول الإحصائيات بتنسيق أفضل
            $pdf->SetTextColor(0, 0, 0); // لون أسود
            $pdf->SetFont('aealarabiya', 'B', 11);

            // الصف الأول من الإحصائيات
            $pdf->SetFillColor(240, 249, 255); // لون أزرق فاتح جدًا (#f0f9ff)
            $pdf->Cell(58, 10, 'إجمالي الطلبات', 1, 0, 'C', true);
            $pdf->SetFont('aealarabiya', '', 11);
            $pdf->Cell(32, 10, $monthlyStats['total_orders'] . ' طلب', 1, 0, 'C');

            $pdf->SetFont('aealarabiya', 'B', 11);
            $pdf->Cell(58, 10, 'إجمالي الإيرادات', 1, 0, 'C', true);
            $pdf->SetFont('aealarabiya', '', 11);
            $pdf->Cell(32, 10, number_format($monthlyStats['total_revenue'], 0) . ' ريال', 1, 1, 'C');

            // الصف الثاني من الإحصائيات
            $pdf->SetFont('aealarabiya', 'B', 11);
            $pdf->Cell(58, 10, 'متوسط قيمة الطلب', 1, 0, 'C', true);
            $pdf->SetFont('aealarabiya', '', 11);
            $pdf->Cell(32, 10, number_format($monthlyStats['average_order_value'], 0) . ' ريال', 1, 0, 'C');

            $pdf->SetFont('aealarabiya', 'B', 11);
            $pdf->Cell(58, 10, 'الطلبات المكتملة', 1, 0, 'C', true);
            $pdf->SetFont('aealarabiya', '', 11);
            $pdf->Cell(32, 10, $monthlyStats['completed_orders'] . ' طلب', 1, 1, 'C');

            // الصف الثالث من الإحصائيات
            $pdf->SetFont('aealarabiya', 'B', 11);
            $pdf->Cell(58, 10, 'الطلبات قيد المعالجة', 1, 0, 'C', true);
            $pdf->SetFont('aealarabiya', '', 11);
            $pdf->Cell(32, 10, $monthlyStats['pending_orders'] . ' طلب', 1, 0, 'C');

            $pdf->SetFont('aealarabiya', 'B', 11);
            $pdf->Cell(58, 10, 'نسبة إكمال الطلبات', 1, 0, 'C', true);
            $pdf->SetFont('aealarabiya', '', 11);
            $pdf->Cell(32, 10, number_format($monthlyStats['total_orders'] > 0 ? ($monthlyStats['completed_orders'] / $monthlyStats['total_orders']) * 100 : 0, 1) . '%', 1, 1, 'C');

            $pdf->Ln(5);
        }

        // إضافة جدول الطلبات
        $pdf->SetTextColor(30, 64, 175); // لون أزرق غامق (#1e40af)
        $pdf->SetFont('aealarabiya', 'B', 14);
        $pdf->Cell(0, 10, 'قائمة الطلبات الشهرية', 0, 1, 'R');

        // تعيين ألوان الجدول
        $headerBgColor = array(59, 130, 246); // لون خلفية الرأس (#3b82f6)
        $headerTextColor = array(255, 255, 255); // لون نص الرأس (أبيض)
        $rowBgColor1 = array(255, 255, 255); // لون الصف الأول (أبيض)
        $rowBgColor2 = array(249, 250, 251); // لون الصف الثاني (#f9fafb)
        $borderColor = array(229, 231, 235); // لون الحدود (#e5e7eb)
        $textColor = array(31, 41, 55); // لون النص (#1f2937)

        // إنشاء جدول الطلبات
        $pdf->SetFont('aealarabiya', 'B', 10);
        $pdf->SetFillColor($headerBgColor[0], $headerBgColor[1], $headerBgColor[2]);
        $pdf->SetTextColor($headerTextColor[0], $headerTextColor[1], $headerTextColor[2]);
        $pdf->SetDrawColor($borderColor[0], $borderColor[1], $borderColor[2]);
        $pdf->SetLineWidth(0.3);

        // تحديد عرض الأعمدة
        $colWidth = array(20, 30, 25, 25, 25, 25, 25);
        $totalWidth = array_sum($colWidth);

        // رؤوس الجدول
        $pdf->Cell($colWidth[0], 10, 'رقم الطلب', 1, 0, 'C', true);
        $pdf->Cell($colWidth[1], 10, 'العميل', 1, 0, 'C', true);
        $pdf->Cell($colWidth[2], 10, 'نوع الخدمة', 1, 0, 'C', true);
        $pdf->Cell($colWidth[3], 10, 'الفرع', 1, 0, 'C', true);
        $pdf->Cell($colWidth[4], 10, 'تاريخ الطلب', 1, 0, 'C', true);
        $pdf->Cell($colWidth[5], 10, 'المبلغ', 1, 0, 'C', true);
        $pdf->Cell($colWidth[6], 10, 'الحالة', 1, 1, 'C', true);

        // بيانات الجدول
        $pdf->SetFont('aealarabiya', '', 9);
        $pdf->SetTextColor($textColor[0], $textColor[1], $textColor[2]);

        $rowCount = 0;
        foreach ($monthlyOrders as $order) {
            // تبديل لون الخلفية بين الصفوف
            $fill = $rowCount % 2 === 0 ? false : true;
            if ($fill) {
                $pdf->SetFillColor($rowBgColor2[0], $rowBgColor2[1], $rowBgColor2[2]);
            } else {
                $pdf->SetFillColor($rowBgColor1[0], $rowBgColor1[1], $rowBgColor1[2]);
            }

            // تعيين لون النص حسب حالة الطلب
            $statusColor = $textColor;
            switch ($order['status']) {
                case 'تم التسليم':
                    $statusColor = array(22, 163, 74); // لون أخضر (#16a34a)
                    break;
                case 'قيد المعالجة':
                    $statusColor = array(234, 179, 8); // لون أصفر (#eab308)
                    break;
                case 'في الطريق':
                    $statusColor = array(59, 130, 246); // لون أزرق (#3b82f6)
                    break;
                case 'ملغى':
                    $statusColor = array(220, 38, 38); // لون أحمر (#dc2626)
                    break;
            }

            // طباعة بيانات الطلب
            $pdf->Cell($colWidth[0], 10, $order['order_number'], 1, 0, 'C', $fill);
            $pdf->Cell($colWidth[1], 10, $order['customer_name'], 1, 0, 'R', $fill);
            $pdf->Cell($colWidth[2], 10, $order['service_type'], 1, 0, 'R', $fill);
            $pdf->Cell($colWidth[3], 10, $order['branch'] ?? 'غير محدد', 1, 0, 'R', $fill);
            $pdf->Cell($colWidth[4], 10, formatDateArabic($order['order_date']), 1, 0, 'C', $fill);
            $pdf->Cell($colWidth[5], 10, number_format(($order['paid_amount'] ?? 0) + ($order['remaining_amount'] ?? 0), 0) . ' ريال', 1, 0, 'C', $fill);

            // تغيير لون النص للحالة
            $pdf->SetTextColor($statusColor[0], $statusColor[1], $statusColor[2]);
            $pdf->Cell($colWidth[6], 10, $order['status'], 1, 1, 'C', $fill);

            // استعادة لون النص الأصلي
            $pdf->SetTextColor($textColor[0], $textColor[1], $textColor[2]);

            $rowCount++;
        }

        if (empty($monthlyOrders)) {
            $pdf->SetFillColor($rowBgColor1[0], $rowBgColor1[1], $rowBgColor1[2]);
            $pdf->Cell($totalWidth, 10, 'لا توجد طلبات في هذا الشهر', 1, 1, 'C', true);
        }
    }

    /**
     * دالة للحصول على اسم الشهر بالعربية
     *
     * @param string $month رقم الشهر
     * @return string اسم الشهر بالعربية
     */
    function getMonthNameArabic($month) {
        $months = [
            '01' => 'يناير',
            '02' => 'فبراير',
            '03' => 'مارس',
            '04' => 'أبريل',
            '05' => 'مايو',
            '06' => 'يونيو',
            '07' => 'يوليو',
            '08' => 'أغسطس',
            '09' => 'سبتمبر',
            '10' => 'أكتوبر',
            '11' => 'نوفمبر',
            '12' => 'ديسمبر'
        ];

        return $months[$month] ?? '';
    }

    // إضافة تذييل الصفحة
    $pdf->SetY(-60);

    // إضافة خط فاصل
    $pdf->SetDrawColor(59, 130, 246); // لون أزرق (#3b82f6)
    $pdf->SetLineWidth(0.5);
    $pdf->Line(15, $pdf->GetY(), 195, $pdf->GetY());
    $pdf->Ln(5);

    // إضافة معلومات التذييل
    $pdf->SetTextColor(102, 102, 102); // لون رمادي (#666666)
    $pdf->SetFont('aealarabiya', '', 10);
    $pdf->Cell(65, 10, 'تم إنشاء هذا التقرير بواسطة: ' . $user['full_name'], 0, 0, 'R');
    $pdf->Cell(65, 10, 'تاريخ التقرير: ' . date('Y-m-d H:i:s'), 0, 0, 'C');
    $pdf->Cell(65, 10, 'صفحة 1 من 1', 0, 1, 'L');

    // إضافة مربعات التوقيع
    $pdf->Ln(5);
    $pdf->SetFont('aealarabiya', 'B', 10);
    $pdf->SetTextColor(30, 64, 175); // لون أزرق غامق (#1e40af)
    $pdf->SetFillColor(240, 249, 255); // لون أزرق فاتح جدًا (#f0f9ff)

    $pdf->Cell(58, 10, 'إعداد', 1, 0, 'C', true);
    $pdf->Cell(58, 10, 'اعتماد', 1, 0, 'C', true);
    $pdf->Cell(58, 10, 'تصديق', 1, 1, 'C', true);

    $pdf->SetTextColor(0, 0, 0); // لون أسود
    $pdf->SetFont('aealarabiya', '', 9);
    $pdf->Cell(58, 20, 'توقيع المسؤول', 1, 0, 'C');
    $pdf->Cell(58, 20, 'ختم الشركة', 1, 0, 'C');
    $pdf->Cell(58, 20, 'توقيع المدير', 1, 1, 'C');

    // إخراج المستند
    $pdf->Output($filename . '.pdf', 'I');
    exit;
}

/**
 * دالة تصدير التقرير بتنسيق Excel
 *
 * @param string $reportType نوع التقرير
 * @param string $weekStart تاريخ بداية الأسبوع
 * @param string $weekEnd تاريخ نهاية الأسبوع
 * @param string $month الشهر
 * @param string $year السنة
 * @param string $branchId معرف الفرع
 * @param string $status حالة الطلب
 * @param array $user معلومات المستخدم
 */
function exportToExcel($reportType, $weekStart, $weekEnd, $month, $year, $branchId, $status, $user, $ordersOnly = false) {
    global $conn, $company_name, $company_logo;

    // بدء إخراج HTML
    echo '<!DOCTYPE html>
    <html dir="rtl">
    <head>
        <meta charset="UTF-8">
        <title>تقرير ' . ($reportType === 'weekly' ? 'أسبوعي' : 'شهري') . '</title>
        <style>
            @import url("https://fonts.googleapis.com/css2?family=Tajawal:wght@400;500;700&display=swap");

            @page {
                size: A4;
                margin: 1cm;
            }

            body {
                font-family: "Tajawal", Arial, sans-serif;
                direction: rtl;
                text-align: right;
                margin: 0;
                padding: 0;
                background-color: white;
                color: #333;
                font-size: 12pt;
            }

            .container {
                width: 21cm;
                min-height: 29.7cm;
                margin: 0 auto;
                background-color: #fff;
                padding: 1.5cm;
                position: relative;
                box-sizing: border-box;
            }

            .watermark {
                position: absolute;
                top: 50%;
                left: 50%;
                transform: translate(-50%, -50%) rotate(-45deg);
                opacity: 0.05;
                z-index: 0;
                width: 80%;
                height: auto;
            }

            .header {
                margin-bottom: 30px;
                padding-bottom: 20px;
                border-bottom: 2px solid #3b82f6;
            }

            .header-flex {
                display: flex;
                align-items: center;
                margin-bottom: 20px;
            }

            .logo {
                max-width: 120px;
                margin-left: 20px;
            }

            .header-text {
                flex: 1;
            }

            .company-info {
                color: #666;
                font-size: 10pt;
                margin-top: 5px;
            }

            .document-number {
                display: flex;
                justify-content: space-between;
                background-color: #f8f9fa;
                padding: 10px 15px;
                border-radius: 5px;
                font-size: 10pt;
                margin-top: 10px;
            }

            h1 {
                font-size: 24px;
                color: #1e40af;
                margin-bottom: 10px;
            }

            h2 {
                font-size: 18px;
                color: #1e40af;
                margin-bottom: 20px;
            }

            h3 {
                font-size: 16px;
                color: #1e40af;
                margin-top: 30px;
                margin-bottom: 15px;
                border-right: 4px solid #3b82f6;
                padding-right: 10px;
            }

            table {
                width: 100%;
                border-collapse: collapse;
                margin-bottom: 30px;
            }

            th {
                background-color: #3b82f6;
                color: white;
                font-weight: 500;
                text-align: right;
                padding: 12px 15px;
                border: 1px solid #3b82f6;
            }

            td {
                padding: 10px 15px;
                border: 1px solid #e5e7eb;
            }

            tr:nth-child(even) {
                background-color: #f9fafb;
            }

            .footer {
                text-align: center;
                margin-top: 40px;
                padding-top: 20px;
                border-top: 1px solid #e5e7eb;
                color: #6b7280;
                font-size: 10pt;
                position: relative;
            }

            .footer-stamp {
                display: flex;
                justify-content: space-between;
                margin-top: 30px;
                margin-bottom: 20px;
            }

            .stamp-section {
                text-align: center;
                width: 30%;
            }

            .stamp-title {
                font-weight: bold;
                margin-bottom: 5px;
            }

            .stamp-box {
                border: 1px solid #ccc;
                height: 80px;
                display: flex;
                align-items: center;
                justify-content: center;
                margin-top: 10px;
                color: #999;
            }

            .page-number {
                position: absolute;
                bottom: 10px;
                left: 0;
                right: 0;
                text-align: center;
                font-size: 9pt;
                color: #999;
            }

            .center { text-align: center; }
            .right { text-align: right; }

            .stats-grid {
                display: grid;
                grid-template-columns: repeat(3, 1fr);
                gap: 20px;
                margin-bottom: 30px;
            }

            .stat-card {
                background-color: #f0f9ff;
                border-radius: 8px;
                padding: 15px;
                border-right: 3px solid #3b82f6;
            }

            .stat-title {
                font-size: 14px;
                color: #6b7280;
                margin-bottom: 5px;
            }

            .stat-value {
                font-size: 24px;
                font-weight: bold;
                color: #1e40af;
            }

            .stat-unit {
                font-size: 14px;
                color: #6b7280;
            }
        </style>
    </head>
    <body>
    <div class="container">
        <img src="' . $company_logo . '" alt="شعار الشركة" class="watermark">
        <div class="header">
            <div class="header-flex">
                <img src="' . $company_logo . '" alt="شعار الشركة" class="logo">
                <div class="header-text">
                    <h1>مجموعة ابراهيم الاحمدي اليمنية</h1>';

    if ($reportType === 'weekly') {
        // الحصول على بيانات التقرير الأسبوعي
        $weeklyOrders = getWeeklyOrders($conn, $weekStart, $weekEnd, $user['id'], $user['role'], $branchId, $status);

        // إضافة عنوان التقرير
        if ($ordersOnly) {
            echo '<h2>قائمة الطلبات الأسبوعية: ' . formatDateArabic($weekStart) . ' - ' . formatDateArabic($weekEnd) . '</h2>';
        } else {
            $weeklyStats = getWeeklyStats($conn, $weekStart, $weekEnd, $user['id'], $user['role']);
            echo '<h2>التقرير الأسبوعي: ' . formatDateArabic($weekStart) . ' - ' . formatDateArabic($weekEnd) . '</h2>';
        }
        echo '<p class="company-info">العنوان: اليمن، حضرموت، المكلا | هاتف: 05318281 | البريد الإلكتروني: ibrahim@al-ahmadigroup.com</p>
                </div>
            </div>
            <div class="document-number">
                <p>رقم المستند: <strong>WR-' . date('Ymd', strtotime($weekStart)) . '</strong></p>
                <p>تاريخ الإصدار: <strong>' . date('Y-m-d') . '</strong></p>
            </div>
        </div>';

        // إضافة ملخص الإحصائيات إذا لم يكن الخيار للطلبات فقط
        if (!$ordersOnly) {
            echo '<h3 class="right">ملخص الإحصائيات</h3>';
            echo '<table>
                <tr>
                    <th>إجمالي الطلبات</th>
                    <th>إجمالي الإيرادات</th>
                    <th>الطلبات المكتملة</th>
                    <th>الطلبات قيد المعالجة</th>
                    <th>متوسط قيمة الطلب</th>
                    <th>نسبة إكمال الطلبات</th>
                </tr>
                <tr>
                    <td>' . $weeklyStats['total_orders'] . '</td>
                    <td>' . number_format($weeklyStats['total_revenue'], 2) . ' ريال</td>
                    <td>' . $weeklyStats['completed_orders'] . '</td>
                    <td>' . $weeklyStats['pending_orders'] . '</td>
                    <td>' . number_format($weeklyStats['average_order_value'], 2) . ' ريال</td>
                    <td>' . number_format($weeklyStats['total_orders'] > 0 ? ($weeklyStats['completed_orders'] / $weeklyStats['total_orders']) * 100 : 0, 1) . '%</td>
                </tr>
            </table>';
        }

        // إضافة جدول الطلبات
        echo '<h3 class="right">قائمة الطلبات الأسبوعية</h3>';
        echo '<table>
            <tr>
                <th style="width: 10%;">رقم الطلب</th>
                <th style="width: 15%;">العميل</th>
                <th style="width: 15%;">نوع الخدمة</th>
                <th style="width: 15%;">الفرع</th>
                <th style="width: 15%;">تاريخ الطلب</th>
                <th style="width: 15%;">المبلغ</th>
                <th style="width: 15%;">الحالة</th>
            </tr>';

        $rowCount = 0;
        foreach ($weeklyOrders as $order) {
            $rowClass = $rowCount % 2 === 0 ? '' : ' style="background-color: #f9fafb;"';
            $statusClass = '';

            // تحديد لون الحالة
            switch ($order['status']) {
                case 'تم التسليم':
                    $statusClass = ' style="color: #16a34a; font-weight: bold;"';
                    break;
                case 'قيد المعالجة':
                    $statusClass = ' style="color: #eab308; font-weight: bold;"';
                    break;
                case 'في الطريق':
                    $statusClass = ' style="color: #3b82f6; font-weight: bold;"';
                    break;
                case 'ملغى':
                    $statusClass = ' style="color: #dc2626; font-weight: bold;"';
                    break;
            }

            echo '<tr' . $rowClass . '>
                <td style="text-align: center;">' . $order['order_number'] . '</td>
                <td>' . $order['client_name'] . '</td>
                <td>' . $order['service_type'] . '</td>
                <td>' . ($order['branch'] ?? 'غير محدد') . '</td>
                <td style="text-align: center;">' . formatDateArabic($order['order_date']) . '</td>
                <td style="text-align: center;">' . number_format(($order['paid_amount'] ?? 0) + ($order['remaining_amount'] ?? 0), 0) . ' ' . ($order['currency'] ?? 'ريال') . '</td>
                <td' . $statusClass . ' style="text-align: center;">' . $order['status'] . '</td>
            </tr>';

            $rowCount++;
        }

        if (empty($weeklyOrders)) {
            echo '<tr><td colspan="7" class="center" style="padding: 20px;">لا توجد طلبات في هذا الأسبوع</td></tr>';
        }

        echo '</table>';
    } elseif ($reportType === 'monthly') {
        // الحصول على بيانات التقرير الشهري
        $monthlyOrders = getMonthlyOrders($conn, $month . '-01', date('Y-m-t', strtotime($month . '-01')), $user['id'], $user['role'], $branchId, $status);

        // الحصول على إحصائيات الشهر فقط إذا لم يكن الخيار للطلبات فقط
        if (!$ordersOnly) {
            $monthlyStats = getMonthlyStats($conn, $month . '-01', date('Y-m-t', strtotime($month . '-01')), $user['id'], $user['role']);
        }

        // إضافة عنوان التقرير
        if ($ordersOnly) {
            echo '<h2>قائمة الطلبات الشهرية: ' . getMonthNameArabic(date('m', strtotime($month . '-01'))) . ' ' . date('Y', strtotime($month . '-01')) . '</h2>';
        } else {
            echo '<h2>التقرير الشهري: ' . getMonthNameArabic(date('m', strtotime($month . '-01'))) . ' ' . date('Y', strtotime($month . '-01')) . '</h2>';
        }
        echo '<p class="company-info">العنوان: اليمن، حضرموت، المكلا | هاتف: 05318281 | البريد الإلكتروني: ibrahim@al-ahmadigroup.com</p>
                </div>
            </div>
            <div class="document-number">
                <p>رقم المستند: <strong>MR-' . date('Ym', strtotime($month . '-01')) . '</strong></p>
                <p>تاريخ الإصدار: <strong>' . date('Y-m-d') . '</strong></p>
            </div>
        </div>';

        // إضافة ملخص الإحصائيات إذا لم يكن الخيار للطلبات فقط
        if (!$ordersOnly) {
            echo '<h3 class="right">ملخص الإحصائيات</h3>';
            echo '<table>
                <tr>
                    <th>إجمالي الطلبات</th>
                    <th>إجمالي الإيرادات</th>
                    <th>الطلبات المكتملة</th>
                    <th>الطلبات قيد المعالجة</th>
                    <th>متوسط قيمة الطلب</th>
                    <th>نسبة إكمال الطلبات</th>
                </tr>
                <tr>
                    <td>' . $monthlyStats['total_orders'] . '</td>
                    <td>' . number_format($monthlyStats['total_revenue'], 2) . ' ' . ($monthlyStats['default_currency'] ?? 'ريال') . '</td>
                    <td>' . $monthlyStats['completed_orders'] . '</td>
                    <td>' . $monthlyStats['pending_orders'] . '</td>
                    <td>' . number_format($monthlyStats['average_order_value'], 2) . ' ' . ($monthlyStats['default_currency'] ?? 'ريال') . '</td>
                    <td>' . number_format($monthlyStats['total_orders'] > 0 ? ($monthlyStats['completed_orders'] / $monthlyStats['total_orders']) * 100 : 0, 1) . '%</td>
                </tr>
            </table>';
        }

        // إضافة جدول الطلبات
        echo '<h3 class="right">قائمة الطلبات الشهرية</h3>';
        echo '<table>
            <tr>
                <th style="width: 10%;">رقم الطلب</th>
                <th style="width: 15%;">العميل</th>
                <th style="width: 15%;">نوع الخدمة</th>
                <th style="width: 15%;">الفرع</th>
                <th style="width: 15%;">تاريخ الطلب</th>
                <th style="width: 15%;">المبلغ</th>
                <th style="width: 15%;">الحالة</th>
            </tr>';

        $rowCount = 0;
        foreach ($monthlyOrders as $order) {
            $rowClass = $rowCount % 2 === 0 ? '' : ' style="background-color: #f9fafb;"';
            $statusClass = '';

            // تحديد لون الحالة
            switch ($order['status']) {
                case 'تم التسليم':
                    $statusClass = ' style="color: #16a34a; font-weight: bold;"';
                    break;
                case 'قيد المعالجة':
                    $statusClass = ' style="color: #eab308; font-weight: bold;"';
                    break;
                case 'في الطريق':
                    $statusClass = ' style="color: #3b82f6; font-weight: bold;"';
                    break;
                case 'ملغى':
                    $statusClass = ' style="color: #dc2626; font-weight: bold;"';
                    break;
            }

            echo '<tr' . $rowClass . '>
                <td style="text-align: center;">' . $order['order_number'] . '</td>
                <td>' . $order['customer_name'] . '</td>
                <td>' . $order['service_type'] . '</td>
                <td>' . ($order['branch'] ?? 'غير محدد') . '</td>
                <td style="text-align: center;">' . formatDateArabic($order['order_date']) . '</td>
                <td style="text-align: center;">' . number_format(($order['paid_amount'] ?? 0) + ($order['remaining_amount'] ?? 0), 0) . ' ' . ($order['currency'] ?? 'ريال') . '</td>
                <td' . $statusClass . ' style="text-align: center;">' . $order['status'] . '</td>
            </tr>';

            $rowCount++;
        }

        if (empty($monthlyOrders)) {
            echo '<tr><td colspan="7" class="center" style="padding: 20px;">لا توجد طلبات في هذا الشهر</td></tr>';
        }

        echo '</table>';
    }

    // إضافة تذييل الصفحة
    echo '<div class="footer">
        <div class="footer-stamp">
            <div class="stamp-section">
                <div class="stamp-title">إعداد</div>
                <div class="stamp-box">توقيع المسؤول</div>
            </div>
            <div class="stamp-section">
                <div class="stamp-title">اعتماد</div>
                <div class="stamp-box">ختم الشركة</div>
            </div>
            <div class="stamp-section">
                <div class="stamp-title">تصديق</div>
                <div class="stamp-box">توقيع المدير</div>
            </div>
        </div>
        <p>تم إنشاء هذا التقرير بواسطة: ' . $user['full_name'] . '</p>
        <p>تاريخ التقرير: ' . date('Y-m-d H:i:s') . '</p>
        <div class="page-number">صفحة 1 من 1</div>
    </div>
    </div>
    </body></html>';
    exit;
}

/**
 * دالة للحصول على إحصائيات الأسبوع
 *
 * @param PDO $conn اتصال قاعدة البيانات
 * @param string $startDate تاريخ بداية الأسبوع
 * @param string $endDate تاريخ نهاية الأسبوع
 * @param int $userId معرف المستخدم
 * @param string $userRole دور المستخدم
 * @return array إحصائيات الأسبوع
 */
function getWeeklyStats($conn, $startDate, $endDate, $userId, $userRole) {
    // استعلام للحصول على إحصائيات الأسبوع
    $sql = "
        SELECT
            COUNT(*) as total_orders,
            SUM(CASE WHEN status = 'تم التسليم' THEN 1 ELSE 0 END) as completed_orders,
            SUM(CASE WHEN status = 'قيد المعالجة' THEN 1 ELSE 0 END) as pending_orders,
            SUM(CASE WHEN status = 'في الطريق' THEN 1 ELSE 0 END) as in_transit_orders,
            SUM(CASE WHEN status = 'ملغى' THEN 1 ELSE 0 END) as cancelled_orders,
            SUM(amount_agreed_with_customer) as total_revenue,
            (SELECT currency FROM orders WHERE order_upload_date BETWEEN ? AND ? ORDER BY id DESC LIMIT 1) as default_currency
        FROM orders
        WHERE order_upload_date BETWEEN ? AND ?
    ";

    // إضافة شرط المستخدم إذا لم يكن مدير
    $params = [$startDate, $endDate, $startDate, $endDate];

    if ($userRole !== 'admin' && $userRole !== 'manager') {
        $sql .= " AND user_id = ?";
        $params[] = $userId;
    }

    $stmt = $conn->prepare($sql);
    $stmt->execute($params);
    $stats = $stmt->fetch(PDO::FETCH_ASSOC);

    // حساب متوسط قيمة الطلب
    $stats['average_order_value'] = $stats['total_orders'] > 0 ?
        $stats['total_revenue'] / $stats['total_orders'] : 0;

    return $stats;
}

/**
 * دالة للحصول على إحصائيات الشهر
 *
 * @param PDO $conn اتصال قاعدة البيانات
 * @param string $startDate تاريخ بداية الشهر
 * @param string $endDate تاريخ نهاية الشهر
 * @param int $userId معرف المستخدم
 * @param string $userRole دور المستخدم
 * @return array إحصائيات الشهر
 */
function getMonthlyStats($conn, $startDate, $endDate, $userId, $userRole) {
    // استعلام للحصول على إحصائيات الشهر
    $sql = "
        SELECT
            COUNT(*) as total_orders,
            SUM(CASE WHEN status = 'تم التسليم' THEN 1 ELSE 0 END) as completed_orders,
            SUM(CASE WHEN status = 'قيد المعالجة' THEN 1 ELSE 0 END) as pending_orders,
            SUM(CASE WHEN status = 'في الطريق' THEN 1 ELSE 0 END) as in_transit_orders,
            SUM(CASE WHEN status = 'ملغى' THEN 1 ELSE 0 END) as cancelled_orders,
            SUM(amount_agreed_with_customer) as total_revenue,
            (SELECT currency FROM orders WHERE order_upload_date BETWEEN ? AND ? ORDER BY id DESC LIMIT 1) as default_currency
        FROM orders
        WHERE order_upload_date BETWEEN ? AND ?
    ";

    // إضافة شرط المستخدم إذا لم يكن مدير
    $params = [$startDate, $endDate, $startDate, $endDate];

    if ($userRole !== 'admin' && $userRole !== 'manager') {
        $sql .= " AND user_id = ?";
        $params[] = $userId;
    }

    $stmt = $conn->prepare($sql);
    $stmt->execute($params);
    $stats = $stmt->fetch(PDO::FETCH_ASSOC);

    // حساب متوسط قيمة الطلب
    $stats['average_order_value'] = $stats['total_orders'] > 0 ?
        $stats['total_revenue'] / $stats['total_orders'] : 0;

    return $stats;
}

/**
 * دالة للحصول على الطلبات الشهرية
 *
 * @param PDO $conn اتصال قاعدة البيانات
 * @param string $startDate تاريخ بداية الشهر
 * @param string $endDate تاريخ نهاية الشهر
 * @param int $userId معرف المستخدم
 * @param string $userRole دور المستخدم
 * @param string $branchId معرف الفرع
 * @param string $status حالة الطلب
 * @return array الطلبات الشهرية
 */
function getMonthlyOrders($conn, $startDate, $endDate, $userId, $userRole, $branchId = 'all', $status = '') {
    // استعلام للحصول على الطلبات الشهرية
    $sql = "
        SELECT
            o.id,
            o.order_number,
            o.client_name as customer_name,
            o.service_type,
            o.goods_type as cargo_type,
            o.goods_name as cargo_name,
            o.delivery_duration as delivery_time,
            o.payment_type,
            o.amount_paid as paid_amount,
            o.remaining_amount,
            o.second_destination,
            o.delivery_area,
            o.order_upload_date as order_date,
            o.scheduled_delivery_date as delivery_date,
            o.status,
            o.currency,
            b.name as branch
        FROM orders o
        LEFT JOIN branches b ON o.branch_id = b.id
        WHERE o.order_upload_date BETWEEN ? AND ?
    ";

    // إضافة شرط المستخدم إذا لم يكن مدير
    $params = [$startDate, $endDate];

    if ($userRole !== 'admin' && $userRole !== 'manager') {
        $sql .= " AND o.user_id = ?";
        $params[] = $userId;
    }

    // إضافة شرط الفرع إذا تم تحديده
    if ($branchId !== 'all') {
        $sql .= " AND o.branch_id = ?";
        $params[] = $branchId;
    }

    // إضافة شرط الحالة إذا تم تحديدها
    if ($status !== '') {
        $sql .= " AND o.status = ?";
        $params[] = $status;
    }

    $sql .= " ORDER BY o.order_upload_date DESC";

    $stmt = $conn->prepare($sql);
    $stmt->execute($params);
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

/**
 * دالة للحصول على الطلبات الأسبوعية
 *
 * @param PDO $conn اتصال قاعدة البيانات
 * @param string $startDate تاريخ بداية الأسبوع
 * @param string $endDate تاريخ نهاية الأسبوع
 * @param int $userId معرف المستخدم
 * @param string $userRole دور المستخدم
 * @param string $branchId معرف الفرع
 * @param string $status حالة الطلب
 * @return array الطلبات الأسبوعية
 */
function getWeeklyOrders($conn, $startDate, $endDate, $userId, $userRole, $branchId = 'all', $status = '') {
    // استعلام للحصول على الطلبات الأسبوعية
    $sql = "
        SELECT
            o.id,
            o.order_number,
            o.client_name,
            o.service_type,
            o.goods_type as cargo_type,
            o.goods_name as cargo_name,
            o.delivery_duration as delivery_time,
            o.payment_type,
            o.amount_paid as paid_amount,
            o.remaining_amount,
            o.second_destination,
            o.delivery_area,
            o.order_upload_date as order_date,
            o.scheduled_delivery_date as delivery_date,
            o.status,
            o.currency,
            b.name as branch
        FROM orders o
        LEFT JOIN branches b ON o.branch_id = b.id
        WHERE o.order_upload_date BETWEEN ? AND ?
    ";

    // إضافة شرط المستخدم إذا لم يكن مدير
    $params = [$startDate, $endDate];

    if ($userRole !== 'admin' && $userRole !== 'manager') {
        $sql .= " AND o.user_id = ?";
        $params[] = $userId;
    }

    // إضافة شرط الفرع إذا تم تحديده
    if ($branchId !== 'all') {
        $sql .= " AND o.branch_id = ?";
        $params[] = $branchId;
    }

    // إضافة شرط الحالة إذا تم تحديدها
    if ($status !== '') {
        $sql .= " AND o.status = ?";
        $params[] = $status;
    }

    $sql .= " ORDER BY o.order_upload_date DESC";

    $stmt = $conn->prepare($sql);
    $stmt->execute($params);
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

/**
 * دالة لتنسيق التاريخ بالعربية
 *
 * @param string $date التاريخ
 * @return string التاريخ المنسق
 */
function formatDateArabic($date) {
    if (empty($date)) return '';

    $months = [
        'January' => 'يناير',
        'February' => 'فبراير',
        'March' => 'مارس',
        'April' => 'أبريل',
        'May' => 'مايو',
        'June' => 'يونيو',
        'July' => 'يوليو',
        'August' => 'أغسطس',
        'September' => 'سبتمبر',
        'October' => 'أكتوبر',
        'November' => 'نوفمبر',
        'December' => 'ديسمبر'
    ];

    $formattedDate = date('d F Y', strtotime($date));
    foreach ($months as $en => $ar) {
        $formattedDate = str_replace($en, $ar, $formattedDate);
    }
    return $formattedDate;
}

/**
 * دالة للحصول على اسم الشهر بالعربية
 *
 * @param string $month رقم الشهر (1-12)
 * @return string اسم الشهر بالعربية
 */
function getMonthNameArabic($month) {
    $months = [
        '01' => 'يناير',
        '02' => 'فبراير',
        '03' => 'مارس',
        '04' => 'أبريل',
        '05' => 'مايو',
        '06' => 'يونيو',
        '07' => 'يوليو',
        '08' => 'أغسطس',
        '09' => 'سبتمبر',
        '10' => 'أكتوبر',
        '11' => 'نوفمبر',
        '12' => 'ديسمبر'
    ];

    // تنظيف رقم الشهر
    $month = str_pad($month, 2, '0', STR_PAD_LEFT);

    return $months[$month] ?? '';
}
