<?php
/**
 * ملف API للحصول على الرقم التسلسلي التالي لفرع معين
 * يستخدم هذا الملف للحصول على الرقم التسلسلي التالي لفرع معين من جدول order_sequences
 * ويضمن أن تبدأ جميع الفروع بترقيم تسلسلي من 01 وليس أرقام عشوائية
 */

// تفعيل عرض الأخطاء للتصحيح (في بيئة التطوير فقط)
ini_set('display_errors', 0); // إيقاف عرض الأخطاء للمستخدم النهائي
ini_set('display_startup_errors', 0);
error_reporting(E_ALL);

// تسجيل الأخطاء في ملف السجل
ini_set('log_errors', 1);
ini_set('error_log', 'logs/api_errors.log');

// تضمين ملف الاتصال بقاعدة البيانات
require_once 'config/config.php';

// السماح بطلبات CORS
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization");

// التحقق من أن الطلب هو طلب GET
if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    header('Content-Type: application/json');
    echo json_encode(['status' => 'error', 'message' => 'يجب استخدام طريقة GET']);
    exit;
}

// الحصول على رمز الفرع من الطلب
$branchCode = isset($_GET['branch_code']) ? $_GET['branch_code'] : null;

if (!$branchCode) {
    header('Content-Type: application/json');
    echo json_encode(['status' => 'error', 'message' => 'يجب تحديد رمز الفرع']);
    exit;
}

try {
    // التحقق من صحة رمز الفرع
    if (!preg_match('/^[A-Z]+$/', $branchCode)) {
        throw new Exception("رمز الفرع غير صالح: $branchCode. يجب أن يحتوي على أحرف كبيرة فقط.");
    }

    // استخراج رمز الفرع الأساسي (مثلاً: MKL25- يصبح MKL)
    // نستخدم تعبير منتظم أكثر دقة للحصول على الحروف فقط من بداية الرمز
    if (preg_match('/^([A-Z]+)/', $branchCode, $matches)) {
        $branchCodeBase = $matches[1];
    } else {
        // إذا لم يتم العثور على حروف، استخدم الرمز كما هو
        $branchCodeBase = $branchCode;
    }

    // التحقق من أن رمز الفرع هو أحد الرموز المعتمدة
    $validBranchCodes = ['MKL', 'SNA', 'ADN', 'SH', 'CN'];
    if (!in_array($branchCodeBase, $validBranchCodes)) {
        throw new Exception("رمز الفرع غير معتمد: $branchCodeBase. الرموز المعتمدة هي: " . implode(', ', $validBranchCodes));
    }

    // تسجيل رمز الفرع الأساسي للتصحيح
    error_log("رمز الفرع الأصلي: $branchCode، رمز الفرع الأساسي: $branchCodeBase");

    // التحقق مما إذا كان يجب حجز الرقم التسلسلي (الافتراضي: نعم)
    $reserve = isset($_GET['reserve']) ? ($_GET['reserve'] === 'true') : true;

    // بدء المعاملة لضمان عدم تداخل العمليات
    $pdo->beginTransaction();

    try {
        // الحصول على آخر رقم تسلسلي للفرع مع قفل الصف
        $stmt = $pdo->prepare("SELECT last_sequence_number FROM order_sequences WHERE branch_code = ? FOR UPDATE");
        $stmt->execute([$branchCodeBase]);
        $sequence = $stmt->fetch();

        // الحصول على السنة والشهر الحالي
        $year = date('y');
        $month = date('m');

        $sequenceNumber = 1;
        if ($sequence) {
            $sequenceNumber = $sequence['last_sequence_number'] + 1;

            // تحديث رقم التسلسل في قاعدة البيانات إذا كان الحجز مطلوباً
            if ($reserve) {
                $stmt = $pdo->prepare("UPDATE order_sequences SET last_sequence_number = ? WHERE branch_code = ?");
                $stmt->execute([$sequenceNumber, $branchCodeBase]);
            }
        } else {
            // إنشاء سجل جديد إذا لم يكن موجوداً
            $stmt = $pdo->prepare("INSERT INTO order_sequences (branch_code, last_sequence_number) VALUES (?, ?)");
            $stmt->execute([$branchCodeBase, $reserve ? $sequenceNumber : 0]);
        }

        // تأكيد المعاملة
        $pdo->commit();

        // تنسيق رقم التسلسل (مثلاً: 01, 02, 03, ...)
        $formattedSequence = str_pad($sequenceNumber, 2, '0', STR_PAD_LEFT);

        // تنسيق رقم الطلب الكامل (مثلاً: MKL25-01, SNA25-01, ...)
        $orderNumber = $branchCodeBase . '25-' . $formattedSequence;
    } catch (PDOException $e) {
        // إلغاء المعاملة في حالة حدوث خطأ
        $pdo->rollBack();
        throw $e;
    }

    // إرجاع رقم الطلب مع معلومات تصحيح إضافية
    header('Content-Type: application/json');
    echo json_encode([
        'status' => 'success',
        'order_number' => $orderNumber,
        'sequence_number' => $sequenceNumber,
        'formatted_sequence' => $formattedSequence,
        'branch_code' => $branchCode,
        'branch_code_base' => $branchCodeBase,
        'reserve' => $reserve
    ]);

} catch (PDOException $e) {
    // تسجيل الخطأ في ملف السجل
    error_log("PDOException: " . $e->getMessage());

    // إرجاع رسالة الخطأ
    header('Content-Type: application/json');
    echo json_encode([
        'status' => 'error',
        'message' => 'فشل في الحصول على الرقم التسلسلي: ' . $e->getMessage()
    ]);
} catch (Exception $e) {
    // تسجيل الخطأ في ملف السجل
    error_log("Exception: " . $e->getMessage());

    // إرجاع رسالة الخطأ
    header('Content-Type: application/json');
    echo json_encode([
        'status' => 'error',
        'message' => $e->getMessage()
    ]);
}
?>
