<?php
/**
 * ملف مشترك لتضمين الملفات الأساسية في النظام
 * 
 * هذا الملف يقوم بتضمين جميع الملفات المشتركة التي يحتاجها النظام
 * ويجب تضمينه في بداية كل صفحة
 * 
 * @version 1.0
 * @author فريق التطوير
 */

// منع الوصول المباشر للملف
if (!defined('SYSTEM_LOADED')) {
    define('SYSTEM_LOADED', true);
} else {
    // تجنب تحميل الملف أكثر من مرة
    return;
}

// تحديد المسار الأساسي للمشروع
if (!defined('BASE_PATH')) {
    define('BASE_PATH', realpath(__DIR__ . '/..'));
}

// تحديد ما إذا كان النظام في وضع التطوير أم الإنتاج
if (!defined('DEVELOPMENT_MODE')) {
    define('DEVELOPMENT_MODE', false);
}

// تضمين ملف حماية النظام
require_once BASE_PATH . '/utils/system_protection.php';

// بدء الجلسة إذا لم تكن قد بدأت بالفعل
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// تضمين ملف الاتصال بقاعدة البيانات
require_once BASE_PATH . '/config/db_connection.php';

// تضمين ملف الوظائف المساعدة
require_once BASE_PATH . '/includes/helpers.php';

// تضمين ملف التكوين
require_once BASE_PATH . '/config/config.php';

// تضمين ملف الترجمة
if (file_exists(BASE_PATH . '/includes/translation.php')) {
    require_once BASE_PATH . '/includes/translation.php';
}

// تضمين ملف المصادقة
require_once BASE_PATH . '/includes/auth.php';

// دالة لتضمين الرأس المشترك
function includeHeader($title = 'نظام الشحن', $additionalStyles = []) {
    global $config;
    
    // تحديد المسار الأساسي للأصول
    $assetsPath = $config['base_url'] . '/assets';
    
    // تحديد ما إذا كان يجب استخدام الملفات المضغوطة
    $minified = !DEVELOPMENT_MODE;
    
    // تحديد ملفات CSS الأساسية
    $cssFiles = [
        $assetsPath . '/css/' . ($minified ? 'main.min.css' : 'main.css'),
        'https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.rtl.min.css',
        'https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css'
    ];
    
    // إضافة ملفات CSS إضافية
    $cssFiles = array_merge($cssFiles, $additionalStyles);
    
    // بدء مخرجات HTML
    ob_start();
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($title); ?> | <?php echo htmlspecialchars($config['site_name']); ?></title>
    
    <!-- ملفات CSS -->
    <?php foreach ($cssFiles as $css): ?>
    <link rel="stylesheet" href="<?php echo htmlspecialchars($css); ?>">
    <?php endforeach; ?>
    
    <!-- أيقونة الموقع -->
    <link rel="icon" href="<?php echo $assetsPath; ?>/images/favicon.ico" type="image/x-icon">
    
    <!-- العلامات الوصفية -->
    <meta name="description" content="<?php echo htmlspecialchars($config['site_description']); ?>">
    <meta name="keywords" content="<?php echo htmlspecialchars($config['site_keywords']); ?>">
    <meta name="author" content="<?php echo htmlspecialchars($config['site_author']); ?>">
    
    <!-- علامات الأمان -->
    <meta http-equiv="X-UA-Compatible" content="IE=edge">
    <meta http-equiv="Content-Security-Policy" content="default-src 'self' 'unsafe-inline' 'unsafe-eval' https://cdn.jsdelivr.net https://cdnjs.cloudflare.com https://fonts.googleapis.com https://fonts.gstatic.com; img-src 'self' data:; connect-src 'self'">
    <meta http-equiv="X-XSS-Protection" content="1; mode=block">
    <meta http-equiv="X-Content-Type-Options" content="nosniff">
    <meta http-equiv="Referrer-Policy" content="strict-origin-when-cross-origin">
    <meta name="robots" content="index, follow">
</head>
<body>
    <!-- شريط التنقل -->
    <nav class="navbar navbar-expand-lg navbar-dark bg-primary">
        <div class="container">
            <a class="navbar-brand" href="<?php echo $config['base_url']; ?>">
                <img src="<?php echo $assetsPath; ?>/images/logo.png" alt="<?php echo htmlspecialchars($config['site_name']); ?>" height="40">
            </a>
            <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navbarNav" aria-controls="navbarNav" aria-expanded="false" aria-label="Toggle navigation">
                <span class="navbar-toggler-icon"></span>
            </button>
            <div class="collapse navbar-collapse" id="navbarNav">
                <ul class="navbar-nav me-auto">
                    <li class="nav-item">
                        <a class="nav-link" href="<?php echo $config['base_url']; ?>">الرئيسية</a>
                    </li>
                    <?php if (isLoggedIn()): ?>
                    <li class="nav-item">
                        <a class="nav-link" href="<?php echo $config['base_url']; ?>/ASDRF43/unified-dashboard.php">لوحة التحكم</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="<?php echo $config['base_url']; ?>/orders/orders.php">الطلبات</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="<?php echo $config['base_url']; ?>/reports/weekly_reports.php">التقارير الأسبوعية</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="<?php echo $config['base_url']; ?>/reports/monthly_reports.php">التقارير الشهرية</a>
                    </li>
                    <?php if (hasPermission('admin')): ?>
                    <li class="nav-item">
                        <a class="nav-link" href="<?php echo $config['base_url']; ?>/admin/users.php">المستخدمين</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="<?php echo $config['base_url']; ?>/admin/branches.php">الفروع</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="<?php echo $config['base_url']; ?>/admin/settings.php">الإعدادات</a>
                    </li>
                    <?php endif; ?>
                    <?php endif; ?>
                </ul>
                <ul class="navbar-nav">
                    <?php if (isLoggedIn()): ?>
                    <li class="nav-item dropdown">
                        <a class="nav-link dropdown-toggle" href="#" id="userDropdown" role="button" data-bs-toggle="dropdown" aria-expanded="false">
                            <i class="fas fa-user-circle me-1"></i>
                            <?php echo htmlspecialchars($_SESSION['username']); ?>
                        </a>
                        <ul class="dropdown-menu dropdown-menu-end" aria-labelledby="userDropdown">
                            <li><a class="dropdown-item" href="<?php echo $config['base_url']; ?>/profile.php">الملف الشخصي</a></li>
                            <li><hr class="dropdown-divider"></li>
                            <li><a class="dropdown-item" href="<?php echo $config['base_url']; ?>/auth/logout.php">تسجيل الخروج</a></li>
                        </ul>
                    </li>
                    <?php else: ?>
                    <li class="nav-item">
                        <a class="nav-link" href="<?php echo $config['base_url']; ?>/auth/login.php">تسجيل الدخول</a>
                    </li>
                    <?php endif; ?>
                </ul>
            </div>
        </div>
    </nav>
    
    <!-- محتوى الصفحة -->
    <div class="container mt-4">
        <?php
        // عرض رسائل الفلاش
        $flashMessage = displayFlashMessage();
        if ($flashMessage) {
            $alertClass = 'alert-info';
            
            switch ($flashMessage['type']) {
                case 'success':
                    $alertClass = 'alert-success';
                    break;
                case 'error':
                    $alertClass = 'alert-danger';
                    break;
                case 'warning':
                    $alertClass = 'alert-warning';
                    break;
            }
        ?>
        <div class="alert <?php echo $alertClass; ?> alert-dismissible fade show" role="alert">
            <?php echo htmlspecialchars($flashMessage['message']); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
        <?php } ?>
<?php
    // إرجاع مخرجات HTML
    return ob_get_clean();
}

// دالة لتضمين التذييل المشترك
function includeFooter($additionalScripts = []) {
    global $config;
    
    // تحديد المسار الأساسي للأصول
    $assetsPath = $config['base_url'] . '/assets';
    
    // تحديد ما إذا كان يجب استخدام الملفات المضغوطة
    $minified = !DEVELOPMENT_MODE;
    
    // تحديد ملفات JavaScript الأساسية
    $jsFiles = [
        'https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js',
        $assetsPath . '/js/' . ($minified ? 'main.min.js' : 'main.js')
    ];
    
    // إضافة ملفات JavaScript إضافية
    $jsFiles = array_merge($jsFiles, $additionalScripts);
    
    // بدء مخرجات HTML
    ob_start();
?>
    </div>
    
    <!-- التذييل -->
    <footer class="bg-dark text-white py-4 mt-5">
        <div class="container">
            <div class="row">
                <div class="col-md-6">
                    <h5><?php echo htmlspecialchars($config['site_name']); ?></h5>
                    <p><?php echo htmlspecialchars($config['site_description']); ?></p>
                </div>
                <div class="col-md-3">
                    <h5>روابط سريعة</h5>
                    <ul class="list-unstyled">
                        <li><a href="<?php echo $config['base_url']; ?>" class="text-white">الرئيسية</a></li>
                        <li><a href="<?php echo $config['base_url']; ?>/about.php" class="text-white">من نحن</a></li>
                        <li><a href="<?php echo $config['base_url']; ?>/contact.php" class="text-white">اتصل بنا</a></li>
                    </ul>
                </div>
                <div class="col-md-3">
                    <h5>تواصل معنا</h5>
                    <ul class="list-unstyled">
                        <li><i class="fas fa-map-marker-alt me-2"></i> <?php echo htmlspecialchars($config['company_address']); ?></li>
                        <li><i class="fas fa-phone me-2"></i> <?php echo htmlspecialchars($config['company_phone']); ?></li>
                        <li><i class="fas fa-envelope me-2"></i> <?php echo htmlspecialchars($config['company_email']); ?></li>
                    </ul>
                </div>
            </div>
            <hr>
            <div class="text-center">
                <p>&copy; <?php echo date('Y'); ?> <?php echo htmlspecialchars($config['site_name']); ?>. جميع الحقوق محفوظة.</p>
            </div>
        </div>
    </footer>
    
    <!-- ملفات JavaScript -->
    <?php foreach ($jsFiles as $js): ?>
    <script src="<?php echo htmlspecialchars($js); ?>"></script>
    <?php endforeach; ?>
</body>
</html>
<?php
    // إرجاع مخرجات HTML
    return ob_get_clean();
}

// دالة للتحقق من تسجيل الدخول
function isLoggedIn() {
    return isset($_SESSION['user_id']) && isset($_SESSION['logged_in']) && $_SESSION['logged_in'] === true;
}

// دالة للتحقق من الصلاحيات
function hasPermission($role) {
    if (!isLoggedIn()) {
        return false;
    }
    
    if ($_SESSION['role'] === 'admin') {
        return true;
    }
    
    if (is_array($role)) {
        return in_array($_SESSION['role'], $role);
    } else {
        return $_SESSION['role'] === $role;
    }
}

// دالة للتحقق من الصلاحيات وإعادة التوجيه إذا لم تكن متوفرة
function requirePermission($role) {
    if (!hasPermission($role)) {
        redirectWithMessage('/auth/login.php', 'يجب تسجيل الدخول للوصول إلى هذه الصفحة', 'error');
    }
}
