<?php
/**
 * ملف الدوال المساعدة
 * يحتوي هذا الملف على دوال مساعدة متنوعة تستخدم في جميع أنحاء التطبيق
 */

// تم نقل هذه الدوال إلى ملف config_new.php
// /**
//  * دالة لتنظيف المدخلات
//  * @param string $input - النص المدخل
//  * @return string - النص بعد التنظيف
//  */
// function sanitizeInput($input) {
//     // إزالة المسافات الزائدة من بداية ونهاية النص
//     $input = trim($input);
//
//     // إزالة الأكواد الضارة
//     $input = htmlspecialchars($input, ENT_QUOTES, 'UTF-8');
//
//     return $input;
// }
//
// /**
//  * دالة لإنشاء رمز CSRF للحماية من هجمات CSRF
//  * @return string - رمز CSRF
//  */
// function generateCSRFToken() {
//     if (!isset($_SESSION['csrf_token'])) {
//         $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
//     }
//
//     return $_SESSION['csrf_token'];
// }
//
// /**
//  * دالة للتحقق من صحة رمز CSRF
//  * @param string $token - الرمز المرسل من النموذج
//  * @return bool - هل الرمز صحيح أم لا
//  */
// function validateCSRFToken($token) {
//     if (!isset($_SESSION['csrf_token']) || $token !== $_SESSION['csrf_token']) {
//         return false;
//     }
//
//     return true;
// }

/**
 * دالة لتنسيق التاريخ بالتنسيق العربي
 * @param string $date - التاريخ بتنسيق Y-m-d
 * @return string - التاريخ بتنسيق d/m/Y
 */
function formatDate($date) {
    if (empty($date)) {
        return '';
    }

    $timestamp = strtotime($date);
    if ($timestamp === false) {
        return $date;
    }

    return date('d/m/Y', $timestamp);
}

/**
 * دالة لتنسيق المبلغ المالي
 * @param float $amount - المبلغ
 * @param string $currency - العملة (اختياري)
 * @return string - المبلغ المنسق
 */
function formatAmount($amount, $currency = '') {
    if (!is_numeric($amount)) {
        return $amount;
    }

    $formattedAmount = number_format($amount, 2, '.', ',');

    if (!empty($currency)) {
        return $formattedAmount . ' ' . $currency;
    }

    return $formattedAmount;
}

/**
 * دالة لتحويل حالة الطلب إلى نص مقروء
 * @param string $status - حالة الطلب
 * @return string - النص المقروء
 */
function formatOrderStatus($status) {
    $statusMap = [
        'pending' => 'قيد المعالجة',
        'processing' => 'قيد المعالجة',
        'in_transit' => 'في الطريق',
        'delivered' => 'تم التسليم',
        'cancelled' => 'ملغى'
    ];

    if (isset($statusMap[$status])) {
        return $statusMap[$status];
    }

    return $status;
}

/**
 * دالة لتحويل حالة الدفع إلى نص مقروء
 * @param string $status - حالة الدفع
 * @return string - النص المقروء
 */
function formatPaymentStatus($status) {
    $statusMap = [
        'paid' => 'مدفوع',
        'partial' => 'جزئي',
        'unpaid' => 'غير مدفوع'
    ];

    if (isset($statusMap[$status])) {
        return $statusMap[$status];
    }

    return $status;
}

/**
 * دالة لتحويل نوع الخدمة إلى نص مقروء
 * @param string $type - نوع الخدمة
 * @return string - النص المقروء
 */
function formatServiceType($type) {
    $typeMap = [
        'shipping' => 'شحن',
        'delivery' => 'توصيل',
        'storage' => 'تخزين',
        'packaging' => 'تغليف',
        'customs_clearance' => 'تخليص جمركي'
    ];

    if (isset($typeMap[$type])) {
        return $typeMap[$type];
    }

    return $type;
}

/**
 * دالة لتحويل نوع الدفع إلى نص مقروء
 * @param string $type - نوع الدفع
 * @return string - النص المقروء
 */
function formatPaymentType($type) {
    $typeMap = [
        'cash' => 'نقدي',
        'bank_transfer' => 'تحويل بنكي',
        'credit_card' => 'بطاقة ائتمان',
        'debit_card' => 'بطاقة خصم',
        'cheque' => 'شيك'
    ];

    if (isset($typeMap[$type])) {
        return $typeMap[$type];
    }

    return $type;
}

/**
 * دالة لتحويل العملة إلى نص مقروء
 * @param string $currency - العملة
 * @return string - النص المقروء
 */
function formatCurrency($currency) {
    $currencyMap = [
        'USD' => 'دولار أمريكي',
        'SAR' => 'ريال سعودي',
        'YER' => 'ريال يمني',
        'AED' => 'درهم إماراتي',
        'OMR' => 'ريال عماني',
        'KWD' => 'دينار كويتي',
        'QAR' => 'ريال قطري',
        'BHD' => 'دينار بحريني'
    ];

    if (isset($currencyMap[$currency])) {
        return $currencyMap[$currency];
    }

    return $currency;
}

/**
 * دالة لتحويل الوزن إلى نص مقروء
 * @param float $weight - الوزن
 * @param string $unit - وحدة الوزن (اختياري)
 * @return string - النص المقروء
 */
function formatWeight($weight, $unit = 'kg') {
    if (!is_numeric($weight)) {
        return $weight;
    }

    $formattedWeight = number_format($weight, 2, '.', ',');

    $unitMap = [
        'kg' => 'كجم',
        'g' => 'جم',
        'lb' => 'رطل',
        'oz' => 'أونصة',
        'ton' => 'طن'
    ];

    $formattedUnit = isset($unitMap[$unit]) ? $unitMap[$unit] : $unit;

    return $formattedWeight . ' ' . $formattedUnit;
}

/**
 * دالة لتحويل الكمية إلى نص مقروء
 * @param int $quantity - الكمية
 * @param string $unit - وحدة الكمية (اختياري)
 * @return string - النص المقروء
 */
function formatQuantity($quantity, $unit = '') {
    if (!is_numeric($quantity)) {
        return $quantity;
    }

    $formattedQuantity = number_format($quantity, 0, '.', ',');

    if (!empty($unit)) {
        return $formattedQuantity . ' ' . $unit;
    }

    return $formattedQuantity;
}

/**
 * دالة لتحويل المدة إلى نص مقروء
 * @param string $duration - المدة
 * @return string - النص المقروء
 */
function formatDuration($duration) {
    // يمكن تنفيذ المزيد من المنطق هنا حسب تنسيق المدة
    return $duration;
}

/**
 * دالة لتحويل رقم الهاتف إلى تنسيق موحد
 * @param string $phone - رقم الهاتف
 * @return string - رقم الهاتف المنسق
 */
function formatPhoneNumber($phone) {
    // إزالة جميع الأحرف غير الرقمية
    $phone = preg_replace('/[^0-9]/', '', $phone);

    // يمكن تنفيذ المزيد من المنطق هنا حسب تنسيق رقم الهاتف
    return $phone;
}

/**
 * دالة لتحويل النص إلى URL آمن
 * @param string $text - النص
 * @return string - URL آمن
 */
function slugify($text) {
    // تحويل النص إلى أحرف صغيرة
    $text = mb_strtolower($text, 'UTF-8');

    // استبدال الأحرف غير الإنجليزية بأحرف إنجليزية
    $text = str_replace(['أ', 'إ', 'آ'], 'a', $text);
    $text = str_replace(['ب'], 'b', $text);
    $text = str_replace(['ت', 'ة'], 't', $text);
    $text = str_replace(['ث'], 'th', $text);
    $text = str_replace(['ج'], 'j', $text);
    $text = str_replace(['ح'], 'h', $text);
    $text = str_replace(['خ'], 'kh', $text);
    $text = str_replace(['د'], 'd', $text);
    $text = str_replace(['ذ'], 'th', $text);
    $text = str_replace(['ر'], 'r', $text);
    $text = str_replace(['ز'], 'z', $text);
    $text = str_replace(['س'], 's', $text);
    $text = str_replace(['ش'], 'sh', $text);
    $text = str_replace(['ص'], 's', $text);
    $text = str_replace(['ض'], 'd', $text);
    $text = str_replace(['ط'], 't', $text);
    $text = str_replace(['ظ'], 'z', $text);
    $text = str_replace(['ع'], 'a', $text);
    $text = str_replace(['غ'], 'gh', $text);
    $text = str_replace(['ف'], 'f', $text);
    $text = str_replace(['ق'], 'q', $text);
    $text = str_replace(['ك'], 'k', $text);
    $text = str_replace(['ل'], 'l', $text);
    $text = str_replace(['م'], 'm', $text);
    $text = str_replace(['ن'], 'n', $text);
    $text = str_replace(['ه'], 'h', $text);
    $text = str_replace(['و'], 'w', $text);
    $text = str_replace(['ي', 'ى', 'ئ'], 'y', $text);

    // استبدال المسافات بشرطات
    $text = preg_replace('/\s+/', '-', $text);

    // إزالة جميع الأحرف غير الأبجدية الرقمية والشرطات
    $text = preg_replace('/[^a-z0-9\-]/', '', $text);

    // إزالة الشرطات المتكررة
    $text = preg_replace('/-+/', '-', $text);

    // إزالة الشرطات من بداية ونهاية النص
    $text = trim($text, '-');

    return $text;
}

/**
 * دالة للتحقق من صحة البريد الإلكتروني
 * @param string $email - البريد الإلكتروني
 * @return bool - هل البريد الإلكتروني صحيح أم لا
 */
function isValidEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

/**
 * دالة للتحقق من قوة كلمة المرور
 * @param string $password - كلمة المرور
 * @return bool - هل كلمة المرور قوية أم لا
 */
function isStrongPassword($password) {
    // التحقق من طول كلمة المرور (6 أحرف على الأقل)
    if (strlen($password) < 6) {
        return false;
    }

    // للتطوير: تم تبسيط شروط قوة كلمة المرور مؤقتًا
    // في بيئة الإنتاج، يجب إعادة تفعيل الشروط التالية

    /*
    // التحقق من وجود حرف كبير واحد على الأقل
    if (!preg_match('/[A-Z]/', $password)) {
        return false;
    }

    // التحقق من وجود حرف صغير واحد على الأقل
    if (!preg_match('/[a-z]/', $password)) {
        return false;
    }

    // التحقق من وجود رقم واحد على الأقل
    if (!preg_match('/[0-9]/', $password)) {
        return false;
    }
    */

    return true;
}

/**
 * دالة لتسجيل دخول المستخدم
 * @param string $email - البريد الإلكتروني
 * @param string $password - كلمة المرور
 * @param bool $remember - تذكر المستخدم
 * @return array - مصفوفة تحتوي على حالة تسجيل الدخول ورسالة
 */
function loginUser($emailOrUsername, $password, $remember = false) {
    global $conn;

    try {
        // التحقق من وجود المستخدم (بالبريد الإلكتروني أو اسم المستخدم)
        $stmt = $conn->prepare("SELECT * FROM users WHERE email = ? OR username = ?");
        $stmt->execute([$emailOrUsername, $emailOrUsername]);
        $user = $stmt->fetch();

        if (!$user) {
            return [
                'success' => false,
                'message' => 'اسم المستخدم أو كلمة المرور غير صحيحة'
            ];
        }

        // التحقق من حالة المستخدم
        if ($user['status'] !== 'active') {
            return [
                'success' => false,
                'message' => 'حسابك غير مفعل. يرجى الانتظار حتى يتم تفعيل حسابك من قبل الإدارة'
            ];
        }

        // التحقق من كلمة المرور
        if (!password_verify($password, $user['password'])) {
            return [
                'success' => false,
                'message' => 'اسم المستخدم أو كلمة المرور غير صحيحة'
            ];
        }

        // تحديث وقت آخر تسجيل دخول
        $updateStmt = $conn->prepare("UPDATE users SET last_login = NOW() WHERE id = ?");
        $updateStmt->execute([$user['id']]);

        // تخزين معلومات المستخدم في الجلسة
        $_SESSION['user_id'] = $user['id'];
        $_SESSION['user_name'] = $user['full_name'];
        $_SESSION['user_email'] = $user['email'];
        $_SESSION['user_role'] = $user['role'];
        $_SESSION['user_branch_id'] = $user['branch_id'];

        // إذا كان المستخدم اختار تذكره
        if ($remember) {
            // إنشاء رمز تذكر
            $token = bin2hex(random_bytes(32));
            $expires = time() + (30 * 24 * 60 * 60); // 30 يوم

            // تخزين الرمز في ملف تعريف الارتباط
            setcookie('remember_token', $token, $expires, '/', '', false, true);

            // تخزين الرمز في قاعدة البيانات (يمكن إنشاء جدول منفصل لتخزين رموز التذكر)
            // هذا مثال بسيط، في التطبيقات الحقيقية يجب استخدام نهج أكثر أمانًا
            $updateTokenStmt = $conn->prepare("UPDATE users SET remember_token = ? WHERE id = ?");
            $updateTokenStmt->execute([$token, $user['id']]);
        }

        return [
            'success' => true,
            'message' => 'تم تسجيل الدخول بنجاح',
            'user' => $user
        ];
    } catch (PDOException $e) {
        return [
            'success' => false,
            'message' => 'حدث خطأ أثناء تسجيل الدخول: ' . $e->getMessage()
        ];
    }
}

/**
 * دالة لتسجيل خروج المستخدم
 */
function logoutUser() {
    // حذف متغيرات الجلسة
    unset($_SESSION['user_id']);
    unset($_SESSION['user_name']);
    unset($_SESSION['user_email']);
    unset($_SESSION['user_role']);
    unset($_SESSION['user_branch_id']);

    // حذف ملف تعريف الارتباط الخاص بتذكر المستخدم
    if (isset($_COOKIE['remember_token'])) {
        setcookie('remember_token', '', time() - 3600, '/', '', false, true);
    }

    // إعادة توجيه المستخدم إلى صفحة تسجيل الدخول
    header('Location: login.php');
    exit;
}

/**
 * دالة لتسجيل مستخدم جديد
 * @param string $fullName - الاسم الكامل
 * @param string $email - البريد الإلكتروني
 * @param string $password - كلمة المرور
 * @param string $role - دور المستخدم (اختياري)
 * @return array - مصفوفة تحتوي على حالة التسجيل ورسالة
 */
function registerUser($fullName, $email, $password, $role = 'user') {
    global $conn;

    // تسجيل بداية الدالة للتصحيح
    error_log("دالة registerUser: بدء التنفيذ");
    error_log("متغير الاتصال conn: " . (isset($conn) ? 'موجود' : 'غير موجود'));

    try {
        // التحقق من صحة البريد الإلكتروني
        if (!isValidEmail($email)) {
            return [
                'success' => false,
                'message' => 'البريد الإلكتروني غير صحيح'
            ];
        }

        // التحقق من قوة كلمة المرور
        if (strlen($password) < 6) {
            return [
                'success' => false,
                'message' => 'كلمة المرور ضعيفة. يجب أن تحتوي على 6 أحرف على الأقل'
            ];
        }

        // إنشاء اسم مستخدم فريد من البريد الإلكتروني
        $username = explode('@', $email)[0];

        // التحقق من وجود المستخدم بالبريد الإلكتروني
        $stmt = $conn->prepare("SELECT id FROM users WHERE email = ?");
        $stmt->execute([$email]);

        if ($stmt->rowCount() > 0) {
            return [
                'success' => false,
                'message' => 'البريد الإلكتروني مستخدم بالفعل'
            ];
        }

        // التحقق من وجود اسم المستخدم
        $stmt = $conn->prepare("SELECT id FROM users WHERE username = ?");
        $stmt->execute([$username]);

        // إذا كان اسم المستخدم موجودًا، أضف رقمًا عشوائيًا إليه
        if ($stmt->rowCount() > 0) {
            $username = $username . rand(100, 999);
        }

        // تشفير كلمة المرور
        $hashedPassword = password_hash($password, PASSWORD_DEFAULT);

        // تاريخ التسجيل الحالي
        $registrationDate = date('Y-m-d H:i:s');

        // إدراج المستخدم الجديد - تعديل أسماء الحقول لتتوافق مع هيكل الجدول الفعلي
        $insertStmt = $conn->prepare("
            INSERT INTO users (username, full_name, email, password, role, status, registration_date, phone)
            VALUES (?, ?, ?, ?, ?, 'pending', ?, '123456789')
        ");

        // تسجيل الاستعلام للتصحيح
        error_log("استعلام إدراج المستخدم: INSERT INTO users (username, full_name, email, password, role, status, registration_date, phone) VALUES ('$username', '$fullName', '$email', '[HIDDEN]', '$role', 'pending', '$registrationDate', '123456789')");

        try {
            $result = $insertStmt->execute([$username, $fullName, $email, $hashedPassword, $role, $registrationDate]);
            error_log("نتيجة تنفيذ الاستعلام: " . ($result ? 'نجاح' : 'فشل'));
        } catch (PDOException $execException) {
            error_log("خطأ أثناء تنفيذ استعلام إدراج المستخدم: " . $execException->getMessage() . " (رمز الخطأ: " . $execException->getCode() . ")");
            throw $execException; // إعادة رمي الاستثناء ليتم التقاطه في الكتلة الخارجية
        }

        if ($result) {
            return [
                'success' => true,
                'message' => 'تم إنشاء الحساب بنجاح. يرجى الانتظار حتى يتم تفعيل حسابك من قبل الإدارة'
            ];
        } else {
            return [
                'success' => false,
                'message' => 'فشل إدراج المستخدم الجديد في قاعدة البيانات'
            ];
        }
    } catch (PDOException $e) {
        // تسجيل الخطأ في ملف السجل للتشخيص
        error_log('خطأ في تسجيل المستخدم: ' . $e->getMessage());

        return [
            'success' => false,
            'message' => 'حدث خطأ أثناء إنشاء الحساب: ' . $e->getMessage() . ' (رمز الخطأ: ' . $e->getCode() . ')'
        ];
    }
}

// تم نقل هذه الدالة إلى ملف config_new.php
// /**
//  * دالة للتحقق من حالة تسجيل الدخول
//  * @return bool - هل المستخدم مسجل الدخول أم لا
//  */
// function isLoggedIn() {
//     return isset($_SESSION['user_id']);
// }

/**
 * دالة للتحقق من صلاحيات المستخدم
 * @param string $requiredRole - الدور المطلوب
 * @return bool - هل المستخدم لديه الصلاحية أم لا
 */
function hasRole($requiredRole) {
    if (!isLoggedIn()) {
        return false;
    }

    $userRole = $_SESSION['user_role'];

    // إذا كان المستخدم مديرًا، فلديه جميع الصلاحيات
    if ($userRole === 'admin') {
        return true;
    }

    // التحقق من الدور المطلوب
    switch ($requiredRole) {
        case 'admin':
            return $userRole === 'admin';
        case 'manager':
            return in_array($userRole, ['admin', 'manager']);
        case 'employee':
            return in_array($userRole, ['admin', 'manager', 'employee']);
        case 'client':
            return in_array($userRole, ['admin', 'manager', 'employee', 'client']);
        default:
            return false;
    }
}
?>
