<?php
/**
 * مدير الإشعارات
 * يحتوي على الدوال المسؤولة عن إدارة الإشعارات في النظام
 */

// التأكد من بدء الجلسة
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// التأكد من وجود اتصال بقاعدة البيانات
if (!isset($pdo)) {
    require_once __DIR__ . '/../config/db_connection.php';
}

/**
 * دالة للتحقق من وجود جدول الإشعارات وإنشائه إذا لم يكن موجودًا
 */
function createNotificationsTableIfNotExists() {
    global $pdo;
    
    try {
        // التحقق من وجود جدول الإشعارات
        $stmt = $pdo->query("SHOW TABLES LIKE 'notifications'");
        $tableExists = $stmt->rowCount() > 0;
        
        if (!$tableExists) {
            // إنشاء جدول الإشعارات
            $pdo->exec("
                CREATE TABLE IF NOT EXISTS `notifications` (
                  `id` INT AUTO_INCREMENT PRIMARY KEY,
                  `user_id` INT DEFAULT NULL COMMENT 'معرف المستخدم المستلم (NULL للكل)',
                  `title` VARCHAR(255) NOT NULL COMMENT 'عنوان الإشعار',
                  `message` TEXT NOT NULL COMMENT 'نص الإشعار',
                  `type` VARCHAR(50) DEFAULT 'info' COMMENT 'نوع الإشعار (info, success, warning, error)',
                  `link` VARCHAR(255) DEFAULT NULL COMMENT 'رابط متعلق بالإشعار',
                  `is_read` TINYINT(1) DEFAULT 0 COMMENT 'هل تم قراءة الإشعار',
                  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP COMMENT 'تاريخ الإنشاء',
                  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP COMMENT 'تاريخ التحديث',
                  INDEX (`user_id`),
                  INDEX (`is_read`),
                  INDEX (`created_at`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
            ");
            
            // إضافة بعض الإشعارات الافتراضية للمدير
            $pdo->exec("
                INSERT INTO `notifications` (`user_id`, `title`, `message`, `type`, `link`) VALUES
                (NULL, 'مرحبًا بك في النظام', 'تم تثبيت النظام بنجاح وهو جاهز للاستخدام', 'success', NULL),
                (NULL, 'تحديث النظام', 'يتوفر تحديث جديد للنظام، يرجى مراجعة صفحة الإعدادات', 'info', 'settings.php'),
                (NULL, 'طلب جديد', 'تم إنشاء طلب شحن جديد برقم MKL25-01', 'info', 'orders/view.php?id=1')
            ");
            
            return true;
        }
        
        return $tableExists;
    } catch (PDOException $e) {
        error_log("خطأ في إنشاء جدول الإشعارات: " . $e->getMessage());
        return false;
    }
}

/**
 * دالة لإضافة إشعار جديد
 * 
 * @param string $title عنوان الإشعار
 * @param string $message نص الإشعار
 * @param string $type نوع الإشعار (info, success, warning, error)
 * @param int|null $userId معرف المستخدم (NULL للكل)
 * @param string|null $link رابط متعلق بالإشعار
 * @return bool نجاح العملية
 */
function addNotification($title, $message, $type = 'info', $userId = null, $link = null) {
    global $pdo;
    
    try {
        // التأكد من وجود جدول الإشعارات
        createNotificationsTableIfNotExists();
        
        // إضافة الإشعار
        $stmt = $pdo->prepare("
            INSERT INTO `notifications` (`user_id`, `title`, `message`, `type`, `link`)
            VALUES (?, ?, ?, ?, ?)
        ");
        
        return $stmt->execute([$userId, $title, $message, $type, $link]);
    } catch (PDOException $e) {
        error_log("خطأ في إضافة إشعار: " . $e->getMessage());
        return false;
    }
}

/**
 * دالة للحصول على إشعارات المستخدم
 * 
 * @param int|null $userId معرف المستخدم (NULL للحصول على الإشعارات العامة)
 * @param int $limit عدد الإشعارات المطلوبة
 * @param bool $includeRead تضمين الإشعارات المقروءة
 * @return array مصفوفة من الإشعارات
 */
function getUserNotifications($userId = null, $limit = 5, $includeRead = false) {
    global $pdo;
    
    try {
        // التأكد من وجود جدول الإشعارات
        if (!createNotificationsTableIfNotExists()) {
            return [];
        }
        
        // بناء الاستعلام
        $sql = "
            SELECT * FROM `notifications`
            WHERE (user_id = ? OR user_id IS NULL)
        ";
        
        // إضافة شرط الإشعارات غير المقروءة إذا لزم الأمر
        if (!$includeRead) {
            $sql .= " AND is_read = 0";
        }
        
        // ترتيب النتائج وتحديد العدد
        $sql .= " ORDER BY created_at DESC LIMIT ?";
        
        // تنفيذ الاستعلام
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$userId, $limit]);
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("خطأ في الحصول على إشعارات المستخدم: " . $e->getMessage());
        return [];
    }
}

/**
 * دالة لتحديث حالة قراءة الإشعار
 * 
 * @param int $notificationId معرف الإشعار
 * @param bool $isRead حالة القراءة
 * @return bool نجاح العملية
 */
function markNotificationAsRead($notificationId, $isRead = true) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            UPDATE `notifications`
            SET is_read = ?
            WHERE id = ?
        ");
        
        return $stmt->execute([$isRead ? 1 : 0, $notificationId]);
    } catch (PDOException $e) {
        error_log("خطأ في تحديث حالة قراءة الإشعار: " . $e->getMessage());
        return false;
    }
}

/**
 * دالة لحساب عدد الإشعارات غير المقروءة للمستخدم
 * 
 * @param int|null $userId معرف المستخدم
 * @return int عدد الإشعارات غير المقروءة
 */
function countUnreadNotifications($userId = null) {
    global $pdo;
    
    try {
        // التأكد من وجود جدول الإشعارات
        if (!createNotificationsTableIfNotExists()) {
            return 0;
        }
        
        $stmt = $pdo->prepare("
            SELECT COUNT(*) as count
            FROM `notifications`
            WHERE (user_id = ? OR user_id IS NULL)
            AND is_read = 0
        ");
        
        $stmt->execute([$userId]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        return (int) $result['count'];
    } catch (PDOException $e) {
        error_log("خطأ في حساب عدد الإشعارات غير المقروءة: " . $e->getMessage());
        return 0;
    }
}
