<?php
/**
 * ملف إدارة الصلاحيات
 * يحتوي هذا الملف على دوال إدارة الصلاحيات والتحقق منها
 *
 * @version 1.0
 * @author System Developer
 */

// تضمين ملف إدارة الجلسات
require_once __DIR__ . '/session_manager.php';

// تعريف الأدوار
define('ROLE_ADMIN', 'admin');         // مدير النظام
define('ROLE_MANAGER', 'manager');     // مشرف
define('ROLE_EMPLOYEE', 'employee');   // موظف
define('ROLE_CLIENT', 'client');       // عميل

// تعريف الصلاحيات
$PERMISSIONS = [
    // صلاحيات لوحة التحكم
    'dashboard_view' => [ROLE_ADMIN, ROLE_MANAGER, ROLE_EMPLOYEE, ROLE_CLIENT],
    'dashboard_admin_view' => [ROLE_ADMIN, ROLE_MANAGER],
    'dashboard_view_own' => [ROLE_EMPLOYEE, ROLE_CLIENT], // صلاحية عرض لوحة التحكم الخاصة بالمستخدم فقط

    // صلاحيات إدارة المستخدمين
    'users_view' => [ROLE_ADMIN, ROLE_MANAGER],
    'users_add' => [ROLE_ADMIN],
    'users_edit' => [ROLE_ADMIN],
    'users_delete' => [ROLE_ADMIN],
    'users_approve' => [ROLE_ADMIN, ROLE_MANAGER],

    // صلاحيات إدارة الطلبات
    'orders_view' => [ROLE_ADMIN, ROLE_MANAGER, ROLE_EMPLOYEE, ROLE_CLIENT],
    'orders_view_all' => [ROLE_ADMIN, ROLE_MANAGER],
    'orders_view_own' => [ROLE_EMPLOYEE, ROLE_CLIENT], // صلاحية عرض الطلبات الخاصة بالمستخدم فقط
    'orders_add' => [ROLE_ADMIN, ROLE_MANAGER, ROLE_EMPLOYEE, ROLE_CLIENT],
    'orders_edit' => [ROLE_ADMIN, ROLE_MANAGER],
    'orders_edit_own' => [ROLE_EMPLOYEE, ROLE_CLIENT], // صلاحية تعديل الطلبات الخاصة بالمستخدم فقط
    'orders_delete' => [ROLE_ADMIN, ROLE_MANAGER],
    'orders_archive' => [ROLE_ADMIN, ROLE_MANAGER],
    'orders_export' => [ROLE_ADMIN, ROLE_MANAGER, ROLE_EMPLOYEE, ROLE_CLIENT],
    'orders_export_own' => [ROLE_EMPLOYEE, ROLE_CLIENT], // صلاحية تصدير الطلبات الخاصة بالمستخدم فقط
    'orders_manage_own' => [ROLE_EMPLOYEE, ROLE_CLIENT], // صلاحية إدارة الطلبات الخاصة بالمستخدم فقط

    // صلاحيات التقارير
    'reports_view' => [ROLE_ADMIN, ROLE_MANAGER],
    'reports_view_all' => [ROLE_ADMIN, ROLE_MANAGER],
    'reports_view_own' => [ROLE_EMPLOYEE, ROLE_CLIENT], // صلاحية عرض التقارير الخاصة بالمستخدم فقط
    'reports_export' => [ROLE_ADMIN, ROLE_MANAGER],
    'reports_export_own' => [ROLE_EMPLOYEE, ROLE_CLIENT], // صلاحية تصدير التقارير الخاصة بالمستخدم فقط
    'weekly_reports_view_own' => [ROLE_EMPLOYEE, ROLE_CLIENT], // صلاحية عرض التقارير الأسبوعية الخاصة بالمستخدم
    'monthly_reports_view_own' => [ROLE_EMPLOYEE, ROLE_CLIENT], // صلاحية عرض التقارير الشهرية الخاصة بالمستخدم

    // صلاحيات إعدادات النظام
    'settings_view' => [ROLE_ADMIN],
    'settings_edit' => [ROLE_ADMIN],

    // صلاحيات الدعم الفني
    'support_view' => [ROLE_ADMIN, ROLE_MANAGER, ROLE_EMPLOYEE, ROLE_CLIENT],
    'support_create' => [ROLE_EMPLOYEE, ROLE_CLIENT], // صلاحية إنشاء طلبات دعم فني
    'support_view_own' => [ROLE_EMPLOYEE, ROLE_CLIENT], // صلاحية عرض طلبات الدعم الفني الخاصة بالمستخدم
    'support_respond' => [ROLE_ADMIN, ROLE_MANAGER],
    'support_contact_admin' => [ROLE_EMPLOYEE, ROLE_CLIENT], // صلاحية التواصل مع الإدارة
];

/**
 * التحقق من صلاحية المستخدم
 *
 * @param string $permission الصلاحية المطلوبة
 * @return bool هل يملك المستخدم الصلاحية
 */
function hasPermission($permission) {
    global $PERMISSIONS;

    // التحقق من تسجيل دخول المستخدم
    if (!isset($_SESSION['user_id'])) {
        return false;
    }

    // الحصول على دور المستخدم
    $userRole = $_SESSION['role'] ?? $_SESSION['user_role'] ?? '';

    // التحقق من وجود الصلاحية
    if (!isset($PERMISSIONS[$permission])) {
        return false;
    }

    // التحقق من وجود دور المستخدم في قائمة الأدوار المسموح لها بالصلاحية
    return in_array($userRole, $PERMISSIONS[$permission]);
}

/**
 * التحقق من دور المستخدم
 *
 * @param string $role الدور المطلوب
 * @return bool هل يملك المستخدم الدور
 */
function hasRole($role) {
    // التحقق من تسجيل دخول المستخدم
    if (!isset($_SESSION['user_id'])) {
        return false;
    }

    // الحصول على دور المستخدم
    $userRole = $_SESSION['role'] ?? $_SESSION['user_role'] ?? '';

    // إذا كان المستخدم مديرًا، فلديه جميع الصلاحيات
    if ($userRole === ROLE_ADMIN) {
        return true;
    }

    // التحقق من الدور المطلوب
    switch ($role) {
        case ROLE_ADMIN:
            return $userRole === ROLE_ADMIN;
        case ROLE_MANAGER:
            return in_array($userRole, [ROLE_ADMIN, ROLE_MANAGER]);
        case ROLE_EMPLOYEE:
            return in_array($userRole, [ROLE_ADMIN, ROLE_MANAGER, ROLE_EMPLOYEE]);
        case ROLE_CLIENT:
            return in_array($userRole, [ROLE_ADMIN, ROLE_MANAGER, ROLE_EMPLOYEE, ROLE_CLIENT]);
        default:
            return false;
    }
}

/**
 * التحقق من امتلاك المستخدم لأي من الأدوار المحددة
 *
 * @param array $roles مصفوفة الأدوار
 * @return bool هل يملك المستخدم أي من الأدوار
 */
function hasAnyRole($roles) {
    // التحقق من كل دور
    foreach ($roles as $role) {
        if (hasRole($role)) {
            return true;
        }
    }

    return false;
}

/**
 * التحقق مما إذا كان الطلب ملكاً للمستخدم الحالي
 *
 * @param int $orderId معرف الطلب
 * @return bool هل الطلب ملك للمستخدم الحالي
 */
function isOrderOwner($orderId) {
    global $conn, $pdo;

    // التحقق من تسجيل دخول المستخدم
    if (!isset($_SESSION['user_id'])) {
        return false;
    }

    // الحصول على معرف المستخدم الحالي
    $userId = $_SESSION['user_id'];

    // المدير والمشرف لديهم صلاحية على جميع الطلبات
    if (hasRole(ROLE_ADMIN) || hasRole(ROLE_MANAGER)) {
        return true;
    }

    try {
        // استخدام PDO إذا كان متاحاً
        if (isset($pdo)) {
            $stmt = $pdo->prepare("SELECT COUNT(*) FROM orders WHERE id = ? AND user_id = ?");
            $stmt->execute([$orderId, $userId]);
            return $stmt->fetchColumn() > 0;
        }
        // استخدام mysqli إذا كان متاحاً
        else if (isset($conn)) {
            $stmt = $conn->prepare("SELECT COUNT(*) FROM orders WHERE id = ? AND user_id = ?");
            $stmt->bind_param("ii", $orderId, $userId);
            $stmt->execute();
            $stmt->bind_result($count);
            $stmt->fetch();
            $stmt->close();
            return $count > 0;
        }
    } catch (Exception $e) {
        error_log("خطأ في التحقق من ملكية الطلب: " . $e->getMessage());
    }

    return false;
}

/**
 * التحقق من صلاحية المستخدم وتوجيهه إذا لم يكن لديه الصلاحية
 *
 * @param string $permission الصلاحية المطلوبة
 * @param string $redirectUrl عنوان URL للتوجيه في حالة عدم وجود الصلاحية (اختياري)
 * @return void
 */
function requirePermission($permission, $redirectUrl = null) {
    // التحقق من تسجيل دخول المستخدم
    if (!isset($_SESSION['user_id'])) {
        redirectToLogin();
    }

    // التحقق من الصلاحية
    if (!hasPermission($permission)) {
        // تسجيل محاولة الوصول غير المصرح به
        $user = $_SESSION['username'] ?? $_SESSION['user_name'] ?? 'غير معروف';
        error_log("محاولة وصول غير مصرح به: المستخدم {$user} حاول الوصول إلى صلاحية {$permission}");

        // إعادة توجيه المستخدم
        if ($redirectUrl) {
            header('Location: ' . $redirectUrl);
        } else {
            // تحديد مسار صفحة رفض الوصول
            $access_denied_path = '/auth/access_denied.php';

            // التحقق من المسار الحالي
            $current_path = $_SERVER['PHP_SELF'];
            $depth = substr_count($current_path, '/') - 1;

            if ($depth > 0) {
                $access_denied_path = str_repeat('../', $depth) . 'auth/access_denied.php';
            }

            header('Location: ' . $access_denied_path);
        }

        exit;
    }
}

/**
 * التحقق من دور المستخدم وتوجيهه إذا لم يكن لديه الدور
 *
 * @param string $role الدور المطلوب
 * @param string $redirectUrl عنوان URL للتوجيه في حالة عدم وجود الدور (اختياري)
 * @return void
 */
function requireRole($role, $redirectUrl = null) {
    // التحقق من تسجيل دخول المستخدم
    if (!isset($_SESSION['user_id'])) {
        redirectToLogin();
    }

    // التحقق من الدور
    if (!hasRole($role)) {
        // تسجيل محاولة الوصول غير المصرح به
        $user = $_SESSION['username'] ?? $_SESSION['user_name'] ?? 'غير معروف';
        error_log("محاولة وصول غير مصرح به: المستخدم {$user} حاول الوصول إلى دور {$role}");

        // إعادة توجيه المستخدم
        if ($redirectUrl) {
            header('Location: ' . $redirectUrl);
        } else {
            // تحديد مسار صفحة رفض الوصول
            $access_denied_path = '/auth/access_denied.php';

            // التحقق من المسار الحالي
            $current_path = $_SERVER['PHP_SELF'];
            $depth = substr_count($current_path, '/') - 1;

            if ($depth > 0) {
                $access_denied_path = str_repeat('../', $depth) . 'auth/access_denied.php';
            }

            header('Location: ' . $access_denied_path);
        }

        exit;
    }
}

/**
 * التحقق من امتلاك المستخدم لأي من الأدوار المحددة وتوجيهه إذا لم يكن لديه أي منها
 *
 * @param array $roles مصفوفة الأدوار
 * @param string $redirectUrl عنوان URL للتوجيه في حالة عدم وجود أي من الأدوار (اختياري)
 * @return void
 */
function requireAnyRole($roles, $redirectUrl = null) {
    // التحقق من تسجيل دخول المستخدم
    if (!isset($_SESSION['user_id'])) {
        redirectToLogin();
    }

    // التحقق من الأدوار
    if (!hasAnyRole($roles)) {
        // تسجيل محاولة الوصول غير المصرح به
        $user = $_SESSION['username'] ?? $_SESSION['user_name'] ?? 'غير معروف';
        $rolesStr = implode(', ', $roles);
        error_log("محاولة وصول غير مصرح به: المستخدم {$user} حاول الوصول إلى أدوار {$rolesStr}");

        // إعادة توجيه المستخدم
        if ($redirectUrl) {
            header('Location: ' . $redirectUrl);
        } else {
            // تحديد مسار صفحة رفض الوصول
            $access_denied_path = '/auth/access_denied.php';

            // التحقق من المسار الحالي
            $current_path = $_SERVER['PHP_SELF'];
            $depth = substr_count($current_path, '/') - 1;

            if ($depth > 0) {
                $access_denied_path = str_repeat('../', $depth) . 'auth/access_denied.php';
            }

            header('Location: ' . $access_denied_path);
        }

        exit;
    }
}
