<?php
/**
 * ملف دوال الأمان الموحد
 * يحتوي على جميع دوال الأمان المستخدمة في النظام
 * 
 * @version 1.0
 */

// تكوين الجلسة
ini_set('session.cookie_httponly', 1);
ini_set('session.use_only_cookies', 1);
if (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on') {
    ini_set('session.cookie_secure', 1);
}
ini_set('session.cookie_samesite', 'Strict');
ini_set('session.gc_maxlifetime', 3600); // مدة الجلسة: ساعة واحدة

// بدء الجلسة إذا لم تكن قد بدأت بالفعل
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

/**
 * دالة لتنظيف المدخلات
 * 
 * @param string|array $data البيانات المراد تنظيفها
 * @return string|array البيانات بعد التنظيف
 */
function sanitizeInput($data) {
    if (is_array($data)) {
        foreach ($data as $key => $value) {
            $data[$key] = sanitizeInput($value);
        }
        return $data;
    }
    
    // تنظيف البيانات
    $data = trim($data);
    $data = stripslashes($data);
    $data = htmlspecialchars($data, ENT_QUOTES, 'UTF-8');
    
    return $data;
}

/**
 * دالة للتحقق من صحة البريد الإلكتروني
 * 
 * @param string $email البريد الإلكتروني المراد التحقق منه
 * @return bool نتيجة التحقق
 */
function validateEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

/**
 * دالة للتحقق من قوة كلمة المرور
 * 
 * @param string $password كلمة المرور المراد التحقق منها
 * @return bool نتيجة التحقق
 */
function validatePassword($password) {
    // كلمة المرور يجب أن تكون على الأقل 8 أحرف
    if (strlen($password) < 8) {
        return false;
    }

    // كلمة المرور يجب أن تحتوي على حرف كبير واحد على الأقل
    if (!preg_match('/[A-Z]/', $password)) {
        return false;
    }

    // كلمة المرور يجب أن تحتوي على حرف صغير واحد على الأقل
    if (!preg_match('/[a-z]/', $password)) {
        return false;
    }

    // كلمة المرور يجب أن تحتوي على رقم واحد على الأقل
    if (!preg_match('/[0-9]/', $password)) {
        return false;
    }

    return true;
}

/**
 * دالة لتشفير كلمة المرور
 * 
 * @param string $password كلمة المرور المراد تشفيرها
 * @return string كلمة المرور المشفرة
 */
function hashPassword($password) {
    return password_hash($password, PASSWORD_DEFAULT);
}

/**
 * دالة للتحقق من كلمة المرور
 * 
 * @param string $password كلمة المرور المدخلة
 * @param string $hash كلمة المرور المشفرة المخزنة
 * @return bool نتيجة التحقق
 */
function verifyPassword($password, $hash) {
    return password_verify($password, $hash);
}

/**
 * دالة لتوليد رمز CSRF
 * 
 * @return string رمز CSRF
 */
function generateCSRFToken() {
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }

    return $_SESSION['csrf_token'];
}

/**
 * دالة للتحقق من رمز CSRF
 * 
 * @param string $token الرمز المراد التحقق منه
 * @return bool نتيجة التحقق
 */
function verifyCSRFToken($token) {
    if (!isset($_SESSION['csrf_token']) || $token !== $_SESSION['csrf_token']) {
        // تسجيل محاولة تزوير CSRF
        error_log("محاولة تزوير CSRF من العنوان: " . $_SERVER['REMOTE_ADDR']);
        return false;
    }

    return true;
}

/**
 * دالة للتحقق من حالة تسجيل الدخول
 * 
 * @return bool حالة تسجيل الدخول
 */
function isLoggedIn() {
    // التحقق من وجود معلومات المستخدم في الجلسة
    if (isset($_SESSION['logged_in']) && $_SESSION['logged_in'] === true) {
        // تحديث وقت آخر نشاط
        $_SESSION['last_activity'] = time();
        return true;
    }
    
    // التحقق من وجود رمز تذكر تسجيل الدخول
    if (isset($_COOKIE['remember_token'])) {
        return loginWithRememberToken($_COOKIE['remember_token']);
    }
    
    return false;
}

/**
 * دالة للتحقق من صلاحيات المستخدم
 * 
 * @param string $permission الصلاحية المطلوبة
 * @return bool نتيجة التحقق
 */
function hasPermission($permission) {
    global $conn;
    
    // التحقق من تسجيل الدخول
    if (!isLoggedIn()) {
        return false;
    }
    
    // المدير لديه جميع الصلاحيات
    if ($_SESSION['role'] === 'admin') {
        return true;
    }
    
    try {
        // التحقق من وجود جدول صلاحيات المستخدمين
        $stmt = $conn->prepare("SHOW TABLES LIKE 'user_permissions'");
        $stmt->execute();
        
        if ($stmt->rowCount() === 0) {
            // إذا لم يكن الجدول موجودًا، نفترض أن المستخدم ليس لديه الصلاحية
            return false;
        }
        
        // التحقق من صلاحيات المستخدم
        $stmt = $conn->prepare("
            SELECT up.id 
            FROM user_permissions up
            JOIN permissions p ON up.permission_id = p.id
            WHERE up.user_id = :user_id 
            AND p.name = :permission
        ");
        $stmt->bindParam(':user_id', $_SESSION['user_id']);
        $stmt->bindParam(':permission', $permission);
        $stmt->execute();
        
        return $stmt->rowCount() > 0;
    } catch (PDOException $e) {
        // تسجيل الخطأ
        error_log('خطأ في التحقق من صلاحيات المستخدم: ' . $e->getMessage());
        
        // في حالة حدوث خطأ، نفترض أن المستخدم ليس لديه الصلاحية
        return false;
    }
}

/**
 * دالة لتسجيل خروج المستخدم
 */
function logout() {
    // تدمير جميع بيانات الجلسة
    $_SESSION = [];

    // تدمير ملف تعريف ارتباط الجلسة
    if (ini_get('session.use_cookies')) {
        $params = session_get_cookie_params();
        setcookie(
            session_name(),
            '',
            time() - 42000,
            $params['path'],
            $params['domain'],
            $params['secure'],
            $params['httponly']
        );
    }

    // تدمير الجلسة
    session_destroy();
}

/**
 * دالة لإعادة توجيه المستخدم مع رسالة
 * 
 * @param string $url الرابط المراد التوجيه إليه
 * @param string $message الرسالة المراد عرضها
 * @param string $type نوع الرسالة (success, error, warning, info)
 */
function redirectWithMessage($url, $message, $type = 'success') {
    $_SESSION['flash_message'] = $message;
    $_SESSION['flash_type'] = $type;
    
    header("Location: $url");
    exit;
}

/**
 * دالة لإعادة توجيه المستخدم إلى صفحة تسجيل الدخول إذا لم يكن مسجل الدخول
 */
function redirectIfNotLoggedIn() {
    if (!isLoggedIn()) {
        redirectWithMessage('auth/login.php', 'يجب تسجيل الدخول للوصول إلى هذه الصفحة', 'error');
    }
}

/**
 * دالة لإعادة توجيه المستخدم إلى لوحة التحكم إذا كان مسجل الدخول
 */
function redirectIfLoggedIn() {
    if (isLoggedIn()) {
        redirectWithMessage('unified-dashboard.php', 'أنت مسجل الدخول بالفعل', 'info');
    }
}

/**
 * دالة لتسجيل نشاط المستخدم
 * 
 * @param int $user_id معرف المستخدم
 * @param string $action الإجراء
 * @param string $details تفاصيل الإجراء
 */
function logActivity($user_id, $action, $details = '') {
    global $conn;
    
    try {
        // التحقق من وجود جدول سجل النشاط
        $stmt = $conn->prepare("SHOW TABLES LIKE 'activity_log'");
        $stmt->execute();
        
        if ($stmt->rowCount() === 0) {
            // إنشاء جدول سجل النشاط إذا لم يكن موجودًا
            $conn->exec("
                CREATE TABLE activity_log (
                    id INT AUTO_INCREMENT PRIMARY KEY,
                    user_id INT NOT NULL,
                    action VARCHAR(100) NOT NULL,
                    details TEXT,
                    ip_address VARCHAR(45),
                    user_agent VARCHAR(255),
                    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
            ");
        }
        
        // تسجيل النشاط
        $stmt = $conn->prepare("
            INSERT INTO activity_log (user_id, action, details, ip_address, user_agent)
            VALUES (:user_id, :action, :details, :ip_address, :user_agent)
        ");
        $stmt->bindParam(':user_id', $user_id);
        $stmt->bindParam(':action', $action);
        $stmt->bindParam(':details', $details);
        $stmt->bindParam(':ip_address', $_SERVER['REMOTE_ADDR']);
        $stmt->bindParam(':user_agent', $_SERVER['HTTP_USER_AGENT']);
        $stmt->execute();
        
        return true;
    } catch (PDOException $e) {
        // تسجيل الخطأ
        error_log('خطأ في تسجيل نشاط المستخدم: ' . $e->getMessage());
        
        return false;
    }
}

/**
 * دالة لتسجيل الأخطاء
 * 
 * @param string $message رسالة الخطأ
 * @param string $type نوع الخطأ
 */
function logError($message, $type = 'ERROR') {
    error_log("[$type] $message");
}
