<?php
/**
 * ملف إدارة الجلسات
 * يحتوي هذا الملف على دوال إدارة الجلسات وتأمينها
 *
 * @version 1.0
 * @author System Developer
 */

// بدء الجلسة إذا لم تكن قد بدأت بالفعل
if (session_status() === PHP_SESSION_NONE) {
    // تعيين خيارات الجلسة للأمان
    $session_options = [
        'cookie_httponly' => true,     // منع الوصول إلى ملف تعريف الارتباط من خلال JavaScript
        'cookie_secure' => isset($_SERVER['HTTPS']), // استخدام HTTPS فقط إذا كان متاحًا
        'cookie_samesite' => 'Lax',    // منع هجمات CSRF
        'use_strict_mode' => true,     // تفعيل الوضع الصارم للجلسة
        'gc_maxlifetime' => 3600,      // مدة صلاحية الجلسة (بالثواني)
    ];

    session_start($session_options);
}

// تعيين ثوابت إدارة الجلسة
if (!defined('SESSION_LIFETIME')) {
    define('SESSION_LIFETIME', 3600);          // مدة صلاحية الجلسة (بالثواني)
}
if (!defined('SESSION_REGENERATE_TIME')) {
    define('SESSION_REGENERATE_TIME', 300);    // وقت تجديد معرف الجلسة (بالثواني)
}
if (!defined('SESSION_IDLE_TIMEOUT')) {
    define('SESSION_IDLE_TIMEOUT', 1800);      // وقت الخمول قبل تسجيل الخروج التلقائي (بالثواني)
}

/**
 * تهيئة الجلسة
 *
 * @return void
 */
function initSession() {
    // تعيين وقت بدء الجلسة إذا لم يكن موجودًا
    if (!isset($_SESSION['session_start_time'])) {
        $_SESSION['session_start_time'] = time();
    }

    // تعيين وقت آخر نشاط إذا لم يكن موجودًا
    if (!isset($_SESSION['last_activity'])) {
        $_SESSION['last_activity'] = time();
    }

    // تعيين وقت آخر تجديد للجلسة إذا لم يكن موجودًا
    if (!isset($_SESSION['last_regeneration'])) {
        $_SESSION['last_regeneration'] = time();
    }

    // تعيين رمز CSRF إذا لم يكن موجودًا
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }

    // التحقق من صلاحية الجلسة
    validateSession();
}

/**
 * التحقق من صلاحية الجلسة
 *
 * @return void
 */
function validateSession() {
    // التحقق من وقت بدء الجلسة
    if (isset($_SESSION['session_start_time'])) {
        // التحقق من انتهاء صلاحية الجلسة
        if (time() - $_SESSION['session_start_time'] > SESSION_LIFETIME) {
            // تسجيل الخروج التلقائي
            destroySession();
            redirectToLogin('انتهت صلاحية الجلسة. يرجى تسجيل الدخول مرة أخرى.');
        }
    }

    // التحقق من وقت آخر نشاط
    if (isset($_SESSION['last_activity'])) {
        // التحقق من تجاوز وقت الخمول
        if (time() - $_SESSION['last_activity'] > SESSION_IDLE_TIMEOUT) {
            // تسجيل الخروج التلقائي
            destroySession();
            redirectToLogin('تم تسجيل خروجك تلقائيًا بسبب عدم النشاط. يرجى تسجيل الدخول مرة أخرى.');
        }

        // تحديث وقت آخر نشاط
        $_SESSION['last_activity'] = time();
    }

    // التحقق من وقت آخر تجديد للجلسة
    if (isset($_SESSION['last_regeneration'])) {
        // التحقق من الحاجة إلى تجديد معرف الجلسة
        if (time() - $_SESSION['last_regeneration'] > SESSION_REGENERATE_TIME) {
            // تجديد معرف الجلسة
            regenerateSession();
        }
    }
}

/**
 * تجديد معرف الجلسة
 *
 * @return void
 */
function regenerateSession() {
    // حفظ بيانات الجلسة
    $old_session_data = $_SESSION;

    // تدمير الجلسة الحالية
    session_destroy();

    // بدء جلسة جديدة
    $session_options = [
        'cookie_httponly' => true,
        'cookie_secure' => isset($_SERVER['HTTPS']),
        'cookie_samesite' => 'Lax',
        'use_strict_mode' => true,
        'gc_maxlifetime' => SESSION_LIFETIME,
    ];

    session_start($session_options);

    // استعادة بيانات الجلسة
    $_SESSION = $old_session_data;

    // تحديث وقت آخر تجديد للجلسة
    $_SESSION['last_regeneration'] = time();

    // تجديد رمز CSRF
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

/**
 * تدمير الجلسة
 *
 * @return void
 */
function destroySession() {
    // حذف ملف تعريف الارتباط الخاص بالجلسة
    if (ini_get('session.use_cookies')) {
        $params = session_get_cookie_params();
        setcookie(
            session_name(),
            '',
            time() - 42000,
            $params['path'],
            $params['domain'],
            $params['secure'],
            $params['httponly']
        );
    }

    // تدمير الجلسة
    session_unset();
    session_destroy();
}

/**
 * إعادة توجيه المستخدم إلى صفحة تسجيل الدخول
 *
 * @param string $message رسالة الخطأ (اختياري)
 * @return void
 */
function redirectToLogin($message = null) {
    // تخزين رسالة الخطأ في الجلسة إذا كانت موجودة
    if ($message) {
        // بدء جلسة جديدة
        session_start();

        // تخزين رسالة الخطأ
        $_SESSION['login_error'] = $message;
    }

    // تحديد مسار صفحة تسجيل الدخول
    $login_path = '/auth/login.php';

    // التحقق من المسار الحالي
    $current_path = $_SERVER['PHP_SELF'];
    $depth = substr_count($current_path, '/') - 1;

    if ($depth > 0) {
        $login_path = str_repeat('../', $depth) . 'auth/login.php';
    }

    // إعادة توجيه المستخدم
    header('Location: ' . $login_path);
    exit;
}

/**
 * إنشاء رمز CSRF
 *
 * @return string رمز CSRF
 */
// تم تعليق هذه الدالة لتجنب التعارض مع التعريف الموجود في config/config.php
// function generateCSRFToken() {
//     // التحقق من وجود رمز CSRF في الجلسة
//     if (!isset($_SESSION['csrf_token'])) {
//         $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
//     }
//
//     return $_SESSION['csrf_token'];
// }

/**
 * التحقق من صحة رمز CSRF
 *
 * @param string $token رمز CSRF المقدم
 * @return bool صحة الرمز
 */
// تم تعليق هذه الدالة لتجنب التعارض مع دالة verifyCSRFToken في config/config.php
// function validateCSRFToken($token) {
//     // التحقق من وجود رمز CSRF في الجلسة
//     if (!isset($_SESSION['csrf_token'])) {
//         return false;
//     }
//
//     // التحقق من تطابق الرمز
//     return hash_equals($_SESSION['csrf_token'], $token);
// }

/**
 * الحصول على معلومات المستخدم الحالي
 *
 * @return array معلومات المستخدم
 */
function getCurrentUser() {
    // التحقق من تسجيل دخول المستخدم
    if (!isset($_SESSION['user_id'])) {
        return null;
    }

    // إنشاء مصفوفة معلومات المستخدم
    $user = [
        'id' => $_SESSION['user_id'] ?? 0,
        'username' => $_SESSION['username'] ?? '',
        'full_name' => $_SESSION['full_name'] ?? $_SESSION['user_name'] ?? 'مستخدم النظام',
        'email' => $_SESSION['email'] ?? $_SESSION['user_email'] ?? '',
        'role' => $_SESSION['role'] ?? $_SESSION['user_role'] ?? 'user',
        'branch_id' => $_SESSION['branch_id'] ?? $_SESSION['user_branch_id'] ?? null
    ];

    return $user;
}

// تهيئة الجلسة
initSession();
