<?php
/**
 * ملف إدارة إعدادات النظام
 * يحتوي هذا الملف على دوال إدارة إعدادات النظام والصلاحيات
 * 
 * @version 1.0
 * @author System Developer
 */

// تضمين ملف الاتصال بقاعدة البيانات إذا لم يكن موجودًا
if (!isset($conn) || !isset($pdo)) {
    require_once __DIR__ . '/../config/database.php';
}

/**
 * الحصول على قيمة إعداد معين
 * 
 * @param string $key مفتاح الإعداد
 * @param mixed $default القيمة الافتراضية إذا لم يكن الإعداد موجودًا
 * @return mixed قيمة الإعداد
 */
function getSetting($key, $default = null) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("SELECT setting_value FROM system_settings WHERE setting_key = :key");
        $stmt->execute(['key' => $key]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        return $result ? $result['setting_value'] : $default;
    } catch (PDOException $e) {
        error_log("خطأ في الحصول على الإعداد: " . $e->getMessage());
        return $default;
    }
}

/**
 * الحصول على مجموعة من الإعدادات
 * 
 * @param string $group مجموعة الإعدادات
 * @return array مصفوفة من الإعدادات
 */
function getSettingsByGroup($group) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("SELECT setting_key, setting_value, description FROM system_settings WHERE setting_group = :group");
        $stmt->execute(['group' => $group]);
        $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        $settings = [];
        foreach ($results as $row) {
            $settings[$row['setting_key']] = [
                'value' => $row['setting_value'],
                'description' => $row['description']
            ];
        }
        
        return $settings;
    } catch (PDOException $e) {
        error_log("خطأ في الحصول على مجموعة الإعدادات: " . $e->getMessage());
        return [];
    }
}

/**
 * الحصول على جميع الإعدادات
 * 
 * @return array مصفوفة من جميع الإعدادات
 */
function getAllSettings() {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("SELECT setting_key, setting_value, setting_group, description FROM system_settings");
        $stmt->execute();
        $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        $settings = [];
        foreach ($results as $row) {
            $settings[$row['setting_key']] = [
                'value' => $row['setting_value'],
                'group' => $row['setting_group'],
                'description' => $row['description']
            ];
        }
        
        return $settings;
    } catch (PDOException $e) {
        error_log("خطأ في الحصول على جميع الإعدادات: " . $e->getMessage());
        return [];
    }
}

/**
 * تحديث قيمة إعداد معين
 * 
 * @param string $key مفتاح الإعداد
 * @param mixed $value القيمة الجديدة
 * @return bool نجاح العملية
 */
function updateSetting($key, $value) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("UPDATE system_settings SET setting_value = :value WHERE setting_key = :key");
        $stmt->execute([
            'key' => $key,
            'value' => $value
        ]);
        
        return $stmt->rowCount() > 0;
    } catch (PDOException $e) {
        error_log("خطأ في تحديث الإعداد: " . $e->getMessage());
        return false;
    }
}

/**
 * تحديث مجموعة من الإعدادات
 * 
 * @param array $settings مصفوفة من الإعدادات (المفتاح => القيمة)
 * @return bool نجاح العملية
 */
function updateSettings($settings) {
    global $pdo;
    
    try {
        $pdo->beginTransaction();
        
        foreach ($settings as $key => $value) {
            $stmt = $pdo->prepare("UPDATE system_settings SET setting_value = :value WHERE setting_key = :key");
            $stmt->execute([
                'key' => $key,
                'value' => $value
            ]);
        }
        
        $pdo->commit();
        return true;
    } catch (PDOException $e) {
        $pdo->rollBack();
        error_log("خطأ في تحديث الإعدادات: " . $e->getMessage());
        return false;
    }
}

/**
 * إضافة إعداد جديد
 * 
 * @param string $key مفتاح الإعداد
 * @param mixed $value قيمة الإعداد
 * @param string $group مجموعة الإعداد
 * @param string $description وصف الإعداد
 * @return bool نجاح العملية
 */
function addSetting($key, $value, $group = 'general', $description = '') {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("INSERT INTO system_settings (setting_key, setting_value, setting_group, description) VALUES (:key, :value, :group, :description)");
        $stmt->execute([
            'key' => $key,
            'value' => $value,
            'group' => $group,
            'description' => $description
        ]);
        
        return $stmt->rowCount() > 0;
    } catch (PDOException $e) {
        error_log("خطأ في إضافة الإعداد: " . $e->getMessage());
        return false;
    }
}

/**
 * حذف إعداد
 * 
 * @param string $key مفتاح الإعداد
 * @return bool نجاح العملية
 */
function deleteSetting($key) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("DELETE FROM system_settings WHERE setting_key = :key");
        $stmt->execute(['key' => $key]);
        
        return $stmt->rowCount() > 0;
    } catch (PDOException $e) {
        error_log("خطأ في حذف الإعداد: " . $e->getMessage());
        return false;
    }
}

/**
 * الحصول على جميع الأدوار
 * 
 * @return array مصفوفة من الأدوار
 */
function getAllRoles() {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("SELECT * FROM roles ORDER BY id");
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("خطأ في الحصول على الأدوار: " . $e->getMessage());
        return [];
    }
}

/**
 * الحصول على جميع الصلاحيات
 * 
 * @return array مصفوفة من الصلاحيات
 */
function getAllPermissions() {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("SELECT * FROM permissions ORDER BY category, id");
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("خطأ في الحصول على الصلاحيات: " . $e->getMessage());
        return [];
    }
}

/**
 * الحصول على صلاحيات دور معين
 * 
 * @param int $roleId معرف الدور
 * @return array مصفوفة من الصلاحيات
 */
function getRolePermissions($roleId) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            SELECT p.* 
            FROM permissions p
            JOIN role_permissions rp ON p.id = rp.permission_id
            WHERE rp.role_id = :role_id
            ORDER BY p.category, p.id
        ");
        $stmt->execute(['role_id' => $roleId]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("خطأ في الحصول على صلاحيات الدور: " . $e->getMessage());
        return [];
    }
}

/**
 * إضافة صلاحية لدور
 * 
 * @param int $roleId معرف الدور
 * @param int $permissionId معرف الصلاحية
 * @return bool نجاح العملية
 */
function addPermissionToRole($roleId, $permissionId) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("INSERT IGNORE INTO role_permissions (role_id, permission_id) VALUES (:role_id, :permission_id)");
        $stmt->execute([
            'role_id' => $roleId,
            'permission_id' => $permissionId
        ]);
        
        return true;
    } catch (PDOException $e) {
        error_log("خطأ في إضافة صلاحية للدور: " . $e->getMessage());
        return false;
    }
}

/**
 * حذف صلاحية من دور
 * 
 * @param int $roleId معرف الدور
 * @param int $permissionId معرف الصلاحية
 * @return bool نجاح العملية
 */
function removePermissionFromRole($roleId, $permissionId) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("DELETE FROM role_permissions WHERE role_id = :role_id AND permission_id = :permission_id");
        $stmt->execute([
            'role_id' => $roleId,
            'permission_id' => $permissionId
        ]);
        
        return true;
    } catch (PDOException $e) {
        error_log("خطأ في حذف صلاحية من الدور: " . $e->getMessage());
        return false;
    }
}

/**
 * تحديث صلاحيات دور
 * 
 * @param int $roleId معرف الدور
 * @param array $permissionIds مصفوفة من معرفات الصلاحيات
 * @return bool نجاح العملية
 */
function updateRolePermissions($roleId, $permissionIds) {
    global $pdo;
    
    try {
        $pdo->beginTransaction();
        
        // حذف جميع صلاحيات الدور الحالية
        $stmt = $pdo->prepare("DELETE FROM role_permissions WHERE role_id = :role_id");
        $stmt->execute(['role_id' => $roleId]);
        
        // إضافة الصلاحيات الجديدة
        foreach ($permissionIds as $permissionId) {
            $stmt = $pdo->prepare("INSERT INTO role_permissions (role_id, permission_id) VALUES (:role_id, :permission_id)");
            $stmt->execute([
                'role_id' => $roleId,
                'permission_id' => $permissionId
            ]);
        }
        
        $pdo->commit();
        return true;
    } catch (PDOException $e) {
        $pdo->rollBack();
        error_log("خطأ في تحديث صلاحيات الدور: " . $e->getMessage());
        return false;
    }
}

/**
 * التحقق من وجود صلاحية لدور
 * 
 * @param int $roleId معرف الدور
 * @param int $permissionId معرف الصلاحية
 * @return bool هل يملك الدور الصلاحية
 */
function roleHasPermission($roleId, $permissionId) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM role_permissions WHERE role_id = :role_id AND permission_id = :permission_id");
        $stmt->execute([
            'role_id' => $roleId,
            'permission_id' => $permissionId
        ]);
        
        return $stmt->fetchColumn() > 0;
    } catch (PDOException $e) {
        error_log("خطأ في التحقق من صلاحية الدور: " . $e->getMessage());
        return false;
    }
}

/**
 * الحصول على جميع وحدات النظام
 * 
 * @return array مصفوفة من وحدات النظام
 */
function getAllModules() {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("SELECT * FROM system_modules ORDER BY `order`");
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("خطأ في الحصول على وحدات النظام: " . $e->getMessage());
        return [];
    }
}

/**
 * تحديث حالة وحدة
 * 
 * @param int $moduleId معرف الوحدة
 * @param string $status الحالة الجديدة
 * @return bool نجاح العملية
 */
function updateModuleStatus($moduleId, $status) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("UPDATE system_modules SET status = :status WHERE id = :id");
        $stmt->execute([
            'id' => $moduleId,
            'status' => $status
        ]);
        
        return $stmt->rowCount() > 0;
    } catch (PDOException $e) {
        error_log("خطأ في تحديث حالة الوحدة: " . $e->getMessage());
        return false;
    }
}

/**
 * تحديث ترتيب وحدة
 * 
 * @param int $moduleId معرف الوحدة
 * @param int $order الترتيب الجديد
 * @return bool نجاح العملية
 */
function updateModuleOrder($moduleId, $order) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("UPDATE system_modules SET `order` = :order WHERE id = :id");
        $stmt->execute([
            'id' => $moduleId,
            'order' => $order
        ]);
        
        return $stmt->rowCount() > 0;
    } catch (PDOException $e) {
        error_log("خطأ في تحديث ترتيب الوحدة: " . $e->getMessage());
        return false;
    }
}
