<?php
/**
 * ملف تثبيت نظام الشحن
 * يستخدم هذا الملف لتثبيت النظام وتهيئة قاعدة البيانات
 */

// عرض جميع الأخطاء للمساعدة في التشخيص
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// تعريف متغيرات CSS للتنسيق
$successColor = "#4CAF50";
$errorColor = "#F44336";
$warningColor = "#FF9800";
$infoColor = "#2196F3";
$primaryColor = "#4c0c70";
$secondaryColor = "#6d28d9";

// تعريف متغيرات لتخزين الرسائل
$messages = [];
$errors = [];
$debugInfo = [];

// دالة لإضافة رسالة
function addMessage($type, $message) {
    global $messages;
    $messages[] = [
        'type' => $type,
        'message' => $message
    ];
}

// دالة لإضافة معلومات تصحيح
function addDebugInfo($info) {
    global $debugInfo;
    $debugInfo[] = $info;
}

// التحقق من وجود ملفات التكوين
$configFile = 'config.php';
$dbCredentialsFile = 'config/db_credentials.php';

if (!file_exists($configFile)) {
    addMessage('error', 'ملف التكوين الرئيسي غير موجود: ' . $configFile);
}

if (!file_exists($dbCredentialsFile)) {
    addMessage('error', 'ملف بيانات الاتصال بقاعدة البيانات غير موجود: ' . $dbCredentialsFile);
}

// التحقق من إمكانية الكتابة في المجلدات
$writableDirs = [
    'uploads',
    'logs',
    'database'
];

foreach ($writableDirs as $dir) {
    if (!file_exists($dir)) {
        if (!mkdir($dir, 0755, true)) {
            addMessage('warning', 'لم يتم إنشاء المجلد: ' . $dir);
        } else {
            addMessage('success', 'تم إنشاء المجلد: ' . $dir);
        }
    }
    
    if (!is_writable($dir)) {
        addMessage('warning', 'المجلد غير قابل للكتابة: ' . $dir);
    }
}

// معالجة النموذج
$dbHost = 'localhost';
$dbName = '';
$dbUser = '';
$dbPass = '';
$adminEmail = 'admin@al-ahmadigroup.com';
$adminPassword = 'admin123';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // الحصول على بيانات النموذج
    $dbHost = $_POST['db_host'] ?? 'localhost';
    $dbName = $_POST['db_name'] ?? '';
    $dbUser = $_POST['db_user'] ?? '';
    $dbPass = $_POST['db_pass'] ?? '';
    $adminEmail = $_POST['admin_email'] ?? 'admin@al-ahmadigroup.com';
    $adminPassword = $_POST['admin_password'] ?? 'admin123';
    
    // التحقق من البيانات
    if (empty($dbName) || empty($dbUser)) {
        addMessage('error', 'يرجى إدخال جميع بيانات قاعدة البيانات المطلوبة.');
    } else {
        // محاولة الاتصال بقاعدة البيانات
        try {
            $dsn = "mysql:host=$dbHost;charset=utf8mb4";
            $options = [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false,
            ];
            
            $pdo = new PDO($dsn, $dbUser, $dbPass, $options);
            
            // التحقق من وجود قاعدة البيانات
            $stmt = $pdo->query("SELECT SCHEMA_NAME FROM INFORMATION_SCHEMA.SCHEMATA WHERE SCHEMA_NAME = '$dbName'");
            $dbExists = $stmt->fetchColumn();
            
            if (!$dbExists) {
                // إنشاء قاعدة البيانات
                $pdo->exec("CREATE DATABASE IF NOT EXISTS `$dbName` DEFAULT CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
                addMessage('success', 'تم إنشاء قاعدة البيانات بنجاح: ' . $dbName);
            } else {
                addMessage('info', 'قاعدة البيانات موجودة بالفعل: ' . $dbName);
            }
            
            // الاتصال بقاعدة البيانات المحددة
            $pdo = new PDO("mysql:host=$dbHost;dbname=$dbName;charset=utf8mb4", $dbUser, $dbPass, $options);
            
            // تنفيذ ملف SQL
            $sqlFile = 'database/shipping_db.sql';
            if (file_exists($sqlFile)) {
                $sql = file_get_contents($sqlFile);
                $statements = explode(';', $sql);
                
                foreach ($statements as $statement) {
                    $statement = trim($statement);
                    if (!empty($statement)) {
                        $pdo->exec($statement);
                    }
                }
                
                addMessage('success', 'تم تهيئة قاعدة البيانات بنجاح.');
            } else {
                addMessage('error', 'ملف SQL غير موجود: ' . $sqlFile);
            }
            
            // تحديث ملفات التكوين
            if (file_exists($dbCredentialsFile)) {
                $content = file_get_contents($dbCredentialsFile);
                $content = preg_replace("/'db_host' => '.*?'/", "'db_host' => '$dbHost'", $content);
                $content = preg_replace("/'db_name' => '.*?'/", "'db_name' => '$dbName'", $content);
                $content = preg_replace("/'db_user' => '.*?'/", "'db_user' => '$dbUser'", $content);
                $content = preg_replace("/'db_pass' => '.*?'/", "'db_pass' => '$dbPass'", $content);
                
                file_put_contents($dbCredentialsFile, $content);
                addMessage('success', 'تم تحديث ملف بيانات الاتصال بقاعدة البيانات بنجاح.');
            }
            
            if (file_exists($configFile)) {
                $content = file_get_contents($configFile);
                $content = preg_replace("/define\('DB_HOST', '.*?'\)/", "define('DB_HOST', '$dbHost')", $content);
                $content = preg_replace("/define\('DB_NAME', '.*?'\)/", "define('DB_NAME', '$dbName')", $content);
                $content = preg_replace("/define\('DB_USER', '.*?'\)/", "define('DB_USER', '$dbUser')", $content);
                $content = preg_replace("/define\('DB_PASS', '.*?'\)/", "define('DB_PASS', '$dbPass')", $content);
                
                file_put_contents($configFile, $content);
                addMessage('success', 'تم تحديث ملف التكوين الرئيسي بنجاح.');
            }
            
            // إنشاء مدير النظام
            $hashedPassword = password_hash($adminPassword, PASSWORD_DEFAULT);
            
            // التحقق من وجود مدير النظام
            $stmt = $pdo->prepare("SELECT * FROM users WHERE email = :email");
            $stmt->bindParam(':email', $adminEmail);
            $stmt->execute();
            $admin = $stmt->fetch();
            
            if (!$admin) {
                // البحث عن المركز الرئيسي
                $stmt = $pdo->prepare("SELECT id FROM branches WHERE name = 'المركز الرئيسي'");
                $stmt->execute();
                $branch = $stmt->fetch();
                
                if (!$branch) {
                    // إنشاء المركز الرئيسي
                    $stmt = $pdo->prepare("INSERT INTO branches (name, code, is_active) VALUES ('المركز الرئيسي', 'HQ', 1)");
                    $stmt->execute();
                    $branchId = $pdo->lastInsertId();
                } else {
                    $branchId = $branch['id'];
                }
                
                // إنشاء مدير النظام
                $stmt = $pdo->prepare("
                    INSERT INTO users (
                        username, 
                        password, 
                        full_name, 
                        email, 
                        role, 
                        branch_id, 
                        status, 
                        is_active, 
                        registration_date
                    ) VALUES (
                        'admin', 
                        :password, 
                        'مدير النظام', 
                        :email, 
                        'admin', 
                        :branch_id, 
                        'active', 
                        1, 
                        NOW()
                    )
                ");
                
                $stmt->bindParam(':password', $hashedPassword);
                $stmt->bindParam(':email', $adminEmail);
                $stmt->bindParam(':branch_id', $branchId);
                
                $stmt->execute();
                
                addMessage('success', 'تم إنشاء مدير النظام بنجاح.');
            } else {
                // تحديث كلمة مرور مدير النظام
                $stmt = $pdo->prepare("UPDATE users SET password = :password WHERE email = :email");
                $stmt->bindParam(':password', $hashedPassword);
                $stmt->bindParam(':email', $adminEmail);
                $stmt->execute();
                
                addMessage('success', 'تم تحديث كلمة مرور مدير النظام بنجاح.');
            }
            
            addMessage('success', 'تم تثبيت النظام بنجاح! يمكنك الآن <a href="auth/login.php">تسجيل الدخول</a>.');
            
        } catch (PDOException $e) {
            addMessage('error', 'خطأ في الاتصال بقاعدة البيانات: ' . $e->getMessage());
        }
    }
}
?>

<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>تثبيت نظام الشحن</title>
    <link href="https://fonts.googleapis.com/css2?family=Tajawal:wght@400;500;700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Tajawal', sans-serif;
            background-color: #f5f5f5;
            margin: 0;
            padding: 20px;
            color: #333;
        }
        .container {
            max-width: 800px;
            margin: 0 auto;
            background-color: #fff;
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 0 20px rgba(0, 0, 0, 0.1);
        }
        h1 {
            color: <?php echo $primaryColor; ?>;
            text-align: center;
            margin-bottom: 30px;
            font-size: 28px;
        }
        .message {
            padding: 15px;
            margin-bottom: 15px;
            border-radius: 5px;
            font-size: 16px;
        }
        .success {
            background-color: rgba(76, 175, 80, 0.1);
            border: 1px solid <?php echo $successColor; ?>;
            color: <?php echo $successColor; ?>;
        }
        .error {
            background-color: rgba(244, 67, 54, 0.1);
            border: 1px solid <?php echo $errorColor; ?>;
            color: <?php echo $errorColor; ?>;
        }
        .warning {
            background-color: rgba(255, 152, 0, 0.1);
            border: 1px solid <?php echo $warningColor; ?>;
            color: <?php echo $warningColor; ?>;
        }
        .info {
            background-color: rgba(33, 150, 243, 0.1);
            border: 1px solid <?php echo $infoColor; ?>;
            color: <?php echo $infoColor; ?>;
        }
        .form-group {
            margin-bottom: 20px;
        }
        label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
            color: #333;
        }
        input[type="text"],
        input[type="password"],
        input[type="email"] {
            width: 100%;
            padding: 10px;
            border: 1px solid #ddd;
            border-radius: 5px;
            font-size: 16px;
            font-family: 'Tajawal', sans-serif;
        }
        button {
            background-color: <?php echo $primaryColor; ?>;
            color: #fff;
            border: none;
            padding: 12px 20px;
            border-radius: 5px;
            font-size: 16px;
            cursor: pointer;
            font-family: 'Tajawal', sans-serif;
            font-weight: bold;
        }
        button:hover {
            background-color: <?php echo $secondaryColor; ?>;
        }
        .section {
            margin-bottom: 30px;
            border-bottom: 1px solid #eee;
            padding-bottom: 20px;
        }
        .section h2 {
            color: <?php echo $primaryColor; ?>;
            font-size: 20px;
            margin-top: 0;
        }
        a {
            color: <?php echo $primaryColor; ?>;
            text-decoration: none;
        }
        a:hover {
            text-decoration: underline;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>تثبيت نظام الشحن</h1>
        
        <?php foreach ($messages as $message): ?>
            <div class="message <?php echo $message['type']; ?>">
                <?php echo $message['message']; ?>
            </div>
        <?php endforeach; ?>
        
        <form method="post" action="">
            <div class="section">
                <h2>إعدادات قاعدة البيانات</h2>
                <div class="form-group">
                    <label for="db_host">خادم قاعدة البيانات:</label>
                    <input type="text" id="db_host" name="db_host" value="<?php echo htmlspecialchars($dbHost); ?>" required>
                </div>
                <div class="form-group">
                    <label for="db_name">اسم قاعدة البيانات:</label>
                    <input type="text" id="db_name" name="db_name" value="<?php echo htmlspecialchars($dbName); ?>" required>
                </div>
                <div class="form-group">
                    <label for="db_user">اسم مستخدم قاعدة البيانات:</label>
                    <input type="text" id="db_user" name="db_user" value="<?php echo htmlspecialchars($dbUser); ?>" required>
                </div>
                <div class="form-group">
                    <label for="db_pass">كلمة مرور قاعدة البيانات:</label>
                    <input type="password" id="db_pass" name="db_pass" value="<?php echo htmlspecialchars($dbPass); ?>">
                </div>
            </div>
            
            <div class="section">
                <h2>إعدادات مدير النظام</h2>
                <div class="form-group">
                    <label for="admin_email">البريد الإلكتروني لمدير النظام:</label>
                    <input type="email" id="admin_email" name="admin_email" value="<?php echo htmlspecialchars($adminEmail); ?>" required>
                </div>
                <div class="form-group">
                    <label for="admin_password">كلمة مرور مدير النظام:</label>
                    <input type="password" id="admin_password" name="admin_password" value="<?php echo htmlspecialchars($adminPassword); ?>" required>
                </div>
            </div>
            
            <button type="submit">تثبيت النظام</button>
        </form>
    </div>
</body>
</html>
