<?php
/**
 * التقارير الشهرية
 * تعرض هذه الصفحة التقارير الشهرية للنظام
 *
 * @version 1.0
 * @author فريق التطوير
 */

// بدء الجلسة
session_start();

// تضمين ملف التكوين
require_once 'config/config.php';

// تفعيل عرض الأخطاء للتصحيح
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// التحقق من حالة تسجيل الدخول
if (!isset($_SESSION['logged_in']) || $_SESSION['logged_in'] !== true) {
    // إعادة توجيه المستخدم إلى صفحة تسجيل الدخول
    header('Location: auth/login.php');
    exit;
}

// الحصول على معلومات المستخدم
$user = [
    'id' => $_SESSION['user_id'] ?? 0,
    'username' => $_SESSION['username'] ?? '',
    'full_name' => $_SESSION['full_name'] ?? 'المستخدم',
    'email' => $_SESSION['email'] ?? '',
    'role' => $_SESSION['role'] ?? 'user',
    'branch_id' => $_SESSION['branch_id'] ?? null
];

// تحديد ما إذا كان المستخدم مديرًا
$isAdmin = (isset($_SESSION['role']) && ($_SESSION['role'] === 'admin' || $_SESSION['role'] === 'manager'));

// التحقق من صلاحيات المستخدم للوصول إلى صفحة التقارير
// السماح للمستخدم العادي بالوصول إلى التقارير الخاصة به فقط

// تحديد الشهر الحالي والسابق
$currentMonth = date('Y-m-01');
$currentMonthEnd = date('Y-m-t', strtotime($currentMonth));
$previousMonth = date('Y-m-01', strtotime('-1 month'));
$previousMonthEnd = date('Y-m-t', strtotime($previousMonth));

// الحصول على الشهر المطلوب من الاستعلام
$selectedMonth = $_GET['month'] ?? $currentMonth;
$selectedMonthEnd = date('Y-m-t', strtotime($selectedMonth));

// الاتصال بقاعدة البيانات
try {
    // الحصول على إحصائيات الشهر
    $monthlyStats = getMonthlyStats($conn, $selectedMonth, $selectedMonthEnd, $user['id'], $user['role']);

    // الحصول على الطلبات الشهرية
    $monthlyOrders = getMonthlyOrders($conn, $selectedMonth, $selectedMonthEnd, $user['id'], $user['role']);
} catch (Exception $e) {
    // تسجيل الخطأ
    error_log("خطأ في الحصول على بيانات التقارير الشهرية: " . $e->getMessage());

    // إنشاء بيانات افتراضية في حالة حدوث خطأ
    $monthlyStats = [
        'total_orders' => 0,
        'total_revenue' => 0,
        'average_order_value' => 0,
        'completed_orders' => 0,
        'pending_orders' => 0,
        'cancelled_orders' => 0,
        'weekly_orders' => [],
        'order_status' => []
    ];

    $monthlyOrders = [];
}

/**
 * دالة للحصول على إحصائيات الشهر
 *
 * @param PDO $conn اتصال قاعدة البيانات
 * @param string $startDate تاريخ بداية الشهر
 * @param string $endDate تاريخ نهاية الشهر
 * @param int $userId معرف المستخدم
 * @param string $userRole دور المستخدم
 * @return array إحصائيات الشهر
 */
function getMonthlyStats($conn, $startDate, $endDate, $userId, $userRole) {
    // استعلام للحصول على إحصائيات الشهر
    $sql = "
        SELECT
            COUNT(*) as total_orders,
            SUM(CASE WHEN status = 'تم التسليم' THEN 1 ELSE 0 END) as completed_orders,
            SUM(CASE WHEN status = 'قيد المعالجة' THEN 1 ELSE 0 END) as pending_orders,
            SUM(CASE WHEN status = 'في الطريق' THEN 1 ELSE 0 END) as in_transit_orders,
            SUM(CASE WHEN status = 'ملغى' THEN 1 ELSE 0 END) as cancelled_orders,
            SUM(amount_agreed_with_customer) as total_revenue
        FROM orders
        WHERE order_upload_date BETWEEN ? AND ?
    ";

    // إضافة شرط المستخدم إذا لم يكن مدير
    $params = [$startDate, $endDate];

    if ($userRole !== 'admin' && $userRole !== 'manager') {
        $sql .= " AND user_id = ?";
        $params[] = $userId;
    }

    $stmt = $conn->prepare($sql);
    $stmt->execute($params);
    $stats = $stmt->fetch(PDO::FETCH_ASSOC);

    // حساب متوسط قيمة الطلب
    $stats['average_order_value'] = $stats['total_orders'] > 0 ?
        $stats['total_revenue'] / $stats['total_orders'] : 0;

    // استعلام للحصول على توزيع الطلبات حسب الأسابيع
    $weeklyOrdersSql = "
        SELECT
            WEEK(order_upload_date) as week_number,
            COUNT(*) as orders_count,
            SUM(amount_agreed_with_customer) as revenue
        FROM orders
        WHERE order_upload_date BETWEEN ? AND ?
    ";

    // إضافة شرط المستخدم إذا لم يكن مدير
    $weeklyOrdersParams = [$startDate, $endDate];

    if ($userRole !== 'admin' && $userRole !== 'manager') {
        $weeklyOrdersSql .= " AND user_id = ?";
        $weeklyOrdersParams[] = $userId;
    }

    $weeklyOrdersSql .= " GROUP BY WEEK(order_upload_date) ORDER BY week_number";

    $weeklyOrdersStmt = $conn->prepare($weeklyOrdersSql);
    $weeklyOrdersStmt->execute($weeklyOrdersParams);
    $weeklyOrdersData = $weeklyOrdersStmt->fetchAll(PDO::FETCH_ASSOC);

    // تحويل بيانات الأسابيع إلى التنسيق المطلوب
    $weeklyOrders = [];
    foreach ($weeklyOrdersData as $index => $week) {
        $weeklyOrders[] = [
            'week' => 'الأسبوع ' . ($index + 1),
            'count' => (int)$week['orders_count'],
            'revenue' => (float)$week['revenue']
        ];
    }

    // إضافة أسابيع فارغة إذا كان عدد الأسابيع أقل من 4
    while (count($weeklyOrders) < 4) {
        $weeklyOrders[] = [
            'week' => 'الأسبوع ' . (count($weeklyOrders) + 1),
            'count' => 0,
            'revenue' => 0
        ];
    }

    // استعلام للحصول على توزيع حالة الطلبات
    $orderStatusSql = "
        SELECT
            status,
            COUNT(*) as count,
            COUNT(*) * 100.0 / (SELECT COUNT(*) FROM orders WHERE order_upload_date BETWEEN ? AND ?) as percentage
        FROM orders
        WHERE order_upload_date BETWEEN ? AND ?
    ";

    // إضافة شرط المستخدم إذا لم يكن مدير
    $orderStatusParams = array_merge([$startDate, $endDate], [$startDate, $endDate]);

    if ($userRole !== 'admin' && $userRole !== 'manager') {
        $orderStatusSql .= " AND user_id = ?";
        $orderStatusParams[] = $userId;
        // إضافة شرط المستخدم للاستعلام الفرعي أيضًا
        $orderStatusSql = str_replace("WHERE order_upload_date BETWEEN ? AND ?", "WHERE order_upload_date BETWEEN ? AND ? AND user_id = ?", $orderStatusSql);
        $orderStatusParams = array_merge([$startDate, $endDate, $userId], [$startDate, $endDate, $userId]);
    }

    $orderStatusSql .= " GROUP BY status";

    $orderStatusStmt = $conn->prepare($orderStatusSql);
    $orderStatusStmt->execute($orderStatusParams);
    $orderStatus = $orderStatusStmt->fetchAll(PDO::FETCH_ASSOC);

    // تجميع البيانات
    return [
        'total_orders' => (int)$stats['total_orders'],
        'total_revenue' => (float)$stats['total_revenue'],
        'average_order_value' => (float)$stats['average_order_value'],
        'completed_orders' => (int)$stats['completed_orders'],
        'pending_orders' => (int)$stats['pending_orders'],
        'cancelled_orders' => (int)$stats['cancelled_orders'],
        'weekly_orders' => $weeklyOrders,
        'order_status' => $orderStatus
    ];
}

/**
 * دالة للحصول على الطلبات الشهرية
 *
 * @param PDO $conn اتصال قاعدة البيانات
 * @param string $startDate تاريخ بداية الشهر
 * @param string $endDate تاريخ نهاية الشهر
 * @param int $userId معرف المستخدم
 * @param string $userRole دور المستخدم
 * @return array الطلبات الشهرية
 */
function getMonthlyOrders($conn, $startDate, $endDate, $userId, $userRole) {
    // استعلام للحصول على الطلبات الشهرية
    $sql = "
        SELECT
            o.id,
            o.order_number,
            o.client_name as customer_name,
            o.service_type,
            o.goods_type as cargo_type,
            o.goods_name as cargo_name,
            o.delivery_duration as delivery_time,
            o.payment_type,
            o.amount_paid as paid_amount,
            o.remaining_amount,
            o.second_destination,
            o.delivery_area,
            o.order_upload_date as order_date,
            o.scheduled_delivery_date as delivery_date,
            o.status,
            o.currency,
            o.employee_name,
            b.name as branch
        FROM orders o
        LEFT JOIN branches b ON o.branch_id = b.id
        WHERE o.order_upload_date BETWEEN ? AND ?
    ";

    // إضافة شرط المستخدم إذا لم يكن مدير
    $params = [$startDate, $endDate];

    if ($userRole !== 'admin' && $userRole !== 'manager') {
        $sql .= " AND o.user_id = ?";
        $params[] = $userId;
    }

    // إضافة شرط الفرع إذا تم تحديده
    if (isset($_GET['branch']) && $_GET['branch'] !== 'all') {
        $sql .= " AND o.branch_id = ?";
        $params[] = $_GET['branch'];
    }

    // إضافة شرط الحالة إذا تم تحديدها
    if (isset($_GET['status']) && $_GET['status'] !== '') {
        $sql .= " AND o.status = ?";
        $params[] = $_GET['status'];
    }

    $sql .= " ORDER BY o.order_upload_date DESC";

    $stmt = $conn->prepare($sql);
    $stmt->execute($params);
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

// تنسيق التاريخ بالعربية
function formatDateArabic($date) {
    $months = [
        'January' => 'يناير',
        'February' => 'فبراير',
        'March' => 'مارس',
        'April' => 'أبريل',
        'May' => 'مايو',
        'June' => 'يونيو',
        'July' => 'يوليو',
        'August' => 'أغسطس',
        'September' => 'سبتمبر',
        'October' => 'أكتوبر',
        'November' => 'نوفمبر',
        'December' => 'ديسمبر'
    ];

    $formattedDate = date('d F Y', strtotime($date));
    foreach ($months as $en => $ar) {
        $formattedDate = str_replace($en, $ar, $formattedDate);
    }
    return $formattedDate;
}

// تنسيق الشهر بالعربية
function formatMonthArabic($date) {
    $months = [
        'January' => 'يناير',
        'February' => 'فبراير',
        'March' => 'مارس',
        'April' => 'أبريل',
        'May' => 'مايو',
        'June' => 'يونيو',
        'July' => 'يوليو',
        'August' => 'أغسطس',
        'September' => 'سبتمبر',
        'October' => 'أكتوبر',
        'November' => 'نوفمبر',
        'December' => 'ديسمبر'
    ];

    $formattedMonth = date('F Y', strtotime($date));
    foreach ($months as $en => $ar) {
        $formattedMonth = str_replace($en, $ar, $formattedMonth);
    }
    return $formattedMonth;
}

// تنسيق المبالغ المالية
function formatCurrency($amount) {
    return number_format($amount, 2) . ' ريال';
}

// تحديد عنوان التقرير
$reportTitle = 'التقرير الشهري: ' . formatMonthArabic($selectedMonth);

// تعيين مسار الشعار
$company_logo = 'assets/images/backgrounds/background.png';
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">

<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>نظام طلبات الشحن | التقارير الشهرية</title>
  <script src="https://cdn.tailwindcss.com"></script>
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
  <link rel="stylesheet" href="assets/css/unified-design.css">
  <style>
    /* تحسين البطاقات الإحصائية */
    .stat-card {
      background: white;
      border-radius: var(--border-radius);
      box-shadow: var(--shadow);
      transition: var(--transition);
      position: relative;
      overflow: hidden;
      border: 1px solid var(--gray-200);
    }

    .stat-card::before {
      content: '';
      position: absolute;
      top: 0;
      right: 0;
      width: 100%;
      height: 4px;
      background: linear-gradient(90deg, var(--primary) 0%, var(--primary-light) 100%);
      opacity: 0;
      transition: var(--transition);
    }

    .stat-card:hover {
      transform: translateY(-5px);
      box-shadow: var(--shadow-md);
    }

    .stat-card:hover::before {
      opacity: 1;
    }

    .stat-card .icon-container {
      background: linear-gradient(135deg, var(--primary-light) 0%, var(--primary) 100%);
      color: white;
      border-radius: 50%;
      width: 3rem;
      height: 3rem;
      display: flex;
      align-items: center;
      justify-content: center;
      box-shadow: var(--shadow);
      transition: var(--transition);
    }

    .stat-card:hover .icon-container {
      transform: scale(1.1) rotate(10deg);
    }

    .stat-card .stat-value {
      font-size: 2rem;
      font-weight: 700;
      color: var(--primary-dark);
      margin-top: 0.5rem;
      margin-bottom: 0.25rem;
    }

    .stat-card .stat-label {
      color: var(--gray-600);
      font-weight: 500;
    }

    /* تحسين الرسوم البيانية */
    .chart-container {
      background-color: white;
      border-radius: var(--border-radius);
      box-shadow: var(--shadow);
      padding: 1.5rem;
      border: 1px solid var(--gray-200);
      transition: var(--transition);
    }

    .chart-container:hover {
      box-shadow: var(--shadow-md);
      border-color: var(--primary-light);
    }

    .chart-title {
      font-size: 1.25rem;
      font-weight: 600;
      color: var(--gray-800);
      margin-bottom: 1.5rem;
      display: flex;
      align-items: center;
      gap: 0.5rem;
    }

    .chart-title i {
      color: var(--primary);
    }

    /* تحسين الجداول */
    .data-table {
      width: 100%;
      border-collapse: separate;
      border-spacing: 0;
    }

    .data-table th {
      background-color: var(--primary);
      color: white;
      font-weight: 500;
      text-align: right;
      padding: 0.75rem 1rem;
    }

    .data-table th:first-child {
      border-top-right-radius: var(--border-radius);
    }

    .data-table th:last-child {
      border-top-left-radius: var(--border-radius);
    }

    .data-table td {
      padding: 0.75rem 1rem;
      border-bottom: 1px solid var(--gray-200);
    }

    .data-table tbody tr:hover {
      background-color: var(--gray-50);
    }

    .data-table tbody tr:last-child td {
      border-bottom: none;
    }

    .data-table tbody tr:last-child td:first-child {
      border-bottom-right-radius: var(--border-radius);
    }

    .data-table tbody tr:last-child td:last-child {
      border-bottom-left-radius: var(--border-radius);
    }

    /* تحسين أزرار التنقل بين الشهور */
    .month-nav {
      display: flex;
      align-items: center;
      gap: 1rem;
    }

    .month-nav-btn {
      display: flex;
      align-items: center;
      gap: 0.5rem;
      padding: 0.5rem 1rem;
      border-radius: var(--border-radius);
      transition: var(--transition);
      font-weight: 500;
    }

    .month-nav-btn.prev {
      background-color: var(--gray-100);
      color: var(--gray-700);
    }

    .month-nav-btn.next {
      background-color: var(--primary-50);
      color: var(--primary);
    }

    .month-nav-btn:hover {
      transform: translateY(-2px);
    }

    .month-nav-btn.prev:hover {
      background-color: var(--gray-200);
    }

    .month-nav-btn.next:hover {
      background-color: var(--primary-100);
    }

    /* تأثيرات الحركة */
    @keyframes fadeIn {
      from {
        opacity: 0;
        transform: translateY(10px);
      }
      to {
        opacity: 1;
        transform: translateY(0);
      }
    }

    .animate-fadeIn {
      animation: fadeIn 0.5s ease-out forwards;
    }

    /* تأثير الشعار */
    .logo-reveal {
      animation: logoReveal 1.5s ease-out forwards;
      transition: all 0.5s cubic-bezier(0.175, 0.885, 0.32, 1.275);
      border-radius: 8px;
      position: relative;
      z-index: 2;
      max-height: 100px;
      width: auto;
    }

    .logo-reveal:hover {
      transform: scale(1.08) rotate(1deg);
      filter: brightness(1.2) contrast(1.1) drop-shadow(0 10px 20px rgba(0, 0, 0, 0.25));
    }

    @keyframes logoReveal {
      0% {
        opacity: 0;
        transform: scale(0.8);
      }
      50% {
        opacity: 0.5;
        transform: scale(1.05);
      }
      100% {
        opacity: 1;
        transform: scale(1);
      }
    }

    /* تأثير إضافي للشعار */
    aside .logo-container::before {
      content: '';
      position: absolute;
      top: 0;
      left: 0;
      width: 100%;
      height: 100%;
      background: linear-gradient(135deg, rgba(255,255,255,0.3) 0%, rgba(255,255,255,0) 50%, rgba(255,255,255,0.3) 100%);
      transform: translateX(-100%);
      transition: transform 1s ease;
    }

    aside .logo-container:hover::before {
      transform: translateX(100%);
    }

    /* أنماط الطباعة */
    @media print {
      body {
        background-color: white;
      }

      aside, header, .overlay, #menuBtn, .flex.items-center.gap-2, .bg-white.p-6.rounded-lg.mb-6.shadow-sm.border.border-gray-200 {
        display: none !important;
      }

      main {
        margin: 0 !important;
        padding: 0 !important;
        width: 100% !important;
      }

      .print-only {
        display: block !important;
        text-align: center;
        margin-bottom: 20px;
      }

      .print-header img {
        height: 80px;
        margin: 0 auto 10px;
      }

      .chart-container {
        box-shadow: none !important;
        border: none !important;
        padding: 0 !important;
        margin-bottom: 20px !important;
      }

      .data-table {
        width: 100%;
        border-collapse: collapse;
      }

      .data-table th {
        background-color: #f3f4f6 !important;
        color: black !important;
        border: 1px solid #e5e7eb;
      }

      .data-table td {
        border: 1px solid #e5e7eb;
      }

      .stat-card {
        page-break-inside: avoid;
      }

      .mt-4.pt-4.border-t.border-gray-200 {
        border-top: 1px solid #e5e7eb !important;
        padding-top: 10px !important;
        margin-top: 10px !important;
      }

      .flex.flex-wrap.items-center.gap-4 {
        display: none !important;
      }
    }
  </style>
</head>

<body>
  <!-- طبقة التراكب للأجهزة المحمولة -->
  <div id="overlay" class="overlay"></div>

  <!-- عنصر رأس الصفحة للطباعة فقط -->
  <div class="print-only print-header" style="display: none;">
    <img src="<?php echo $company_logo; ?>" alt="شعار الشركة" class="logo-reveal">
    <h1 style="font-size: 24pt; font-weight: bold; margin-bottom: 10px;">مجموعة ابراهيم الاحمدي اليمنية</h1>
    <h2 style="font-size: 18pt; margin-bottom: 5px;"><?php echo $reportTitle; ?></h2>
    <p style="font-size: 12pt; margin-bottom: 5px;">تاريخ التقرير: <?php echo date('Y-m-d'); ?></p>

    <?php if (isset($_GET['branch']) && $_GET['branch'] !== 'all' || isset($_GET['status']) && $_GET['status'] !== ''): ?>
    <div style="font-size: 12pt; margin-bottom: 15px; color: #444;">
      <?php if (isset($_GET['branch']) && $_GET['branch'] !== 'all'):
        $branchName = '';
        foreach ($branches as $branch) {
          if ($branch['id'] == $_GET['branch']) {
            $branchName = $branch['name'];
            break;
          }
        }
      ?>
      <span style="margin-left: 15px;">الفرع: <strong><?php echo $branchName; ?></strong></span>
      <?php endif; ?>

      <?php if (isset($_GET['status']) && $_GET['status'] !== ''): ?>
      <span>الحالة: <strong><?php echo $_GET['status']; ?></strong></span>
      <?php endif; ?>
    </div>
    <?php endif; ?>

    <hr style="border: 1px solid #000; margin-bottom: 20px;">
  </div>

  <div class="flex min-h-screen">
    <!-- القائمة الجانبية -->
    <aside class="w-64 hidden md:block">
      <div class="p-6">
        <div class="logo-container flex items-center justify-center mb-8 p-2">
          <img src="<?php echo $company_logo; ?>" alt="شعار النظام" class="h-40 w-auto object-contain logo-reveal">
        </div>

        <div class="user-info mb-8">
          <div class="flex items-center space-x-4 space-x-reverse mb-3">
            <div class="user-avatar">
              <?php echo mb_substr($user['full_name'], 0, 1, 'UTF-8'); ?>
            </div>
            <div>
              <h3 class="text-white font-semibold"><?php echo htmlspecialchars($user['full_name']); ?></h3>
              <p class="text-indigo-200 text-sm"><?php echo $isAdmin ? 'مدير النظام' : 'مستخدم'; ?></p>
            </div>
          </div>
          <div class="bg-white/10 h-0.5 rounded-full mb-3"></div>
        </div>

        <nav>
          <ul class="space-y-3">
            <!-- لوحة التحكم -->
            <li>
              <a href="unified-dashboard.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white">
                <i class="fas fa-tachometer-alt ml-3 text-lg"></i>
                <span>لوحة التحكم</span>
              </a>
            </li>

            <!-- إنشاء طلب جديد -->
            <li>
              <a href="create-order-unified.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white">
                <i class="fas fa-plus-circle ml-3 text-lg"></i>
                <span>إنشاء طلب جديد</span>
              </a>
            </li>

            <!-- قائمة الطلبات -->
            <li>
              <a href="orders.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white">
                <i class="fas fa-box ml-3 text-lg"></i>
                <span>قائمة الطلبات</span>
              </a>
            </li>

            <?php if ($isAdmin): ?>
            <!-- قائمة التقارير المنسدلة -->
            <li>
              <a href="#" class="flex items-center p-3 rounded-lg text-white dropdown-toggle active" id="reportsDropdown">
                <i class="fas fa-chart-bar ml-3 text-lg"></i>
                <span>التقارير</span>
              </a>
              <ul class="dropdown-menu active" id="reportsMenu">
                <li>
                  <a href="weekly-reports.php" class="flex items-center p-2 rounded-lg text-white/80 hover:text-white">
                    <i class="fas fa-calendar-week ml-2 text-sm"></i>
                    <span>التقارير الأسبوعية</span>
                  </a>
                </li>
                <li>
                  <a href="monthly-reports.php" class="flex items-center p-2 rounded-lg text-white active">
                    <i class="fas fa-calendar-alt ml-2 text-sm"></i>
                    <span>التقارير الشهرية</span>
                  </a>
                </li>
              </ul>
            </li>

            <!-- إدارة المستخدمين -->
            <li>
              <a href="users.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white">
                <i class="fas fa-users ml-3 text-lg"></i>
                <span>إدارة المستخدمين</span>
              </a>
            </li>

            <!-- الفروع -->
            <li>
              <a href="branches.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white">
                <i class="fas fa-code-branch ml-3 text-lg"></i>
                <span>الفروع</span>
              </a>
            </li>

            <!-- إعدادات النظام -->
            <li>
              <a href="settings.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white">
                <i class="fas fa-cog ml-3 text-lg"></i>
                <span>إعدادات النظام</span>
              </a>
            </li>
            <?php endif; ?>
          </ul>

          <div class="pt-6 mt-6 border-t border-white/10">
            <a href="auth/logout.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white">
              <i class="fas fa-sign-out-alt ml-3 text-lg"></i>
              <span>تسجيل الخروج</span>
            </a>
          </div>
        </nav>
      </div>
    </aside>

    <!-- المحتوى الرئيسي -->
    <main class="flex-1 overflow-x-hidden overflow-y-auto bg-gray-50">
      <!-- شريط التنقل العلوي -->
      <header class="bg-white shadow-sm">
        <div class="flex items-center justify-between p-4">
          <!-- زر القائمة للأجهزة المحمولة -->
          <button id="menuBtn" class="md:hidden text-gray-500 hover:text-gray-600 focus:outline-none">
            <i class="fas fa-bars text-xl"></i>
          </button>

          <!-- عنوان الصفحة -->
          <h1 class="text-xl font-semibold text-gray-800"><?php echo $reportTitle; ?></h1>

          <!-- أزرار التنقل بين الشهور -->
          <div class="month-nav">
            <?php
            $prevMonth = date('Y-m-01', strtotime($selectedMonth . ' -1 month'));
            $nextMonth = date('Y-m-01', strtotime($selectedMonth . ' +1 month'));
            $isCurrentMonth = (date('Y-m', strtotime($selectedMonth)) === date('Y-m'));
            ?>
            <a href="?month=<?php echo $prevMonth; ?>" class="month-nav-btn prev">
              <i class="fas fa-chevron-right"></i>
              <span>الشهر السابق</span>
            </a>
            <?php if (!$isCurrentMonth): ?>
            <a href="?month=<?php echo $nextMonth; ?>" class="month-nav-btn next">
              <span>الشهر التالي</span>
              <i class="fas fa-chevron-left"></i>
            </a>
            <?php endif; ?>
          </div>
        </div>
      </header>

      <!-- محتوى الصفحة -->
      <div class="p-6">
        <!-- البطاقات الإحصائية -->
        <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
          <!-- إجمالي الطلبات -->
          <div class="stat-card p-6 animate-fadeIn" style="animation-delay: 0.1s;">
            <div class="flex items-center justify-between">
              <div class="icon-container">
                <i class="fas fa-box"></i>
              </div>
              <div class="text-xs font-medium text-gray-500">
                <i class="fas fa-chart-line ml-1"></i>
                <span>الشهر الحالي</span>
              </div>
            </div>
            <div class="stat-value"><?php echo $monthlyStats['total_orders']; ?></div>
            <div class="stat-label">إجمالي الطلبات</div>
          </div>

          <!-- إجمالي الإيرادات -->
          <div class="stat-card p-6 animate-fadeIn" style="animation-delay: 0.2s;">
            <div class="flex items-center justify-between">
              <div class="icon-container">
                <i class="fas fa-money-bill-wave"></i>
              </div>
              <div class="text-xs font-medium text-gray-500">
                <i class="fas fa-chart-line ml-1"></i>
                <span>الشهر الحالي</span>
              </div>
            </div>
            <div class="stat-value"><?php echo number_format($monthlyStats['total_revenue']); ?></div>
            <div class="stat-label">إجمالي الإيرادات (ريال)</div>
          </div>

          <!-- متوسط قيمة الطلب -->
          <div class="stat-card p-6 animate-fadeIn" style="animation-delay: 0.3s;">
            <div class="flex items-center justify-between">
              <div class="icon-container">
                <i class="fas fa-calculator"></i>
              </div>
              <div class="text-xs font-medium text-gray-500">
                <i class="fas fa-chart-line ml-1"></i>
                <span>الشهر الحالي</span>
              </div>
            </div>
            <div class="stat-value"><?php echo number_format($monthlyStats['average_order_value']); ?></div>
            <div class="stat-label">متوسط قيمة الطلب (ريال)</div>
          </div>

          <!-- نسبة إكمال الطلبات -->
          <div class="stat-card p-6 animate-fadeIn" style="animation-delay: 0.4s;">
            <div class="flex items-center justify-between">
              <div class="icon-container">
                <i class="fas fa-check-circle"></i>
              </div>
              <div class="text-xs font-medium text-gray-500">
                <i class="fas fa-chart-line ml-1"></i>
                <span>الشهر الحالي</span>
              </div>
            </div>
            <div class="stat-value"><?php echo number_format(($monthlyStats['completed_orders'] / $monthlyStats['total_orders']) * 100, 1); ?>%</div>
            <div class="stat-label">نسبة إكمال الطلبات</div>
          </div>
        </div>

        <!-- قائمة الطلبات الشهرية -->
        <div class="chart-container animate-fadeIn" style="animation-delay: 0.9s;">
          <div class="flex items-center justify-between mb-4">
            <h3 class="chart-title mb-0">
              <i class="fas fa-list"></i>
              <span>قائمة الطلبات الشهرية</span>
              <div class="mr-2 text-sm text-gray-500">
                <?php echo formatMonthArabic($selectedMonth); ?>
              </div>
            </h3>

            <!-- أزرار التصدير -->
            <div class="flex items-center gap-2">
              <button class="flex items-center gap-1 px-4 py-2 bg-blue-500 text-white rounded-lg hover:bg-blue-600 transition-all" onclick="printReport()">
                <i class="fas fa-print"></i>
                <span>طباعة</span>
              </button>

              <a href="export.php?type=pdf&report=monthly&month=<?php echo $selectedMonth; ?><?php echo isset($_GET['branch']) && $_GET['branch'] !== 'all' ? '&branch=' . $_GET['branch'] : ''; ?><?php echo isset($_GET['status']) && $_GET['status'] !== '' ? '&status=' . urlencode($_GET['status']) : ''; ?>" class="flex items-center gap-1 px-4 py-2 bg-indigo-500 text-white rounded-lg hover:bg-indigo-600 transition-all">
                <i class="fas fa-file-pdf"></i>
                <span>تصدير PDF</span>
              </a>

              <a href="export.php?type=excel&report=monthly&month=<?php echo $selectedMonth; ?><?php echo isset($_GET['branch']) && $_GET['branch'] !== 'all' ? '&branch=' . $_GET['branch'] : ''; ?><?php echo isset($_GET['status']) && $_GET['status'] !== '' ? '&status=' . urlencode($_GET['status']) : ''; ?>" class="flex items-center gap-1 px-4 py-2 bg-green-500 text-white rounded-lg hover:bg-green-600 transition-all">
                <i class="fas fa-file-excel"></i>
                <span>تصدير Excel</span>
              </a>

              <a href="export.php?type=orders_list&report=monthly&month=<?php echo $selectedMonth; ?><?php echo isset($_GET['branch']) && $_GET['branch'] !== 'all' ? '&branch=' . $_GET['branch'] : ''; ?><?php echo isset($_GET['status']) && $_GET['status'] !== '' ? '&status=' . urlencode($_GET['status']) : ''; ?>" class="flex items-center gap-1 px-4 py-2 bg-purple-500 text-white rounded-lg hover:bg-purple-600 transition-all">
                <i class="fas fa-list"></i>
                <span>عرض قائمة الطلبات</span>
              </a>

              <script>
                function printReport() {
                  // إضافة معلومات الفلترة إلى عنوان الطباعة
                  let printTitle = document.querySelector('.print-header h2');
                  if (printTitle) {
                    let originalTitle = printTitle.innerText;
                    <?php if (isset($_GET['branch']) && $_GET['branch'] !== 'all'):
                      $branchName = '';
                      foreach ($branches as $branch) {
                        if ($branch['id'] == $_GET['branch']) {
                          $branchName = $branch['name'];
                          break;
                        }
                      }
                    ?>
                    originalTitle += ' (فرع: <?php echo $branchName; ?>)';
                    <?php endif; ?>

                    <?php if (isset($_GET['status']) && $_GET['status'] !== ''): ?>
                    originalTitle += ' (الحالة: <?php echo $_GET['status']; ?>)';
                    <?php endif; ?>

                    printTitle.innerText = originalTitle;
                  }

                  window.print();
                }
              </script>
            </div>
          </div>

          <!-- فلترة الطلبات -->
          <div class="bg-white p-6 rounded-lg mb-6 shadow-sm border border-gray-200">
            <div class="flex items-center mb-4">
              <i class="fas fa-filter text-primary ml-2 text-lg"></i>
              <h3 class="text-lg font-semibold text-gray-800">فلترة التقارير</h3>
              <?php if (isset($_GET['branch']) && $_GET['branch'] !== 'all' || isset($_GET['status']) && $_GET['status'] !== ''): ?>
              <div class="mr-auto">
                <a href="monthly-reports.php?month=<?php echo $selectedMonth; ?>" class="text-sm text-gray-500 hover:text-primary flex items-center">
                  <i class="fas fa-times-circle ml-1"></i>
                  <span>إعادة ضبط الفلاتر</span>
                </a>
              </div>
              <?php endif; ?>
            </div>

            <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
              <!-- فلتر الفروع -->
              <div>
                <label for="branchFilter" class="block text-sm font-medium text-gray-700 mb-2">الفرع</label>
                <div class="relative">
                  <select id="branchFilter" name="branch" class="form-input pl-10 w-full">
                    <option value="all" class="font-bold text-primary">كل الفروع</option>
                    <option disabled>──────────</option>
                    <?php
                    // استعلام للحصول على قائمة الفروع
                    $branchesSql = "SELECT DISTINCT id, name FROM branches ORDER BY name";
                    $branchesStmt = $conn->prepare($branchesSql);
                    $branchesStmt->execute();
                    $allBranches = $branchesStmt->fetchAll(PDO::FETCH_ASSOC);

                    // تصفية الفروع المكررة باستخدام PHP (في حالة وجود فروع بنفس الاسم ولكن بمعرفات مختلفة)
                    $uniqueBranches = [];
                    $uniqueBranchNames = [];

                    foreach ($allBranches as $branch) {
                        if (!in_array($branch['name'], $uniqueBranchNames)) {
                            $uniqueBranchNames[] = $branch['name'];
                            $uniqueBranches[] = $branch;
                        }
                    }

                    // الفروع المعتمدة فقط
                    $approvedBranchNames = ['المكلا', 'الصين', 'عدن', 'صنعاء', 'شحن'];

                    // تصفية الفروع حسب الفروع المعتمدة فقط
                    $branches = [];
                    foreach ($uniqueBranches as $branch) {
                        // تحقق مما إذا كان اسم الفرع يحتوي على أحد أسماء الفروع المعتمدة
                        $isApproved = false;
                        foreach ($approvedBranchNames as $approvedName) {
                            if (strpos($branch['name'], $approvedName) !== false) {
                                $isApproved = true;
                                break;
                            }
                        }

                        if ($isApproved) {
                            $branches[] = $branch;
                        }
                    }

                    // عدد الفروع المستبعدة
                    $excludedBranchesCount = count($uniqueBranches) - count($branches);

                    $selectedBranch = $_GET['branch'] ?? 'all';

                    // ترتيب الفروع المعتمدة حسب الترتيب المحدد
                    $orderedBranchNames = ['المكلا', 'صنعاء', 'عدن', 'شحن', 'الصين'];

                    // ترتيب الفروع حسب الترتيب المحدد
                    usort($branches, function($a, $b) use ($orderedBranchNames) {
                        $aIndex = -1;
                        $bIndex = -1;

                        // البحث عن الفرع في قائمة الفروع المرتبة
                        foreach ($orderedBranchNames as $index => $name) {
                            if (strpos($a['name'], $name) !== false) {
                                $aIndex = $index;
                            }
                            if (strpos($b['name'], $name) !== false) {
                                $bIndex = $index;
                            }
                        }

                        // ترتيب حسب الفهرس
                        if ($aIndex !== -1 && $bIndex !== -1) {
                            return $aIndex - $bIndex;
                        } elseif ($aIndex !== -1) {
                            return -1;
                        } elseif ($bIndex !== -1) {
                            return 1;
                        } else {
                            return strcmp($a['name'], $b['name']);
                        }
                    });

                    // إضافة عنوان للفروع المعتمدة
                    echo "<optgroup label=\"الفروع المعتمدة\">";

                    // عرض الفروع المعتمدة
                    foreach ($branches as $branch) {
                        $selected = ($selectedBranch == $branch['id']) ? 'selected' : '';

                        // تحديد أيقونة لكل فرع
                        $icon = '';
                        if (strpos($branch['name'], 'المكلا') !== false) {
                            $icon = '🏢 ';
                        } elseif (strpos($branch['name'], 'صنعاء') !== false) {
                            $icon = '🏙️ ';
                        } elseif (strpos($branch['name'], 'عدن') !== false) {
                            $icon = '🌊 ';
                        } elseif (strpos($branch['name'], 'شحن') !== false) {
                            $icon = '🚚 ';
                        } elseif (strpos($branch['name'], 'الصين') !== false) {
                            $icon = '🇨🇳 ';
                        }

                        echo "<option value=\"{$branch['id']}\" {$selected}>{$icon}{$branch['name']}</option>";
                    }

                    echo "</optgroup>";
                    ?>
                  </select>
                  <div class="absolute inset-y-0 right-0 flex items-center pr-3 pointer-events-none">
                    <i class="fas fa-code-branch text-gray-400"></i>
                  </div>
                </div>
                <p class="mt-1 text-xs text-green-600">
                  <i class="fas fa-check-circle ml-1"></i>
                  تم عرض الفروع المعتمدة فقط (<?php echo count($branches); ?> فروع)
                </p>

                <?php if ($excludedBranchesCount > 0): ?>
                <p class="mt-1 text-xs text-gray-500">
                  <i class="fas fa-info-circle ml-1"></i>
                  تم استبعاد <?php echo $excludedBranchesCount; ?> فروع غير معتمدة
                </p>
                <?php endif; ?>
              </div>

              <!-- فلتر الحالة -->
              <div>
                <label for="statusFilter" class="block text-sm font-medium text-gray-700 mb-2">حالة الطلب</label>
                <div class="relative">
                  <select id="statusFilter" name="status" class="form-input pl-10 w-full">
                    <option value="">كل الحالات</option>
                    <?php
                    // قائمة حالات الطلبات
                    $statuses = ['تم التسليم', 'قيد المعالجة', 'في الطريق', 'ملغى'];

                    $selectedStatus = $_GET['status'] ?? '';

                    foreach ($statuses as $status) {
                        $selected = ($selectedStatus == $status) ? 'selected' : '';
                        echo "<option value=\"{$status}\" {$selected}>{$status}</option>";
                    }
                    ?>
                  </select>
                  <div class="absolute inset-y-0 right-0 flex items-center pr-3 pointer-events-none">
                    <i class="fas fa-tasks text-gray-400"></i>
                  </div>
                </div>
                <p class="mt-1 text-xs text-gray-500">اختر حالة الطلبات لفلترة التقارير</p>
              </div>

              <!-- زر التصفية -->
              <div class="flex items-end">
                <button id="filterBtn" class="btn btn-primary w-full py-2.5" onclick="applyFilters()">
                  <i class="fas fa-filter ml-2"></i>
                  <span>تطبيق الفلتر</span>
                </button>
              </div>
            </div>

            <?php if (isset($_GET['branch']) && $_GET['branch'] !== 'all' || isset($_GET['status']) && $_GET['status'] !== ''): ?>
            <div class="mt-4 pt-4 border-t border-gray-200">
              <div class="flex items-center text-sm text-gray-600">
                <i class="fas fa-info-circle ml-2 text-primary"></i>
                <span>الفلاتر النشطة:</span>

                <?php if (isset($_GET['branch']) && $_GET['branch'] !== 'all'):
                  $branchName = '';
                  foreach ($branches as $branch) {
                    if ($branch['id'] == $_GET['branch']) {
                      $branchName = $branch['name'];
                      break;
                    }
                  }
                ?>
                <span class="mr-2 bg-blue-100 text-blue-800 px-2 py-1 rounded-full text-xs">
                  الفرع: <?php echo $branchName; ?>
                </span>
                <?php endif; ?>

                <?php if (isset($_GET['status']) && $_GET['status'] !== ''): ?>
                <span class="mr-2 bg-green-100 text-green-800 px-2 py-1 rounded-full text-xs">
                  الحالة: <?php echo $_GET['status']; ?>
                </span>
                <?php endif; ?>

                <div class="mr-auto text-xs">
                  <span>سيتم تطبيق نفس الفلاتر عند تصدير التقارير</span>
                </div>
              </div>
            </div>
            <?php endif; ?>

            <script>
              function applyFilters() {
                const branch = document.getElementById('branchFilter').value;
                const status = document.getElementById('statusFilter').value;
                const month = '<?php echo $selectedMonth; ?>';

                let url = 'monthly-reports.php?month=' + month;

                if (branch && branch !== 'all') {
                  url += '&branch=' + branch;
                }

                if (status) {
                  url += '&status=' + encodeURIComponent(status);
                }

                window.location.href = url;
              }
            </script>
          </div>

          <!-- جدول الطلبات -->
          <div class="overflow-x-auto">
            <table class="data-table">
              <thead>
                <tr>
                  <th>رقم الطلب</th>
                  <th>العميل</th>
                  <th>نوع الخدمة</th>
                  <th>الفرع</th>
                  <th>تاريخ الطلب</th>
                  <th>تاريخ التسليم</th>
                  <th>المبلغ</th>
                  <th>الموظف المسؤول</th>
                  <th>الحالة</th>
                  <th>الإجراءات</th>
                </tr>
              </thead>
              <tbody>
                <?php if (empty($monthlyOrders)): ?>
                <tr>
                  <td colspan="10" class="text-center py-8">
                    <div class="flex flex-col items-center justify-center text-gray-500">
                      <i class="fas fa-inbox text-4xl mb-3 text-gray-300"></i>
                      <p>لا توجد طلبات في هذا الشهر</p>
                    </div>
                  </td>
                </tr>
                <?php else: ?>
                  <?php foreach ($monthlyOrders as $order): ?>
                  <tr data-order-id="<?php echo $order['id']; ?>">
                    <td class="font-medium"><?php echo $order['order_number']; ?></td>
                    <td><?php echo $order['customer_name']; ?></td>
                    <td><?php echo $order['service_type']; ?></td>
                    <td><?php echo $order['branch'] ?? 'غير محدد'; ?></td>
                    <td><?php echo formatDateArabic($order['order_date']); ?></td>
                    <td><?php echo !empty($order['delivery_date']) ? formatDateArabic($order['delivery_date']) : 'غير محدد'; ?></td>
                    <td><?php echo number_format(($order['paid_amount'] ?? 0) + ($order['remaining_amount'] ?? 0), 2); ?> <?php echo $order['currency'] ?? 'ريال'; ?></td>
                    <td><?php echo $order['employee_name'] ? $order['employee_name'] : 'غير محدد'; ?></td>
                    <td>
                      <?php
                      $statusClass = '';
                      $statusIcon = '';

                      switch ($order['status']) {
                          case 'تم التسليم':
                              $statusClass = 'bg-green-100 text-green-800';
                              $statusIcon = 'check-circle';
                              break;
                          case 'قيد المعالجة':
                              $statusClass = 'bg-yellow-100 text-yellow-800';
                              $statusIcon = 'clock';
                              break;
                          case 'في الطريق':
                              $statusClass = 'bg-blue-100 text-blue-800';
                              $statusIcon = 'truck';
                              break;
                          case 'ملغى':
                              $statusClass = 'bg-red-100 text-red-800';
                              $statusIcon = 'times-circle';
                              break;
                          default:
                              $statusClass = 'bg-gray-100 text-gray-800';
                              $statusIcon = 'circle';
                      }
                      ?>
                      <span class="px-2 py-1 rounded-full text-xs font-medium <?php echo $statusClass; ?>">
                        <i class="fas fa-<?php echo $statusIcon; ?> ml-1"></i>
                        <?php echo $order['status']; ?>
                      </span>
                    </td>
                    <td>
                      <div class="flex items-center space-x-2 space-x-reverse">
                        <a href="order-details.php?id=<?php echo $order['id']; ?>" class="p-1 text-primary hover:bg-primary/10 rounded-full transition-all" title="عرض التفاصيل">
                          <i class="fas fa-eye"></i>
                        </a>
                        <a href="edit-order.php?id=<?php echo $order['id']; ?>" class="p-1 text-info hover:bg-info/10 rounded-full transition-all" title="تعديل الطلب">
                          <i class="fas fa-edit"></i>
                        </a>
                        <div class="relative">
                          <button onclick="toggleDropdown(<?php echo $order['id']; ?>)" class="p-1 text-gray-400 hover:bg-gray-100 rounded-full transition-all" title="المزيد من الخيارات">
                            <i class="fas fa-ellipsis-v"></i>
                          </button>
                          <div id="dropdown-<?php echo $order['id']; ?>" class="absolute left-0 mt-2 w-48 bg-white rounded-lg shadow-lg z-10 hidden">
                            <ul class="py-1">
                              <li>
                                <a href="print-order.php?id=<?php echo $order['id']; ?>" class="block px-4 py-2 text-sm text-gray-700 hover:bg-gray-100">
                                  <i class="fas fa-print ml-2"></i> طباعة الطلب
                                </a>
                              </li>
                              <li>
                                <a href="export.php?type=pdf&order_id=<?php echo $order['id']; ?>" class="block px-4 py-2 text-sm text-gray-700 hover:bg-gray-100">
                                  <i class="fas fa-file-pdf ml-2"></i> تصدير PDF
                                </a>
                              </li>
                              <li>
                                <a href="#" onclick="changeStatus(<?php echo $order['id']; ?>)" class="block px-4 py-2 text-sm text-gray-700 hover:bg-gray-100">
                                  <i class="fas fa-exchange-alt ml-2"></i> تغيير الحالة
                                </a>
                              </li>
                            </ul>
                          </div>
                        </div>
                      </div>
                    </td>
                  </tr>
                  <?php endforeach; ?>
                <?php endif; ?>
              </tbody>
            </table>
          </div>

          <!-- تذييل الجدول -->
          <div class="mt-4 pt-4 border-t border-gray-200 flex flex-col md:flex-row justify-between items-start md:items-center gap-4">
            <div class="text-sm text-gray-500">
              <div class="flex items-center mb-2">
                <i class="fas fa-info-circle ml-2 text-primary"></i>
                <span>إجمالي الطلبات: <span class="font-medium text-primary"><?php echo count($monthlyOrders); ?></span></span>
              </div>

              <?php if (isset($_GET['branch']) && $_GET['branch'] !== 'all'):
                $branchName = '';
                foreach ($branches as $branch) {
                  if ($branch['id'] == $_GET['branch']) {
                    $branchName = $branch['name'];
                    break;
                  }
                }
              ?>
              <div class="flex items-center text-xs text-gray-500 mb-1">
                <i class="fas fa-filter ml-1 text-blue-500"></i>
                <span>تم فلترة النتائج حسب الفرع: <span class="font-medium"><?php echo $branchName; ?></span></span>
              </div>
              <?php endif; ?>

              <?php if (isset($_GET['status']) && $_GET['status'] !== ''): ?>
              <div class="flex items-center text-xs text-gray-500">
                <i class="fas fa-filter ml-1 text-green-500"></i>
                <span>تم فلترة النتائج حسب الحالة: <span class="font-medium"><?php echo $_GET['status']; ?></span></span>
              </div>
              <?php endif; ?>
            </div>

            <div class="flex flex-wrap items-center gap-4">
              <a href="monthly-reports.php?month=<?php echo $currentMonth; ?><?php echo isset($_GET['branch']) && $_GET['branch'] !== 'all' ? '&branch=' . $_GET['branch'] : ''; ?><?php echo isset($_GET['status']) && $_GET['status'] !== '' ? '&status=' . urlencode($_GET['status']) : ''; ?>" class="text-primary hover:text-primary-dark text-sm font-medium flex items-center">
                <span>العودة للشهر الحالي</span>
                <i class="fas fa-calendar-alt mr-1"></i>
              </a>

              <div class="border-r border-gray-300 h-6 mx-1 hidden md:block"></div>

              <a href="orders.php<?php echo isset($_GET['branch']) && $_GET['branch'] !== 'all' ? '?branch=' . $_GET['branch'] : ''; ?><?php echo isset($_GET['status']) && $_GET['status'] !== '' ? (isset($_GET['branch']) && $_GET['branch'] !== 'all' ? '&' : '?') . 'status=' . urlencode($_GET['status']) : ''; ?>" class="text-primary hover:text-primary-dark text-sm font-medium flex items-center">
                <span>عرض جميع الطلبات</span>
                <i class="fas fa-arrow-left mr-1"></i>
              </a>

              <?php if (isset($_GET['branch']) && $_GET['branch'] !== 'all' || isset($_GET['status']) && $_GET['status'] !== ''): ?>
              <div class="border-r border-gray-300 h-6 mx-1 hidden md:block"></div>

              <a href="monthly-reports.php?month=<?php echo $selectedMonth; ?>" class="text-red-500 hover:text-red-600 text-sm font-medium flex items-center">
                <span>إزالة الفلاتر</span>
                <i class="fas fa-times-circle mr-1"></i>
              </a>
              <?php endif; ?>
            </div>
          </div>
        </div>
      </div>
    </main>
  </div>

  <!-- سكريبت القائمة الجانبية للأجهزة المحمولة -->
  <script>
    // تبديل القائمة الجانبية للأجهزة المحمولة
    const menuBtn = document.getElementById('menuBtn');
    const sidebar = document.querySelector('aside');
    const overlay = document.getElementById('overlay');

    if (menuBtn && sidebar && overlay) {
      menuBtn.addEventListener('click', () => {
        sidebar.classList.toggle('hidden');
        sidebar.classList.toggle('fixed');
        sidebar.classList.toggle('inset-0');
        sidebar.classList.toggle('z-40');
        sidebar.classList.toggle('w-64');
        overlay.classList.toggle('hidden');
      });

      overlay.addEventListener('click', () => {
        sidebar.classList.add('hidden');
        sidebar.classList.remove('fixed', 'inset-0', 'z-40', 'w-64');
        overlay.classList.add('hidden');
      });
    }

    // تبديل القوائم المنسدلة
    const dropdownToggles = document.querySelectorAll('.dropdown-toggle');

    dropdownToggles.forEach(toggle => {
      toggle.addEventListener('click', (e) => {
        e.preventDefault();
        const menuId = toggle.getAttribute('id');
        const menu = document.getElementById(menuId.replace('Dropdown', 'Menu'));

        if (menu) {
          menu.classList.toggle('active');
        }
      });
    });

    // فلترة الطلبات
    const filterBtn = document.getElementById('filterBtn');
    const branchFilter = document.getElementById('branchFilter');
    const statusFilter = document.getElementById('statusFilter');
    const orderRows = document.querySelectorAll('.data-table tbody tr');

    if (filterBtn && branchFilter && statusFilter) {
      filterBtn.addEventListener('click', () => {
        const selectedBranch = branchFilter.value;
        const selectedStatus = statusFilter.value;

        orderRows.forEach(row => {
          const branch = row.querySelector('td:nth-child(4)').textContent;
          const status = row.querySelector('td:nth-child(9) span').textContent.trim();

          const branchMatch = !selectedBranch || branch === selectedBranch;
          const statusMatch = !selectedStatus || status === selectedStatus;

          if (branchMatch && statusMatch) {
            row.style.display = '';
          } else {
            row.style.display = 'none';
          }
        });
      });
    }

    // تبديل القائمة المنسدلة للإجراءات
    function toggleDropdown(orderId) {
      const dropdown = document.getElementById(`dropdown-${orderId}`);

      // إغلاق جميع القوائم المنسدلة الأخرى
      document.querySelectorAll('[id^="dropdown-"]').forEach(el => {
        if (el.id !== `dropdown-${orderId}`) {
          el.classList.add('hidden');
        }
      });

      // تبديل حالة القائمة المنسدلة الحالية
      dropdown.classList.toggle('hidden');

      // إضافة مستمع حدث لإغلاق القائمة عند النقر في أي مكان آخر
      document.addEventListener('click', function closeDropdown(e) {
        if (!dropdown.contains(e.target) && e.target.id !== `dropdown-toggle-${orderId}`) {
          dropdown.classList.add('hidden');
          document.removeEventListener('click', closeDropdown);
        }
      });
    }

    // تغيير حالة الطلب
    function changeStatus(orderId) {
      // إنشاء مودال لتغيير الحالة
      const modal = document.createElement('div');
      modal.className = 'fixed inset-0 flex items-center justify-center z-50';
      modal.innerHTML = `
        <div class="fixed inset-0 bg-black opacity-50"></div>
        <div class="bg-white rounded-lg shadow-lg p-6 w-96 z-10">
          <h3 class="text-lg font-semibold mb-4">تغيير حالة الطلب</h3>
          <form id="changeStatusForm">
            <div class="mb-4">
              <label class="block text-gray-700 mb-2">الحالة الجديدة</label>
              <select id="newStatus" class="form-input w-full">
                <option value="تم التسليم">تم التسليم</option>
                <option value="قيد المعالجة">قيد المعالجة</option>
                <option value="في الطريق">في الطريق</option>
                <option value="ملغى">ملغى</option>
              </select>
            </div>
            <div class="flex justify-end gap-2">
              <button type="button" onclick="closeModal()" class="px-4 py-2 bg-gray-200 text-gray-700 rounded-lg hover:bg-gray-300">إلغاء</button>
              <button type="button" onclick="saveStatus(${orderId})" class="px-4 py-2 bg-primary text-white rounded-lg hover:bg-primary-dark">حفظ</button>
            </div>
          </form>
        </div>
      `;

      document.body.appendChild(modal);

      // إغلاق المودال
      window.closeModal = function() {
        document.body.removeChild(modal);
      };

      // حفظ الحالة الجديدة
      window.saveStatus = function(id) {
        const newStatus = document.getElementById('newStatus').value;

        // إرسال طلب AJAX لتحديث حالة الطلب
        fetch('api/update-order-status.php', {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
          },
          body: JSON.stringify({
            order_id: id,
            status: newStatus
          }),
        })
        .then(response => response.json())
        .then(data => {
          if (data.success) {
            // تحديث واجهة المستخدم
            const statusCell = document.querySelector(`tr[data-order-id="${id}"] td:nth-child(9) span`);
            if (statusCell) {
              // تحديث النص
              statusCell.textContent = newStatus;

              // تحديث الألوان والأيقونة
              statusCell.className = 'px-2 py-1 rounded-full text-xs font-medium';
              let statusIcon = '';

              switch (newStatus) {
                case 'تم التسليم':
                  statusCell.classList.add('bg-green-100', 'text-green-800');
                  statusIcon = 'check-circle';
                  break;
                case 'قيد المعالجة':
                  statusCell.classList.add('bg-yellow-100', 'text-yellow-800');
                  statusIcon = 'clock';
                  break;
                case 'في الطريق':
                  statusCell.classList.add('bg-blue-100', 'text-blue-800');
                  statusIcon = 'truck';
                  break;
                case 'ملغى':
                  statusCell.classList.add('bg-red-100', 'text-red-800');
                  statusIcon = 'times-circle';
                  break;
                default:
                  statusCell.classList.add('bg-gray-100', 'text-gray-800');
                  statusIcon = 'circle';
              }

              // تحديث الأيقونة
              const iconElement = statusCell.querySelector('i');
              if (iconElement) {
                iconElement.className = `fas fa-${statusIcon} ml-1`;
              } else {
                statusCell.innerHTML = `<i class="fas fa-${statusIcon} ml-1"></i> ${newStatus}`;
              }
            }

            // إغلاق المودال
            closeModal();

            // عرض رسالة نجاح
            alert('تم تحديث حالة الطلب بنجاح');
          } else {
            alert('حدث خطأ أثناء تحديث حالة الطلب');
          }
        })
        .catch(error => {
          console.error('Error:', error);
          alert('حدث خطأ أثناء تحديث حالة الطلب');
        });
      };
    }
  </script>
</body>
</html>
