<?php
// تضمين ملف التحقق من تسجيل الدخول
require_once '../includes/check_auth.php';

// التحقق من صلاحيات المستخدم (متاح فقط للإدارة)
$userRole = isset($_SESSION['role']) ? $_SESSION['role'] : 'user';
if ($userRole !== 'admin' && $userRole !== 'manager') {
    // إعادة توجيه المستخدم إلى الصفحة الرئيسية
    header('Location: ../index.php');
    exit;
}

// الحصول على السنة الحالية
$current_year = date('Y');
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">

<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>نظام طلبات الشحن | أرشيف الطلبات</title>
  <script src="https://cdn.tailwindcss.com"></script>
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
  <style>
    @import url('https://fonts.googleapis.com/css2?family=Tajawal:wght@300;400;500;600;700&display=swap');

    body {
      font-family: 'Tajawal', sans-serif;
      background-color: #f1f5f9;
      color: #1e293b;
    }

    /* تصميم المحتوى الرئيسي */
    .main-content {
      transition: margin-right 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    }

    /* كروت الإحصائيات */
    .stat-card {
      background: white;
      border-radius: 12px;
      box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.05);
      transition: transform 0.3s ease, box-shadow 0.3s ease;
    }

    .stat-card:hover {
      transform: translateY(-5px);
      box-shadow: 0 10px 15px -3px rgba(0, 0, 0, 0.1);
    }

    /* أزرار */
    .btn-primary {
      background: linear-gradient(135deg, #4f46e5 0%, #818cf8 100%);
      transition: all 0.3s ease;
      box-shadow: 0 4px 6px -1px rgba(79, 70, 229, 0.3);
    }

    .btn-primary:hover {
      transform: translateY(-2px);
      box-shadow: 0 10px 15px -3px rgba(79, 70, 229, 0.4);
    }

    /* حقول الإدخال */
    .form-input {
      transition: all 0.3s ease;
      border: 1px solid #e2e8f0;
      background-color: #f8fafc;
    }

    .form-input:focus {
      border-color: #818cf8;
      box-shadow: 0 0 0 3px rgba(129, 140, 248, 0.3);
      background-color: white;
    }

    /* تأثيرات الحركة */
    @keyframes fadeIn {
      from {
        opacity: 0;
        transform: translateY(10px);
      }

      to {
        opacity: 1;
        transform: translateY(0);
      }
    }

    .animate-fadeIn {
      animation: fadeIn 0.5s ease-out forwards;
    }
  </style>
</head>

<body class="bg-gray-50">

  <!-- تضمين القائمة الجانبية المشتركة -->
  <?php include_once '../includes/sidebar.php'; ?>

  <!-- المحتوى الرئيسي -->
  <main class="main-content min-h-screen">
    <!-- شريط التنقل العلوي -->
    <header class="bg-white shadow-sm sticky top-0 z-20">
      <div class="flex items-center justify-between p-4">
        <!-- زر القائمة للجوال -->
        <button id="mobileMenuButton" class="md:hidden text-gray-600 hover:text-indigo-600">
          <i class="fas fa-bars text-xl"></i>
        </button>

        <!-- شريط البحث -->
        <div class="relative flex-1 max-w-md mx-4">
          <input type="text" placeholder="ابحث في الأرشيف..." class="form-input w-full pl-10 pr-4 py-2 rounded-full">
          <i class="fas fa-search absolute right-3 top-3 text-gray-400"></i>
        </div>

        <!-- إشعارات وملف المستخدم -->
        <div class="flex items-center space-x-4 space-x-reverse">
          <button class="relative text-gray-600 hover:text-indigo-600">
            <i class="fas fa-bell text-xl"></i>
            <span
              class="absolute -top-1 -right-1 bg-red-500 text-white text-xs rounded-full h-5 w-5 flex items-center justify-center">3</span>
          </button>

          <div class="flex items-center space-x-2 space-x-reverse">
            <div class="text-right">
              <p class="font-medium text-sm"><?php echo htmlspecialchars($currentUser['full_name']); ?></p>
              <p class="text-xs text-gray-500"><?php echo htmlspecialchars($currentUser['role']); ?></p>
            </div>
            <div
              class="h-10 w-10 rounded-full bg-indigo-100 flex items-center justify-center text-indigo-600 font-bold">
              <?php echo mb_substr($currentUser['full_name'], 0, 1, 'UTF-8'); ?>
            </div>
          </div>
        </div>
      </div>
    </header>

    <!-- محتوى الصفحة -->
    <div class="p-6">
      <!-- رأس الصفحة -->
      <div class="flex flex-col md:flex-row md:items-center md:justify-between mb-6">
        <div>
          <h1 class="text-2xl font-bold text-gray-800 flex items-center">
            <i class="fas fa-archive text-indigo-600 ml-3"></i>
            أرشيف الطلبات
          </h1>
          <p class="text-gray-600 mt-1">عرض وإدارة الطلبات المؤرشفة (متاح فقط للإدارة)</p>
        </div>

        <div class="mt-4 md:mt-0">
          <div class="flex items-center space-x-4 space-x-reverse">
            <button id="refreshBtn" onclick="loadArchivedOrders()" class="bg-green-600 hover:bg-green-700 text-white px-4 py-2 rounded-lg font-medium flex items-center transition-colors duration-300">
              <i class="fas fa-sync-alt ml-2"></i>
              <span>تحديث</span>
            </button>
            <button onclick="window.location.href='orders.php'" class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded-lg font-medium flex items-center transition-colors duration-300">
              <i class="fas fa-boxes ml-2"></i>
              <span>الطلبات النشطة</span>
            </button>
          </div>
        </div>
      </div>

      <!-- أدوات التصفية -->
      <div class="bg-white rounded-xl shadow-sm p-5 mb-6 fade-in">
        <h2 class="text-lg font-bold mb-4 flex items-center">
          <i class="fas fa-filter text-indigo-600 ml-2"></i>
          <span>تصفية الأرشيف</span>
        </h2>
        <div class="flex flex-wrap items-center gap-4">
          <div>
            <label for="yearFilter" class="block text-gray-500 mb-1 text-sm">السنة</label>
            <select id="yearFilter" class="bg-gray-50 border border-gray-300 text-gray-700 py-2 px-4 rounded-lg focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 transition-all duration-300">
              <option value="">جميع السنوات</option>
              <?php for ($i = $current_year; $i >= $current_year - 5; $i--): ?>
                <option value="<?php echo $i; ?>"><?php echo $i; ?></option>
              <?php endfor; ?>
            </select>
          </div>
          <div>
            <label for="monthFilter" class="block text-gray-500 mb-1 text-sm">الشهر</label>
            <select id="monthFilter" class="bg-gray-50 border border-gray-300 text-gray-700 py-2 px-4 rounded-lg focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 transition-all duration-300">
              <option value="">جميع الشهور</option>
              <option value="1">يناير</option>
              <option value="2">فبراير</option>
              <option value="3">مارس</option>
              <option value="4">أبريل</option>
              <option value="5">مايو</option>
              <option value="6">يونيو</option>
              <option value="7">يوليو</option>
              <option value="8">أغسطس</option>
              <option value="9">سبتمبر</option>
              <option value="10">أكتوبر</option>
              <option value="11">نوفمبر</option>
              <option value="12">ديسمبر</option>
            </select>
          </div>
          <div>
            <label for="branchFilter" class="block text-gray-500 mb-1 text-sm">الفرع</label>
            <select id="branchFilter" class="bg-gray-50 border border-gray-300 text-gray-700 py-2 px-4 rounded-lg focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 transition-all duration-300">
              <option value="">جميع الفروع</option>
              <!-- سيتم تعبئة هذه القائمة ديناميكياً -->
            </select>
          </div>
          <div>
            <label for="searchInput" class="block text-gray-500 mb-1 text-sm">بحث</label>
            <div class="relative">
              <input type="text" id="searchInput" placeholder="بحث..." class="bg-gray-50 border border-gray-300 text-gray-700 py-2 pl-10 pr-4 rounded-lg focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 transition-all duration-300 w-full">
              <i class="fas fa-search absolute right-3 top-3 text-gray-400"></i>
            </div>
          </div>
          <div class="mt-auto">
            <button id="filterButton" class="bg-indigo-600 hover:bg-indigo-700 text-white px-4 py-2 rounded-lg font-medium flex items-center transition-colors duration-300">
              <i class="fas fa-filter ml-2"></i>
              <span>تصفية</span>
            </button>
          </div>
        </div>
      </div>

      <!-- ملاحظة الأرشفة -->
      <div class="bg-blue-50 border border-blue-200 rounded-lg p-3 mb-6 text-sm text-blue-700 flex items-start">
        <i class="fas fa-info-circle text-blue-500 mt-1 ml-2 text-lg"></i>
        <p>يتم أرشفة الطلبات تلقائياً بعد شهر من تاريخ إنشائها أو يدوياً عن طريق المستخدم. الطلبات المؤرشفة متاحة فقط للإدارة.</p>
      </div>

      <!-- جدول الطلبات المؤرشفة -->
      <div class="bg-white rounded-xl shadow-sm overflow-hidden animate-fadeIn">
        <div class="p-5 bg-gray-50 border-b border-gray-200 flex items-center justify-between">
          <h3 class="text-lg font-bold text-gray-800 flex items-center">
            <i class="fas fa-archive text-indigo-600 ml-2"></i>
            <span>الطلبات المؤرشفة</span>
          </h3>
          <div id="totalOrdersCount" class="text-sm text-gray-500">
            إجمالي الطلبات: <span class="font-bold">0</span>
          </div>
        </div>
        <div class="overflow-x-auto">
          <table class="min-w-full divide-y divide-gray-200">
            <thead class="bg-gray-50">
              <tr>
                <th scope="col" class="px-3 py-3 text-center text-xs font-medium text-gray-500 uppercase tracking-wider">
                  #
                </th>
                <th scope="col" class="px-3 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                  الرقم التسلسلي
                </th>
                <th scope="col" class="px-3 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                  اسم العميل
                </th>
                <th scope="col" class="px-3 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                  نوع الخدمة
                </th>
                <th scope="col" class="px-3 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                  تاريخ الطلب
                </th>
                <th scope="col" class="px-3 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                  تاريخ الأرشفة
                </th>
                <th scope="col" class="px-3 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                  الحالة
                </th>
                <th scope="col" class="px-3 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                  المبلغ
                </th>
                <th scope="col" class="px-3 py-3 text-center text-xs font-medium text-gray-500 uppercase tracking-wider">
                  إجراء
                </th>
              </tr>
            </thead>
            <tbody id="archiveTableBody" class="bg-white divide-y divide-gray-200">
              <!-- سيتم إضافة الطلبات المؤرشفة هنا ديناميكياً -->
              <tr>
                <td colspan="9" class="px-3 py-4 text-center text-gray-500">
                  <div class="flex flex-col items-center justify-center py-6">
                    <div class="animate-spin rounded-full h-12 w-12 border-b-2 border-indigo-500 mb-3"></div>
                    <p class="text-gray-500">جاري تحميل البيانات...</p>
                  </div>
                </td>
              </tr>
            </tbody>
          </table>
        </div>

        <!-- تذييل الجدول - التنقل بين الصفحات -->
        <div id="pagination" class="bg-gray-50 px-6 py-4 flex items-center justify-between border-t border-gray-200">
          <div>
            <p class="text-sm text-gray-700">
              عرض <span id="pageStart" class="font-medium">0</span> إلى <span id="pageEnd" class="font-medium">0</span> من <span id="totalItems" class="font-medium">0</span> طلب
            </p>
          </div>
          <div>
            <nav class="relative z-0 inline-flex rounded-md shadow-sm -space-x-px space-x-reverse" aria-label="Pagination">
              <button id="prevPageBtn" class="relative inline-flex items-center px-3 py-2 rounded-r-md border border-gray-300 bg-white text-sm font-medium text-gray-500 hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed">
                <span class="sr-only">السابق</span>
                <i class="fas fa-chevron-right ml-1"></i>
                السابق
              </button>

              <div id="pageNumbers" class="flex items-center">
                <!-- سيتم إضافة أرقام الصفحات هنا ديناميكياً -->
              </div>

              <button id="nextPageBtn" class="relative inline-flex items-center px-3 py-2 rounded-l-md border border-gray-300 bg-white text-sm font-medium text-gray-500 hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed">
                التالي
                <i class="fas fa-chevron-left mr-1"></i>
              </button>
            </nav>
          </div>
        </div>
      </div>
    </div>
  </main>

  <!-- تضمين مكتبة SweetAlert2 للتنبيهات الجميلة -->
  <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>

  <!-- سكريبت إدارة الأرشيف -->
  <script>
    // متغيرات عامة
    let currentPage = 1;
    let totalPages = 1;
    let pageSize = 10;
    let archivedOrders = [];
    let branches = [];

    // عند تحميل الصفحة
    document.addEventListener('DOMContentLoaded', function() {
      // تحميل الفروع
      loadBranches();

      // تحميل الطلبات المؤرشفة
      loadArchivedOrders();

      // إضافة مستمعي الأحداث
      setupEventListeners();
    });

    // تحميل الفروع
    function loadBranches() {
      fetch('../api/branches.php?action=get_branches')
        .then(response => response.json())
        .then(data => {
          if (data.status === 'success') {
            branches = data.branches;
            populateBranchesDropdown();
          }
        })
        .catch(error => {
          console.error('خطأ في تحميل الفروع:', error);
        });
    }

    // تعبئة قائمة الفروع
    function populateBranchesDropdown() {
      const branchFilter = document.getElementById('branchFilter');
      if (branchFilter && branches.length > 0) {
        branches.forEach(branch => {
          const option = document.createElement('option');
          option.value = branch.name;
          option.textContent = branch.name;
          branchFilter.appendChild(option);
        });
      }
    }

    // تحميل الطلبات المؤرشفة
    function loadArchivedOrders() {
      // إظهار مؤشر التحميل
      showLoading(true);

      // الحصول على معلمات التصفية
      const year = document.getElementById('yearFilter').value;
      const month = document.getElementById('monthFilter').value;
      const branch = document.getElementById('branchFilter').value;
      const search = document.getElementById('searchInput').value;

      // بناء عنوان URL
      let url = `../api/get_archived_orders.php?page=${currentPage}&limit=${pageSize}`;
      if (year) url += `&year=${year}`;
      if (month) url += `&month=${month}`;
      if (branch) url += `&branch=${branch}`;
      if (search) url += `&search=${encodeURIComponent(search)}`;

      // إرسال الطلب
      fetch(url)
        .then(response => {
          if (!response.ok) {
            throw new Error('فشل في جلب الطلبات المؤرشفة');
          }
          return response.json();
        })
        .then(data => {
          if (data.status === 'success') {
            // تخزين البيانات
            archivedOrders = data.orders;
            totalPages = data.totalPages;

            // تحديث واجهة المستخدم
            updateUI(data);
          } else {
            throw new Error(data.message || 'فشل في جلب الطلبات المؤرشفة');
          }
        })
        .catch(error => {
          console.error('خطأ:', error);
          showError(error.message);
        })
        .finally(() => {
          // إخفاء مؤشر التحميل
          showLoading(false);
        });
    }

    // تحديث واجهة المستخدم
    function updateUI(data) {
      // تحديث جدول الطلبات
      updateOrdersTable(data.orders);

      // تحديث معلومات الصفحات
      updatePagination(data.page, data.totalPages, data.total);

      // تحديث إجمالي عدد الطلبات
      document.querySelector('#totalOrdersCount span').textContent = data.total;
    }

    // تحديث جدول الطلبات
    function updateOrdersTable(orders) {
      const tableBody = document.getElementById('archiveTableBody');

      if (!orders || orders.length === 0) {
        tableBody.innerHTML = `
          <tr>
            <td colspan="9" class="px-3 py-4 text-center text-gray-500">
              <i class="fas fa-archive ml-2"></i>
              لا توجد طلبات مؤرشفة تطابق معايير البحث
            </td>
          </tr>
        `;
        return;
      }

      let html = '';
      orders.forEach((order, index) => {
        // تحديد لون حالة الطلب
        let statusClass = '';
        switch (order.status) {
          case 'تم التسليم':
            statusClass = 'bg-green-100 text-green-800';
            break;
          case 'قيد المعالجة':
            statusClass = 'bg-yellow-100 text-yellow-800';
            break;
          case 'في الطريق':
            statusClass = 'bg-blue-100 text-blue-800';
            break;
          case 'ملغى':
            statusClass = 'bg-red-100 text-red-800';
            break;
          default:
            statusClass = 'bg-gray-100 text-gray-800';
        }

        html += `
          <tr class="hover:bg-gray-50 transition-colors duration-200">
            <td class="px-3 py-3 text-center">${(currentPage - 1) * pageSize + index + 1}</td>
            <td class="px-3 py-3">${order.order_number}</td>
            <td class="px-3 py-3">${order.client_name}</td>
            <td class="px-3 py-3">${order.service_type || '-'}</td>
            <td class="px-3 py-3">${order.order_date || '-'}</td>
            <td class="px-3 py-3">${order.archive_date || '-'}</td>
            <td class="px-3 py-3">
              <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full ${statusClass}">
                ${order.status}
              </span>
            </td>
            <td class="px-3 py-3">${order.amount ? order.amount + ' ' + (order.currency || '') : '-'}</td>
            <td class="px-3 py-3 text-center">
              <div class="flex items-center justify-center space-x-2 space-x-reverse">
                <button onclick="viewOrder(${order.id})" class="text-indigo-600 hover:text-indigo-900" title="عرض">
                  <i class="fas fa-eye"></i>
                </button>
                <button onclick="unarchiveOrder(${order.id})" class="text-blue-600 hover:text-blue-900" title="إلغاء الأرشفة">
                  <i class="fas fa-box-open"></i>
                </button>
                <button onclick="printOrder(${order.id})" class="text-gray-600 hover:text-gray-900" title="طباعة">
                  <i class="fas fa-print"></i>
                </button>
              </div>
            </td>
          </tr>
        `;
      });

      tableBody.innerHTML = html;
    }

    // تحديث معلومات الصفحات
    function updatePagination(currentPage, totalPages, totalItems) {
      // تحديث أرقام الصفحات
      const pageStart = (currentPage - 1) * pageSize + 1;
      const pageEnd = Math.min(currentPage * pageSize, totalItems);

      document.getElementById('pageStart').textContent = pageStart;
      document.getElementById('pageEnd').textContent = pageEnd;
      document.getElementById('totalItems').textContent = totalItems;

      // تحديث أزرار التنقل
      const prevBtn = document.getElementById('prevPageBtn');
      const nextBtn = document.getElementById('nextPageBtn');

      prevBtn.disabled = currentPage <= 1;
      nextBtn.disabled = currentPage >= totalPages;

      // تحديث أرقام الصفحات
      const pageNumbers = document.getElementById('pageNumbers');
      pageNumbers.innerHTML = '';

      // تحديد نطاق الصفحات للعرض
      let startPage = Math.max(1, currentPage - 2);
      let endPage = Math.min(totalPages, startPage + 4);

      if (endPage - startPage < 4 && totalPages > 4) {
        startPage = Math.max(1, endPage - 4);
      }

      // إضافة زر الصفحة الأولى
      if (startPage > 1) {
        const firstPageBtn = document.createElement('button');
        firstPageBtn.className = 'relative inline-flex items-center px-4 py-2 border border-gray-300 bg-white text-sm font-medium text-gray-700 hover:bg-gray-50';
        firstPageBtn.textContent = '1';
        firstPageBtn.addEventListener('click', () => goToPage(1));
        pageNumbers.appendChild(firstPageBtn);

        if (startPage > 2) {
          const ellipsis = document.createElement('span');
          ellipsis.className = 'relative inline-flex items-center px-4 py-2 border border-gray-300 bg-white text-sm font-medium text-gray-700';
          ellipsis.textContent = '...';
          pageNumbers.appendChild(ellipsis);
        }
      }

      // إضافة أزرار الصفحات
      for (let i = startPage; i <= endPage; i++) {
        const pageBtn = document.createElement('button');
        pageBtn.className = `relative inline-flex items-center px-4 py-2 border ${i === currentPage ? 'border-indigo-500 bg-indigo-50 text-indigo-600' : 'border-gray-300 bg-white text-gray-700 hover:bg-gray-50'}`;
        pageBtn.textContent = i;
        pageBtn.addEventListener('click', () => goToPage(i));
        pageNumbers.appendChild(pageBtn);
      }

      // إضافة زر الصفحة الأخيرة
      if (endPage < totalPages) {
        if (endPage < totalPages - 1) {
          const ellipsis = document.createElement('span');
          ellipsis.className = 'relative inline-flex items-center px-4 py-2 border border-gray-300 bg-white text-sm font-medium text-gray-700';
          ellipsis.textContent = '...';
          pageNumbers.appendChild(ellipsis);
        }

        const lastPageBtn = document.createElement('button');
        lastPageBtn.className = 'relative inline-flex items-center px-4 py-2 border border-gray-300 bg-white text-sm font-medium text-gray-700 hover:bg-gray-50';
        lastPageBtn.textContent = totalPages;
        lastPageBtn.addEventListener('click', () => goToPage(totalPages));
        pageNumbers.appendChild(lastPageBtn);
      }
    }

    // الانتقال إلى صفحة معينة
    function goToPage(page) {
      currentPage = page;
      loadArchivedOrders();
    }

    // إضافة مستمعي الأحداث
    function setupEventListeners() {
      // زر التصفية
      const filterButton = document.getElementById('filterButton');
      if (filterButton) {
        filterButton.addEventListener('click', () => {
          currentPage = 1;
          loadArchivedOrders();
        });
      }

      // أزرار التنقل بين الصفحات
      const prevPageBtn = document.getElementById('prevPageBtn');
      const nextPageBtn = document.getElementById('nextPageBtn');

      if (prevPageBtn) {
        prevPageBtn.addEventListener('click', () => {
          if (currentPage > 1) {
            currentPage--;
            loadArchivedOrders();
          }
        });
      }

      if (nextPageBtn) {
        nextPageBtn.addEventListener('click', () => {
          if (currentPage < totalPages) {
            currentPage++;
            loadArchivedOrders();
          }
        });
      }

      // حقل البحث
      const searchInput = document.getElementById('searchInput');
      if (searchInput) {
        searchInput.addEventListener('keypress', (e) => {
          if (e.key === 'Enter') {
            currentPage = 1;
            loadArchivedOrders();
          }
        });
      }
    }

    // إظهار/إخفاء مؤشر التحميل
    function showLoading(show) {
      const tableBody = document.getElementById('archiveTableBody');

      if (show) {
        tableBody.innerHTML = `
          <tr>
            <td colspan="9" class="px-3 py-4 text-center text-gray-500">
              <div class="flex flex-col items-center justify-center py-6">
                <div class="animate-spin rounded-full h-12 w-12 border-b-2 border-indigo-500 mb-3"></div>
                <p class="text-gray-500">جاري تحميل البيانات...</p>
              </div>
            </td>
          </tr>
        `;
      }
    }

    // إظهار رسالة خطأ
    function showError(message) {
      const tableBody = document.getElementById('archiveTableBody');

      tableBody.innerHTML = `
        <tr>
          <td colspan="9" class="px-3 py-4 text-center text-red-500">
            <i class="fas fa-exclamation-triangle ml-2"></i>
            ${message}
          </td>
        </tr>
      `;
    }

    // عرض تفاصيل الطلب
    function viewOrder(orderId) {
      window.location.href = `view-order.php?id=${orderId}&archived=1`;
    }

    // إلغاء أرشفة الطلب
    function unarchiveOrder(orderId) {
      Swal.fire({
        title: 'تأكيد إلغاء الأرشفة',
        text: 'هل أنت متأكد من رغبتك في إلغاء أرشفة هذا الطلب؟',
        icon: 'question',
        showCancelButton: true,
        confirmButtonText: 'نعم، إلغاء الأرشفة',
        cancelButtonText: 'إلغاء',
        confirmButtonColor: '#3085d6',
        cancelButtonColor: '#d33'
      }).then((result) => {
        if (result.isConfirmed) {
          // إظهار مؤشر التحميل
          Swal.fire({
            title: 'جاري إلغاء الأرشفة...',
            text: 'يرجى الانتظار',
            allowOutsideClick: false,
            didOpen: () => {
              Swal.showLoading();
            }
          });

          // إرسال طلب إلغاء الأرشفة
          fetch('../api/unarchive_order.php', {
            method: 'POST',
            headers: {
              'Content-Type': 'application/json'
            },
            body: JSON.stringify({ order_id: orderId })
          })
          .then(response => response.json())
          .then(data => {
            if (data.status === 'success') {
              Swal.fire({
                icon: 'success',
                title: 'تم إلغاء الأرشفة بنجاح',
                text: data.message,
                showConfirmButton: false,
                timer: 1500
              });

              // إعادة تحميل الطلبات
              loadArchivedOrders();
            } else {
              throw new Error(data.message || 'فشل في إلغاء أرشفة الطلب');
            }
          })
          .catch(error => {
            console.error('خطأ:', error);
            Swal.fire({
              icon: 'error',
              title: 'خطأ',
              text: error.message
            });
          });
        }
      });
    }

    // طباعة الطلب
    function printOrder(orderId) {
      window.open(`print-order.php?id=${orderId}`, '_blank');
    }

    // تصدير إلى PDF
    function exportToPDF() {
      Swal.fire({
        title: 'جاري التصدير...',
        text: 'يرجى الانتظار',
        allowOutsideClick: false,
        didOpen: () => {
          Swal.showLoading();

          setTimeout(() => {
            Swal.fire({
              icon: 'success',
              title: 'تم التصدير بنجاح',
              text: 'تم تصدير الطلبات المؤرشفة إلى ملف PDF',
              showConfirmButton: false,
              timer: 1500
            });
          }, 1500);
        }
      });
    }

    // تصدير إلى Excel
    function exportToExcel() {
      Swal.fire({
        title: 'جاري التصدير...',
        text: 'يرجى الانتظار',
        allowOutsideClick: false,
        didOpen: () => {
          Swal.showLoading();

          setTimeout(() => {
            Swal.fire({
              icon: 'success',
              title: 'تم التصدير بنجاح',
              text: 'تم تصدير الطلبات المؤرشفة إلى ملف Excel',
              showConfirmButton: false,
              timer: 1500
            });
          }, 1500);
        }
      });
    }
  </script>
</body>

</html>
