<?php
// تضمين ملف التحقق من تسجيل الدخول
require_once 'check_auth.php';

// التحقق من وجود معرف الطلب
if (!isset($_GET['id']) || empty($_GET['id'])) {
    // إعادة توجيه المستخدم إلى صفحة قائمة الطلبات
    header('Location: orders.php');
    exit;
}

// الحصول على معرف الطلب
$orderId = (int)$_GET['id'];

// الحصول على معلومات الطلب
$stmt = $pdo->prepare("
    SELECT 
        o.*, 
        b.name as branch_name
    FROM orders o
    LEFT JOIN branches b ON o.branch_id = b.id
    WHERE o.id = ?
");
$stmt->execute([$orderId]);
$order = $stmt->fetch();

// التحقق من وجود الطلب
if (!$order) {
    // إعادة توجيه المستخدم إلى صفحة قائمة الطلبات
    header('Location: orders.php');
    exit;
}

// التحقق من صلاحيات المستخدم
$canEditOrder = ($currentUser['role'] === 'admin' || $currentUser['role'] === 'manager' || $order['user_id'] == $currentUser['id']);

if (!$canEditOrder) {
    // إعادة توجيه المستخدم إلى صفحة قائمة الطلبات مع رسالة خطأ
    $_SESSION['error_message'] = 'ليس لديك صلاحية لتعديل هذا الطلب';
    header('Location: orders.php');
    exit;
}

// الحصول على قائمة الفروع
$branchesStmt = $pdo->query("SELECT id, name, code FROM branches WHERE is_active = 1 ORDER BY name");
$branches = $branchesStmt->fetchAll();

// معالجة تحديث الطلب
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_order'])) {
    // التحقق من رمز CSRF
    if (!isset($_POST['csrf_token']) || !verifyCSRFToken($_POST['csrf_token'])) {
        $error_message = 'خطأ في التحقق من الأمان. يرجى المحاولة مرة أخرى.';
    } else {
        try {
            // تنظيف وتحقق من البيانات المدخلة
            $clientName = sanitizeInput($_POST['clientName'] ?? '');
            $serviceType = sanitizeInput($_POST['serviceType'] ?? '');
            $branchId = null;
            
            // الحصول على معرف الفرع من اسم الفرع
            $branchName = sanitizeInput($_POST['branch'] ?? '');
            if (!empty($branchName)) {
                $stmt = $pdo->prepare("SELECT id FROM branches WHERE name = ?");
                $stmt->execute([$branchName]);
                $branch = $stmt->fetch();
                if ($branch) {
                    $branchId = $branch['id'];
                }
            }
            
            $departureArea = sanitizeInput($_POST['departureArea'] ?? '');
            $secondDestination = sanitizeInput($_POST['secondDestination'] ?? '');
            $deliveryArea = sanitizeInput($_POST['deliveryArea'] ?? '');
            $goodsName = sanitizeInput($_POST['goodsName'] ?? '');
            $goodsType = sanitizeInput($_POST['goodsType'] ?? '');
            $countryOfOrigin = sanitizeInput($_POST['countryOfOrigin'] ?? '');
            $weight = floatval($_POST['weight'] ?? 0);
            $deliveryDuration = intval($_POST['deliveryDuration'] ?? 0);
            $orderDate = sanitizeInput($_POST['orderDate'] ?? date('Y-m-d'));
            $expectedDeliveryDate = sanitizeInput($_POST['expectedDeliveryDate'] ?? '');
            $receiverName = sanitizeInput($_POST['receiverName'] ?? '');
            $phoneNumber = sanitizeInput($_POST['phoneNumber'] ?? '');
            $serviceFee = floatval($_POST['serviceFee'] ?? 0);
            $currency = sanitizeInput($_POST['currency'] ?? '');
            $clientAmount = floatval($_POST['clientAmount'] ?? 0);
            $agentAmount = floatval($_POST['agentAmount'] ?? 0);
            $paidAmount = floatval($_POST['paidAmount'] ?? 0);
            $remainingAmount = floatval($_POST['remainingAmount'] ?? 0);
            $otherExpenses = floatval($_POST['otherExpenses'] ?? 0);
            $profit = floatval($_POST['profit'] ?? 0);
            $responsibleEmployee = sanitizeInput($_POST['responsibleEmployee'] ?? '');
            $orderStatus = sanitizeInput($_POST['orderStatus'] ?? 'قيد المعالجة');
            $notes = sanitizeInput($_POST['notes'] ?? '');
            
            // تحديد حالة الدفع
            $paymentStatus = 'غير مدفوع';
            if ($paidAmount >= $clientAmount) {
                $paymentStatus = 'مدفوع';
            } elseif ($paidAmount > 0) {
                $paymentStatus = 'جزئي';
            }
            
            // تحديث الطلب في قاعدة البيانات
            $stmt = $pdo->prepare("
                UPDATE orders SET
                    client_name = ?, service_type = ?, branch_id = ?,
                    departure_area = ?, second_destination = ?, delivery_area = ?,
                    goods_name = ?, goods_type = ?, country_of_origin = ?,
                    weight = ?, delivery_duration = ?,
                    order_upload_date = ?, scheduled_delivery_date = ?,
                    recipient_name = ?, phone_number = ?,
                    service_fees = ?, currency = ?,
                    amount_agreed_with_customer = ?, amount_agreed_with_agent = ?,
                    amount_paid = ?, remaining_amount = ?, other_expenses = ?, profit = ?,
                    employee_name = ?, status = ?, payment_status = ?, notes = ?,
                    updated_at = NOW()
                WHERE id = ?
            ");
            
            $stmt->execute([
                $clientName, $serviceType, $branchId,
                $departureArea, $secondDestination, $deliveryArea,
                $goodsName, $goodsType, $countryOfOrigin,
                $weight, $deliveryDuration,
                $orderDate, $expectedDeliveryDate,
                $receiverName, $phoneNumber,
                $serviceFee, $currency,
                $clientAmount, $agentAmount,
                $paidAmount, $remainingAmount, $otherExpenses, $profit,
                $responsibleEmployee, $orderStatus, $paymentStatus, $notes,
                $orderId
            ]);
            
            // تحويل المستخدم إلى صفحة تفاصيل الطلب مع رسالة نجاح
            $_SESSION['success_message'] = 'تم تحديث الطلب بنجاح';
            header('Location: order-details.php?id=' . $orderId);
            exit;
            
        } catch (PDOException $e) {
            $error_message = 'حدث خطأ أثناء تحديث الطلب: ' . $e->getMessage();
        }
    }
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">

<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>نظام طلبات الشحن | تعديل الطلب</title>
  <script src="https://cdn.tailwindcss.com"></script>
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
  <link rel="stylesheet" href="../assets/css/unified-design.css">
</head>

<body>
  <!-- طبقة التراكب للأجهزة المحمولة -->
  <div id="overlay" class="overlay"></div>

  <div class="flex min-h-screen">
    <!-- القائمة الجانبية -->
    <?php include '../includes/sidebar.php'; ?>

    <!-- المحتوى الرئيسي -->
    <main class="flex-1">
      <!-- الشريط العلوي -->
      <?php include '../includes/header.php'; ?>

      <!-- محتوى الصفحة -->
      <div class="p-6">
        <!-- رأس الصفحة -->
        <div class="flex flex-col md:flex-row md:items-center md:justify-between mb-8">
          <div class="animate-fadeIn" style="animation-delay: 0.1s;">
            <h1 class="text-3xl font-bold text-gray-800 flex items-center">
              <span class="bg-gradient-to-r from-primary to-primary-light bg-clip-text text-transparent">
                تعديل الطلب
              </span>
            </h1>
            <p class="text-gray-500 mt-2 flex items-center">
              <i class="fas fa-edit ml-2 text-primary"></i>
              <span>تعديل بيانات الطلب رقم: <?php echo htmlspecialchars($order['order_number']); ?></span>
            </p>
          </div>

          <div class="mt-4 md:mt-0 animate-fadeIn" style="animation-delay: 0.2s;">
            <div class="flex items-center space-x-4 space-x-reverse">
              <a href="order-details.php?id=<?php echo $order['id']; ?>" class="btn btn-secondary">
                <i class="fas fa-arrow-right ml-2"></i>
                <span>العودة للتفاصيل</span>
              </a>
            </div>
          </div>
        </div>

        <!-- عرض رسائل الخطأ إن وجدت -->
        <?php if (isset($error_message)): ?>
        <div class="bg-red-100 border-r-4 border-red-500 text-red-700 p-4 mb-6 rounded-lg">
          <p class="font-bold">خطأ!</p>
          <p><?php echo $error_message; ?></p>
        </div>
        <?php endif; ?>

        <!-- نموذج تعديل الطلب -->
        <form id="editOrderForm" method="post" action="" class="bg-white rounded-xl shadow-sm overflow-hidden animate-fadeIn">
          <div class="p-6 max-w-full overflow-x-auto">
            <!-- المعلومات الأساسية -->
            <div class="form-section bg-white rounded-lg shadow-sm p-6 mb-6 border border-gray-200 hover:border-indigo-300 transition-all duration-300">
              <h3 class="form-section-title">
                <i class="fas fa-info-circle text-indigo-500"></i>
                <span>المعلومات الأساسية للطلب</span>
              </h3>
              <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
                <div>
                  <label class="block text-sm font-medium text-gray-700 mb-1">رقم الطلب</label>
                  <div class="relative">
                    <input type="text" id="orderNumber" name="orderNumber" readonly class="form-input w-full bg-gray-50 text-indigo-700 font-medium" value="<?php echo htmlspecialchars($order['order_number']); ?>">
                    <i class="fas fa-hashtag absolute left-3 top-3 text-gray-400"></i>
                  </div>
                </div>
                <div>
                  <label class="block text-sm font-medium text-gray-700 mb-1">اسم العميل <span class="text-red-500">*</span></label>
                  <div class="relative">
                    <input type="text" id="clientName" name="clientName" required class="form-input w-full" placeholder="أدخل اسم العميل" value="<?php echo htmlspecialchars($order['client_name']); ?>">
                    <i class="fas fa-user absolute left-3 top-3 text-gray-400"></i>
                  </div>
                </div>
                <div>
                  <label class="block text-sm font-medium text-gray-700 mb-1">نوع الخدمة <span class="text-red-500">*</span></label>
                  <div class="relative">
                    <input type="text" id="serviceType" name="serviceType" required class="form-input w-full" placeholder="أدخل نوع الخدمة" value="<?php echo htmlspecialchars($order['service_type'] ?? ''); ?>">
                    <i class="fas fa-cog absolute left-3 top-3 text-gray-400"></i>
                  </div>
                </div>
              </div>
            </div>

            <!-- معلومات البضاعة -->
            <div class="form-section bg-white rounded-lg shadow-sm p-6 mb-6 border border-gray-200 hover:border-indigo-300 transition-all duration-300">
              <h3 class="form-section-title">
                <i class="fas fa-box text-amber-500"></i>
                <span>معلومات البضاعة</span>
              </h3>
              <div class="grid grid-cols-1 md:grid-cols-4 gap-6">
                <div>
                  <label class="block text-sm font-medium text-gray-700 mb-1">اسم البضاعة <span class="text-red-500">*</span></label>
                  <input type="text" id="goodsName" name="goodsName" required class="form-input w-full" placeholder="أدخل اسم البضاعة" value="<?php echo htmlspecialchars($order['goods_name'] ?? ''); ?>">
                </div>
                <div>
                  <label class="block text-sm font-medium text-gray-700 mb-1">نوع البضاعة <span class="text-red-500">*</span></label>
                  <input type="text" id="goodsType" name="goodsType" required class="form-input w-full" placeholder="أدخل نوع البضاعة" value="<?php echo htmlspecialchars($order['goods_type'] ?? ''); ?>">
                </div>
                <div>
                  <label class="block text-sm font-medium text-gray-700 mb-1">بلد المنشأ <span class="text-red-500">*</span></label>
                  <div class="relative">
                    <input type="text" id="countryOfOrigin" name="countryOfOrigin" required class="form-input w-full" placeholder="أدخل بلد المنشأ" value="<?php echo htmlspecialchars($order['country_of_origin'] ?? ''); ?>">
                    <i class="fas fa-globe absolute left-3 top-3 text-gray-400"></i>
                  </div>
                </div>
                <div>
                  <label class="block text-sm font-medium text-gray-700 mb-1">الوزن <span class="text-red-500">*</span></label>
                  <div class="relative">
                    <input type="number" id="weight" name="weight" step="0.1" required class="form-input w-full" placeholder="أدخل الوزن" value="<?php echo $order['weight'] ?? ''; ?>">
                    <i class="fas fa-weight absolute left-3 top-3 text-gray-400"></i>
                  </div>
                </div>
              </div>
            </div>

            <!-- معلومات الفرع والمناطق -->
            <div class="form-section bg-white rounded-lg shadow-sm p-6 mb-6 border border-gray-200 hover:border-indigo-300 transition-all duration-300">
              <h3 class="form-section-title">
                <i class="fas fa-map-marker-alt text-red-500"></i>
                <span>معلومات الفرع والمناطق</span>
              </h3>
              <div class="grid grid-cols-1 md:grid-cols-4 gap-6">
                <div>
                  <label class="block text-sm font-medium text-gray-700 mb-1">الفرع <span class="text-red-500">*</span></label>
                  <div class="relative">
                    <select id="branch" name="branch" required class="form-input w-full">
                      <option value="">اختر الفرع</option>
                      <?php foreach ($branches as $branch): ?>
                      <option value="<?php echo htmlspecialchars($branch['name']); ?>" data-code="<?php echo htmlspecialchars($branch['code']); ?>" <?php echo ($branch['name'] === $order['branch_name']) ? 'selected' : ''; ?>><?php echo htmlspecialchars($branch['name']); ?></option>
                      <?php endforeach; ?>
                    </select>
                    <i class="fas fa-building absolute left-3 top-3 text-gray-400"></i>
                  </div>
                </div>
                <div>
                  <label class="block text-sm font-medium text-gray-700 mb-1">منطقة المغادرة <span class="text-red-500">*</span></label>
                  <input type="text" id="departureArea" name="departureArea" required class="form-input w-full" placeholder="أدخل منطقة المغادرة" value="<?php echo htmlspecialchars($order['departure_area'] ?? ''); ?>">
                </div>
                <div>
                  <label class="block text-sm font-medium text-gray-700 mb-1">الوجهة الثانية</label>
                  <input type="text" id="secondDestination" name="secondDestination" class="form-input w-full" placeholder="أدخل الوجهة الثانية" value="<?php echo htmlspecialchars($order['second_destination'] ?? ''); ?>">
                </div>
                <div>
                  <label class="block text-sm font-medium text-gray-700 mb-1">منطقة التسليم <span class="text-red-500">*</span></label>
                  <input type="text" id="deliveryArea" name="deliveryArea" required class="form-input w-full" placeholder="أدخل منطقة التسليم" value="<?php echo htmlspecialchars($order['delivery_area'] ?? ''); ?>">
                </div>
              </div>
            </div>

            <!-- أزرار الإجراءات -->
            <div class="flex flex-col md:flex-row gap-4 mt-6 sticky bottom-0 bg-white p-4 rounded-lg shadow-lg border border-gray-200 z-10">
              <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">
              <button type="submit" name="update_order" class="btn btn-primary flex-1">
                <i class="fas fa-save ml-2"></i>
                <span>حفظ التعديلات</span>
              </button>
              <a href="order-details.php?id=<?php echo $order['id']; ?>" class="btn btn-secondary flex-1 text-center">
                <i class="fas fa-times ml-2"></i>
                <span>إلغاء</span>
              </a>
            </div>
          </div>
        </form>
      </div>
    </main>
  </div>

  <!-- تضمين مكتبة SweetAlert2 للتنبيهات الجميلة -->
  <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>

  <script>
    document.addEventListener('DOMContentLoaded', function() {
      // تهيئة القائمة الجانبية للأجهزة المحمولة
      const toggleSidebar = document.getElementById('toggleSidebar');
      const sidebar = document.querySelector('aside');
      const overlay = document.getElementById('overlay');

      if (toggleSidebar && sidebar && overlay) {
        toggleSidebar.addEventListener('click', function() {
          sidebar.classList.toggle('hidden');
          sidebar.classList.toggle('active');
          overlay.classList.toggle('hidden');
          overlay.classList.toggle('active');
          document.body.classList.toggle('overflow-hidden');
        });

        overlay.addEventListener('click', function() {
          sidebar.classList.add('hidden');
          sidebar.classList.remove('active');
          overlay.classList.add('hidden');
          overlay.classList.remove('active');
          document.body.classList.remove('overflow-hidden');
        });
      }

      // التحقق من صحة النموذج قبل الإرسال
      const editOrderForm = document.getElementById('editOrderForm');
      if (editOrderForm) {
        editOrderForm.addEventListener('submit', function(event) {
          const requiredFields = document.querySelectorAll('input[required], select[required]');
          let isValid = true;

          requiredFields.forEach(field => {
            if (!field.value.trim()) {
              field.classList.add('border-red-500');
              isValid = false;
            } else {
              field.classList.remove('border-red-500');
            }
          });

          if (!isValid) {
            event.preventDefault();
            Swal.fire({
              title: 'خطأ في النموذج',
              text: 'يرجى ملء جميع الحقول المطلوبة',
              icon: 'error',
              confirmButtonText: 'حسناً'
            });
          }
        });
      }
    });
  </script>
</body>

</html>
