/**
 * ملف JavaScript لإدارة نموذج إنشاء الطلب
 */

document.addEventListener('DOMContentLoaded', function() {
    // تهيئة النموذج
    initOrderForm();
});

/**
 * دالة لتهيئة نموذج إنشاء الطلب
 */
function initOrderForm() {
    // الحصول على عناصر النموذج
    const form = document.getElementById('shipmentForm');
    const branchSelect = document.getElementById('branch');
    const orderNumberField = document.getElementById('orderNumber');
    const orderUploadDateField = document.getElementById('orderUploadDate');
    const serviceFeeField = document.getElementById('serviceFees');
    const amountPaidField = document.getElementById('amountPaid');
    const remainingAmountField = document.getElementById('remainingAmount');
    const amountWithCustomerField = document.getElementById('amountAgreedWithCustomer');
    const amountWithAgentField = document.getElementById('amountAgreedWithAgent');
    const otherExpensesField = document.getElementById('otherExpenses');
    const profitField = document.getElementById('profit');
    const submitBtn = document.getElementById('submitOrderBtn');

    // تعيين تاريخ اليوم كتاريخ افتراضي لرفع الطلب
    if (orderUploadDateField) {
        orderUploadDateField.valueAsDate = new Date();
    }

    // إنشاء رقم الطلب عند تغيير الفرع
    if (branchSelect) {
        branchSelect.addEventListener('change', function() {
            generateOrderNumber();
        });
    }

    // حساب المبلغ المتبقي عند تغيير رسوم الخدمة أو المبلغ المدفوع
    if (serviceFeeField && amountPaidField) {
        serviceFeeField.addEventListener('input', calculateRemainingAmount);
        amountPaidField.addEventListener('input', calculateRemainingAmount);
    }

    // حساب الربح عند تغيير المبالغ
    if (amountWithCustomerField && amountWithAgentField && otherExpensesField) {
        amountWithCustomerField.addEventListener('input', calculateProfit);
        amountWithAgentField.addEventListener('input', calculateProfit);
        otherExpensesField.addEventListener('input', calculateProfit);
    }

    // إرسال النموذج
    if (submitBtn) {
        submitBtn.addEventListener('click', function() {
            submitOrderForm();
        });
    }
}

/**
 * دالة لإنشاء رقم طلب جديد
 */
function generateOrderNumber() {
    const branchSelect = document.getElementById('branch');
    const orderNumberField = document.getElementById('orderNumber');

    if (!branchSelect || !orderNumberField) return;

    const selectedOption = branchSelect.options[branchSelect.selectedIndex];
    if (!selectedOption || !selectedOption.value) return;

    // الحصول على رمز الفرع
    const branchCode = selectedOption.getAttribute('data-code');
    if (!branchCode) return;

    // إظهار مؤشر التحميل
    orderNumberField.value = 'جاري إنشاء رقم الطلب...';
    orderNumberField.classList.add('bg-indigo-50');

    // إنشاء رقم الطلب
    const year = new Date().getFullYear().toString().substr(-2);
    const randomNum = Math.floor(Math.random() * 100).toString().padStart(2, '0');
    const orderNumber = `${branchCode}${year}-${randomNum}`;

    // تعيين رقم الطلب
    setTimeout(() => {
        orderNumberField.value = orderNumber;
        orderNumberField.classList.remove('bg-indigo-50');
        orderNumberField.classList.add('bg-green-50');
        setTimeout(() => {
            orderNumberField.classList.remove('bg-green-50');
        }, 1000);
    }, 500);
}

/**
 * دالة لحساب المبلغ المتبقي
 */
function calculateRemainingAmount() {
    const serviceFeeField = document.getElementById('serviceFees');
    const amountPaidField = document.getElementById('amountPaid');
    const remainingAmountField = document.getElementById('remainingAmount');

    if (!serviceFeeField || !amountPaidField || !remainingAmountField) return;

    const serviceFee = parseFloat(serviceFeeField.value) || 0;
    const amountPaid = parseFloat(amountPaidField.value) || 0;
    const remainingAmount = serviceFee - amountPaid;

    remainingAmountField.value = remainingAmount.toFixed(2);
}

/**
 * دالة لحساب الربح
 */
function calculateProfit() {
    const amountWithCustomerField = document.getElementById('amountAgreedWithCustomer');
    const amountWithAgentField = document.getElementById('amountAgreedWithAgent');
    const otherExpensesField = document.getElementById('otherExpenses');
    const profitField = document.getElementById('profit');

    if (!amountWithCustomerField || !amountWithAgentField || !otherExpensesField || !profitField) return;

    const amountWithCustomer = parseFloat(amountWithCustomerField.value) || 0;
    const amountWithAgent = parseFloat(amountWithAgentField.value) || 0;
    const otherExpenses = parseFloat(otherExpensesField.value) || 0;
    const profit = amountWithCustomer - amountWithAgent - otherExpenses;

    profitField.value = profit.toFixed(2);
}

/**
 * دالة لإرسال نموذج الطلب
 */
function submitOrderForm() {
    const form = document.getElementById('shipmentForm');
    
    if (!form) return;
    
    // التحقق من صحة النموذج
    if (!validateOrderForm()) {
        Swal.fire({
            icon: 'error',
            title: 'خطأ في النموذج',
            text: 'يرجى ملء جميع الحقول المطلوبة',
            confirmButtonText: 'حسناً'
        });
        return;
    }
    
    // إنشاء كائن FormData
    const formData = new FormData(form);
    
    // إضافة حقول إضافية
    formData.append('status', 'قيد المعالجة');
    
    // عرض مؤشر التحميل
    const submitBtn = document.getElementById('submitOrderBtn');
    if (submitBtn) {
        submitBtn.disabled = true;
        submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin ml-2"></i> <span>جاري الحفظ...</span>';
    }
    
    // محاكاة إرسال البيانات (في الإصدار النهائي، يجب استبدال هذا بطلب حقيقي إلى الخادم)
    setTimeout(() => {
        // إعادة تفعيل الزر
        if (submitBtn) {
            submitBtn.disabled = false;
            submitBtn.innerHTML = '<i class="fas fa-paper-plane ml-2"></i> <span>إرسال الطلب</span>';
        }
        
        // عرض رسالة نجاح
        Swal.fire({
            icon: 'success',
            title: 'تم إنشاء الطلب بنجاح',
            text: `تم إنشاء الطلب برقم: ${document.getElementById('orderNumber').value}`,
            confirmButtonText: 'حسناً'
        }).then(() => {
            // إعادة توجيه المستخدم إلى صفحة إدارة الطلبات
            window.location.href = 'orders.php';
        });
    }, 1500);
}

/**
 * دالة للتحقق من صحة نموذج الطلب
 */
function validateOrderForm() {
    const requiredFields = [
        'clientName',
        'branch',
        'goodsName',
        'goodsType',
        'departureArea',
        'deliveryArea',
        'serviceFees',
        'currency',
        'recipientName',
        'phoneNumber'
    ];
    
    let isValid = true;
    
    requiredFields.forEach(fieldId => {
        const field = document.getElementById(fieldId);
        if (field && !field.value.trim()) {
            field.classList.add('border-red-500');
            isValid = false;
        } else if (field) {
            field.classList.remove('border-red-500');
        }
    });
    
    return isValid;
}
