<?php
/**
 * متحكم الطلبات
 * يتعامل مع إدارة الطلبات (إنشاء، عرض، تعديل، حذف)
 *
 * @version 1.0
 * @author فريق التطوير
 */

namespace App\Controllers;

use App\Core\Auth;
use App\Models\Order;
use App\Models\Branch;

class OrderController {
    private $auth;
    private $orderModel;
    private $branchModel;

    /**
     * إنشاء متحكم الطلبات
     */
    public function __construct() {
        $this->auth = Auth::getInstance();
        $this->orderModel = new Order();
        $this->branchModel = new Branch();

        // التحقق من تسجيل الدخول
        if (!$this->auth->isLoggedIn()) {
            header('Location: /login');
            exit;
        }
    }

    /**
     * عرض قائمة الطلبات
     */
    public function index() {
        // الحصول على بيانات المستخدم الحالي
        $user = $this->auth->getUser();

        // تحديد ما إذا كان المستخدم مديرًا
        $isAdmin = ($user['role'] === 'admin' || $user['role'] === 'manager');

        // تحديد معلمات الفلترة
        $filters = [];

        // إذا كان المستخدم ليس مديرًا، فلتر الطلبات حسب الفرع
        if (!$isAdmin) {
            $filters['branch_id'] = $user['branch_id'];
        }

        // الحصول على معلمات البحث
        $search = $_GET['search'] ?? '';
        $status = $_GET['status'] ?? '';
        $startDate = $_GET['start_date'] ?? '';
        $endDate = $_GET['end_date'] ?? '';
        $branchId = $_GET['branch_id'] ?? '';

        // إضافة معلمات البحث إلى الفلاتر
        if (!empty($search)) {
            $filters['search'] = $search;
        }

        if (!empty($status)) {
            $filters['status'] = $status;
        }

        if (!empty($startDate)) {
            $filters['start_date'] = $startDate;
        }

        if (!empty($endDate)) {
            $filters['end_date'] = $endDate;
        }

        if (!empty($branchId) && $isAdmin) {
            $filters['branch_id'] = $branchId;
        }

        // تحديد الصفحة الحالية
        $page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
        $perPage = 20;
        $offset = ($page - 1) * $perPage;

        // الحصول على الطلبات
        $orders = $this->orderModel->getAll($perPage, $offset, $filters);

        // الحصول على إجمالي عدد الطلبات
        $totalOrders = $this->orderModel->count($filters);

        // حساب عدد الصفحات
        $totalPages = ceil($totalOrders / $perPage);

        // الحصول على قائمة الفروع للفلترة
        $branches = $isAdmin ? $this->branchModel->getActive() : [];

        // تعيين عنوان الصفحة والصفحة النشطة
        $pageTitle = 'قائمة الطلبات';
        $activePage = 'orders';

        // تمرير متغير $auth إلى الصفحة
        $auth = $this->auth;
        $config = require dirname(__DIR__, 2) . '/config/config.php';

        // عرض صفحة قائمة الطلبات
        include dirname(__DIR__) . '/views/orders/list.php';
    }

    /**
     * عرض صفحة إنشاء طلب جديد
     */
    public function create() {
        // الحصول على بيانات المستخدم الحالي
        $user = $this->auth->getUser();

        // تحديد ما إذا كان المستخدم مديرًا
        $isAdmin = ($user['role'] === 'admin' || $user['role'] === 'manager');

        // الحصول على قائمة الفروع
        $branches = $isAdmin ? $this->branchModel->getActive() : [$this->branchModel->findById($user['branch_id'])];

        // تعيين عنوان الصفحة والصفحة النشطة
        $pageTitle = 'إنشاء طلب جديد';
        $activePage = 'create-order';

        // تمرير متغير $auth إلى الصفحة
        $auth = $this->auth;
        $config = require dirname(__DIR__, 2) . '/config/config.php';

        // عرض صفحة إنشاء طلب جديد
        include dirname(__DIR__) . '/views/orders/create.php';
    }

    /**
     * حفظ طلب جديد
     */
    public function store() {
        // التحقق من طريقة الطلب
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /orders/create');
            exit;
        }

        // التحقق من رمز CSRF
        if (!isset($_POST['csrf_token']) || !$this->auth->verifyCSRFToken($_POST['csrf_token'])) {
            $_SESSION['flash_message'] = 'خطأ في التحقق من الأمان. يرجى المحاولة مرة أخرى.';
            $_SESSION['flash_type'] = 'error';
            header('Location: /orders/create');
            exit;
        }

        // الحصول على بيانات المستخدم الحالي
        $user = $this->auth->getUser();

        // تحديد ما إذا كان المستخدم مديرًا
        $isAdmin = ($user['role'] === 'admin' || $user['role'] === 'manager');

        // الحصول على بيانات النموذج
        $branchId = $_POST['branch_id'] ?? ($isAdmin ? null : $user['branch_id']);
        $customerName = $_POST['customer_name'] ?? '';
        $customerPhone = $_POST['customer_phone'] ?? '';
        $customerAddress = $_POST['customer_address'] ?? '';
        $orderDetails = $_POST['order_details'] ?? '';
        $amountPaid = $_POST['amount_paid'] ?? 0;
        $status = $_POST['status'] ?? 'قيد المعالجة';
        $employeeName = $_POST['employee_name'] ?? $user['full_name'];

        // التحقق من البيانات
        if (empty($branchId) || empty($customerName) || empty($orderDetails)) {
            $_SESSION['flash_message'] = 'يرجى ملء جميع الحقول المطلوبة.';
            $_SESSION['flash_type'] = 'error';
            header('Location: /orders/create');
            exit;
        }

        // الحصول على معلومات الفرع
        $branch = $this->branchModel->findById($branchId);
        if (!$branch) {
            $_SESSION['flash_message'] = 'الفرع غير موجود.';
            $_SESSION['flash_type'] = 'error';
            header('Location: /orders/create');
            exit;
        }

        // الحصول على رقم طلب جديد
        $orderNumber = $this->orderModel->getNewOrderNumber($branch['code']);
        if (!$orderNumber) {
            $_SESSION['flash_message'] = 'فشل إنشاء رقم طلب جديد.';
            $_SESSION['flash_type'] = 'error';
            header('Location: /orders/create');
            exit;
        }

        // إعداد بيانات الطلب
        $orderData = [
            'branch_id' => $branchId,
            'order_number' => $orderNumber,
            'customer_name' => $customerName,
            'customer_phone' => $customerPhone,
            'customer_address' => $customerAddress,
            'order_details' => $orderDetails,
            'amount_paid' => $amountPaid,
            'status' => $status,
            'employee_name' => $employeeName,
            'created_by' => $user['id'],
            'created_at' => date('Y-m-d H:i:s')
        ];

        // إنشاء الطلب
        $result = $this->orderModel->create($orderData);

        if ($result['success']) {
            $_SESSION['flash_message'] = 'تم إنشاء الطلب بنجاح.';
            $_SESSION['flash_type'] = 'success';
            header('Location: /orders/view/' . $result['order_id']);
            exit;
        } else {
            $_SESSION['flash_message'] = 'فشل إنشاء الطلب: ' . $result['message'];
            $_SESSION['flash_type'] = 'error';
            header('Location: /orders/create');
            exit;
        }
    }

    /**
     * عرض تفاصيل الطلب
     */
    public function view($id) {
        // الحصول على بيانات المستخدم الحالي
        $user = $this->auth->getUser();

        // تحديد ما إذا كان المستخدم مديرًا
        $isAdmin = ($user['role'] === 'admin' || $user['role'] === 'manager');

        // الحصول على بيانات الطلب
        $order = $this->orderModel->findById($id);

        // التحقق من وجود الطلب
        if (!$order) {
            $_SESSION['flash_message'] = 'الطلب غير موجود.';
            $_SESSION['flash_type'] = 'error';
            header('Location: /orders');
            exit;
        }

        // التحقق من صلاحية الوصول إلى الطلب
        if (!$isAdmin && $order['branch_id'] != $user['branch_id']) {
            $_SESSION['flash_message'] = 'ليس لديك صلاحية للوصول إلى هذا الطلب.';
            $_SESSION['flash_type'] = 'error';
            header('Location: /orders');
            exit;
        }

        // تعيين عنوان الصفحة والصفحة النشطة
        $pageTitle = 'تفاصيل الطلب: ' . $order['order_number'];
        $activePage = 'orders';

        // تمرير متغير $auth إلى الصفحة
        $auth = $this->auth;
        $config = require dirname(__DIR__, 2) . '/config/config.php';

        // عرض صفحة تفاصيل الطلب
        include dirname(__DIR__) . '/views/orders/view.php';
    }

    /**
     * تحديث حالة الطلب
     */
    public function updateStatus($id) {
        // التحقق من طريقة الطلب
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /orders/view/' . $id);
            exit;
        }

        // التحقق من رمز CSRF
        if (!isset($_POST['csrf_token']) || !$this->auth->verifyCSRFToken($_POST['csrf_token'])) {
            $_SESSION['flash_message'] = 'خطأ في التحقق من الأمان. يرجى المحاولة مرة أخرى.';
            $_SESSION['flash_type'] = 'error';
            header('Location: /orders/view/' . $id);
            exit;
        }

        // الحصول على بيانات المستخدم الحالي
        $user = $this->auth->getUser();

        // تحديد ما إذا كان المستخدم مديرًا
        $isAdmin = ($user['role'] === 'admin' || $user['role'] === 'manager');

        // الحصول على بيانات الطلب
        $order = $this->orderModel->findById($id);

        // التحقق من وجود الطلب
        if (!$order) {
            $_SESSION['flash_message'] = 'الطلب غير موجود.';
            $_SESSION['flash_type'] = 'error';
            header('Location: /orders');
            exit;
        }

        // التحقق من صلاحية الوصول إلى الطلب
        if (!$isAdmin && $order['branch_id'] != $user['branch_id']) {
            $_SESSION['flash_message'] = 'ليس لديك صلاحية للوصول إلى هذا الطلب.';
            $_SESSION['flash_type'] = 'error';
            header('Location: /orders');
            exit;
        }

        // الحصول على الحالة الجديدة
        $newStatus = $_POST['status'] ?? '';

        if (empty($newStatus)) {
            $_SESSION['flash_message'] = 'يرجى تحديد الحالة الجديدة.';
            $_SESSION['flash_type'] = 'error';
            header('Location: /orders/view/' . $id);
            exit;
        }

        // تحديث حالة الطلب
        $result = $this->orderModel->update($id, [
            'status' => $newStatus,
            'updated_by' => $user['id'],
            'updated_at' => date('Y-m-d H:i:s')
        ]);

        if ($result['success']) {
            $_SESSION['flash_message'] = 'تم تحديث حالة الطلب بنجاح.';
            $_SESSION['flash_type'] = 'success';
        } else {
            $_SESSION['flash_message'] = 'فشل تحديث حالة الطلب: ' . $result['message'];
            $_SESSION['flash_type'] = 'error';
        }

        header('Location: /orders/view/' . $id);
        exit;
    }
}
