<?php
/**
 * فئة المصادقة
 * تتعامل مع تسجيل الدخول والتحقق من الصلاحيات
 *
 * @version 1.0
 * @author فريق التطوير
 */

namespace App\Core;

use App\Models\User;

class Auth {
    private static $instance;
    private $user = null;
    private $isLoggedIn = false;

    /**
     * إنشاء كائن المصادقة
     */
    private function __construct() {
        // بدء الجلسة إذا لم تكن قد بدأت
        if (session_status() == PHP_SESSION_NONE) {
            session_start();
        }
        
        // التحقق من حالة تسجيل الدخول
        $this->checkLoginStatus();
    }

    /**
     * الحصول على نسخة واحدة من الفئة (نمط Singleton)
     */
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * التحقق من حالة تسجيل الدخول
     */
    private function checkLoginStatus() {
        if (isset($_SESSION['user_id']) && !empty($_SESSION['user_id'])) {
            $this->isLoggedIn = true;
            
            // إنشاء كائن المستخدم من بيانات الجلسة
            $this->user = [
                'id' => $_SESSION['user_id'],
                'username' => $_SESSION['username'] ?? '',
                'full_name' => $_SESSION['full_name'] ?? '',
                'email' => $_SESSION['email'] ?? '',
                'role' => $_SESSION['role'] ?? 'user',
                'branch_id' => $_SESSION['branch_id'] ?? null,
                'branch_name' => $_SESSION['branch_name'] ?? '',
                'branch_code' => $_SESSION['branch_code'] ?? ''
            ];
        }
    }

    /**
     * تسجيل دخول المستخدم
     */
    public function login($email, $password, $branch_id = null) {
        // استخدام نموذج المستخدم للتحقق من بيانات الاعتماد
        $userModel = new User();
        $user = $userModel->findByEmail($email);
        
        if (!$user) {
            return false;
        }
        
        // التحقق من كلمة المرور
        if (!password_verify($password, $user['password'])) {
            return false;
        }
        
        // التحقق من الفرع إذا كان مطلوبًا
        if ($user['role'] !== 'admin' && $branch_id === null) {
            return false;
        }
        
        // تعيين بيانات المستخدم في الجلسة
        $_SESSION['user_id'] = $user['id'];
        $_SESSION['username'] = $user['username'];
        $_SESSION['full_name'] = $user['full_name'];
        $_SESSION['email'] = $user['email'];
        $_SESSION['role'] = $user['role'];
        $_SESSION['logged_in'] = true;
        
        // تعيين معلومات الفرع إذا كان مطلوبًا
        if ($branch_id !== null && $user['role'] !== 'admin') {
            // الحصول على معلومات الفرع
            $branchModel = new \App\Models\Branch();
            $branch = $branchModel->findById($branch_id);
            
            if ($branch) {
                $_SESSION['branch_id'] = $branch['id'];
                $_SESSION['branch_name'] = $branch['name'];
                $_SESSION['branch_code'] = $branch['code'];
            }
        }
        
        // تحديث حالة تسجيل الدخول
        $this->isLoggedIn = true;
        $this->checkLoginStatus();
        
        // تسجيل نشاط تسجيل الدخول
        $this->logActivity($user['id'], 'تسجيل الدخول');
        
        return true;
    }

    /**
     * تسجيل خروج المستخدم
     */
    public function logout() {
        // تسجيل نشاط تسجيل الخروج إذا كان المستخدم مسجل الدخول
        if ($this->isLoggedIn && isset($_SESSION['user_id'])) {
            $this->logActivity($_SESSION['user_id'], 'تسجيل الخروج');
        }
        
        // حذف بيانات الجلسة
        session_unset();
        session_destroy();
        
        // إعادة تعيين حالة تسجيل الدخول
        $this->isLoggedIn = false;
        $this->user = null;
        
        return true;
    }

    /**
     * التحقق مما إذا كان المستخدم مسجل الدخول
     */
    public function isLoggedIn() {
        return $this->isLoggedIn;
    }

    /**
     * الحصول على بيانات المستخدم الحالي
     */
    public function getUser() {
        return $this->user;
    }

    /**
     * التحقق من صلاحيات المستخدم
     */
    public function hasPermission($permission) {
        if (!$this->isLoggedIn) {
            return false;
        }
        
        // المدير لديه جميع الصلاحيات
        if ($this->user['role'] === 'admin') {
            return true;
        }
        
        // يمكن تنفيذ المزيد من التحقق من الصلاحيات هنا
        // على سبيل المثال، يمكن التحقق من جدول الصلاحيات في قاعدة البيانات
        
        return false;
    }

    /**
     * تسجيل نشاط المستخدم
     */
    private function logActivity($userId, $action, $details = '') {
        $db = Database::getInstance();
        
        try {
            $data = [
                'user_id' => $userId,
                'action' => $action,
                'details' => $details,
                'ip_address' => $_SERVER['REMOTE_ADDR'] ?? 'غير معروف',
                'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'غير معروف',
                'created_at' => date('Y-m-d H:i:s')
            ];
            
            $db->insert('activity_log', $data);
            return true;
        } catch (\Exception $e) {
            error_log("فشل تسجيل النشاط: " . $e->getMessage());
            return false;
        }
    }

    /**
     * توليد رمز CSRF
     */
    public function generateCSRFToken() {
        if (!isset($_SESSION['csrf_token'])) {
            $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
        }
        
        return $_SESSION['csrf_token'];
    }

    /**
     * التحقق من رمز CSRF
     */
    public function verifyCSRFToken($token) {
        if (!isset($_SESSION['csrf_token']) || $token !== $_SESSION['csrf_token']) {
            // تسجيل محاولة تزوير CSRF
            error_log("محاولة تزوير CSRF من العنوان: " . $_SERVER['REMOTE_ADDR']);
            return false;
        }
        
        return true;
    }
}
