<?php
/**
 * نموذج الفرع
 * يتعامل مع بيانات الفروع في قاعدة البيانات
 *
 * @version 1.0
 * @author فريق التطوير
 */

namespace App\Models;

use App\Core\Database;

class Branch {
    private $db;
    private $table = 'branches';

    /**
     * إنشاء نموذج الفرع
     */
    public function __construct() {
        $this->db = Database::getInstance();
    }

    /**
     * الحصول على جميع الفروع
     */
    public function getAll() {
        return $this->db->fetchAll("SELECT * FROM {$this->table} ORDER BY id ASC");
    }

    /**
     * الحصول على الفروع النشطة
     */
    public function getActive() {
        return $this->db->fetchAll("SELECT * FROM {$this->table} WHERE is_active = 1 ORDER BY id ASC");
    }

    /**
     * البحث عن فرع بواسطة المعرف
     */
    public function findById($id) {
        return $this->db->fetchOne("SELECT * FROM {$this->table} WHERE id = ?", [$id]);
    }

    /**
     * البحث عن فرع بواسطة الرمز
     */
    public function findByCode($code) {
        return $this->db->fetchOne("SELECT * FROM {$this->table} WHERE code = ?", [$code]);
    }

    /**
     * البحث عن فرع بواسطة الاسم
     */
    public function findByName($name) {
        return $this->db->fetchOne("SELECT * FROM {$this->table} WHERE name = ?", [$name]);
    }

    /**
     * إنشاء فرع جديد
     */
    public function create($data) {
        // التحقق من وجود الرمز أو الاسم
        $existingCode = $this->findByCode($data['code']);
        if ($existingCode) {
            return ['success' => false, 'message' => 'رمز الفرع مستخدم بالفعل'];
        }

        $existingName = $this->findByName($data['name']);
        if ($existingName) {
            return ['success' => false, 'message' => 'اسم الفرع مستخدم بالفعل'];
        }

        // إضافة تاريخ الإنشاء
        $data['created_at'] = date('Y-m-d H:i:s');
        
        // إدراج الفرع في قاعدة البيانات
        $branchId = $this->db->insert($this->table, $data);
        
        if ($branchId) {
            // إنشاء سجل تسلسل الطلبات للفرع الجديد
            $this->db->insert('order_sequences', [
                'branch_code' => $data['code'],
                'last_sequence_number' => 0
            ]);
            
            return ['success' => true, 'branch_id' => $branchId];
        } else {
            return ['success' => false, 'message' => 'فشل إنشاء الفرع'];
        }
    }

    /**
     * تحديث بيانات الفرع
     */
    public function update($id, $data) {
        // التحقق من وجود الفرع
        $branch = $this->findById($id);
        if (!$branch) {
            return ['success' => false, 'message' => 'الفرع غير موجود'];
        }

        // التحقق من الرمز
        if (isset($data['code']) && $data['code'] !== $branch['code']) {
            $existingCode = $this->findByCode($data['code']);
            if ($existingCode) {
                return ['success' => false, 'message' => 'رمز الفرع مستخدم بالفعل'];
            }
            
            // تحديث رمز الفرع في جدول تسلسل الطلبات
            $this->db->update('order_sequences', 
                ['branch_code' => $data['code']], 
                'branch_code = ?', 
                [$branch['code']]
            );
        }

        // التحقق من الاسم
        if (isset($data['name']) && $data['name'] !== $branch['name']) {
            $existingName = $this->findByName($data['name']);
            if ($existingName && $existingName['id'] != $id) {
                return ['success' => false, 'message' => 'اسم الفرع مستخدم بالفعل'];
            }
        }
        
        // إضافة تاريخ التحديث
        $data['updated_at'] = date('Y-m-d H:i:s');
        
        // تحديث بيانات الفرع
        $success = $this->db->update($this->table, $data, 'id = ?', [$id]);
        
        if ($success) {
            return ['success' => true];
        } else {
            return ['success' => false, 'message' => 'فشل تحديث بيانات الفرع'];
        }
    }

    /**
     * حذف فرع
     */
    public function delete($id) {
        // التحقق من وجود الفرع
        $branch = $this->findById($id);
        if (!$branch) {
            return ['success' => false, 'message' => 'الفرع غير موجود'];
        }
        
        // التحقق من وجود طلبات مرتبطة بالفرع
        $orderCount = $this->db->fetchColumn("SELECT COUNT(*) FROM orders WHERE branch_id = ?", [$id]);
        if ($orderCount > 0) {
            return ['success' => false, 'message' => 'لا يمكن حذف الفرع لأنه يحتوي على طلبات'];
        }
        
        // حذف سجل تسلسل الطلبات للفرع
        $this->db->delete('order_sequences', 'branch_code = ?', [$branch['code']]);
        
        // حذف الفرع
        $success = $this->db->delete($this->table, 'id = ?', [$id]);
        
        if ($success) {
            return ['success' => true];
        } else {
            return ['success' => false, 'message' => 'فشل حذف الفرع'];
        }
    }

    /**
     * الحصول على إحصائيات الفرع
     */
    public function getStats($id, $startDate = null, $endDate = null) {
        $sql = "SELECT 
                COUNT(o.id) as total_orders,
                COALESCE(SUM(o.amount_paid), 0) as total_revenue
                FROM {$this->table} b
                LEFT JOIN orders o ON o.branch_id = b.id
                WHERE b.id = ?";
        
        $params = [$id];
        
        // إضافة فلتر تاريخ البداية
        if ($startDate !== null) {
            $sql .= " AND o.created_at >= ?";
            $params[] = $startDate;
        }
        
        // إضافة فلتر تاريخ النهاية
        if ($endDate !== null) {
            $sql .= " AND o.created_at <= ?";
            $params[] = $endDate . ' 23:59:59';
        }
        
        return $this->db->fetchOne($sql, $params);
    }

    /**
     * الحصول على أفضل الفروع أداءً
     */
    public function getTopBranches($limit = 5, $startDate = null, $endDate = null) {
        $sql = "SELECT 
                b.id,
                b.name,
                b.code,
                COUNT(o.id) as total_orders,
                COALESCE(SUM(o.amount_paid), 0) as total_revenue
                FROM {$this->table} b
                LEFT JOIN orders o ON o.branch_id = b.id";
        
        $params = [];
        $whereClause = ["b.is_active = 1"];
        
        // إضافة فلتر تاريخ البداية
        if ($startDate !== null) {
            $whereClause[] = "o.created_at >= ?";
            $params[] = $startDate;
        }
        
        // إضافة فلتر تاريخ النهاية
        if ($endDate !== null) {
            $whereClause[] = "o.created_at <= ?";
            $params[] = $endDate . ' 23:59:59';
        }
        
        $sql .= " WHERE " . implode(" AND ", $whereClause);
        $sql .= " GROUP BY b.id, b.name, b.code";
        $sql .= " ORDER BY total_orders DESC, total_revenue DESC";
        $sql .= " LIMIT ?";
        $params[] = $limit;
        
        return $this->db->fetchAll($sql, $params);
    }
}
