<?php
/**
 * نموذج المستخدم
 * يتعامل مع بيانات المستخدمين في قاعدة البيانات
 *
 * @version 1.0
 * @author فريق التطوير
 */

namespace App\Models;

use App\Core\Database;

class User {
    private $db;
    private $table = 'users';

    /**
     * إنشاء نموذج المستخدم
     */
    public function __construct() {
        $this->db = Database::getInstance();
    }

    /**
     * الحصول على جميع المستخدمين
     */
    public function getAll() {
        return $this->db->fetchAll("SELECT * FROM {$this->table} ORDER BY id DESC");
    }

    /**
     * البحث عن مستخدم بواسطة المعرف
     */
    public function findById($id) {
        return $this->db->fetchOne("SELECT * FROM {$this->table} WHERE id = ?", [$id]);
    }

    /**
     * البحث عن مستخدم بواسطة البريد الإلكتروني
     */
    public function findByEmail($email) {
        return $this->db->fetchOne("SELECT * FROM {$this->table} WHERE email = ?", [$email]);
    }

    /**
     * البحث عن مستخدم بواسطة اسم المستخدم
     */
    public function findByUsername($username) {
        return $this->db->fetchOne("SELECT * FROM {$this->table} WHERE username = ?", [$username]);
    }

    /**
     * إنشاء مستخدم جديد
     */
    public function create($data) {
        // التحقق من وجود البريد الإلكتروني أو اسم المستخدم
        $existingUser = $this->findByEmail($data['email']);
        if ($existingUser) {
            return ['success' => false, 'message' => 'البريد الإلكتروني مستخدم بالفعل'];
        }

        if (isset($data['username'])) {
            $existingUsername = $this->findByUsername($data['username']);
            if ($existingUsername) {
                return ['success' => false, 'message' => 'اسم المستخدم مستخدم بالفعل'];
            }
        }

        // تشفير كلمة المرور
        $data['password'] = password_hash($data['password'], PASSWORD_DEFAULT);
        
        // إضافة تاريخ الإنشاء
        $data['created_at'] = date('Y-m-d H:i:s');
        
        // إدراج المستخدم في قاعدة البيانات
        $userId = $this->db->insert($this->table, $data);
        
        if ($userId) {
            return ['success' => true, 'user_id' => $userId];
        } else {
            return ['success' => false, 'message' => 'فشل إنشاء المستخدم'];
        }
    }

    /**
     * تحديث بيانات المستخدم
     */
    public function update($id, $data) {
        // التحقق من وجود المستخدم
        $user = $this->findById($id);
        if (!$user) {
            return ['success' => false, 'message' => 'المستخدم غير موجود'];
        }

        // التحقق من البريد الإلكتروني
        if (isset($data['email'])) {
            $existingUser = $this->findByEmail($data['email']);
            if ($existingUser && $existingUser['id'] != $id) {
                return ['success' => false, 'message' => 'البريد الإلكتروني مستخدم بالفعل'];
            }
        }

        // التحقق من اسم المستخدم
        if (isset($data['username'])) {
            $existingUsername = $this->findByUsername($data['username']);
            if ($existingUsername && $existingUsername['id'] != $id) {
                return ['success' => false, 'message' => 'اسم المستخدم مستخدم بالفعل'];
            }
        }

        // تشفير كلمة المرور إذا تم تغييرها
        if (isset($data['password']) && !empty($data['password'])) {
            $data['password'] = password_hash($data['password'], PASSWORD_DEFAULT);
        } else {
            // إزالة كلمة المرور من البيانات إذا كانت فارغة
            unset($data['password']);
        }
        
        // إضافة تاريخ التحديث
        $data['updated_at'] = date('Y-m-d H:i:s');
        
        // تحديث بيانات المستخدم
        $success = $this->db->update($this->table, $data, 'id = ?', [$id]);
        
        if ($success) {
            return ['success' => true];
        } else {
            return ['success' => false, 'message' => 'فشل تحديث بيانات المستخدم'];
        }
    }

    /**
     * حذف مستخدم
     */
    public function delete($id) {
        // التحقق من وجود المستخدم
        $user = $this->findById($id);
        if (!$user) {
            return ['success' => false, 'message' => 'المستخدم غير موجود'];
        }
        
        // حذف المستخدم
        $success = $this->db->delete($this->table, 'id = ?', [$id]);
        
        if ($success) {
            return ['success' => true];
        } else {
            return ['success' => false, 'message' => 'فشل حذف المستخدم'];
        }
    }

    /**
     * التحقق من صحة كلمة المرور
     */
    public function verifyPassword($userId, $password) {
        $user = $this->findById($userId);
        if (!$user) {
            return false;
        }
        
        return password_verify($password, $user['password']);
    }

    /**
     * تغيير كلمة المرور
     */
    public function changePassword($userId, $newPassword) {
        $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
        
        $data = [
            'password' => $hashedPassword,
            'updated_at' => date('Y-m-d H:i:s')
        ];
        
        return $this->db->update($this->table, $data, 'id = ?', [$userId]);
    }
}
