<?php
// تضمين ملف التحقق من تسجيل الدخول
require_once 'check_auth.php';

// تحديد نوع التقرير (مستخدم أو إدارة)
$report_type = isset($_GET['type']) ? $_GET['type'] : 'user';
$is_admin_report = ($report_type === 'admin');

// التحقق من الصلاحيات للتقارير الإدارية
if ($is_admin_report && (!isset($_SESSION['role']) || ($_SESSION['role'] !== 'admin' && $_SESSION['role'] !== 'manager'))) {
    // إعادة توجيه المستخدم إلى صفحة التقارير العادية
    header('Location: monthly_reports.php?type=user');
    exit;
}

// عنوان الصفحة
$page_title = $is_admin_report ? 'التقارير الشهرية للإدارة' : 'التقارير الشهرية';

// الحصول على الشهر والسنة الحالية
$current_month = isset($_GET['month']) ? intval($_GET['month']) : date('n');
$current_year = isset($_GET['year']) ? intval($_GET['year']) : date('Y');

// أسماء الأشهر بالعربية
$months_ar = [
    1 => 'يناير',
    2 => 'فبراير',
    3 => 'مارس',
    4 => 'أبريل',
    5 => 'مايو',
    6 => 'يونيو',
    7 => 'يوليو',
    8 => 'أغسطس',
    9 => 'سبتمبر',
    10 => 'أكتوبر',
    11 => 'نوفمبر',
    12 => 'ديسمبر'
];

// الحصول على اسم الشهر بالعربية
$month_name = $months_ar[$current_month];

// الحصول على عدد أيام الشهر
$days_in_month = cal_days_in_month(CAL_GREGORIAN, $current_month, $current_year);

// الحصول على الشهر السابق والسنة
$prev_month = $current_month - 1;
$prev_year = $current_year;
if ($prev_month < 1) {
    $prev_month = 12;
    $prev_year--;
}

// الحصول على الشهر التالي والسنة
$next_month = $current_month + 1;
$next_year = $current_year;
if ($next_month > 12) {
    $next_month = 1;
    $next_year++;
}

// تحديد معرف المستخدم الحالي
$current_user_id = isset($_SESSION['user_id']) ? $_SESSION['user_id'] : 0;
$current_user_role = isset($_SESSION['role']) ? $_SESSION['role'] : 'user';

// بيانات إحصائية افتراضية (سيتم استبدالها بالبيانات الفعلية من API)
$stats = [
    'total_orders' => 0,
    'total_growth' => 0,
    'completed_orders' => 0,
    'completed_percentage' => 0,
    'completed_growth' => 0,
    'pending_orders' => 0,
    'pending_percentage' => 0,
    'pending_growth' => 0,
    'in_transit_orders' => 0,
    'in_transit_percentage' => 0,
    'cancelled_orders' => 0,
    'cancelled_percentage' => 0,
    'daily_average' => 0,
    'daily_average_growth' => 0,
    'prev_month_orders' => 0,
    'weekly_data' => [
        ['week' => 'الأسبوع 1', 'orders' => 0],
        ['week' => 'الأسبوع 2', 'orders' => 0],
        ['week' => 'الأسبوع 3', 'orders' => 0],
        ['week' => 'الأسبوع 4', 'orders' => 0]
    ],
    'branches' => [
        ['name' => 'المكلا', 'percentage' => 0],
        ['name' => 'صنعاء', 'percentage' => 0],
        ['name' => 'عدن', 'percentage' => 0],
        ['name' => 'شحن', 'percentage' => 0],
        ['name' => 'الصين', 'percentage' => 0]
    ]
];

// تحديد ما إذا كان المستخدم يمكنه رؤية تقارير الإدارة
$can_view_admin_reports = ($current_user_role === 'admin' || $current_user_role === 'manager');
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">

<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>نظام طلبات الشحن | <?php echo $page_title; ?></title>
  <script src="https://cdn.tailwindcss.com"></script>
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/chart.js@3.9.1/dist/chart.min.css">
  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/animate.css@4.1.1/animate.min.css">
  <style>
    :root {
      --primary: #6366f1;
      --primary-light: #818cf8;
      --secondary: #f59e0b;
      --dark: #1e293b;
      --light: #f8fafc;
      --success: #10b981;
      --warning: #f59e0b;
      --danger: #ef4444;
      --info: #3b82f6;
    }

    @import url('https://fonts.googleapis.com/css2?family=Tajawal:wght@300;400;500;600;700&display=swap');

    body {
      font-family: 'Tajawal', sans-serif;
      background-color: #f1f5f9;
      color: var(--dark);
    }

    /* تصميم المحتوى الرئيسي */
    .main-content {
      transition: margin-right 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    }

    /* تصميم الرسوم البيانية */
    .chart-container {
      position: relative;
      height: 300px;
      width: 100%;
    }

    /* تصميم البطاقات الإحصائية */
    .stat-card {
      border-radius: 16px;
      box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
      transition: all 0.3s ease;
      position: relative;
      overflow: hidden;
    }

    .stat-card::before {
      content: '';
      position: absolute;
      top: 0;
      left: 0;
      width: 100%;
      height: 100%;
      background: linear-gradient(135deg, rgba(255, 255, 255, 0.1) 0%, rgba(255, 255, 255, 0) 100%);
      z-index: 1;
      opacity: 0;
      transition: opacity 0.3s ease;
    }

    .stat-card:hover {
      transform: translateY(-5px);
      box-shadow: 0 8px 15px rgba(0, 0, 0, 0.1);
    }

    .stat-card:hover::before {
      opacity: 1;
    }

    .stat-icon {
      width: 48px;
      height: 48px;
      border-radius: 50%;
      display: flex;
      align-items: center;
      justify-content: center;
      transition: transform 0.3s ease;
    }

    .stat-card:hover .stat-icon {
      transform: scale(1.1);
    }

    .progress-bar {
      height: 6px;
      border-radius: 3px;
      background-color: #e5e7eb;
      margin-top: 8px;
      overflow: hidden;
    }

    .progress-value {
      height: 6px;
      border-radius: 3px;
      transition: width 1s ease-in-out;
    }

    .badge {
      display: inline-flex;
      align-items: center;
      padding: 0.25rem 0.5rem;
      border-radius: 9999px;
      font-size: 0.75rem;
      font-weight: 500;
      box-shadow: 0 1px 2px rgba(0, 0, 0, 0.1);
    }

    .badge-success {
      background-color: rgba(16, 185, 129, 0.1);
      color: #10b981;
    }

    .badge-danger {
      background-color: rgba(239, 68, 68, 0.1);
      color: #ef4444;
    }

    .circle-number {
      width: 24px;
      height: 24px;
      border-radius: 50%;
      display: flex;
      align-items: center;
      justify-content: center;
      background-color: #e0e7ff;
      color: #6366f1;
      font-size: 0.75rem;
      font-weight: 600;
      box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
    }

    /* تأثيرات الانتقال */
    .fade-in {
      animation: fadeIn 0.5s ease-in-out;
    }

    .slide-up {
      animation: slideInUp 0.5s ease-in-out;
    }

    .slide-right {
      animation: slideInRight 0.5s ease-in-out;
    }

    /* تصميم الأزرار */
    .btn-primary {
      background-color: var(--primary);
      color: white;
      transition: all 0.3s ease;
      position: relative;
      overflow: hidden;
    }

    .btn-primary::after {
      content: '';
      position: absolute;
      top: 50%;
      left: 50%;
      width: 5px;
      height: 5px;
      background: rgba(255, 255, 255, 0.5);
      opacity: 0;
      border-radius: 100%;
      transform: scale(1, 1) translate(-50%, -50%);
      transform-origin: 50% 50%;
    }

    .btn-primary:hover {
      background-color: #5558e3;
      box-shadow: 0 4px 10px rgba(99, 102, 241, 0.3);
    }

    .btn-primary:active::after {
      animation: ripple 0.6s ease-out;
    }

    @keyframes ripple {
      0% {
        transform: scale(0, 0);
        opacity: 0.5;
      }
      100% {
        transform: scale(20, 20);
        opacity: 0;
      }
    }

    /* تصميم القائمة الجانبية المحسنة */
    .sidebar {
      box-shadow: 0 0 20px rgba(0, 0, 0, 0.1);
    }

    .nav-item {
      transition: all 0.3s ease;
      border-radius: 8px;
      position: relative;
      overflow: hidden;
    }

    .nav-item::before {
      content: '';
      position: absolute;
      top: 0;
      left: 0;
      width: 3px;
      height: 100%;
      background-color: var(--primary);
      transform: scaleY(0);
      transition: transform 0.3s ease;
    }

    .nav-item:hover::before,
    .nav-item.active::before {
      transform: scaleY(1);
    }

    .nav-item.active {
      background-color: rgba(255, 255, 255, 0.1);
    }

    .nav-item:hover {
      background-color: rgba(255, 255, 255, 0.05);
    }

    /* تصميم الرسوم البيانية المحسنة */
    .chart-container {
      box-shadow: 0 2px 10px rgba(0, 0, 0, 0.05);
      border-radius: 16px;
      padding: 1rem;
      background-color: white;
      transition: all 0.3s ease;
    }

    .chart-container:hover {
      box-shadow: 0 5px 15px rgba(0, 0, 0, 0.1);
      transform: translateY(-3px);
    }
  </style>
</head>

<body class="bg-gray-50 reports-page">
  <!-- زر القائمة للجوال -->
  <button id="mobileMenuButton" class="fixed top-4 right-4 z-50 bg-indigo-600 text-white p-2 rounded-lg shadow-lg md:hidden">
    <i class="fas fa-bars"></i>
  </button>

  <!-- تضمين القائمة الجانبية المشتركة -->
  <?php include_once '../includes/sidebar.php'; ?>

  <!-- المحتوى الرئيسي -->
  <main class="main-content p-6 md:p-8">
    <!-- شريط التنقل العلوي -->
    <div class="bg-white rounded-xl shadow-sm p-4 mb-6 flex flex-wrap items-center justify-between fade-in">
      <div class="flex items-center">
        <div class="w-10 h-10 rounded-full bg-indigo-100 flex items-center justify-center text-indigo-600 ml-3">
          <i class="fas fa-calendar-alt"></i>
        </div>
        <div>
          <h1 class="text-xl font-bold text-gray-800"><?php echo $page_title; ?></h1>
          <p class="text-sm text-gray-500">
            تقرير شهري لشهر <?php echo $month_name . ' ' . $current_year; ?>
          </p>
        </div>
      </div>
      <div class="flex mt-4 md:mt-0 space-x-2 space-x-reverse">
        <button onclick="window.print()" class="flex items-center bg-gray-100 hover:bg-gray-200 text-gray-700 px-3 py-2 rounded-lg transition-colors duration-300">
          <i class="fas fa-print ml-2"></i>
          <span>طباعة</span>
        </button>
        <button onclick="exportToPDF()" class="flex items-center bg-gray-100 hover:bg-gray-200 text-gray-700 px-3 py-2 rounded-lg transition-colors duration-300">
          <i class="fas fa-file-pdf ml-2"></i>
          <span>PDF</span>
        </button>
        <button class="flex items-center bg-gray-100 hover:bg-gray-200 text-gray-700 px-3 py-2 rounded-lg transition-colors duration-300">
          <i class="fas fa-file-excel ml-2"></i>
          <span>Excel</span>
        </button>
      </div>
    </div>

    <!-- اختيار الشهر والسنة -->
    <div class="bg-white rounded-xl shadow-sm p-5 mb-6 slide-up">
      <h2 class="text-lg font-bold mb-4 flex items-center">
        <i class="fas fa-filter text-indigo-600 ml-2"></i>
        <span>تصفية التقرير</span>
      </h2>
      <div class="flex flex-wrap items-center justify-between">
        <div class="flex flex-wrap items-center gap-4">
          <div>
            <label for="monthSelector" class="block text-gray-500 mb-1 text-sm">الشهر</label>
            <select id="monthSelector" class="bg-gray-50 border border-gray-300 text-gray-700 py-2 px-4 rounded-lg focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 transition-all duration-300">
              <?php for ($i = 1; $i <= 12; $i++): ?>
                <option value="<?php echo $i; ?>" <?php echo $i == $current_month ? 'selected' : ''; ?>>
                  <?php echo $months_ar[$i]; ?>
                </option>
              <?php endfor; ?>
            </select>
          </div>
          <div>
            <label for="yearSelector" class="block text-gray-500 mb-1 text-sm">السنة</label>
            <select id="yearSelector" class="bg-gray-50 border border-gray-300 text-gray-700 py-2 px-4 rounded-lg focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 transition-all duration-300">
              <?php for ($i = date('Y') - 2; $i <= date('Y') + 2; $i++): ?>
                <option value="<?php echo $i; ?>" <?php echo $i == $current_year ? 'selected' : ''; ?>>
                  <?php echo $i; ?>
                </option>
              <?php endfor; ?>
            </select>
          </div>
          <div>
            <label for="branchSelector" class="block text-gray-500 mb-1 text-sm">الفرع</label>
            <select id="branchSelector" class="bg-gray-50 border border-gray-300 text-gray-700 py-2 px-4 rounded-lg focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 transition-all duration-300">
              <option value="all" selected>جميع الفروع</option>
              <?php foreach ($stats['branches'] as $branch): ?>
              <option value="<?php echo strtolower($branch['name']); ?>"><?php echo $branch['name']; ?></option>
              <?php endforeach; ?>
            </select>
          </div>
        </div>
        <button id="showReportBtn" class="btn-primary px-6 py-2 rounded-lg font-medium mt-4 md:mt-0">
          <i class="fas fa-search ml-2"></i>
          عرض التقرير
        </button>
      </div>
    </div>

    <!-- حاوية الأخطاء -->
    <div id="errorContainer" class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded relative mb-6 hidden" role="alert">
      <strong class="font-bold ml-1">خطأ!</strong>
      <span class="block sm:inline">حدث خطأ أثناء جلب البيانات.</span>
    </div>

    <!-- بطاقات الإحصائيات -->
    <div class="grid grid-cols-1 md:grid-cols-4 gap-6 mb-8">
      <!-- إجمالي الطلبات -->
      <div class="stat-card bg-white p-6 relative overflow-hidden slide-up" style="animation-delay: 0.1s">
        <div class="flex justify-between items-start mb-3">
          <div class="stat-icon bg-gradient-to-br from-blue-500 to-blue-600 text-white">
            <i class="fas fa-box text-xl"></i>
          </div>
          <span class="badge badge-success">
            <i class="fas fa-arrow-up ml-1"></i>
            <?php echo $stats['total_growth']; ?>%
          </span>
        </div>
        <h3 class="text-gray-500 text-sm font-medium mb-2">إجمالي الطلبات</h3>
        <div class="flex items-baseline content-element">
          <span id="totalOrders" class="text-4xl font-bold text-gray-800">0</span>
        </div>
        <!-- مؤشر التحميل -->
        <div class="loading-indicator flex items-center justify-center py-2" style="display: none;">
          <div class="animate-spin rounded-full h-5 w-5 border-b-2 border-indigo-500 mr-2"></div>
          <span class="text-sm text-gray-500">جاري التحميل...</span>
        </div>
        <div class="flex items-center mt-2">
          <span class="text-xs text-gray-500">
            <i class="fas fa-arrow-up text-green-500 ml-1"></i>
            <?php echo $stats['total_growth']; ?>% مقارنة بالشهر السابق
          </span>
        </div>
        <div class="progress-bar mt-3">
          <div class="progress-value bg-gradient-to-r from-blue-400 to-blue-600" style="width: 100%"></div>
        </div>
        <!-- زخرفة الخلفية -->
        <div class="absolute -bottom-6 -left-6 w-24 h-24 rounded-full bg-blue-50 opacity-20"></div>
        <div class="absolute -top-6 -right-6 w-16 h-16 rounded-full bg-blue-50 opacity-20"></div>
      </div>

      <!-- الطلبات المكتملة -->
      <div class="stat-card bg-white p-6 relative overflow-hidden slide-up" style="animation-delay: 0.2s">
        <div class="flex justify-between items-start mb-3">
          <div class="stat-icon bg-gradient-to-br from-green-500 to-green-600 text-white">
            <i class="fas fa-check text-xl"></i>
          </div>
          <span class="badge badge-success">
            <i class="fas fa-arrow-up ml-1"></i>
            <?php echo $stats['completed_growth']; ?>%
          </span>
        </div>
        <h3 class="text-gray-500 text-sm font-medium mb-2">الطلبات المكتملة</h3>
        <div class="flex items-baseline content-element">
          <span id="completedOrders" class="text-4xl font-bold text-gray-800">0</span>
          <span id="completedPercentage" class="text-sm text-gray-500 mr-2">(0%)</span>
        </div>
        <!-- مؤشر التحميل -->
        <div class="loading-indicator flex items-center justify-center py-2" style="display: none;">
          <div class="animate-spin rounded-full h-5 w-5 border-b-2 border-indigo-500 mr-2"></div>
          <span class="text-sm text-gray-500">جاري التحميل...</span>
        </div>
        <div class="flex items-center mt-2">
          <span class="text-xs text-gray-500">
            <i class="fas fa-arrow-up text-green-500 ml-1"></i>
            <?php echo $stats['completed_growth']; ?>% مقارنة بالشهر السابق
          </span>
        </div>
        <div class="progress-bar mt-3">
          <div id="completedProgressBar" class="progress-value bg-gradient-to-r from-green-400 to-green-600" style="width: 0%"></div>
        </div>
        <!-- زخرفة الخلفية -->
        <div class="absolute -bottom-6 -left-6 w-24 h-24 rounded-full bg-green-50 opacity-20"></div>
        <div class="absolute -top-6 -right-6 w-16 h-16 rounded-full bg-green-50 opacity-20"></div>
      </div>

      <!-- الطلبات المعلقة -->
      <div class="stat-card bg-white p-6 relative overflow-hidden slide-up" style="animation-delay: 0.3s">
        <div class="flex justify-between items-start mb-3">
          <div class="stat-icon bg-gradient-to-br from-yellow-500 to-yellow-600 text-white">
            <i class="fas fa-clock text-xl"></i>
          </div>
          <span class="badge badge-danger">
            <i class="fas fa-arrow-down ml-1"></i>
            <?php echo abs($stats['pending_growth']); ?>%
          </span>
        </div>
        <h3 class="text-gray-500 text-sm font-medium mb-2">الطلبات المعلقة</h3>
        <div class="flex items-baseline content-element">
          <span id="pendingOrders" class="text-4xl font-bold text-gray-800">0</span>
          <span id="pendingPercentage" class="text-sm text-gray-500 mr-2">(0%)</span>
        </div>
        <!-- مؤشر التحميل -->
        <div class="loading-indicator flex items-center justify-center py-2" style="display: none;">
          <div class="animate-spin rounded-full h-5 w-5 border-b-2 border-indigo-500 mr-2"></div>
          <span class="text-sm text-gray-500">جاري التحميل...</span>
        </div>
        <div class="flex items-center mt-2">
          <span class="text-xs text-gray-500">
            <i class="fas fa-arrow-down text-red-500 ml-1"></i>
            <?php echo abs($stats['pending_growth']); ?>% مقارنة بالشهر السابق
          </span>
        </div>
        <div class="progress-bar mt-3">
          <div id="pendingProgressBar" class="progress-value bg-gradient-to-r from-yellow-400 to-yellow-600" style="width: 0%"></div>
        </div>
        <!-- زخرفة الخلفية -->
        <div class="absolute -bottom-6 -left-6 w-24 h-24 rounded-full bg-yellow-50 opacity-20"></div>
        <div class="absolute -top-6 -right-6 w-16 h-16 rounded-full bg-yellow-50 opacity-20"></div>
      </div>

      <!-- متوسط الطلبات اليومية -->
      <div class="stat-card bg-white p-6 relative overflow-hidden slide-up" style="animation-delay: 0.4s">
        <div class="flex justify-between items-start mb-3">
          <div class="stat-icon bg-gradient-to-br from-purple-500 to-purple-600 text-white">
            <i class="fas fa-chart-line text-xl"></i>
          </div>
          <span class="badge badge-success">
            <i class="fas fa-arrow-up ml-1"></i>
            <?php echo $stats['daily_average_growth']; ?>%
          </span>
        </div>
        <h3 class="text-gray-500 text-sm font-medium mb-2">متوسط الطلبات اليومية</h3>
        <div class="flex items-baseline content-element">
          <span id="dailyAverage" class="text-4xl font-bold text-gray-800">0</span>
          <span class="text-sm text-gray-500 mr-2">طلب/يوم</span>
        </div>
        <!-- مؤشر التحميل -->
        <div class="loading-indicator flex items-center justify-center py-2" style="display: none;">
          <div class="animate-spin rounded-full h-5 w-5 border-b-2 border-indigo-500 mr-2"></div>
          <span class="text-sm text-gray-500">جاري التحميل...</span>
        </div>
        <div class="flex items-center mt-2">
          <span class="text-xs text-gray-500">
            <i class="fas fa-arrow-up text-green-500 ml-1"></i>
            <?php echo $stats['daily_average_growth']; ?>% مقارنة بالشهر السابق
          </span>
        </div>
        <div class="progress-bar mt-3">
          <div class="progress-value bg-gradient-to-r from-purple-400 to-purple-600" style="width: 80%"></div>
        </div>
        <!-- زخرفة الخلفية -->
        <div class="absolute -bottom-6 -left-6 w-24 h-24 rounded-full bg-purple-50 opacity-20"></div>
        <div class="absolute -top-6 -right-6 w-16 h-16 rounded-full bg-purple-50 opacity-20"></div>
      </div>
    </div>

    <!-- تفاصيل الطلبات في الشهر -->
    <div class="bg-white rounded-xl shadow-sm p-6 mb-8 slide-up" style="animation-delay: 0.5s">
      <div class="flex justify-between items-center mb-6">
        <h2 class="text-xl font-bold flex items-center">
          <i class="fas fa-chart-pie text-indigo-600 ml-2"></i>
          <span>تفاصيل الطلبات في الشهر</span>
        </h2>
        <div class="flex space-x-2 space-x-reverse">
          <button class="bg-gray-100 hover:bg-gray-200 text-gray-700 px-3 py-2 rounded-lg flex items-center transition-colors duration-300">
            <i class="fas fa-sync-alt ml-1"></i>
            <span>تحديث</span>
          </button>
          <button class="bg-gray-100 hover:bg-gray-200 text-gray-700 px-3 py-2 rounded-lg flex items-center transition-colors duration-300">
            <i class="fas fa-filter ml-1"></i>
            <span>تصفية</span>
          </button>
        </div>
      </div>

      <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
        <!-- توزيع الطلبات حسب الحالة -->
        <div class="bg-gradient-to-br from-white to-gray-50 p-5 rounded-xl shadow-sm hover:shadow-md transition-all duration-300">
          <h3 class="text-lg font-medium mb-4 flex items-center">
            <i class="fas fa-chart-pie text-green-500 ml-2"></i>
            <span>توزيع الطلبات حسب الحالة</span>
          </h3>
          <div class="flex items-center mb-4">
            <div class="flex items-center">
              <span class="w-4 h-4 rounded-full bg-gradient-to-r from-green-400 to-green-600 ml-2 shadow-sm"></span>
              <span class="text-sm">مكتملة</span>
            </div>
            <span class="mr-auto text-sm font-medium">(<?php echo $stats['completed_percentage']; ?>%) <?php echo $stats['completed_orders']; ?></span>
          </div>
          <div class="flex items-center mb-4">
            <div class="flex items-center">
              <span class="w-4 h-4 rounded-full bg-gradient-to-r from-yellow-400 to-yellow-600 ml-2 shadow-sm"></span>
              <span class="text-sm">معلقة</span>
            </div>
            <span class="mr-auto text-sm font-medium">(<?php echo $stats['pending_percentage']; ?>%) <?php echo $stats['pending_orders']; ?></span>
          </div>
          <div class="mt-4 bg-gray-200 rounded-full h-3 overflow-hidden shadow-inner">
            <div class="bg-gradient-to-r from-green-400 to-green-600 h-3 rounded-full transition-all duration-1000" style="width: <?php echo $stats['completed_percentage']; ?>%"></div>
          </div>
          <div class="mt-4 pt-4 border-t border-gray-100">
            <div class="flex justify-between items-center">
              <span class="text-xs text-gray-500">إجمالي الطلبات</span>
              <span class="text-sm font-medium"><?php echo $stats['total_orders']; ?></span>
            </div>
          </div>
        </div>

        <!-- توزيع الطلبات حسب الأسابيع -->
        <div class="bg-gradient-to-br from-white to-gray-50 p-5 rounded-xl shadow-sm hover:shadow-md transition-all duration-300">
          <h3 class="text-lg font-medium mb-4 flex items-center">
            <i class="fas fa-calendar-week text-blue-500 ml-2"></i>
            <span>توزيع الطلبات حسب الأسابيع</span>
          </h3>
          <div class="weekly-data-container">
            <?php foreach ($stats['weekly_data'] as $index => $week): ?>
            <div class="flex items-center mb-3 p-2 rounded-lg hover:bg-white transition-colors duration-300">
              <div class="circle-number ml-3 shadow-sm"><?php echo $index + 1; ?></div>
              <span class="text-sm font-medium"><?php echo $week['week']; ?></span>
              <div class="mr-auto flex items-center">
                <div class="w-16 h-2 bg-gray-200 rounded-full overflow-hidden ml-2">
                  <div class="h-full bg-gradient-to-r from-blue-400 to-blue-600 rounded-full" style="width: <?php echo (max(array_column($stats['weekly_data'], 'orders')) > 0) ? ($week['orders'] / max(array_column($stats['weekly_data'], 'orders'))) * 100 : 0; ?>%"></div>
                </div>
                <span class="text-sm font-medium"><?php echo $week['orders']; ?> طلب</span>
              </div>
            </div>
            <?php endforeach; ?>
          </div>
          <div class="mt-4 pt-4 border-t border-gray-100">
            <p class="text-xs text-gray-500 flex items-center">
              <i class="fas fa-info-circle ml-1 text-blue-500"></i>
              الأسبوع الثاني هو الأكثر نشاطاً بنسبة 28% من إجمالي الطلبات
            </p>
          </div>
        </div>

        <!-- معدل نمو الطلبات -->
        <div class="bg-gradient-to-br from-white to-gray-50 p-5 rounded-xl shadow-sm hover:shadow-md transition-all duration-300">
          <h3 class="text-lg font-medium mb-4 flex items-center">
            <i class="fas fa-chart-line text-indigo-500 ml-2"></i>
            <span>معدل نمو الطلبات</span>
          </h3>
          <div class="flex items-center justify-center mb-6">
            <div class="text-center">
              <div class="relative inline-block">
                <span class="text-4xl font-bold text-indigo-600 growth-value">
                  <?php echo $stats['total_growth'] >= 0 ? '+' : '-'; ?><?php echo abs($stats['total_growth']); ?>%
                </span>
                <div class="absolute -top-1 -right-1 w-3 h-3 bg-indigo-100 rounded-full"></div>
                <div class="absolute -bottom-1 -left-1 w-2 h-2 bg-indigo-100 rounded-full"></div>
              </div>
              <p class="text-sm text-gray-500 mt-2">مقارنة بالشهر السابق</p>
            </div>
          </div>
          <div class="flex items-center justify-between mt-4 bg-gray-50 p-3 rounded-lg">
            <div class="text-center">
              <span class="text-sm font-medium text-gray-500">الشهر السابق</span>
              <p class="text-lg font-bold"><span id="prevMonthOrders"><?php echo $stats['prev_month_orders']; ?></span> <span class="text-xs">طلب</span></p>
            </div>
            <div class="w-10 h-10 rounded-full bg-indigo-50 flex items-center justify-center text-indigo-600">
              <i class="fas fa-arrow-left"></i>
            </div>
            <div class="text-center">
              <span class="text-sm font-medium text-gray-500">الشهر الحالي</span>
              <p class="text-lg font-bold"><span id="currentMonthOrders"><?php echo $stats['total_orders']; ?></span> <span class="text-xs">طلب</span></p>
            </div>
          </div>
          <div class="mt-4 pt-4 border-t border-gray-100">
            <div class="flex justify-between items-center">
              <span class="text-xs text-gray-500">معدل النمو السنوي</span>
              <span class="text-sm font-medium text-green-500">+22%</span>
            </div>
          </div>
        </div>
      </div>
    </div>

    <!-- الرسوم البيانية -->
    <div class="grid grid-cols-1 md:grid-cols-2 gap-8 mb-8">
      <!-- الطلبات الأسبوعية -->
      <div class="bg-white rounded-xl shadow-sm p-6 slide-up" style="animation-delay: 0.6s">
        <div class="flex justify-between items-center mb-4">
          <h3 class="text-lg font-medium flex items-center">
            <i class="fas fa-chart-bar text-blue-500 ml-2"></i>
            <span>الطلبات الأسبوعية</span>
          </h3>
          <div class="flex">
            <button class="text-gray-400 hover:text-gray-600 ml-2 transition-colors duration-300">
              <i class="fas fa-expand-alt"></i>
            </button>
            <button class="text-gray-400 hover:text-gray-600 transition-colors duration-300">
              <i class="fas fa-ellipsis-v"></i>
            </button>
          </div>
        </div>
        <div class="chart-container bg-white p-2 rounded-lg">
          <canvas id="weeklyOrdersChart"></canvas>
        </div>
        <div class="mt-4 pt-4 border-t border-gray-100 flex justify-between items-center">
          <div class="flex items-center">
            <span class="w-3 h-3 rounded-full bg-blue-500 ml-1"></span>
            <span class="text-xs text-gray-500">عدد الطلبات</span>
          </div>
          <span class="text-xs text-gray-500"><?php echo $month_name; ?> <?php echo $current_year; ?></span>
        </div>
      </div>

      <!-- توزيع الطلبات حسب الفروع -->
      <div class="bg-white rounded-xl shadow-sm p-6 slide-up" style="animation-delay: 0.7s">
        <div class="flex justify-between items-center mb-4">
          <h3 class="text-lg font-medium flex items-center">
            <i class="fas fa-chart-pie text-indigo-500 ml-2"></i>
            <span>توزيع الطلبات حسب الفروع</span>
          </h3>
          <div class="flex">
            <button class="text-gray-400 hover:text-gray-600 ml-2 transition-colors duration-300">
              <i class="fas fa-expand-alt"></i>
            </button>
            <button class="text-gray-400 hover:text-gray-600 transition-colors duration-300">
              <i class="fas fa-ellipsis-v"></i>
            </button>
          </div>
        </div>
        <div class="chart-container bg-white p-2 rounded-lg">
          <canvas id="branchesChart"></canvas>
        </div>
        <div class="mt-4 pt-4 border-t border-gray-100 grid grid-cols-5 gap-2">
          <?php foreach ($stats['branches'] as $index => $branch): ?>
          <div class="text-center">
            <div class="w-3 h-3 rounded-full bg-<?php echo ['indigo', 'green', 'yellow', 'red', 'purple'][$index]; ?>-500 mx-auto mb-1"></div>
            <span class="text-xs text-gray-500"><?php echo $branch['name']; ?></span>
          </div>
          <?php endforeach; ?>
        </div>
      </div>
    </div>

    <!-- جدول أحدث الطلبات -->
    <div class="bg-white rounded-xl shadow-sm p-6 mb-8 slide-up" style="animation-delay: 0.8s">
      <div class="flex justify-between items-center mb-6">
        <h2 class="text-xl font-bold flex items-center">
          <i class="fas fa-list text-indigo-600 ml-2"></i>
          <span>أحدث الطلبات</span>
        </h2>
        <a href="../orders/orders.php" class="text-indigo-600 hover:text-indigo-800 flex items-center transition-colors duration-300">
          <span>عرض الكل</span>
          <i class="fas fa-chevron-left mr-1"></i>
        </a>
      </div>

      <div class="overflow-x-auto">
        <table class="min-w-full divide-y divide-gray-200">
          <thead>
            <tr class="bg-gray-50">
              <th class="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">رقم الطلب</th>
              <th class="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">العميل</th>
              <th class="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">التاريخ</th>
              <th class="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">المبلغ</th>
              <th class="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">الموظف المسؤول</th>
              <th class="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">الحالة</th>
              <th class="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">الإجراءات</th>
            </tr>
          </thead>
          <tbody class="bg-white divide-y divide-gray-200">
            <tr id="loading-row">
              <td colspan="7" class="px-4 py-8 text-center">
                <div class="flex flex-col items-center justify-center text-gray-500">
                  <div class="animate-spin rounded-full h-8 w-8 border-b-2 border-indigo-500 mb-3"></div>
                  <p>جاري تحميل الطلبات...</p>
                </div>
              </td>
            </tr>
            <!-- سيتم إضافة الطلبات هنا ديناميكيًا -->
          </tbody>
        </table>
      </div>
    </div>
  </main>

  <!-- سكريبت الرسوم البيانية -->
  <script src="https://cdn.jsdelivr.net/npm/chart.js@3.9.1/dist/chart.min.js"></script>
  <script src="https://cdn.jsdelivr.net/npm/chartjs-plugin-datalabels@2.0.0"></script>
  <script>
    // متغيرات عامة
    let reportData = null;
    const reportType = '<?php echo $report_type; ?>';
    const currentMonth = <?php echo $current_month; ?>;
    const currentYear = <?php echo $current_year; ?>;
    const currentUserId = <?php echo $current_user_id; ?>;
    const currentUserRole = '<?php echo $current_user_role; ?>';
    const canViewAdminReports = <?php echo $can_view_admin_reports ? 'true' : 'false'; ?>;

    document.addEventListener('DOMContentLoaded', function() {
      // جلب بيانات التقرير
      fetchReportData();

      // تهيئة زر عرض التقرير
      initReportButton();

      // تهيئة تأثيرات التمرير
      initScrollAnimations();

      // تهيئة أزرار التصدير
      initExportButtons();
    });

    // دالة جلب بيانات التقرير
    function fetchReportData() {
      // إظهار مؤشر التحميل
      showLoading(true);

      // بناء عنوان URL للطلب
      const url = `../api/reports.php?report_type=monthly&data_type=${reportType}&year=${currentYear}&period=${currentMonth}&branch=all`;

      // إرسال الطلب
      fetch(url)
        .then(response => {
          if (!response.ok) {
            throw new Error('فشل في جلب بيانات التقرير');
          }
          return response.json();
        })
        .then(data => {
          if (data.status === 'success') {
            // تخزين البيانات
            reportData = data.data;

            // تحديث واجهة المستخدم
            updateUI(reportData);

            // تهيئة الرسوم البيانية
            initCharts(reportData);
          } else {
            throw new Error(data.message || 'فشل في جلب بيانات التقرير');
          }
        })
        .catch(error => {
          console.error('خطأ:', error);
          showError(error.message);
        })
        .finally(() => {
          // إخفاء مؤشر التحميل
          showLoading(false);
        });
    }

    // دالة تحديث واجهة المستخدم
    function updateUI(data) {
      console.log('تحديث واجهة المستخدم بالبيانات:', data);

      // تحديث بطاقات الإحصائيات
      document.getElementById('totalOrders').textContent = data.stats.total_orders;
      document.getElementById('completedOrders').textContent = data.stats.completed_orders;
      document.getElementById('pendingOrders').textContent = data.stats.pending_orders;
      document.getElementById('dailyAverage').textContent = data.stats.daily_average;

      // تحديث النسب المئوية
      document.getElementById('completedPercentage').textContent = `(${data.stats.completed_percentage}%)`;
      document.getElementById('pendingPercentage').textContent = `(${data.stats.pending_percentage}%)`;

      // تحديث شرائط التقدم
      document.getElementById('completedProgressBar').style.width = `${data.stats.completed_percentage}%`;
      document.getElementById('pendingProgressBar').style.width = `${data.stats.pending_percentage}%`;

      // تحديث بيانات النمو
      const totalGrowthElements = document.querySelectorAll('.total-growth');
      totalGrowthElements.forEach(element => {
        const growthValue = data.stats.total_growth || 0;
        const isPositive = growthValue >= 0;

        element.innerHTML = `
          <i class="fas fa-arrow-${isPositive ? 'up' : 'down'} ml-1"></i>
          ${Math.abs(growthValue)}%
        `;

        element.className = `badge badge-${isPositive ? 'success' : 'danger'} total-growth`;
      });

      // تحديث بيانات الشهر السابق
      if (document.getElementById('prevMonthOrders')) {
        document.getElementById('prevMonthOrders').textContent = data.stats.prev_month_orders || 0;
      }

      // تحديث بيانات الأسابيع
      if (data.weekly_data && data.weekly_data.length > 0) {
        const weeklyDataContainer = document.querySelector('.weekly-data-container');
        if (weeklyDataContainer) {
          let weeklyDataHTML = '';

          data.weekly_data.forEach((week, index) => {
            const percentage = data.weekly_data.reduce((max, w) => Math.max(max, w.orders), 0) > 0
              ? (week.orders / data.weekly_data.reduce((max, w) => Math.max(max, w.orders), 0)) * 100
              : 0;

            weeklyDataHTML += `
              <div class="flex items-center mb-3 p-2 rounded-lg hover:bg-white transition-colors duration-300">
                <div class="circle-number ml-3 shadow-sm">${index + 1}</div>
                <span class="text-sm font-medium">${week.week}</span>
                <div class="mr-auto flex items-center">
                  <div class="w-16 h-2 bg-gray-200 rounded-full overflow-hidden ml-2">
                    <div class="h-full bg-gradient-to-r from-blue-400 to-blue-600 rounded-full" style="width: ${percentage}%"></div>
                  </div>
                  <span class="text-sm font-medium">${week.orders} طلب</span>
                </div>
              </div>
            `;
          });

          weeklyDataContainer.innerHTML = weeklyDataHTML;
        }
      }

      // تحديث بيانات الفروع
      if (data.branches && data.branches.length > 0) {
        // تحديث قائمة الفروع في حقل التصفية
        const branchSelector = document.getElementById('branchSelector');
        if (branchSelector) {
          // حفظ القيمة المحددة حاليًا
          const selectedValue = branchSelector.value;

          // إنشاء الخيار الافتراضي
          let branchOptionsHTML = '<option value="all" selected>جميع الفروع</option>';

          // إضافة خيارات الفروع
          data.branches.forEach(branch => {
            if (branch.name && branch.name !== 'غير محدد') {
              branchOptionsHTML += `<option value="${branch.name}" ${selectedValue === branch.name ? 'selected' : ''}>${branch.name}</option>`;
            }
          });

          branchSelector.innerHTML = branchOptionsHTML;
        }
      }

      // تحديث جدول الطلبات
      updateOrdersTable(data.orders || []);
    }

    // دالة تحديث جدول الطلبات
    function updateOrdersTable(orders) {
      const tableBody = document.querySelector('table tbody');
      if (!tableBody) return;

      // إذا لم تكن هناك طلبات، عرض رسالة
      if (!orders || orders.length === 0) {
        tableBody.innerHTML = `
          <tr>
            <td colspan="7" class="px-4 py-8 text-center">
              <div class="flex flex-col items-center justify-center text-gray-500">
                <i class="fas fa-inbox text-4xl mb-3 text-gray-300"></i>
                <p>لا توجد طلبات في هذا الشهر</p>
              </div>
            </td>
          </tr>
        `;
        return;
      }

      // إنشاء صفوف الجدول
      let tableHTML = '';

      // عرض أحدث 5 طلبات فقط
      const displayOrders = orders.slice(0, 5);

      displayOrders.forEach(order => {
        // تحديد لون وأيقونة حالة الطلب
        let statusClass, statusIcon, statusText;

        switch(order.status) {
          case 'تم التسليم':
            statusClass = 'bg-green-100 text-green-800';
            statusIcon = 'fa-check-circle';
            statusText = 'مكتمل';
            break;
          case 'قيد المعالجة':
            statusClass = 'bg-yellow-100 text-yellow-800';
            statusIcon = 'fa-clock';
            statusText = 'معلق';
            break;
          case 'في الطريق':
            statusClass = 'bg-blue-100 text-blue-800';
            statusIcon = 'fa-truck';
            statusText = 'في الطريق';
            break;
          case 'ملغى':
            statusClass = 'bg-red-100 text-red-800';
            statusIcon = 'fa-times-circle';
            statusText = 'ملغى';
            break;
          default:
            statusClass = 'bg-gray-100 text-gray-800';
            statusIcon = 'fa-question-circle';
            statusText = order.status || 'غير معروف';
        }

        // تنسيق التاريخ
        const orderDate = new Date(order.order_upload_date);
        const formattedDate = orderDate.toLocaleDateString('ar-SA');

        // تنسيق المبلغ
        const amount = parseFloat(order.amount_agreed_with_customer) || 0;
        const formattedAmount = amount.toLocaleString('ar-SA') + ' ر.س';

        tableHTML += `
          <tr class="hover:bg-gray-50 transition-colors duration-300">
            <td class="px-4 py-3 whitespace-nowrap text-sm font-medium text-gray-900">${order.order_number}</td>
            <td class="px-4 py-3 whitespace-nowrap text-sm text-gray-500">${order.client_name}</td>
            <td class="px-4 py-3 whitespace-nowrap text-sm text-gray-500">${formattedDate}</td>
            <td class="px-4 py-3 whitespace-nowrap text-sm text-gray-500">${formattedAmount}</td>
            <td class="px-4 py-3 whitespace-nowrap text-sm text-gray-500">${order.employee_name || 'غير محدد'}</td>
            <td class="px-4 py-3 whitespace-nowrap">
              <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full ${statusClass}">
                ${statusText}
              </span>
            </td>
            <td class="px-4 py-3 whitespace-nowrap text-sm text-gray-500">
              <div class="flex space-x-2 space-x-reverse">
                <a href="../orders/order-details.php?id=${order.id}" class="text-blue-600 hover:text-blue-800">
                  <i class="fas fa-eye"></i>
                </a>
                <a href="../orders/edit-order.php?id=${order.id}" class="text-indigo-600 hover:text-indigo-800">
                  <i class="fas fa-edit"></i>
                </a>
              </div>
            </td>
          </tr>
        `;
      });

      tableBody.innerHTML = tableHTML;
    }

    // دالة إظهار/إخفاء مؤشر التحميل
    function showLoading(show) {
      const loadingIndicators = document.querySelectorAll('.loading-indicator');
      loadingIndicators.forEach(indicator => {
        indicator.style.display = show ? 'flex' : 'none';
      });

      const contentElements = document.querySelectorAll('.content-element');
      contentElements.forEach(element => {
        element.style.display = show ? 'none' : 'block';
      });
    }

    // دالة إظهار رسالة خطأ
    function showError(message) {
      const errorContainer = document.getElementById('errorContainer');
      if (errorContainer) {
        errorContainer.textContent = message;
        errorContainer.style.display = 'block';
      } else {
        alert(message);
      }
    }

    // تهيئة الرسوم البيانية
    function initCharts(data) {
      Chart.register(ChartDataLabels);

      // إذا كانت الرسوم البيانية موجودة بالفعل، قم بتدميرها
      if (window.weeklyOrdersChart) {
        window.weeklyOrdersChart.destroy();
      }

      if (window.branchesChart) {
        window.branchesChart.destroy();
      }

      // بيانات الطلبات الأسبوعية
      const weeklyOrdersCtx = document.getElementById('weeklyOrdersChart').getContext('2d');

      // استخدام البيانات من API إذا كانت متوفرة
      const weeklyData = data && data.weekly_data ? data.weekly_data : [
        <?php foreach ($stats['weekly_data'] as $week): ?>
        { week: '<?php echo $week['week']; ?>', orders: <?php echo $week['orders']; ?> },
        <?php endforeach; ?>
      ];

      window.weeklyOrdersChart = new Chart(weeklyOrdersCtx, {
        type: 'bar',
        data: {
          labels: weeklyData.map(week => week.week),
          datasets: [{
            label: 'عدد الطلبات',
            data: weeklyData.map(week => week.orders),
            backgroundColor: weeklyData.map((week, index) => {
              // تحديد أعلى أسبوع
              const maxOrders = Math.max(...weeklyData.map(w => w.orders));
              return `rgba(99, 102, 241, ${week.orders === maxOrders ? '0.9' : '0.7'})`;
            }),
            borderColor: weeklyData.map(() => 'rgba(99, 102, 241, 1)'),
            borderWidth: 1,
            borderRadius: 6,
            hoverBackgroundColor: 'rgba(99, 102, 241, 0.9)',
            hoverBorderColor: 'rgba(99, 102, 241, 1)',
            hoverBorderWidth: 2
          }]
        },
        options: {
          responsive: true,
          maintainAspectRatio: false,
          animation: {
            duration: 1000,
            easing: 'easeOutQuart'
          },
          plugins: {
            legend: {
              display: false
            },
            tooltip: {
              backgroundColor: 'rgba(255, 255, 255, 0.9)',
              titleColor: '#1e293b',
              bodyColor: '#1e293b',
              borderColor: 'rgba(99, 102, 241, 0.3)',
              borderWidth: 1,
              cornerRadius: 8,
              padding: 12,
              boxPadding: 6,
              usePointStyle: true,
              callbacks: {
                title: function(context) {
                  return context[0].label;
                },
                label: function(context) {
                  return `عدد الطلبات: ${context.raw} طلب`;
                }
              }
            },
            datalabels: {
              color: '#6366f1',
              anchor: 'end',
              align: 'top',
              offset: 0,
              font: {
                weight: 'bold',
                size: 10
              },
              formatter: function(value) {
                return value;
              },
              display: function(context) {
                return context.dataIndex === 1; // عرض القيمة فقط للأسبوع الثاني (أعلى قيمة)
              }
            }
          },
          scales: {
            y: {
              beginAtZero: true,
              grid: {
                display: true,
                color: 'rgba(0, 0, 0, 0.05)'
              },
              ticks: {
                font: {
                  family: 'Tajawal'
                }
              }
            },
            x: {
              grid: {
                display: false
              },
              ticks: {
                font: {
                  family: 'Tajawal'
                }
              }
            }
          }
        }
      });

      // بيانات توزيع الطلبات حسب الفروع
      const branchesCtx = document.getElementById('branchesChart').getContext('2d');

      // استخدام البيانات من API إذا كانت متوفرة
      const branchesData = data && data.branches ? data.branches : [
        <?php foreach ($stats['branches'] as $branch): ?>
        { name: '<?php echo $branch['name']; ?>', percentage: <?php echo $branch['percentage']; ?> },
        <?php endforeach; ?>
      ];

      window.branchesChart = new Chart(branchesCtx, {
        type: 'doughnut',
        data: {
          labels: branchesData.map(branch => branch.name),
          datasets: [{
            data: branchesData.map(branch => branch.percentage),
            backgroundColor: [
              'rgba(99, 102, 241, 0.8)',
              'rgba(16, 185, 129, 0.8)',
              'rgba(245, 158, 11, 0.8)',
              'rgba(239, 68, 68, 0.8)',
              'rgba(139, 92, 246, 0.8)'
            ],
            borderColor: [
              'rgba(99, 102, 241, 1)',
              'rgba(16, 185, 129, 1)',
              'rgba(245, 158, 11, 1)',
              'rgba(239, 68, 68, 1)',
              'rgba(139, 92, 246, 1)'
            ],
            borderWidth: 1,
            hoverOffset: 10
          }]
        },
        options: {
          responsive: true,
          maintainAspectRatio: false,
          animation: {
            animateRotate: true,
            animateScale: true,
            duration: 1000,
            easing: 'easeOutQuart'
          },
          plugins: {
            legend: {
              display: false
            },
            tooltip: {
              backgroundColor: 'rgba(255, 255, 255, 0.9)',
              titleColor: '#1e293b',
              bodyColor: '#1e293b',
              borderColor: 'rgba(99, 102, 241, 0.3)',
              borderWidth: 1,
              cornerRadius: 8,
              padding: 12,
              boxPadding: 6,
              usePointStyle: true,
              callbacks: {
                label: function(context) {
                  return `${context.label}: ${context.raw}%`;
                }
              }
            },
            datalabels: {
              color: '#ffffff',
              font: {
                weight: 'bold',
                size: 11
              },
              formatter: function(value, context) {
                return value + '%';
              },
              display: function(context) {
                return context.dataset.data[context.dataIndex] > 10; // عرض النسبة فقط للقيم الأكبر من 10%
              }
            }
          },
          cutout: '65%'
        }
      });
    }

    // تهيئة زر عرض التقرير
    function initReportButton() {
      const showReportBtn = document.getElementById('showReportBtn');
      const monthSelector = document.getElementById('monthSelector');
      const yearSelector = document.getElementById('yearSelector');
      const branchSelector = document.getElementById('branchSelector');

      if (showReportBtn && monthSelector && yearSelector) {
        showReportBtn.addEventListener('click', function(event) {
          // إضافة تأثير نقرة الزر
          addRippleEffect(this, event);

          const selectedMonth = monthSelector.value;
          const selectedYear = yearSelector.value;
          const selectedBranch = branchSelector ? branchSelector.value : 'all';
          const reportType = '<?php echo $report_type; ?>';

          // إظهار مؤشر التحميل
          showReportBtn.innerHTML = '<i class="fas fa-spinner fa-spin ml-2"></i> جاري التحميل...';
          showReportBtn.disabled = true;

          // إظهار مؤشر التحميل في البطاقات
          showLoading(true);

          // بناء عنوان URL للطلب
          const url = `../api/reports.php?report_type=monthly&data_type=${reportType}&year=${selectedYear}&period=${selectedMonth}&branch=${selectedBranch}`;

          // إرسال الطلب
          fetch(url)
            .then(response => {
              if (!response.ok) {
                throw new Error('فشل في جلب بيانات التقرير');
              }
              return response.json();
            })
            .then(data => {
              if (data.status === 'success') {
                // تخزين البيانات
                reportData = data.data;

                // تحديث واجهة المستخدم
                updateUI(reportData);

                // تهيئة الرسوم البيانية
                initCharts(reportData);

                // تحديث عنوان URL بدون إعادة تحميل الصفحة
                const newUrl = `monthly_reports.php?type=${reportType}&month=${selectedMonth}&year=${selectedYear}&branch=${selectedBranch}`;
                window.history.pushState({ path: newUrl }, '', newUrl);
              } else {
                throw new Error(data.message || 'فشل في جلب بيانات التقرير');
              }
            })
            .catch(error => {
              console.error('خطأ:', error);
              showError(error.message);
            })
            .finally(() => {
              // إخفاء مؤشر التحميل
              showLoading(false);

              // إعادة زر التقرير إلى حالته الأصلية
              showReportBtn.innerHTML = '<i class="fas fa-search ml-2"></i> عرض التقرير';
              showReportBtn.disabled = false;
            });
        });
      }

      // تهيئة حقول التصفية
      initFilterFields();
    }

    // تهيئة حقول التصفية
    function initFilterFields() {
      const monthSelector = document.getElementById('monthSelector');
      const yearSelector = document.getElementById('yearSelector');
      const branchSelector = document.getElementById('branchSelector');

      // إضافة تأثيرات التغيير للحقول
      [monthSelector, yearSelector, branchSelector].forEach(selector => {
        if (selector) {
          selector.addEventListener('change', function() {
            this.classList.add('ring-2');
            setTimeout(() => {
              this.classList.remove('ring-2');
            }, 300);
          });
        }
      });
    }

    // تهيئة تأثيرات التمرير
    function initScrollAnimations() {
      // تحديد العناصر التي تحتاج إلى تأثيرات التمرير
      const animatedElements = document.querySelectorAll('.slide-up, .fade-in, .slide-right');

      // تهيئة مراقب التمرير
      const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
          if (entry.isIntersecting) {
            entry.target.style.opacity = '1';
            entry.target.style.transform = 'translateY(0)';
            observer.unobserve(entry.target);
          }
        });
      }, {
        threshold: 0.1
      });

      // إضافة العناصر إلى المراقب
      animatedElements.forEach(element => {
        element.style.opacity = '0';
        element.style.transform = 'translateY(20px)';
        element.style.transition = 'opacity 0.5s ease, transform 0.5s ease';
        observer.observe(element);
      });
    }

    // تهيئة أزرار التصدير
    function initExportButtons() {
      // تصدير إلى PDF
      const pdfButton = document.querySelector('button[onclick="exportToPDF()"]');
      if (pdfButton) {
        pdfButton.onclick = function(event) {
          event.preventDefault();
          addRippleEffect(this, event);
          exportToPDF();
        };
      }

      // تصدير إلى Excel
      const excelButton = document.querySelector('button i.fa-file-excel').closest('button');
      if (excelButton) {
        excelButton.onclick = function(event) {
          event.preventDefault();
          addRippleEffect(this, event);
          exportToExcel();
        };
      }

      // طباعة التقرير
      const printButton = document.querySelector('button[onclick="window.print()"]');
      if (printButton) {
        printButton.onclick = function(event) {
          event.preventDefault();
          addRippleEffect(this, event);
          setTimeout(() => {
            window.print();
          }, 300);
        };
      }
    }

    // إضافة تأثير نقرة الزر
    function addRippleEffect(button, event) {
      const ripple = document.createElement('span');
      const rect = button.getBoundingClientRect();

      const x = event.clientX - rect.left;
      const y = event.clientY - rect.top;

      ripple.style.position = 'absolute';
      ripple.style.width = '5px';
      ripple.style.height = '5px';
      ripple.style.borderRadius = '50%';
      ripple.style.backgroundColor = 'rgba(255, 255, 255, 0.5)';
      ripple.style.left = x + 'px';
      ripple.style.top = y + 'px';
      ripple.style.transform = 'scale(0)';
      ripple.style.animation = 'ripple 0.6s ease-out';

      button.style.position = 'relative';
      button.style.overflow = 'hidden';
      button.appendChild(ripple);

      setTimeout(() => {
        ripple.remove();
      }, 600);
    }

    // دالة تصدير التقرير إلى PDF
    function exportToPDF() {
      // هنا يمكن إضافة كود لتصدير التقرير إلى PDF
      const button = document.querySelector('button[onclick="exportToPDF()"]');
      if (button) {
        const originalText = button.innerHTML;
        button.innerHTML = '<i class="fas fa-spinner fa-spin ml-2"></i> جاري التصدير...';
        button.disabled = true;

        setTimeout(() => {
          button.innerHTML = '<i class="fas fa-check ml-2"></i> تم التصدير';

          setTimeout(() => {
            button.innerHTML = originalText;
            button.disabled = false;
          }, 1500);
        }, 1500);
      }
    }

    // دالة تصدير التقرير إلى Excel
    function exportToExcel() {
      // هنا يمكن إضافة كود لتصدير التقرير إلى Excel
      const button = document.querySelector('button i.fa-file-excel').closest('button');
      if (button) {
        const originalText = button.innerHTML;
        button.innerHTML = '<i class="fas fa-spinner fa-spin ml-2"></i> جاري التصدير...';
        button.disabled = true;

        setTimeout(() => {
          button.innerHTML = '<i class="fas fa-check ml-2"></i> تم التصدير';

          setTimeout(() => {
            button.innerHTML = originalText;
            button.disabled = false;
          }, 1500);
        }, 1500);
      }
    }

    // تهيئة تأثير الـ ripple للأزرار
    @keyframes ripple {
      0% {
        transform: scale(0);
        opacity: 0.5;
      }
      100% {
        transform: scale(40);
        opacity: 0;
      }
    }
  </script>

  <!-- تم نقل تهيئة القائمة الجانبية إلى ملف sidebar.php -->
</body>

</html>
