<?php
/**
 * ملف إعادة تعيين كلمة مرور مدير النظام
 * يقوم هذا الملف بإعادة تعيين كلمة مرور مدير النظام إلى القيمة الافتراضية
 *
 * @version 1.0
 * @author System Developer
 */

// عرض جميع الأخطاء للمساعدة في التشخيص
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// تضمين ملف التكوين الرئيسي
require_once 'config/config.php';

// تضمين ملف الاتصال بقاعدة البيانات
require_once 'config/db_connection.php';

// تعريف متغيرات CSS للتنسيق
$successColor = "#4CAF50";
$errorColor = "#F44336";
$warningColor = "#FF9800";
$infoColor = "#2196F3";
$primaryColor = "#4c0c70";
$secondaryColor = "#6d28d9";

// تعريف متغيرات لبيانات المدير
$adminUsername = "admin";
$adminEmail = "admin@al-ahmadigroup.com";
$adminPassword = "admin123"; // كلمة المرور الافتراضية

// تعريف متغير لتخزين الرسائل
$messages = [];
$errors = [];
$debugInfo = [];

// دالة لإضافة رسالة
function addMessage($type, $message) {
    global $messages;
    $messages[] = [
        'type' => $type,
        'message' => $message
    ];
}

// دالة لإضافة معلومات تصحيح
function addDebugInfo($info) {
    global $debugInfo;
    $debugInfo[] = $info;
}

// دالة للتحقق من وجود مدير النظام
function checkAdmin($conn) {
    global $adminUsername, $adminEmail;

    try {
        // البحث عن مدير النظام باستخدام اسم المستخدم أو البريد الإلكتروني
        $stmt = $conn->prepare("SELECT * FROM users WHERE username = :username OR email = :email");
        $stmt->bindParam(':username', $adminUsername);
        $stmt->bindParam(':email', $adminEmail);
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        addMessage('error', 'خطأ في التحقق من وجود مدير النظام: ' . $e->getMessage());
        return false;
    }
}

// دالة لإعادة تعيين كلمة مرور مدير النظام
function resetAdminPassword($conn, $adminId, $newPassword) {
    try {
        // تشفير كلمة المرور الجديدة
        $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);

        // تحديث كلمة المرور في قاعدة البيانات
        $stmt = $conn->prepare("UPDATE users SET password = :password WHERE id = :id");
        $stmt->bindParam(':password', $hashedPassword);
        $stmt->bindParam(':id', $adminId);
        $stmt->execute();

        addMessage('success', 'تم إعادة تعيين كلمة مرور مدير النظام بنجاح.');
        return true;
    } catch (PDOException $e) {
        addMessage('error', 'خطأ في إعادة تعيين كلمة مرور مدير النظام: ' . $e->getMessage());
        return false;
    }
}

// دالة لإنشاء مدير النظام إذا لم يكن موجودًا
function createAdmin($conn, $adminData) {
    try {
        // البحث عن المركز الرئيسي
        $stmt = $conn->prepare("SELECT id FROM branches WHERE name = 'المركز الرئيسي'");
        $stmt->execute();
        $branch = $stmt->fetch(PDO::FETCH_ASSOC);

        // إذا لم يكن المركز الرئيسي موجودًا، قم بإنشائه
        if (!$branch) {
            $stmt = $conn->prepare("INSERT INTO branches (name, code, is_active) VALUES ('المركز الرئيسي', 'HQ', 1)");
            $stmt->execute();
            $branchId = $conn->lastInsertId();
        } else {
            $branchId = $branch['id'];
        }

        // تشفير كلمة المرور
        $hashedPassword = password_hash($adminData['password'], PASSWORD_DEFAULT);

        // إنشاء المدير
        $stmt = $conn->prepare("
            INSERT INTO users (
                username,
                password,
                full_name,
                email,
                role,
                branch_id,
                status,
                is_active,
                registration_date
            ) VALUES (
                :username,
                :password,
                'مدير النظام',
                :email,
                'admin',
                :branch_id,
                'active',
                1,
                NOW()
            )
        ");

        $stmt->bindParam(':username', $adminData['username']);
        $stmt->bindParam(':password', $hashedPassword);
        $stmt->bindParam(':email', $adminData['email']);
        $stmt->bindParam(':branch_id', $branchId);

        $stmt->execute();

        addMessage('success', 'تم إنشاء مدير النظام بنجاح.');
        return $conn->lastInsertId();
    } catch (PDOException $e) {
        addMessage('error', 'خطأ في إنشاء مدير النظام: ' . $e->getMessage());
        return false;
    }
}

// بدء العملية
try {
    // التحقق من وجود مدير النظام
    $admin = checkAdmin($conn);

    // إضافة معلومات تصحيح
    if ($admin) {
        addDebugInfo('تم العثور على مدير النظام (المعرف: ' . $admin['id'] . ')');
        addDebugInfo('اسم المستخدم: ' . $admin['username']);
        addDebugInfo('البريد الإلكتروني: ' . $admin['email']);
        addDebugInfo('الدور: ' . $admin['role']);
        addDebugInfo('الحالة: ' . ($admin['status'] ?? 'غير محدد'));
        addDebugInfo('نشط: ' . ($admin['is_active'] ?? 'غير محدد'));

        // إعادة تعيين كلمة المرور
        $result = resetAdminPassword($conn, $admin['id'], $adminPassword);
        if (!$result) {
            throw new Exception('فشل في إعادة تعيين كلمة مرور مدير النظام.');
        }
    } else {
        addDebugInfo('لم يتم العثور على مدير النظام. سيتم إنشاء مدير جديد.');

        // إنشاء مدير النظام
        $adminData = [
            'username' => $adminUsername,
            'password' => $adminPassword,
            'email' => $adminEmail
        ];

        $adminId = createAdmin($conn, $adminData);
        if (!$adminId) {
            throw new Exception('فشل في إنشاء مدير النظام.');
        }
    }

    // إضافة رسالة نجاح نهائية
    addMessage('success', 'تم إعادة تعيين كلمة مرور مدير النظام بنجاح. يمكنك الآن تسجيل الدخول باستخدام البيانات التالية:');

} catch (Exception $e) {
    addMessage('error', 'حدث خطأ: ' . $e->getMessage());
}
?>

<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>إعادة تعيين كلمة مرور مدير النظام</title>
    <link href="https://fonts.googleapis.com/css2?family=Tajawal:wght@400;500;700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Tajawal', sans-serif;
            background-color: #f5f5f5;
            margin: 0;
            padding: 20px;
            color: #333;
        }
        .container {
            max-width: 800px;
            margin: 0 auto;
            background-color: #fff;
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 0 20px rgba(0, 0, 0, 0.1);
        }
        h1 {
            color: <?php echo $primaryColor; ?>;
            text-align: center;
            margin-bottom: 30px;
            font-size: 28px;
        }
        .message {
            padding: 15px;
            margin-bottom: 15px;
            border-radius: 5px;
            font-size: 16px;
        }
        .success {
            background-color: rgba(76, 175, 80, 0.1);
            border: 1px solid <?php echo $successColor; ?>;
            color: <?php echo $successColor; ?>;
        }
        .error {
            background-color: rgba(244, 67, 54, 0.1);
            border: 1px solid <?php echo $errorColor; ?>;
            color: <?php echo $errorColor; ?>;
        }
        .warning {
            background-color: rgba(255, 152, 0, 0.1);
            border: 1px solid <?php echo $warningColor; ?>;
            color: <?php echo $warningColor; ?>;
        }
        .info {
            background-color: rgba(33, 150, 243, 0.1);
            border: 1px solid <?php echo $infoColor; ?>;
            color: <?php echo $infoColor; ?>;
        }
        .admin-info {
            background-color: rgba(76, 12, 112, 0.05);
            border: 1px solid #ddd;
            padding: 20px;
            border-radius: 5px;
            margin-top: 20px;
        }
        .admin-info h2 {
            color: <?php echo $primaryColor; ?>;
            margin-top: 0;
            font-size: 20px;
        }
        .admin-info p {
            margin: 10px 0;
            font-size: 16px;
        }
        .admin-info strong {
            color: <?php echo $secondaryColor; ?>;
        }
        .back-link {
            display: block;
            text-align: center;
            margin-top: 30px;
            color: <?php echo $primaryColor; ?>;
            text-decoration: none;
            font-weight: bold;
            font-size: 16px;
        }
        .back-link:hover {
            text-decoration: underline;
        }
        .debug-info {
            background-color: #f8f9fa;
            border: 1px solid #ddd;
            padding: 15px;
            border-radius: 5px;
            margin-top: 20px;
            font-family: monospace;
            font-size: 14px;
            white-space: pre-wrap;
        }
        .debug-info h3 {
            margin-top: 0;
            color: #333;
            font-size: 16px;
        }
        .debug-info p {
            margin: 5px 0;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>إعادة تعيين كلمة مرور مدير النظام</h1>

        <?php foreach ($messages as $message): ?>
            <div class="message <?php echo $message['type']; ?>">
                <?php echo $message['message']; ?>
            </div>
        <?php endforeach; ?>

        <div class="admin-info">
            <h2>معلومات مدير النظام</h2>
            <p><strong>اسم المستخدم:</strong> <?php echo $adminUsername; ?></p>
            <p><strong>البريد الإلكتروني:</strong> <?php echo $adminEmail; ?></p>
            <p><strong>كلمة المرور:</strong> <?php echo $adminPassword; ?> (كلمة المرور الجديدة)</p>
        </div>

        <?php if (!empty($debugInfo)): ?>
        <div class="debug-info">
            <h3>معلومات التصحيح</h3>
            <?php foreach ($debugInfo as $info): ?>
                <p><?php echo $info; ?></p>
            <?php endforeach; ?>
        </div>
        <?php endif; ?>

        <a href="./auth/login.php" class="back-link">العودة إلى صفحة تسجيل الدخول</a>
    </div>
</body>
</html>
