<?php
/**
 * ملف للتحقق من حالة النظام على المجال الفرعي
 * 
 * هذا الملف يقوم بالتحقق من حالة النظام على المجال الفرعي والتأكد من أن جميع المكونات تعمل بشكل صحيح.
 */

// تعيين رأس الصفحة
header('Content-Type: text/html; charset=utf-8');

// تعريف الدوال المساعدة
function check_file_exists($file) {
    return file_exists($file);
}

function check_directory_exists($directory) {
    return is_dir($directory);
}

function check_writable($path) {
    return is_writable($path);
}

function check_php_version() {
    return version_compare(PHP_VERSION, '7.0.0', '>=');
}

function check_extension_loaded($extension) {
    return extension_loaded($extension);
}

function check_database_connection() {
    // محاولة الاتصال بقاعدة البيانات
    if (file_exists('config.php')) {
        try {
            include_once 'config.php';
            
            if (isset($pdo)) {
                return true;
            }
            
            return false;
        } catch (Exception $e) {
            return false;
        }
    }
    
    return false;
}

function check_api_endpoints() {
    $api_endpoints = [
        'api/branches.php',
        'api/orders.php'
    ];
    
    $results = [];
    
    foreach ($api_endpoints as $endpoint) {
        $results[$endpoint] = file_exists($endpoint);
    }
    
    return $results;
}

function check_html_pages() {
    $html_pages = [
        'dashboard.html',
        'orders.html',
        'create-order.html',
        'settings.html',
        'archive.html'
    ];
    
    $results = [];
    
    foreach ($html_pages as $page) {
        $results[$page] = file_exists($page);
    }
    
    return $results;
}

function get_server_info() {
    $server_info = [
        'server_software' => $_SERVER['SERVER_SOFTWARE'] ?? 'غير معروف',
        'server_name' => $_SERVER['SERVER_NAME'] ?? 'غير معروف',
        'server_protocol' => $_SERVER['SERVER_PROTOCOL'] ?? 'غير معروف',
        'server_port' => $_SERVER['SERVER_PORT'] ?? 'غير معروف',
        'document_root' => $_SERVER['DOCUMENT_ROOT'] ?? 'غير معروف',
        'https' => isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'نعم' : 'لا',
        'remote_addr' => $_SERVER['REMOTE_ADDR'] ?? 'غير معروف',
        'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'غير معروف',
        'script_filename' => $_SERVER['SCRIPT_FILENAME'] ?? 'غير معروف',
        'current_dir' => dirname($_SERVER['SCRIPT_FILENAME']) ?? 'غير معروف',
    ];
    
    return $server_info;
}

function get_php_info() {
    $php_info = [
        'version' => PHP_VERSION,
        'os' => PHP_OS,
        'sapi' => php_sapi_name(),
        'memory_limit' => ini_get('memory_limit'),
        'max_execution_time' => ini_get('max_execution_time'),
        'upload_max_filesize' => ini_get('upload_max_filesize'),
        'post_max_size' => ini_get('post_max_size'),
        'display_errors' => ini_get('display_errors'),
        'error_reporting' => ini_get('error_reporting'),
        'date.timezone' => ini_get('date.timezone'),
    ];
    
    return $php_info;
}

// إجراء الفحوصات
$checks = [
    'files' => [
        'config.php' => check_file_exists('config.php'),
        '.htaccess' => check_file_exists('.htaccess'),
        'dashboard.html' => check_file_exists('dashboard.html'),
        'orders.html' => check_file_exists('orders.html'),
        'create-order.html' => check_file_exists('create-order.html'),
        'settings.html' => check_file_exists('settings.html'),
        'archive.html' => check_file_exists('archive.html'),
    ],
    'directories' => [
        'api' => check_directory_exists('api'),
        'images' => check_directory_exists('images'),
        'css' => check_directory_exists('css'),
        'js' => check_directory_exists('js'),
    ],
    'permissions' => [
        'current_directory' => check_writable('.'),
        'api_directory' => check_directory_exists('api') ? check_writable('api') : false,
        'images_directory' => check_directory_exists('images') ? check_writable('images') : false,
    ],
    'php' => [
        'version' => check_php_version(),
        'pdo_mysql' => check_extension_loaded('pdo_mysql'),
        'json' => check_extension_loaded('json'),
        'mbstring' => check_extension_loaded('mbstring'),
    ],
    'database' => check_database_connection(),
    'api_endpoints' => check_api_endpoints(),
    'html_pages' => check_html_pages(),
];

$server_info = get_server_info();
$php_info = get_php_info();

// عرض النتائج
?>
<!DOCTYPE html>
<html dir="rtl" lang="ar">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>التحقق من حالة النظام على المجال الفرعي</title>
    <style>
        body {
            font-family: 'Tajawal', sans-serif;
            margin: 20px;
            line-height: 1.6;
        }
        h1, h2, h3 {
            color: #0a2056;
        }
        .container {
            max-width: 1000px;
            margin: 0 auto;
            padding: 20px;
            background-color: #f8fafc;
            border-radius: 10px;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
        }
        .section {
            margin-bottom: 20px;
            padding: 15px;
            background-color: #fff;
            border-radius: 5px;
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.05);
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin: 10px 0;
        }
        th, td {
            border: 1px solid #ddd;
            padding: 8px;
            text-align: right;
        }
        th {
            background-color: #f2f2f2;
        }
        .success {
            color: green;
        }
        .error {
            color: red;
        }
        .warning {
            color: orange;
        }
        .summary {
            margin-top: 20px;
            padding: 15px;
            background-color: #e6f7e6;
            border-radius: 5px;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>التحقق من حالة النظام على المجال الفرعي</h1>
        
        <div class="section">
            <h2>معلومات الخادم</h2>
            <table>
                <tr>
                    <th>المعلومة</th>
                    <th>القيمة</th>
                </tr>
                <?php foreach ($server_info as $key => $value): ?>
                <tr>
                    <td><?php echo htmlspecialchars($key); ?></td>
                    <td><?php echo htmlspecialchars($value); ?></td>
                </tr>
                <?php endforeach; ?>
            </table>
        </div>
        
        <div class="section">
            <h2>معلومات PHP</h2>
            <table>
                <tr>
                    <th>المعلومة</th>
                    <th>القيمة</th>
                </tr>
                <?php foreach ($php_info as $key => $value): ?>
                <tr>
                    <td><?php echo htmlspecialchars($key); ?></td>
                    <td><?php echo htmlspecialchars($value); ?></td>
                </tr>
                <?php endforeach; ?>
            </table>
        </div>
        
        <div class="section">
            <h2>فحص الملفات</h2>
            <table>
                <tr>
                    <th>الملف</th>
                    <th>الحالة</th>
                </tr>
                <?php foreach ($checks['files'] as $file => $exists): ?>
                <tr>
                    <td><?php echo htmlspecialchars($file); ?></td>
                    <td class="<?php echo $exists ? 'success' : 'error'; ?>">
                        <?php echo $exists ? 'موجود' : 'غير موجود'; ?>
                    </td>
                </tr>
                <?php endforeach; ?>
            </table>
        </div>
        
        <div class="section">
            <h2>فحص المجلدات</h2>
            <table>
                <tr>
                    <th>المجلد</th>
                    <th>الحالة</th>
                </tr>
                <?php foreach ($checks['directories'] as $directory => $exists): ?>
                <tr>
                    <td><?php echo htmlspecialchars($directory); ?></td>
                    <td class="<?php echo $exists ? 'success' : 'error'; ?>">
                        <?php echo $exists ? 'موجود' : 'غير موجود'; ?>
                    </td>
                </tr>
                <?php endforeach; ?>
            </table>
        </div>
        
        <div class="section">
            <h2>فحص الصلاحيات</h2>
            <table>
                <tr>
                    <th>المسار</th>
                    <th>الحالة</th>
                </tr>
                <?php foreach ($checks['permissions'] as $path => $writable): ?>
                <tr>
                    <td><?php echo htmlspecialchars($path); ?></td>
                    <td class="<?php echo $writable ? 'success' : 'error'; ?>">
                        <?php echo $writable ? 'قابل للكتابة' : 'غير قابل للكتابة'; ?>
                    </td>
                </tr>
                <?php endforeach; ?>
            </table>
        </div>
        
        <div class="section">
            <h2>فحص PHP</h2>
            <table>
                <tr>
                    <th>المتطلب</th>
                    <th>الحالة</th>
                </tr>
                <?php foreach ($checks['php'] as $requirement => $met): ?>
                <tr>
                    <td><?php echo htmlspecialchars($requirement); ?></td>
                    <td class="<?php echo $met ? 'success' : 'error'; ?>">
                        <?php echo $met ? 'متوفر' : 'غير متوفر'; ?>
                    </td>
                </tr>
                <?php endforeach; ?>
            </table>
        </div>
        
        <div class="section">
            <h2>فحص قاعدة البيانات</h2>
            <p class="<?php echo $checks['database'] ? 'success' : 'error'; ?>">
                <?php echo $checks['database'] ? 'الاتصال بقاعدة البيانات ناجح' : 'فشل الاتصال بقاعدة البيانات'; ?>
            </p>
        </div>
        
        <div class="section">
            <h2>فحص نقاط النهاية API</h2>
            <table>
                <tr>
                    <th>نقطة النهاية</th>
                    <th>الحالة</th>
                </tr>
                <?php foreach ($checks['api_endpoints'] as $endpoint => $exists): ?>
                <tr>
                    <td><?php echo htmlspecialchars($endpoint); ?></td>
                    <td class="<?php echo $exists ? 'success' : 'error'; ?>">
                        <?php echo $exists ? 'موجود' : 'غير موجود'; ?>
                    </td>
                </tr>
                <?php endforeach; ?>
            </table>
        </div>
        
        <div class="section">
            <h2>فحص صفحات HTML</h2>
            <table>
                <tr>
                    <th>الصفحة</th>
                    <th>الحالة</th>
                </tr>
                <?php foreach ($checks['html_pages'] as $page => $exists): ?>
                <tr>
                    <td><?php echo htmlspecialchars($page); ?></td>
                    <td class="<?php echo $exists ? 'success' : 'error'; ?>">
                        <?php echo $exists ? 'موجود' : 'غير موجود'; ?>
                    </td>
                </tr>
                <?php endforeach; ?>
            </table>
        </div>
        
        <div class="summary">
            <h2>ملخص</h2>
            <?php
            $all_checks_passed = true;
            
            foreach ($checks['files'] as $exists) {
                if (!$exists) {
                    $all_checks_passed = false;
                    break;
                }
            }
            
            if ($all_checks_passed) {
                foreach ($checks['directories'] as $exists) {
                    if (!$exists) {
                        $all_checks_passed = false;
                        break;
                    }
                }
            }
            
            if ($all_checks_passed) {
                foreach ($checks['permissions'] as $writable) {
                    if (!$writable) {
                        $all_checks_passed = false;
                        break;
                    }
                }
            }
            
            if ($all_checks_passed) {
                foreach ($checks['php'] as $met) {
                    if (!$met) {
                        $all_checks_passed = false;
                        break;
                    }
                }
            }
            
            if ($all_checks_passed && !$checks['database']) {
                $all_checks_passed = false;
            }
            ?>
            
            <?php if ($all_checks_passed): ?>
            <p class="success">جميع الفحوصات ناجحة! النظام جاهز للاستخدام.</p>
            <?php else: ?>
            <p class="warning">بعض الفحوصات فشلت. يرجى مراجعة النتائج أعلاه وإصلاح المشكلات قبل استخدام النظام.</p>
            <?php endif; ?>
        </div>
    </div>
</body>
</html>
