<?php
/**
 * ملف استيراد قاعدة البيانات
 * يستخدم هذا الملف لإنشاء قاعدة البيانات وجداولها
 */

// تفعيل عرض الأخطاء للتصحيح
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// تضمين ملف الاتصال بقاعدة البيانات
require_once 'config/db_connection.php';

// دالة لتنفيذ ملف SQL
function execute_sql_file($pdo, $file_path) {
    if (!file_exists($file_path)) {
        return [
            'success' => false,
            'message' => "الملف $file_path غير موجود."
        ];
    }
    
    try {
        $sql = file_get_contents($file_path);
        
        // تقسيم الملف إلى استعلامات منفصلة
        $queries = explode(';', $sql);
        
        // تنفيذ كل استعلام
        foreach ($queries as $query) {
            $query = trim($query);
            
            if (!empty($query)) {
                $pdo->exec($query);
            }
        }
        
        return [
            'success' => true,
            'message' => "تم تنفيذ ملف $file_path بنجاح."
        ];
    } catch (PDOException $e) {
        return [
            'success' => false,
            'message' => "حدث خطأ أثناء تنفيذ ملف $file_path: " . $e->getMessage()
        ];
    }
}

// دالة لإنشاء قاعدة البيانات وجداولها
function setup_database() {
    global $pdo, $conn, $dbname;
    
    $results = [];
    
    try {
        // التحقق من وجود جدول الفروع
        if (!tableExists('branches')) {
            // تنفيذ ملف SQL لإنشاء جدول الفروع
            $branches_result = execute_sql_file($pdo, 'database/create_orders_table.sql');
            $results[] = $branches_result;
        } else {
            $results[] = [
                'success' => true,
                'message' => "جدول الفروع موجود بالفعل."
            ];
        }
        
        // التحقق من وجود جدول تسلسل أرقام الطلبات
        if (!tableExists('order_sequences')) {
            // تنفيذ ملف SQL لإنشاء جدول تسلسل أرقام الطلبات
            $sequences_result = execute_sql_file($pdo, 'database/create_order_sequences_table.sql');
            $results[] = $sequences_result;
        } else {
            $results[] = [
                'success' => true,
                'message' => "جدول تسلسل أرقام الطلبات موجود بالفعل."
            ];
        }
        
        // التحقق من وجود جدول المستخدمين
        if (!tableExists('users')) {
            // تنفيذ ملف SQL لإنشاء جدول المستخدمين
            $users_result = execute_sql_file($pdo, 'database/create_users_table.sql');
            $results[] = $users_result;
        } else {
            $results[] = [
                'success' => true,
                'message' => "جدول المستخدمين موجود بالفعل."
            ];
        }
        
        // إنشاء مستخدم مسؤول افتراضي إذا لم يكن موجودًا
        $stmt = $pdo->query("SELECT COUNT(*) as count FROM users WHERE username = 'admin'");
        $result = $stmt->fetch();
        
        if ($result['count'] == 0) {
            // إنشاء مستخدم مسؤول افتراضي (كلمة المرور: admin123)
            $defaultAdminPassword = password_hash('admin123', PASSWORD_DEFAULT);
            $stmt = $pdo->prepare("
                INSERT INTO users (username, password, full_name, email, role, is_active, is_approved)
                VALUES ('admin', :password, 'مدير النظام', 'admin@example.com', 'admin', 1, 1)
            ");
            $stmt->bindParam(':password', $defaultAdminPassword);
            $stmt->execute();
            
            $results[] = [
                'success' => true,
                'message' => "تم إنشاء مستخدم مسؤول افتراضي."
            ];
        } else {
            $results[] = [
                'success' => true,
                'message' => "مستخدم المسؤول موجود بالفعل."
            ];
        }
        
        return $results;
    } catch (PDOException $e) {
        return [
            [
                'success' => false,
                'message' => "حدث خطأ أثناء إعداد قاعدة البيانات: " . $e->getMessage()
            ]
        ];
    }
}

// بدء الصفحة HTML
?>
<!DOCTYPE html>
<html dir="rtl" lang="ar">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>استيراد قاعدة البيانات - نظام طلبات الشحن</title>
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            line-height: 1.6;
            margin: 20px;
            background-color: #f8f9fa;
            color: #333;
        }
        h1, h2 {
            color: #0056b3;
        }
        .container {
            max-width: 800px;
            margin: 0 auto;
            background-color: white;
            padding: 20px;
            border-radius: 5px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        }
        .success {
            color: #28a745;
            background-color: #d4edda;
            padding: 10px;
            border-radius: 5px;
            margin: 10px 0;
        }
        .error {
            color: #dc3545;
            background-color: #f8d7da;
            padding: 10px;
            border-radius: 5px;
            margin: 10px 0;
        }
        .warning {
            color: #856404;
            background-color: #fff3cd;
            padding: 10px;
            border-radius: 5px;
            margin: 10px 0;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin: 15px 0;
        }
        th, td {
            padding: 8px;
            text-align: right;
            border: 1px solid #ddd;
        }
        th {
            background-color: #f2f2f2;
        }
        .btn {
            display: inline-block;
            padding: 8px 16px;
            background-color: #0056b3;
            color: white;
            text-decoration: none;
            border-radius: 4px;
            margin-top: 10px;
            margin-left: 5px;
        }
        .btn:hover {
            background-color: #003d7a;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>استيراد قاعدة البيانات - نظام طلبات الشحن</h1>
        
        <?php
        // التحقق من الاتصال بقاعدة البيانات
        try {
            $pdo->query("SELECT 1");
            echo "<div class='success'>تم الاتصال بقاعدة البيانات بنجاح.</div>";
            
            // إعداد قاعدة البيانات
            $results = setup_database();
            
            // عرض نتائج الإعداد
            echo "<h2>نتائج إعداد قاعدة البيانات</h2>";
            echo "<ul>";
            foreach ($results as $result) {
                $class = $result['success'] ? 'success' : 'error';
                echo "<li class='$class'>{$result['message']}</li>";
            }
            echo "</ul>";
            
            // عرض معلومات الجداول
            echo "<h2>معلومات الجداول</h2>";
            echo "<table>";
            echo "<tr><th>اسم الجدول</th><th>عدد السجلات</th></tr>";
            
            $tables = ['branches', 'orders', 'users', 'order_sequences'];
            foreach ($tables as $table) {
                if (tableExists($table)) {
                    $stmt = $pdo->query("SELECT COUNT(*) as count FROM $table");
                    $result = $stmt->fetch();
                    echo "<tr><td>$table</td><td>{$result['count']}</td></tr>";
                } else {
                    echo "<tr><td>$table</td><td class='error'>الجدول غير موجود</td></tr>";
                }
            }
            
            echo "</table>";
            
            // عرض روابط مفيدة
            echo "<div>";
            echo "<a href='index.html' class='btn'>الصفحة الرئيسية</a>";
            echo "<a href='login.php' class='btn'>صفحة تسجيل الدخول</a>";
            echo "<a href='test_db.php' class='btn'>اختبار قاعدة البيانات</a>";
            echo "</div>";
            
        } catch (PDOException $e) {
            echo "<div class='error'>فشل الاتصال بقاعدة البيانات: " . $e->getMessage() . "</div>";
            echo "<div class='warning'>تأكد من تعديل معلومات الاتصال بقاعدة البيانات في ملف config/db_connection.php.</div>";
        }
        ?>
    </div>
</body>
</html>
