<?php
/**
 * ملف لإعداد قاعدة البيانات وإنشاء الجداول اللازمة
 */

// تفعيل عرض الأخطاء للتصحيح
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// تضمين ملف الاتصال بقاعدة البيانات
require_once 'config/db_connection.php';

// دالة لتنفيذ ملف SQL
function executeSqlFile($conn, $filename) {
    $result = [
        'status' => 'success',
        'message' => "تم تنفيذ الملف $filename بنجاح"
    ];

    try {
        // التحقق من وجود الملف
        if (!file_exists($filename)) {
            return [
                'status' => 'error',
                'message' => "الملف $filename غير موجود"
            ];
        }

        // قراءة محتوى الملف
        $sql = file_get_contents($filename);
        if ($sql === false) {
            return [
                'status' => 'error',
                'message' => "فشل في قراءة محتوى الملف $filename"
            ];
        }

        // تقسيم الاستعلامات
        $queries = explode(';', $sql);

        // تنفيذ كل استعلام
        foreach ($queries as $query) {
            $query = trim($query);
            if (!empty($query)) {
                // استبدال DELIMITER لتجنب الأخطاء
                if (strpos($query, 'DELIMITER') === 0) {
                    continue;
                }

                // تجاهل الأسطر التي تبدأ بـ -- (تعليقات SQL)
                if (strpos($query, '--') === 0) {
                    continue;
                }

                $conn->exec($query);
            }
        }
    } catch (PDOException $e) {
        $result = [
            'status' => 'error',
            'message' => "فشل في تنفيذ الملف $filename: " . $e->getMessage() . " في السطر: " . $e->getLine()
        ];
    } catch (Exception $e) {
        $result = [
            'status' => 'error',
            'message' => "حدث خطأ غير متوقع: " . $e->getMessage()
        ];
    }

    return $result;
}

// دالة للتحقق من وجود جدول
function tableExists($conn, $tableName) {
    try {
        $result = $conn->query("SELECT 1 FROM $tableName LIMIT 1");
        return true;
    } catch (Exception $e) {
        return false;
    }
}

// تنفيذ ملفات SQL
$results = [];

// التحقق من اتصال قاعدة البيانات
if (!$conn) {
    $results[] = [
        'status' => 'error',
        'message' => 'فشل الاتصال بقاعدة البيانات. يرجى التحقق من إعدادات الاتصال.'
    ];
} else {
    // عرض معلومات قاعدة البيانات
    $results[] = [
        'status' => 'info',
        'message' => 'تم الاتصال بقاعدة البيانات بنجاح: ' . $dbname
    ];

    // إنشاء جدول الفروع
    $branchesFile = 'create_branches_table.sql';
    $results[] = executeSqlFile($conn, $branchesFile);

    // إنشاء جدول الطلبات
    $ordersFile = 'create_orders_table.sql';
    $results[] = executeSqlFile($conn, $ordersFile);

    // التحقق من وجود الجداول بعد التنفيذ
    $tables = ['branches', 'order_sequences', 'orders', 'archived_orders'];
    foreach ($tables as $table) {
        $exists = tableExists($conn, $table);
        $results[] = [
            'status' => $exists ? 'success' : 'warning',
            'message' => 'جدول ' . $table . ': ' . ($exists ? 'تم إنشاؤه بنجاح' : 'لم يتم إنشاؤه')
        ];
    }
}

// عرض النتائج
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>إعداد قاعدة البيانات</title>
    <style>
        body {
            font-family: 'Tajawal', sans-serif;
            background-color: #f8f9fa;
            margin: 0;
            padding: 20px;
        }
        .container {
            max-width: 800px;
            margin: 0 auto;
            background-color: #fff;
            padding: 20px;
            border-radius: 10px;
            box-shadow: 0 0 10px rgba(0,0,0,0.1);
        }
        h1 {
            color: #4f46e5;
            text-align: center;
            margin-bottom: 30px;
        }
        .result {
            margin-bottom: 15px;
            padding: 15px;
            border-radius: 5px;
        }
        .success {
            background-color: #d1fae5;
            border: 1px solid #10b981;
            color: #047857;
        }
        .error {
            background-color: #fee2e2;
            border: 1px solid #ef4444;
            color: #b91c1c;
        }
        .btn {
            display: inline-block;
            padding: 10px 20px;
            background-color: #4f46e5;
            color: #fff;
            text-decoration: none;
            border-radius: 5px;
            margin-top: 20px;
            text-align: center;
        }
        .btn:hover {
            background-color: #4338ca;
        }
        .table-info {
            margin-top: 30px;
            border-top: 1px solid #e5e7eb;
            padding-top: 20px;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 10px;
        }
        table, th, td {
            border: 1px solid #e5e7eb;
        }
        th, td {
            padding: 10px;
            text-align: right;
        }
        th {
            background-color: #f3f4f6;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>إعداد قاعدة البيانات</h1>

        <?php foreach ($results as $result): ?>
            <div class="result <?php
                $class = 'info';
                switch ($result['status']) {
                    case 'success': $class = 'success'; break;
                    case 'error': $class = 'error'; break;
                    case 'warning': $class = 'warning'; break;
                    case 'info': $class = 'info'; break;
                }
                echo $class;
            ?>">
                <strong><?php echo $result['message']; ?></strong>
            </div>
        <?php endforeach; ?>

        <div class="table-info">
            <h2>حالة الجداول</h2>
            <table>
                <tr>
                    <th>اسم الجدول</th>
                    <th>الحالة</th>
                </tr>
                <tr>
                    <td>branches</td>
                    <td><?php echo tableExists($conn, 'branches') ? 'موجود' : 'غير موجود'; ?></td>
                </tr>
                <tr>
                    <td>order_sequences</td>
                    <td><?php echo tableExists($conn, 'order_sequences') ? 'موجود' : 'غير موجود'; ?></td>
                </tr>
                <tr>
                    <td>orders</td>
                    <td><?php echo tableExists($conn, 'orders') ? 'موجود' : 'غير موجود'; ?></td>
                </tr>
                <tr>
                    <td>archived_orders</td>
                    <td><?php echo tableExists($conn, 'archived_orders') ? 'موجود' : 'غير موجود'; ?></td>
                </tr>
            </table>
        </div>

        <div style="display: flex; justify-content: space-between; margin-top: 20px;">
            <a href="add_test_orders.php" class="btn">إضافة طلبات تجريبية</a>
            <a href="orders.php" class="btn">العودة إلى صفحة الطلبات</a>
        </div>
    </div>
</body>
</html>
