# Simple script to convert SVG files to PNG using System.Drawing
# This script uses a basic approach that works for simple SVG files

# Add System.Drawing assembly
Add-Type -AssemblyName System.Drawing

# Function to create a PNG from SVG using basic drawing
function Convert-SvgToPng {
    param (
        [string]$svgPath,
        [string]$pngPath,
        [int]$width = 800,
        [int]$height = 600
    )
    
    try {
        # Read SVG content
        $svgContent = Get-Content -Path $svgPath -Raw
        
        # Extract width and height from SVG if available
        $widthMatch = [regex]::Match($svgContent, 'width="([0-9]+)"')
        $heightMatch = [regex]::Match($svgContent, 'height="([0-9]+)"')
        
        if ($widthMatch.Success -and $heightMatch.Success) {
            $width = [int]$widthMatch.Groups[1].Value
            $height = [int]$heightMatch.Groups[1].Value
        }
        
        # Create a bitmap
        $bitmap = New-Object System.Drawing.Bitmap($width, $height)
        $graphics = [System.Drawing.Graphics]::FromImage($bitmap)
        
        # Set background color (optional)
        $graphics.Clear([System.Drawing.Color]::White)
        
        # Create a simple representation of the SVG
        # This is a very basic approach and won't work for complex SVGs
        $font = New-Object System.Drawing.Font("Arial", 12)
        $brush = New-Object System.Drawing.SolidBrush([System.Drawing.Color]::Black)
        $graphics.DrawString("SVG Preview (Basic Conversion)", $font, $brush, 10, 10)
        
        # Save as PNG
        $bitmap.Save($pngPath, [System.Drawing.Imaging.ImageFormat]::Png)
        
        # Clean up
        $graphics.Dispose()
        $bitmap.Dispose()
        
        Write-Host "Created basic PNG representation of $svgPath at $pngPath"
        return $true
    }
    catch {
        Write-Host "Error creating PNG from $svgPath: $_" -ForegroundColor Red
        return $false
    }
}

# Main function to process all SVG files
function Process-SvgFiles {
    # Get all SVG files
    $svgFiles = @(
        "images\backgrounds\login-bg.svg",
        "images\countries\sa.svg",
        "images\countries\ye.svg",
        "images\logo\company-logo.svg",
        "pages\images\flags\china.svg",
        "pages\images\flags\oman.svg",
        "pages\images\flags\saudi.svg",
        "pages\images\flags\yemen.svg"
    )
    
    foreach ($svgFile in $svgFiles) {
        # Create PNG path by replacing extension
        $pngFile = $svgFile -replace "\.svg$", ".png"
        
        Write-Host "Processing $svgFile to $pngFile..."
        
        # Ensure directory exists
        $pngDir = Split-Path -Path $pngFile -Parent
        if (-not (Test-Path $pngDir)) {
            New-Item -Path $pngDir -ItemType Directory -Force | Out-Null
        }
        
        $success = Convert-SvgToPng -svgPath $svgFile -pngPath $pngFile
        
        if ($success) {
            Write-Host "Successfully created PNG for $svgFile" -ForegroundColor Green
        }
        else {
            Write-Host "Failed to create PNG for $svgFile" -ForegroundColor Red
        }
    }
}

# Run the main function
Process-SvgFiles
