<?php
/**
 * سكريبت تنزيل أعلام الدول
 * يقوم هذا السكريبت بتنزيل صور أعلام الدول من الإنترنت وحفظها في المجلد المناسب
 */

// تضمين ملف التكوين
require_once 'config/config.php';

// تضمين ملف الأمان
require_once 'config/security.php';

// تضمين ملف الدوال المساعدة
require_once 'includes/functions.php';

// التحقق من حالة تسجيل الدخول
if (!isLoggedIn()) {
    // إعادة توجيه المستخدم إلى صفحة تسجيل الدخول
    header('Location: login.php');
    exit;
}

// التحقق من صلاحيات المستخدم
if ($_SESSION['role'] !== 'admin') {
    echo 'ليس لديك صلاحية للوصول إلى هذه الصفحة.';
    exit;
}

// مجلد حفظ صور الأعلام
$flagsDir = 'assets/images/flags/';

// التأكد من وجود المجلد
if (!is_dir($flagsDir)) {
    mkdir($flagsDir, 0755, true);
}

// قائمة الدول ورموزها
$countries = [
    ['name' => 'الصين', 'code' => 'CN'],
    ['name' => 'اليمن', 'code' => 'YE'],
    ['name' => 'عمان', 'code' => 'OM'],
    ['name' => 'السعودية', 'code' => 'SA'],
    ['name' => 'الإمارات', 'code' => 'AE']
];

// عنوان URL لتنزيل صور الأعلام
$flagsApiUrl = 'https://flagcdn.com/w320/';

// تنزيل صور الأعلام
$results = [];
foreach ($countries as $country) {
    $code = strtolower($country['code']);
    $name = $country['name'];
    $fileName = slugify($name) . '.png';
    $filePath = $flagsDir . $fileName;
    $fileUrl = $flagsApiUrl . $code . '.png';
    
    // تنزيل الصورة
    $result = ['name' => $name, 'status' => 'error', 'message' => ''];
    
    try {
        // التحقق من وجود الملف
        if (file_exists($filePath)) {
            $result['status'] = 'info';
            $result['message'] = 'الملف موجود بالفعل.';
        } else {
            // تنزيل الصورة
            $imageContent = @file_get_contents($fileUrl);
            if ($imageContent === false) {
                $result['message'] = 'فشل في تنزيل الصورة.';
            } else {
                // حفظ الصورة
                if (file_put_contents($filePath, $imageContent) !== false) {
                    $result['status'] = 'success';
                    $result['message'] = 'تم تنزيل الصورة بنجاح.';
                    
                    // تحديث مسار العلم في قاعدة البيانات
                    try {
                        $stmt = $pdo->prepare("UPDATE countries SET flag_path = ? WHERE code = ?");
                        $stmt->execute([$filePath, $country['code']]);
                    } catch (PDOException $e) {
                        // تجاهل الخطأ
                    }
                } else {
                    $result['message'] = 'فشل في حفظ الصورة.';
                }
            }
        }
    } catch (Exception $e) {
        $result['message'] = 'حدث خطأ: ' . $e->getMessage();
    }
    
    $results[] = $result;
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>تنزيل أعلام الدول | نظام طلبات الشحن</title>
    <link rel="stylesheet" href="assets/css/styles.css">
    <style>
        .container {
            max-width: 800px;
            margin: 0 auto;
            padding: 20px;
        }
        
        h1 {
            color: #0a2056;
            margin-bottom: 30px;
        }
        
        .results {
            margin-top: 30px;
        }
        
        .result-item {
            padding: 10px;
            margin-bottom: 10px;
            border-radius: 4px;
        }
        
        .result-success {
            background-color: #d1fae5;
            color: #047857;
        }
        
        .result-error {
            background-color: #fee2e2;
            color: #b91c1c;
        }
        
        .result-info {
            background-color: #e0f2fe;
            color: #0369a1;
        }
        
        .back-link {
            display: inline-block;
            margin-top: 20px;
            color: #0a2056;
            text-decoration: none;
        }
        
        .back-link:hover {
            text-decoration: underline;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>تنزيل أعلام الدول</h1>
        
        <div class="results">
            <?php foreach ($results as $result): ?>
            <div class="result-item result-<?php echo $result['status']; ?>">
                <strong><?php echo $result['name']; ?>:</strong> <?php echo $result['message']; ?>
            </div>
            <?php endforeach; ?>
        </div>
        
        <a href="upload_flags.php" class="back-link">العودة إلى صفحة تحميل الأعلام</a>
    </div>
</body>
</html>
