<?php
/**
 * أداة إصلاح أرقام الطلبات
 * تستخدم هذه الأداة لإصلاح أرقام الطلبات في قاعدة البيانات
 * وضمان أنها تتبع التنسيق الصحيح: MKL25-01, SNA25-01, إلخ
 */

// تضمين ملف الاتصال بقاعدة البيانات
require_once '../config/config.php';

// تعيين رأس الصفحة
header('Content-Type: text/html; charset=utf-8');
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>إصلاح أرقام الطلبات</title>
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            margin: 20px;
            background-color: #f8f9fa;
            color: #333;
        }
        h1, h2 {
            color: #3730a3;
        }
        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 20px;
            background-color: white;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin-bottom: 20px;
        }
        th, td {
            padding: 12px;
            text-align: right;
            border: 1px solid #ddd;
        }
        th {
            background-color: #4f46e5;
            color: white;
        }
        tr:nth-child(even) {
            background-color: #f2f2f2;
        }
        .btn {
            display: inline-block;
            padding: 10px 15px;
            background-color: #4f46e5;
            color: white;
            text-decoration: none;
            border-radius: 5px;
            margin-right: 10px;
            border: none;
            cursor: pointer;
            font-size: 14px;
        }
        .btn-danger {
            background-color: #ef4444;
        }
        .btn-success {
            background-color: #10b981;
        }
        .btn-info {
            background-color: #3b82f6;
        }
        .alert {
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
        }
        .alert-success {
            background-color: #d1fae5;
            border: 1px solid #10b981;
            color: #065f46;
        }
        .alert-warning {
            background-color: #fef3c7;
            border: 1px solid #f59e0b;
            color: #92400e;
        }
        .alert-danger {
            background-color: #fee2e2;
            border: 1px solid #ef4444;
            color: #b91c1c;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>إصلاح أرقام الطلبات</h1>
        <p>تستخدم هذه الأداة لإصلاح أرقام الطلبات في قاعدة البيانات وضمان أنها تتبع التنسيق الصحيح.</p>

        <?php
        // التحقق من وجود طلب إصلاح
        $fix = isset($_GET['fix']) && $_GET['fix'] === 'true';
        $analyze = isset($_GET['analyze']) && $_GET['analyze'] === 'true';

        // تحديد أنماط أرقام الطلبات الصحيحة
        $validPatterns = [
            'MKL25-\d{2}',   // المكلا
            'SNA25-\d{2}',   // صنعاء
            'ADN25-\d{2}',   // عدن
            'SH25-\d{2}',    // شحن
            'CN25-\d{2}'     // الصين
        ];

        // تحديد رموز الفروع
        $branchCodes = [
            'MKL' => 'المكلا',
            'SNA' => 'صنعاء',
            'ADN' => 'عدن',
            'SH' => 'شحن',
            'CN' => 'الصين'
        ];

        // دالة للتحقق من صحة رقم الطلب
        function isValidOrderNumber($orderNumber) {
            global $validPatterns;
            foreach ($validPatterns as $pattern) {
                if (preg_match('/^' . $pattern . '/', $orderNumber)) {
                    return true;
                }
            }
            return false;
        }

        // دالة لإصلاح رقم الطلب
        function fixOrderNumber($orderNumber) {
            global $branchCodes;
            
            // التعبير المنتظم لاستخراج رمز الفرع والرقم التسلسلي
            if (preg_match('/^([A-Z]+)(\d*)[-_]?(\d+)/', $orderNumber, $matches)) {
                $branchCode = $matches[1];
                $sequenceNumber = intval($matches[3]);
                
                // التحقق من أن رمز الفرع صالح
                if (array_key_exists($branchCode, $branchCodes)) {
                    // تنسيق رقم الطلب الجديد
                    return $branchCode . '25-' . str_pad($sequenceNumber, 2, '0', STR_PAD_LEFT);
                }
            }
            
            // إذا لم يتم التعرف على النمط، إرجاع الرقم الأصلي
            return $orderNumber;
        }

        // الحصول على جميع الطلبات
        $stmt = $pdo->query("SELECT id, order_number, client_name, created_at FROM orders ORDER BY id");
        $orders = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // تحليل أرقام الطلبات
        $invalidOrders = [];
        $fixableOrders = [];
        
        foreach ($orders as $order) {
            if (!isValidOrderNumber($order['order_number'])) {
                $invalidOrders[] = $order;
                
                // التحقق مما إذا كان يمكن إصلاح الرقم
                $fixedNumber = fixOrderNumber($order['order_number']);
                if ($fixedNumber !== $order['order_number']) {
                    $fixableOrders[] = [
                        'id' => $order['id'],
                        'old_number' => $order['order_number'],
                        'new_number' => $fixedNumber,
                        'client_name' => $order['client_name']
                    ];
                }
            }
        }

        // إصلاح أرقام الطلبات إذا تم طلب ذلك
        if ($fix && !empty($fixableOrders)) {
            $fixedCount = 0;
            $errorCount = 0;
            
            foreach ($fixableOrders as $order) {
                try {
                    // التحقق من عدم وجود تكرار
                    $checkStmt = $pdo->prepare("SELECT COUNT(*) FROM orders WHERE order_number = ?");
                    $checkStmt->execute([$order['new_number']]);
                    $exists = $checkStmt->fetchColumn() > 0;
                    
                    if (!$exists) {
                        $updateStmt = $pdo->prepare("UPDATE orders SET order_number = ? WHERE id = ?");
                        $updateStmt->execute([$order['new_number'], $order['id']]);
                        $fixedCount++;
                    } else {
                        echo "<div class='alert alert-warning'>";
                        echo "تعذر إصلاح الطلب #{$order['id']} لأن الرقم الجديد {$order['new_number']} موجود بالفعل.";
                        echo "</div>";
                        $errorCount++;
                    }
                } catch (PDOException $e) {
                    echo "<div class='alert alert-danger'>";
                    echo "خطأ في إصلاح الطلب #{$order['id']}: " . $e->getMessage();
                    echo "</div>";
                    $errorCount++;
                }
            }
            
            if ($fixedCount > 0) {
                echo "<div class='alert alert-success'>";
                echo "تم إصلاح {$fixedCount} من أرقام الطلبات بنجاح.";
                echo "</div>";
            }
            
            if ($errorCount > 0) {
                echo "<div class='alert alert-warning'>";
                echo "تعذر إصلاح {$errorCount} من أرقام الطلبات.";
                echo "</div>";
            }
            
            // إعادة تحميل الصفحة لتحديث البيانات
            echo "<script>window.location.href = 'fix_order_numbers.php?analyze=true';</script>";
        }

        // عرض أزرار التحليل والإصلاح
        echo "<div style='margin: 20px 0;'>";
        echo "<a href='fix_order_numbers.php?analyze=true' class='btn btn-info'>تحليل أرقام الطلبات</a>";
        
        if (!empty($fixableOrders)) {
            echo "<a href='fix_order_numbers.php?fix=true' class='btn btn-success' onclick=\"return confirm('هل أنت متأكد من إصلاح أرقام الطلبات؟')\">إصلاح أرقام الطلبات</a>";
        }
        
        echo "<a href='test_branch_sequences.php' class='btn'>اختبار تسلسل أرقام الطلبات</a>";
        echo "</div>";

        // عرض نتائج التحليل
        if ($analyze) {
            echo "<h2>نتائج تحليل أرقام الطلبات:</h2>";
            
            if (empty($invalidOrders)) {
                echo "<div class='alert alert-success'>";
                echo "جميع أرقام الطلبات صحيحة وتتبع التنسيق المطلوب.";
                echo "</div>";
            } else {
                echo "<p>تم العثور على " . count($invalidOrders) . " طلب برقم غير صالح.</p>";
                
                if (!empty($fixableOrders)) {
                    echo "<h3>الطلبات التي يمكن إصلاحها:</h3>";
                    echo "<table>";
                    echo "<tr><th>المعرف</th><th>اسم العميل</th><th>الرقم الحالي</th><th>الرقم الجديد</th></tr>";
                    
                    foreach ($fixableOrders as $order) {
                        echo "<tr>";
                        echo "<td>{$order['id']}</td>";
                        echo "<td>{$order['client_name']}</td>";
                        echo "<td>{$order['old_number']}</td>";
                        echo "<td>{$order['new_number']}</td>";
                        echo "</tr>";
                    }
                    
                    echo "</table>";
                }
                
                $unfixableOrders = array_filter($invalidOrders, function($order) use ($fixableOrders) {
                    foreach ($fixableOrders as $fixable) {
                        if ($fixable['id'] === $order['id']) {
                            return false;
                        }
                    }
                    return true;
                });
                
                if (!empty($unfixableOrders)) {
                    echo "<h3>الطلبات التي لا يمكن إصلاحها تلقائياً:</h3>";
                    echo "<table>";
                    echo "<tr><th>المعرف</th><th>اسم العميل</th><th>رقم الطلب</th><th>تاريخ الإنشاء</th></tr>";
                    
                    foreach ($unfixableOrders as $order) {
                        echo "<tr>";
                        echo "<td>{$order['id']}</td>";
                        echo "<td>{$order['client_name']}</td>";
                        echo "<td>{$order['order_number']}</td>";
                        echo "<td>{$order['created_at']}</td>";
                        echo "</tr>";
                    }
                    
                    echo "</table>";
                }
            }
        }
        ?>
    </div>
</body>
</html>
