<?php
/**
 * أداة لمزامنة أرقام الطلبات الاحتياطية مع قاعدة البيانات
 * تستخدم هذه الأداة لمزامنة أرقام الطلبات التي تم إنشاؤها أثناء وجود مشكلة في الاتصال بالخادم
 * مع قاعدة البيانات عند استعادة الاتصال
 */

// تفعيل عرض الأخطاء للتصحيح
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// تضمين ملف الاتصال بقاعدة البيانات
require_once '../config/config.php';

// التحقق من وجود ملفات السجل
$backupLogFile = '../logs/backup_order_numbers.log';
$randomLogFile = '../logs/random_order_numbers.log';
$processedLogFile = '../logs/processed_order_numbers.log';

// إنشاء ملفات السجل إذا لم تكن موجودة
if (!file_exists(dirname($backupLogFile))) {
    mkdir(dirname($backupLogFile), 0777, true);
}

if (!file_exists($backupLogFile)) {
    file_put_contents($backupLogFile, "");
}

if (!file_exists($randomLogFile)) {
    file_put_contents($randomLogFile, "");
}

if (!file_exists($processedLogFile)) {
    file_put_contents($processedLogFile, "");
}

// دالة لتحديث جدول order_sequences بناءً على رقم الطلب
function updateOrderSequence($orderNumber) {
    global $pdo;
    
    // استخراج رمز الفرع والرقم التسلسلي من رقم الطلب
    if (preg_match('/^([A-Z]+)(\d{2})(\d{2})-(\d{3})/', $orderNumber, $matches)) {
        $branchCode = $matches[1];
        $sequenceNumber = intval($matches[4]);
        
        try {
            // التحقق من آخر رقم تسلسلي للفرع
            $stmt = $pdo->prepare("SELECT last_sequence_number FROM order_sequences WHERE branch_code = ?");
            $stmt->execute([$branchCode]);
            $sequence = $stmt->fetch();
            
            if ($sequence) {
                $currentSequence = $sequence['last_sequence_number'];
                
                // تحديث الرقم التسلسلي فقط إذا كان الرقم الجديد أكبر من الرقم الحالي
                if ($sequenceNumber > $currentSequence) {
                    $stmt = $pdo->prepare("UPDATE order_sequences SET last_sequence_number = ? WHERE branch_code = ?");
                    $stmt->execute([$sequenceNumber, $branchCode]);
                    return true;
                }
            } else {
                // إنشاء سجل جديد إذا لم يكن موجوداً
                $stmt = $pdo->prepare("INSERT INTO order_sequences (branch_code, last_sequence_number) VALUES (?, ?)");
                $stmt->execute([$branchCode, $sequenceNumber]);
                return true;
            }
        } catch (PDOException $e) {
            error_log("خطأ في تحديث جدول order_sequences: " . $e->getMessage());
            return false;
        }
    }
    
    return false;
}

// دالة لمعالجة ملف السجل
function processLogFile($logFile, $processedLogFile) {
    if (!file_exists($logFile)) {
        return ['processed' => 0, 'failed' => 0];
    }
    
    $logContent = file_get_contents($logFile);
    $lines = explode("\n", $logContent);
    $processed = 0;
    $failed = 0;
    
    foreach ($lines as $line) {
        if (empty(trim($line))) {
            continue;
        }
        
        // استخراج رقم الطلب من السجل
        if (preg_match('/تم استخدام رقم طلب احتياطي: ([A-Z0-9-]+)/', $line, $matches) || 
            preg_match('/تم إنشاء رقم طلب عشوائي: ([A-Z0-9-]+)/', $line, $matches)) {
            
            $orderNumber = $matches[1];
            
            // تحديث جدول order_sequences
            if (updateOrderSequence($orderNumber)) {
                $processed++;
                
                // تسجيل المعالجة في ملف السجل
                $logMessage = date('Y-m-d H:i:s') . " - تمت معالجة رقم الطلب: " . $orderNumber . "\n";
                file_put_contents($processedLogFile, $logMessage, FILE_APPEND);
            } else {
                $failed++;
            }
        }
    }
    
    // مسح محتوى ملف السجل بعد المعالجة
    file_put_contents($logFile, "");
    
    return ['processed' => $processed, 'failed' => $failed];
}

// معالجة ملفات السجل
$backupResult = processLogFile($backupLogFile, $processedLogFile);
$randomResult = processLogFile($randomLogFile, $processedLogFile);

// عرض نتائج المزامنة
echo "<h1>نتائج مزامنة أرقام الطلبات الاحتياطية</h1>";

echo "<h2>أرقام الطلبات الاحتياطية:</h2>";
echo "<p>تمت معالجة: {$backupResult['processed']} رقم طلب</p>";
echo "<p>فشل في معالجة: {$backupResult['failed']} رقم طلب</p>";

echo "<h2>أرقام الطلبات العشوائية:</h2>";
echo "<p>تمت معالجة: {$randomResult['processed']} رقم طلب</p>";
echo "<p>فشل في معالجة: {$randomResult['failed']} رقم طلب</p>";

// عرض حالة جدول order_sequences بعد المزامنة
$stmt = $pdo->query("SELECT * FROM order_sequences");
$sequences = $stmt->fetchAll(PDO::FETCH_ASSOC);

echo "<h2>حالة جدول order_sequences بعد المزامنة:</h2>";
echo "<table border='1' cellpadding='5' style='border-collapse: collapse;'>";
echo "<tr><th>المعرف</th><th>رمز الفرع</th><th>آخر رقم تسلسلي</th><th>تاريخ الإنشاء</th><th>تاريخ التحديث</th></tr>";

foreach ($sequences as $sequence) {
    echo "<tr>";
    echo "<td>{$sequence['id']}</td>";
    echo "<td>{$sequence['branch_code']}</td>";
    echo "<td>{$sequence['last_sequence_number']}</td>";
    echo "<td>{$sequence['created_at']}</td>";
    echo "<td>{$sequence['updated_at']}</td>";
    echo "</tr>";
}

echo "</table>";

// عرض رابط للعودة إلى صفحة الأدوات
echo "<p><a href='test_sequence_api.php' style='display: inline-block; padding: 10px 15px; background-color: #4f46e5; color: white; text-decoration: none; border-radius: 5px; margin-top: 20px;'>العودة إلى اختبار API</a></p>";
?>
