<?php
/**
 * لوحة التحكم الموحدة
 * تعرض هذه الصفحة لوحة التحكم الموحدة للنظام مع اختلاف الصلاحيات بين الإدارة والمستخدم العادي
 * تم ربط لوحة التحكم بقاعدة البيانات الفعلية لعرض البيانات الحقيقية
 * تم تحسين أداء الاستعلامات وإضافة التخزين المؤقت لتحسين سرعة تحميل الصفحة
 *
 * @version 1.2
 * @author فريق التطوير
 * @update تم ربط لوحة التحكم بقاعدة البيانات
 * @update تم تحسين أداء الاستعلامات وإضافة التخزين المؤقت
 *
 * ملاحظات التحسينات:
 * 1. تم إضافة نظام التخزين المؤقت للاستعلامات المتكررة
 * 2. تم تحسين استعلامات SQL باستخدام تقنيات أفضل
 * 3. تم إضافة حدود زمنية للاستعلامات لتقليل حجم البيانات
 * 4. تم إضافة معالجة أفضل للأخطاء والبيانات الفارغة
 * 5. تم إنشاء ملف لإضافة الفهارس (database/add_indexes.php) لتحسين أداء قاعدة البيانات
 */

// بدء الجلسة
session_start();

// تضمين ملف التكوين
require_once 'config/config.php';

// تضمين ملف الاتصال بقاعدة البيانات
require_once 'config/database.php';

// تفعيل عرض الأخطاء للتصحيح
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

/**
 * دالة للتخزين المؤقت للاستعلامات
 * تستخدم هذه الدالة لتخزين نتائج الاستعلامات المتكررة وتحسين الأداء
 *
 * @param string $key مفتاح التخزين المؤقت
 * @param callable $callback دالة لتنفيذ الاستعلام إذا لم يكن موجودًا في التخزين المؤقت
 * @param int $ttl مدة صلاحية التخزين المؤقت بالثواني (الافتراضي: 300 ثانية = 5 دقائق)
 * @return mixed نتيجة الاستعلام
 */
function queryCache($key, $callback, $ttl = 300) {
    $cacheDir = __DIR__ . '/cache';
    $cacheFile = $cacheDir . '/' . md5($key) . '.cache';

    // إنشاء مجلد التخزين المؤقت إذا لم يكن موجودًا
    if (!file_exists($cacheDir)) {
        mkdir($cacheDir, 0777, true);
    }

    // التحقق مما إذا كان ملف التخزين المؤقت موجودًا وصالحًا
    if (file_exists($cacheFile) && (time() - filemtime($cacheFile) < $ttl)) {
        // استرجاع البيانات من التخزين المؤقت
        $data = file_get_contents($cacheFile);
        return unserialize($data);
    }

    // تنفيذ الاستعلام
    $result = $callback();

    // حفظ النتيجة في التخزين المؤقت
    file_put_contents($cacheFile, serialize($result));

    return $result;
}

// تعيين مسار الشعار
$company_logo = 'assets/images/backgrounds/background.png';

// التحقق من حالة تسجيل الدخول
if (!isset($_SESSION['logged_in']) || $_SESSION['logged_in'] !== true) {
    // إعادة توجيه المستخدم إلى صفحة تسجيل الدخول
    header('Location: auth/login.php');
    exit;
}

// الحصول على معلومات المستخدم
$user = [
    'id' => $_SESSION['user_id'] ?? 0,
    'username' => $_SESSION['username'] ?? '',
    'full_name' => $_SESSION['full_name'] ?? 'المستخدم',
    'email' => $_SESSION['email'] ?? '',
    'role' => $_SESSION['role'] ?? 'user',
    'branch_id' => $_SESSION['branch_id'] ?? null,
    'branch_name' => $_SESSION['branch_name'] ?? 'غير محدد',
    'branch_code' => $_SESSION['branch_code'] ?? ''
];

// تحديد ما إذا كان المستخدم مديرًا
$isAdmin = (isset($_SESSION['role']) && ($_SESSION['role'] === 'admin' || $_SESSION['role'] === 'manager'));

// بيانات الإحصائيات الافتراضية
$totalOrders = 0;
$pendingOrders = 0;
$inTransitOrders = 0;
$deliveredOrders = 0;
$cancelledOrders = 0;
$totalUsers = 0;
$totalBranches = 0;

// الحصول على البيانات الفعلية من قاعدة البيانات
try {
    // تحديد ما إذا كان يجب تصفية البيانات حسب الفرع
    $branchFilter = '';
    $branchParams = [];

    // إذا كان المستخدم ليس مديرًا، أو إذا كان مديرًا ولكن تم تحديد فرع
    if (!$isAdmin || ($isAdmin && isset($_SESSION['branch_id']) && $_SESSION['branch_id'])) {
        $branchFilter = ' AND branch_id = :branch_id';
        $branchParams[':branch_id'] = $_SESSION['branch_id'];
    }

    // استخدام التخزين المؤقت للإحصائيات
    $cacheKey = 'dashboard_order_stats_' . ($branchFilter ? $_SESSION['branch_id'] : 'all');
    $orderStats = queryCache($cacheKey, function() use ($conn, $branchFilter, $branchParams) {
        // استعلام واحد للحصول على إحصائيات الطلبات حسب الحالة
        $query = "
            SELECT
                COUNT(*) as total,
                SUM(CASE WHEN status = 'قيد المعالجة' THEN 1 ELSE 0 END) as pending,
                SUM(CASE WHEN status = 'قيد التوصيل' THEN 1 ELSE 0 END) as in_transit,
                SUM(CASE WHEN status = 'تم التسليم' THEN 1 ELSE 0 END) as delivered,
                SUM(CASE WHEN status = 'ملغى' THEN 1 ELSE 0 END) as cancelled
            FROM
                orders
            WHERE
                created_at >= DATE_SUB(NOW(), INTERVAL 90 DAY)
                $branchFilter
        ";

        $stmt = $conn->prepare($query);

        // تعيين المعلمات إذا كانت موجودة
        foreach ($branchParams as $param => $value) {
            $stmt->bindValue($param, $value);
        }

        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }, 300); // تخزين مؤقت لمدة 5 دقائق

    // تعيين قيم الإحصائيات من نتيجة الاستعلام
    $totalOrders = $orderStats['total'] ?? 0;
    $pendingOrders = $orderStats['pending'] ?? 0;
    $inTransitOrders = $orderStats['in_transit'] ?? 0;
    $deliveredOrders = $orderStats['delivered'] ?? 0;
    $cancelledOrders = $orderStats['cancelled'] ?? 0;

    // استخدام التخزين المؤقت لعدد المستخدمين
    $totalUsers = queryCache('dashboard_users_count', function() use ($conn) {
        $stmt = $conn->query("SELECT COUNT(*) FROM users");
        return $stmt->fetchColumn();
    }, 600); // تخزين مؤقت لمدة 10 دقائق

    // استخدام التخزين المؤقت لعدد الفروع
    $totalBranches = queryCache('dashboard_branches_count', function() use ($conn) {
        $stmt = $conn->query("SELECT COUNT(*) FROM branches");
        return $stmt->fetchColumn();
    }, 3600); // تخزين مؤقت لمدة ساعة (الفروع لا تتغير كثيرًا)

} catch (PDOException $e) {
    // في حالة حدوث خطأ، استخدم بيانات افتراضية
    error_log("خطأ في الاتصال بقاعدة البيانات: " . $e->getMessage());

    // تعيين بيانات افتراضية للإحصائيات
    $totalOrders = 0;
    $pendingOrders = 0;
    $inTransitOrders = 0;
    $deliveredOrders = 0;
    $cancelledOrders = 0;
    $totalUsers = 0;
    $totalBranches = 0;
}

// بيانات أفضل الموظفين أداءً
$topEmployees = [];

try {
    // استخدام التخزين المؤقت للحصول على أفضل الموظفين أداءً
    $cacheKey = 'dashboard_top_employees_' . ($branchFilter ? $_SESSION['branch_id'] : 'all');
    $topEmployees = queryCache($cacheKey, function() use ($conn, $branchFilter, $branchParams) {
        // استعلام محسن للحصول على أفضل الموظفين أداءً
        $query = "
            SELECT
                employee_name as name,
                COUNT(*) as orders,
                COALESCE(SUM(amount_paid), 0) as revenue
            FROM
                orders
            WHERE
                employee_name IS NOT NULL
                AND employee_name != ''
                AND created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY) -- فقط الطلبات من آخر 30 يوم
                $branchFilter
            GROUP BY
                employee_name
            ORDER BY
                orders DESC, revenue DESC
            LIMIT 4
        ";

        $stmt = $conn->prepare($query);

        // تعيين المعلمات إذا كانت موجودة
        foreach ($branchParams as $param => $value) {
            $stmt->bindValue($param, $value);
        }

        $stmt->execute();
        $result = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // إذا لم يتم العثور على بيانات، استخدم بيانات افتراضية
        if (empty($result)) {
            $result = [
                [
                    'name' => 'أحمد محمد',
                    'orders' => 0,
                    'revenue' => 0
                ],
                [
                    'name' => 'محمد علي',
                    'orders' => 0,
                    'revenue' => 0
                ],
                [
                    'name' => 'فاطمة أحمد',
                    'orders' => 0,
                    'revenue' => 0
                ],
                [
                    'name' => 'خالد عبدالله',
                    'orders' => 0,
                    'revenue' => 0
                ]
            ];
        }

        return $result;
    }, 600); // تخزين مؤقت لمدة 10 دقائق

} catch (PDOException $e) {
    error_log("خطأ في استعلام أفضل الموظفين: " . $e->getMessage());

    // استخدام بيانات افتراضية في حالة حدوث خطأ
    $topEmployees = [
        [
            'name' => 'أحمد محمد',
            'orders' => 0,
            'revenue' => 0
        ],
        [
            'name' => 'محمد علي',
            'orders' => 0,
            'revenue' => 0
        ],
        [
            'name' => 'فاطمة أحمد',
            'orders' => 0,
            'revenue' => 0
        ],
        [
            'name' => 'خالد عبدالله',
            'orders' => 0,
            'revenue' => 0
        ]
    ];
}

// بيانات أفضل الفروع أداءً
$topBranches = [];

try {
    // استخدام التخزين المؤقت للحصول على أفضل الفروع أداءً
    $cacheKey = 'dashboard_top_branches_' . ($branchFilter ? $_SESSION['branch_id'] : 'all');

    // إذا كان المستخدم ليس مديرًا، نعرض فقط الفرع الحالي
    if (!$isAdmin) {
        $branch_id = $_SESSION['branch_id'];
        $topBranches = queryCache($cacheKey, function() use ($conn, $branch_id) {
            // استعلام للحصول على معلومات الفرع الحالي فقط
            $stmt = $conn->prepare("
                SELECT
                    b.name,
                    COUNT(o.id) as orders,
                    COALESCE(SUM(o.amount_paid), 0) as revenue
                FROM
                    branches b
                LEFT JOIN
                    orders o ON o.branch_id = b.id AND o.created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
                WHERE
                    b.id = :branch_id
                GROUP BY
                    b.id, b.name
                LIMIT 1
            ");

            $stmt->bindValue(':branch_id', $branch_id);
            $stmt->execute();
            $result = $stmt->fetchAll(PDO::FETCH_ASSOC);

            return $result;
        }, 900); // تخزين مؤقت لمدة 15 دقيقة
    } else {
        // للمدير، نعرض جميع الفروع
        $topBranches = queryCache($cacheKey, function() use ($conn, $branchFilter, $branchParams) {
            // استعلام محسن للحصول على أفضل الفروع أداءً
            $query = "
                SELECT
                    b.name,
                    COUNT(o.id) as orders,
                    COALESCE(SUM(o.amount_paid), 0) as revenue
                FROM
                    branches b
                LEFT JOIN
                    orders o ON o.branch_id = b.id AND o.created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
                " . ($branchFilter ? "WHERE b.id = :branch_id" : "") . "
                GROUP BY
                    b.id, b.name
                ORDER BY
                    orders DESC, revenue DESC
                LIMIT 4
            ";

            $stmt = $conn->prepare($query);

            // تعيين المعلمات إذا كانت موجودة
            foreach ($branchParams as $param => $value) {
                $stmt->bindValue($param, $value);
            }

            $stmt->execute();
            $result = $stmt->fetchAll(PDO::FETCH_ASSOC);

            return $result;
        }, 900); // تخزين مؤقت لمدة 15 دقيقة
    }

    // التحقق من النتيجة
    if (empty($topBranches)) {
        // استخدام بيانات افتراضية في حالة عدم وجود بيانات
        if (!$isAdmin) {
            // للمستخدم العادي، نعرض فقط الفرع الحالي
            $topBranches = [
                [
                    'name' => $_SESSION['branch_name'] ?? 'الفرع الحالي',
                    'orders' => 0,
                    'revenue' => 0
                ]
            ];
        } else {
            // للمدير، نعرض الفروع الافتراضية
            $topBranches = [
                [
                    'name' => 'المكلا',
                    'orders' => 0,
                    'revenue' => 0
                ],
                [
                    'name' => 'صنعاء',
                    'orders' => 0,
                    'revenue' => 0
                ],
                [
                    'name' => 'عدن',
                    'orders' => 0,
                    'revenue' => 0
                ],
                [
                    'name' => 'الصين',
                    'orders' => 0,
                    'revenue' => 0
                ]
            ];
        }
    }

} catch (PDOException $e) {
    error_log("خطأ في استعلام أفضل الفروع: " . $e->getMessage());

    // استخدام بيانات افتراضية في حالة حدوث خطأ
    $topBranches = [
        [
            'name' => 'المكلا',
            'orders' => 0,
            'revenue' => 0
        ],
        [
            'name' => 'صنعاء',
            'orders' => 0,
            'revenue' => 0
        ],
        [
            'name' => 'عدن',
            'orders' => 0,
            'revenue' => 0
        ],
        [
            'name' => 'الصين',
            'orders' => 0,
            'revenue' => 0
        ]
    ];
}

// الحصول على الطلبات الأخيرة من قاعدة البيانات
$recentOrders = [];

try {
    // استخدام التخزين المؤقت للحصول على الطلبات الأخيرة
    // نستخدم وقت تخزين مؤقت أقل (60 ثانية) لأن الطلبات الأخيرة تتغير بشكل متكرر
    $cacheKey = 'dashboard_recent_orders_' . ($branchFilter ? $_SESSION['branch_id'] : 'all');
    $recentOrders = queryCache($cacheKey, function() use ($conn, $branchFilter, $branchParams) {
        // استعلام محسن للحصول على آخر الطلبات مع اسم الفرع
        $query = "
            SELECT
                o.id,
                o.order_number,
                o.created_at,
                o.customer_name,
                b.name as branch_name,
                o.status
            FROM
                orders o
            LEFT JOIN
                branches b ON o.branch_id = b.id
            WHERE 1=1
                $branchFilter
            ORDER BY
                o.created_at DESC
            LIMIT 5
        ";

        $stmt = $conn->prepare($query);

        // تعيين المعلمات إذا كانت موجودة
        foreach ($branchParams as $param => $value) {
            $stmt->bindValue($param, $value);
        }

        $stmt->execute();
        $result = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // معالجة البيانات قبل إرجاعها
        foreach ($result as &$order) {
            // التأكد من أن اسم الفرع موجود
            if (empty($order['branch_name'])) {
                $order['branch_name'] = 'غير محدد';
            }

            // تنسيق التاريخ
            if (!empty($order['created_at'])) {
                $date = new DateTime($order['created_at']);
                $order['created_at'] = $date->format('Y-m-d H:i:s');
            }
        }

        return $result;
    }, 60); // تخزين مؤقت لمدة دقيقة واحدة فقط

    // إذا لم يتم العثور على طلبات، استخدم مصفوفة فارغة
    if (empty($recentOrders)) {
        $recentOrders = [];
    }
} catch (PDOException $e) {
    error_log("خطأ في استعلام الطلبات الأخيرة: " . $e->getMessage());

    // استخدام بيانات افتراضية في حالة حدوث خطأ
    $recentOrders = [
        [
            'id' => 0,
            'order_number' => '-',
            'created_at' => date('Y-m-d H:i:s'),
            'customer_name' => '-',
            'branch_name' => '-',
            'status' => '-'
        ]
    ];
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">

<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>نظام طلبات الشحن | لوحة التحكم</title>
  <script src="https://cdn.tailwindcss.com"></script>
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
  <link rel="stylesheet" href="assets/css/unified-design.css">
  <style>
    /* تحسين البطاقات الإحصائية */
    .stat-card {
      background: white;
      border-radius: var(--border-radius);
      box-shadow: var(--shadow);
      transition: var(--transition);
      position: relative;
      overflow: hidden;
      border: 1px solid var(--gray-200);
    }

    .stat-card::before {
      content: '';
      position: absolute;
      top: 0;
      right: 0;
      width: 100%;
      height: 4px;
      background: linear-gradient(90deg, var(--primary) 0%, var(--primary-light) 100%);
      opacity: 0;
      transition: var(--transition);
    }

    .stat-card:hover {
      transform: translateY(-5px);
      box-shadow: var(--shadow-md);
    }

    .stat-card:hover::before {
      opacity: 1;
    }

    .stat-card .icon-container {
      background: linear-gradient(135deg, var(--primary-light) 0%, var(--primary) 100%);
      color: white;
      border-radius: 50%;
      width: 3rem;
      height: 3rem;
      display: flex;
      align-items: center;
      justify-content: center;
      box-shadow: var(--shadow);
      transition: var(--transition);
    }

    .stat-card:hover .icon-container {
      transform: scale(1.1) rotate(10deg);
    }

    .stat-card .stat-value {
      font-size: 2rem;
      font-weight: 700;
      color: var(--primary-dark);
      margin-top: 0.5rem;
      margin-bottom: 0.25rem;
    }

    .stat-card .stat-label {
      color: var(--gray-600);
      font-weight: 500;
    }

    /* تنسيق القائمة المنسدلة */
    .dropdown-menu {
      max-height: 0;
      overflow: hidden;
      transition: max-height 0.3s ease;
      margin-right: 2.5rem;
    }

    .dropdown-menu.active {
      max-height: 200px;
    }

    .dropdown-toggle {
      position: relative;
    }

    .dropdown-toggle::after {
      content: '\f107';
      font-family: 'Font Awesome 5 Free';
      font-weight: 900;
      position: absolute;
      left: 1rem;
      top: 50%;
      transform: translateY(-50%);
      transition: transform 0.3s ease;
    }

    .dropdown-toggle.active::after {
      transform: translateY(-50%) rotate(180deg);
    }

    /* تأثيرات إضافية */
    @keyframes logoReveal {
      0% {
        opacity: 0;
        transform: scale(0.8) translateY(20px);
        filter: blur(10px) brightness(0.5);
      }
      60% {
        opacity: 1;
        transform: scale(1.1) translateY(-5px);
        filter: blur(0) brightness(1.3);
      }
      100% {
        opacity: 1;
        transform: scale(1) translateY(0);
        filter: blur(0) brightness(1);
      }
    }

    .logo-reveal {
      animation: logoReveal 1.5s ease-out forwards;
    }

    @keyframes pulse {
      0% {
        box-shadow: 0 0 0 0 rgba(79, 70, 229, 0.4);
      }
      70% {
        box-shadow: 0 0 0 10px rgba(79, 70, 229, 0);
      }
      100% {
        box-shadow: 0 0 0 0 rgba(79, 70, 229, 0);
      }
    }

    @keyframes spin-slow {
      from {
        transform: rotate(0deg);
      }
      to {
        transform: rotate(360deg);
      }
    }

    .animate-spin-slow {
      animation: spin-slow 3s linear infinite;
    }

    .btn-secondary:hover {
      transform: translateY(-2px);
      box-shadow: 0 10px 15px -3px rgba(245, 158, 11, 0.4);
    }

    /* تحسين الشريط العلوي */
    .top-bar {
      background-color: white;
      box-shadow: var(--shadow-sm);
      border-bottom: 1px solid var(--gray-200);
      padding: 1rem;
      display: flex;
      align-items: center;
      justify-content: space-between;
    }

    .user-profile {
      display: flex;
      align-items: center;
      gap: 0.75rem;
      padding: 0.5rem 1rem;
      border-radius: var(--border-radius);
      transition: var(--transition);
    }

    .user-profile:hover {
      background-color: var(--gray-100);
    }

    .user-avatar {
      width: 2.5rem;
      height: 2.5rem;
      border-radius: 50%;
      background: linear-gradient(135deg, var(--primary-light) 0%, var(--primary) 100%);
      color: white;
      display: flex;
      align-items: center;
      justify-content: center;
      font-weight: 600;
      box-shadow: var(--shadow);
    }

    /* تحسين الرسوم البيانية */
    .chart-container {
      background-color: white;
      border-radius: var(--border-radius);
      box-shadow: var(--shadow);
      padding: 1.5rem;
      border: 1px solid var(--gray-200);
    }

    .chart-title {
      font-size: 1.25rem;
      font-weight: 600;
      color: var(--gray-800);
      margin-bottom: 1.5rem;
      display: flex;
      align-items: center;
      gap: 0.5rem;
    }

    .chart-title i {
      color: var(--primary);
    }

    /* تحسين الحالات */
    .status-badge {
      padding: 0.25rem 0.75rem;
      border-radius: 9999px;
      font-size: 0.75rem;
      font-weight: 600;
      display: inline-flex;
      align-items: center;
      gap: 0.25rem;
    }

    .status-pending {
      background-color: rgba(245, 158, 11, 0.1);
      color: var(--warning);
    }

    .status-in-transit {
      background-color: rgba(59, 130, 246, 0.1);
      color: var(--info);
    }

    .status-delivered {
      background-color: rgba(16, 185, 129, 0.1);
      color: var(--success);
    }

    .status-cancelled {
      background-color: rgba(239, 68, 68, 0.1);
      color: var(--danger);
    }

    /* تحسين التجاوب مع الشاشات المختلفة */
    @media (max-width: 768px) {
      aside {
        position: fixed;
        right: -280px;
        top: 0;
        height: 100vh;
        z-index: 50;
        width: 280px;
      }

      aside.active {
        right: 0;
      }

      .overlay {
        position: fixed;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background-color: rgba(0, 0, 0, 0.5);
        z-index: 40;
        opacity: 0;
        pointer-events: none;
        transition: var(--transition);
      }

      .overlay.active {
        opacity: 1;
        pointer-events: auto;
      }

      .main-content {
        margin-right: 0 !important;
      }
    }
  </style>
</head>

<body>
  <!-- طبقة التراكب للأجهزة المحمولة -->
  <div id="overlay" class="overlay"></div>

  <div class="flex min-h-screen">
    <!-- القائمة الجانبية -->
    <aside class="w-64 hidden md:block">
      <div class="p-6">
        <div class="logo-container flex items-center justify-center mb-8 p-2">
          <img src="<?php echo $company_logo; ?>" alt="شعار النظام" class="h-40 w-auto object-contain logo-reveal">
        </div>

        <div class="user-info mb-8">
          <div class="flex items-center space-x-4 space-x-reverse mb-3">
            <div class="user-avatar">
              <?php echo mb_substr($user['full_name'], 0, 1, 'UTF-8'); ?>
            </div>
            <div>
              <h3 class="text-white font-semibold"><?php echo htmlspecialchars($user['full_name']); ?></h3>
              <p class="text-indigo-200 text-sm"><?php echo $isAdmin ? 'مدير النظام' : 'مستخدم'; ?></p>
              <?php if (isset($_SESSION['branch_name']) && !empty($_SESSION['branch_name'])): ?>
              <div class="mt-1 flex items-center">
                <span class="bg-indigo-900/30 text-indigo-100 text-xs px-2 py-0.5 rounded-full flex items-center">
                  <i class="fas fa-building ml-1 text-xs"></i>
                  <?php echo htmlspecialchars($_SESSION['branch_name']); ?>
                </span>
              </div>
              <?php endif; ?>
            </div>
          </div>
          <div class="bg-white/10 h-0.5 rounded-full mb-3"></div>
        </div>

        <nav>
          <ul class="space-y-3">
            <!-- لوحة التحكم -->
            <li>
              <a href="unified-dashboard.php" class="flex items-center p-3 rounded-lg text-white active">
                <i class="fas fa-tachometer-alt ml-3 text-lg"></i>
                <span>لوحة التحكم</span>
              </a>
            </li>

            <!-- إنشاء طلب جديد -->
            <li>
              <a href="create-order-unified.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white">
                <i class="fas fa-plus-circle ml-3 text-lg"></i>
                <span>إنشاء طلب جديد</span>
              </a>
            </li>

            <!-- قائمة الطلبات -->
            <li>
              <a href="orders.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white">
                <i class="fas fa-box ml-3 text-lg"></i>
                <span>قائمة الطلبات</span>
              </a>
            </li>

            <?php if ($isAdmin): ?>
            <!-- قائمة التقارير المنسدلة -->
            <li>
              <a href="#" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white dropdown-toggle" id="reportsDropdown">
                <i class="fas fa-chart-bar ml-3 text-lg"></i>
                <span>التقارير</span>
              </a>
              <ul class="dropdown-menu" id="reportsMenu">
                <li>
                  <a href="weekly-reports.php" class="flex items-center p-2 rounded-lg text-white/80 hover:text-white">
                    <i class="fas fa-calendar-week ml-2 text-sm"></i>
                    <span>التقارير الأسبوعية</span>
                  </a>
                </li>
                <li>
                  <a href="monthly-reports.php" class="flex items-center p-2 rounded-lg text-white/80 hover:text-white">
                    <i class="fas fa-calendar-alt ml-2 text-sm"></i>
                    <span>التقارير الشهرية</span>
                  </a>
                </li>
              </ul>
            </li>

            <!-- إدارة المستخدمين -->
            <li>
              <a href="users.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white">
                <i class="fas fa-users ml-3 text-lg"></i>
                <span>إدارة المستخدمين</span>
              </a>
            </li>

            <!-- الفروع -->
            <li>
              <a href="branches.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white">
                <i class="fas fa-code-branch ml-3 text-lg"></i>
                <span>الفروع</span>
              </a>
            </li>

            <!-- إعدادات النظام -->
            <li>
              <a href="settings.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white">
                <i class="fas fa-cog ml-3 text-lg"></i>
                <span>إعدادات النظام</span>
              </a>
            </li>
            <?php endif; ?>
          </ul>

          <div class="pt-6 mt-6 border-t border-white/10">
            <a href="auth/logout.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white">
              <i class="fas fa-sign-out-alt ml-3 text-lg"></i>
              <span>تسجيل الخروج</span>
            </a>
          </div>
        </nav>
      </div>
    </aside>

    <!-- المحتوى الرئيسي -->
    <main class="flex-1">
      <!-- الشريط العلوي -->
      <header class="top-bar">
        <div class="flex items-center">
          <button id="toggleSidebar" class="md:hidden text-gray-600 hover:text-primary p-2 rounded-lg">
            <i class="fas fa-bars text-xl"></i>
          </button>

          <!-- شعار النظام في الشريط العلوي (يظهر فقط في الأجهزة المحمولة) -->
          <div class="flex items-center md:hidden ml-3">
            <img src="<?php echo $company_logo; ?>" alt="شعار النظام" class="h-16 w-auto object-contain logo-reveal">
          </div>

          <div class="relative ml-4 hidden md:block">
            <input type="text" placeholder="البحث..." class="bg-gray-100 rounded-full py-2 px-4 pl-10 w-64 focus:outline-none focus:ring-2 focus:ring-primary/30 focus:bg-white transition-all">
            <i class="fas fa-search absolute right-3 top-3 text-gray-400"></i>
          </div>
        </div>

        <!-- معلومات المستخدم والإشعارات -->
        <div class="flex items-center space-x-4 space-x-reverse">
          <?php if ($isAdmin): ?>
          <div class="relative">
            <button class="relative p-2 rounded-full hover:bg-gray-100 transition-all">
              <i class="fas fa-bell text-gray-600"></i>
              <span class="absolute -top-1 -right-1 bg-red-500 text-white text-xs rounded-full h-5 w-5 flex items-center justify-center animate-pulse">3</span>
            </button>
          </div>
          <?php endif; ?>

          <div class="relative">
            <button class="relative p-2 rounded-full hover:bg-gray-100 transition-all">
              <i class="fas fa-envelope text-gray-600"></i>
              <?php if ($isAdmin): ?>
              <span class="absolute -top-1 -right-1 bg-primary text-white text-xs rounded-full h-5 w-5 flex items-center justify-center">2</span>
              <?php endif; ?>
            </button>
          </div>

          <div class="border-r border-gray-300 h-8 mx-2"></div>

          <div class="user-profile">
            <div class="user-avatar">
              <?php echo mb_substr($user['full_name'], 0, 1, 'UTF-8'); ?>
            </div>
            <div>
              <p class="font-medium text-sm"><?php echo htmlspecialchars($user['full_name']); ?></p>
              <div class="flex items-center">
                <p class="text-xs text-gray-500"><?php echo $isAdmin ? 'مدير النظام' : 'مستخدم'; ?></p>
                <?php if (isset($_SESSION['branch_name']) && !empty($_SESSION['branch_name'])): ?>
                <span class="mr-2 bg-blue-100 text-blue-800 text-xs px-1.5 py-0.5 rounded-full">
                  <?php echo htmlspecialchars($_SESSION['branch_name']); ?>
                </span>
                <?php endif; ?>
              </div>
            </div>
          </div>
        </div>
      </header>

      <!-- محتوى الصفحة -->
      <div class="p-6">
        <!-- رأس الصفحة -->
        <div class="flex flex-col md:flex-row md:items-center md:justify-between mb-8">
          <div class="animate-fadeIn" style="animation-delay: 0.1s;">
            <h1 class="text-3xl font-bold text-gray-800 flex items-center">
              <span class="bg-gradient-to-r from-primary to-primary-light bg-clip-text text-transparent">
                <?php echo $isAdmin ? 'لوحة تحكم الإدارة' : 'لوحة التحكم'; ?>
              </span>
            </h1>
            <div class="flex flex-col md:flex-row md:items-center mt-2">
              <p class="text-gray-500 flex items-center">
                <i class="fas fa-calendar-alt ml-2 text-primary"></i>
                <span><?php echo date('l، d F Y'); ?></span>
              </p>
              <?php if (isset($_SESSION['branch_name']) && !empty($_SESSION['branch_name'])): ?>
              <div class="md:mr-4 mt-2 md:mt-0 flex items-center">
                <span class="bg-blue-100 text-blue-800 text-xs font-medium px-3 py-1 rounded-full flex items-center">
                  <i class="fas fa-building ml-1"></i>
                  <?php echo htmlspecialchars($_SESSION['branch_name']); ?>
                  <?php if (isset($_SESSION['branch_code']) && !empty($_SESSION['branch_code'])): ?>
                  <span class="mr-1 text-blue-600">(<?php echo htmlspecialchars($_SESSION['branch_code']); ?>)</span>
                  <?php endif; ?>
                </span>
              </div>
              <?php endif; ?>
            </div>
          </div>

          <div class="mt-4 md:mt-0 animate-fadeIn" style="animation-delay: 0.2s;">
            <?php if ($isAdmin): ?>
            <div class="flex items-center space-x-4 space-x-reverse">
              <div class="text-sm bg-blue-50 text-blue-800 px-3 py-2 rounded-full flex items-center">
                <i class="fas fa-sync-alt ml-2 animate-spin-slow"></i>
                <span>آخر تحديث: <?php echo date('H:i'); ?></span>
              </div>
              <a href="reports.php" class="btn btn-primary">
                <i class="fas fa-chart-bar ml-2"></i>
                <span>عرض التقارير</span>
              </a>
            </div>
            <?php else: ?>
            <a href="create-order-unified.php" class="btn btn-primary">
              <i class="fas fa-plus-circle ml-2"></i>
              <span>إنشاء طلب جديد</span>
            </a>
            <?php endif; ?>
          </div>
        </div>

        <!-- كروت الإحصائيات -->
        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
          <div class="stat-card p-6 animate-fadeIn" style="animation-delay: 0.1s;">
            <div class="flex flex-col">
              <div class="icon-container mb-4">
                <i class="fas fa-box text-xl"></i>
              </div>
              <p class="stat-label">إجمالي الطلبات</p>
              <p class="stat-value"><?php echo $totalOrders; ?></p>
              <div class="flex items-center mt-2 text-sm text-green-600">
                <i class="fas fa-arrow-up ml-1"></i>
                <span>12% زيادة</span>
                <span class="text-gray-400 mr-2 text-xs">مقارنة بالشهر الماضي</span>
              </div>
            </div>
          </div>

          <div class="stat-card p-6 animate-fadeIn" style="animation-delay: 0.2s;">
            <div class="flex flex-col">
              <div class="icon-container mb-4" style="background: linear-gradient(135deg, #fbbf24 0%, #d97706 100%);">
                <i class="fas fa-clock text-xl"></i>
              </div>
              <p class="stat-label">قيد المعالجة</p>
              <p class="stat-value" style="color: var(--warning);"><?php echo $pendingOrders; ?></p>
              <div class="flex items-center mt-2 text-sm text-gray-600">
                <i class="fas fa-minus ml-1"></i>
                <span>3% تغيير</span>
                <span class="text-gray-400 mr-2 text-xs">مقارنة بالأسبوع الماضي</span>
              </div>
            </div>
          </div>

          <div class="stat-card p-6 animate-fadeIn" style="animation-delay: 0.3s;">
            <div class="flex flex-col">
              <div class="icon-container mb-4" style="background: linear-gradient(135deg, #60a5fa 0%, #3b82f6 100%);">
                <i class="fas fa-truck text-xl"></i>
              </div>
              <p class="stat-label">قيد التوصيل</p>
              <p class="stat-value" style="color: var(--info);"><?php echo $inTransitOrders; ?></p>
              <div class="flex items-center mt-2 text-sm text-blue-600">
                <i class="fas fa-arrow-up ml-1"></i>
                <span>8% زيادة</span>
                <span class="text-gray-400 mr-2 text-xs">مقارنة بالأسبوع الماضي</span>
              </div>
            </div>
          </div>

          <div class="stat-card p-6 animate-fadeIn" style="animation-delay: 0.4s;">
            <div class="flex flex-col">
              <div class="icon-container mb-4" style="background: linear-gradient(135deg, #34d399 0%, #10b981 100%);">
                <i class="fas fa-check text-xl"></i>
              </div>
              <p class="stat-label">تم التسليم</p>
              <p class="stat-value" style="color: var(--success);"><?php echo $deliveredOrders; ?></p>
              <div class="flex items-center mt-2 text-sm text-green-600">
                <i class="fas fa-arrow-up ml-1"></i>
                <span>18% زيادة</span>
                <span class="text-gray-400 mr-2 text-xs">مقارنة بالشهر الماضي</span>
              </div>
            </div>
          </div>
        </div>

        <?php if ($isAdmin): ?>
        <!-- إحصائيات إضافية للمدير -->
        <div class="grid grid-cols-1 md:grid-cols-2 gap-6 mb-8">
          <div class="stat-card p-6 animate-fadeIn" style="animation-delay: 0.5s;">
            <div class="flex items-center">
              <div class="icon-container mr-4" style="background: linear-gradient(135deg, #c084fc 0%, #8b5cf6 100%);">
                <i class="fas fa-users text-xl"></i>
              </div>
              <div>
                <p class="stat-label">إجمالي المستخدمين</p>
                <p class="stat-value" style="color: #8b5cf6;"><?php echo $totalUsers; ?></p>
                <div class="flex items-center mt-2 text-sm text-purple-600">
                  <i class="fas fa-arrow-up ml-1"></i>
                  <span>5 مستخدمين جدد</span>
                  <span class="text-gray-400 mr-2 text-xs">هذا الشهر</span>
                </div>
              </div>
            </div>
          </div>

          <div class="stat-card p-6 animate-fadeIn" style="animation-delay: 0.6s;">
            <div class="flex items-center">
              <div class="icon-container mr-4" style="background: linear-gradient(135deg, #f472b6 0%, #db2777 100%);">
                <i class="fas fa-code-branch text-xl"></i>
              </div>
              <div>
                <p class="stat-label">إجمالي الفروع</p>
                <p class="stat-value" style="color: #db2777;"><?php echo $totalBranches; ?></p>
                <div class="flex items-center mt-2 text-sm text-pink-600">
                  <i class="fas fa-plus ml-1"></i>
                  <span>فرع جديد</span>
                  <span class="text-gray-400 mr-2 text-xs">تمت إضافته مؤخرًا</span>
                </div>
              </div>
            </div>
          </div>
        </div>
        <?php endif; ?>

        <!-- آخر الطلبات -->
        <div class="chart-container mb-8 animate-fadeIn" style="animation-delay: 0.7s;">
          <div class="flex justify-between items-center mb-6">
            <h2 class="chart-title">
              <i class="fas fa-list-ul"></i>
              <span>آخر الطلبات</span>
            </h2>
            <a href="orders.php" class="text-primary hover:text-primary-dark text-sm font-medium transition-all flex items-center">
              عرض جميع الطلبات
              <i class="fas fa-arrow-left mr-2 transition-transform group-hover:translate-x-1"></i>
            </a>
          </div>

          <div class="overflow-x-auto">
            <table class="min-w-full data-table">
              <thead>
                <tr>
                  <th class="text-right">رقم الطلب</th>
                  <th class="text-right">التاريخ</th>
                  <?php if ($isAdmin): ?>
                  <th class="text-right">العميل</th>
                  <?php endif; ?>
                  <th class="text-right">الفرع</th>
                  <th class="text-right">الحالة</th>
                  <th class="text-right">الإجراءات</th>
                </tr>
              </thead>
              <tbody>
                <?php if (empty($recentOrders)): ?>
                <tr>
                  <td colspan="<?php echo $isAdmin ? '6' : '5'; ?>" class="text-center py-8">
                    <div class="flex flex-col items-center justify-center text-gray-500">
                      <i class="fas fa-inbox text-4xl mb-3 text-gray-300"></i>
                      <p>لا توجد طلبات حاليًا</p>
                      <a href="create-order-unified.php" class="mt-3 text-primary hover:text-primary-dark text-sm">إنشاء طلب جديد</a>
                    </div>
                  </td>
                </tr>
                <?php else: ?>
                  <?php foreach ($recentOrders as $order): ?>
                  <tr>
                    <td class="font-medium"><?php echo htmlspecialchars($order['order_number'] ?? '-'); ?></td>
                    <td><?php echo htmlspecialchars(date('Y-m-d', strtotime($order['created_at'] ?? 'now'))); ?></td>
                    <?php if ($isAdmin): ?>
                    <td>
                      <div class="flex items-center">
                        <div class="w-8 h-8 rounded-full bg-gray-200 flex items-center justify-center text-gray-600 ml-2">
                          <?php echo mb_substr($order['customer_name'] ?? 'م', 0, 1, 'UTF-8'); ?>
                        </div>
                        <span><?php echo htmlspecialchars($order['customer_name'] ?? '-'); ?></span>
                      </div>
                    </td>
                    <?php endif; ?>
                    <td><?php echo htmlspecialchars($order['branch_name'] ?? '-'); ?></td>
                    <td>
                      <?php
                      $statusClass = '';
                      $statusIcon = '';
                      $status = $order['status'] ?? '';

                      switch ($status) {
                          case 'قيد المعالجة':
                              $statusClass = 'status-pending';
                              $statusIcon = 'clock';
                              break;
                          case 'قيد التوصيل':
                              $statusClass = 'status-in-transit';
                              $statusIcon = 'truck';
                              break;
                          case 'تم التسليم':
                              $statusClass = 'status-delivered';
                              $statusIcon = 'check-circle';
                              break;
                          case 'ملغى':
                              $statusClass = 'status-cancelled';
                              $statusIcon = 'times-circle';
                              break;
                          default:
                              $statusClass = 'bg-gray-100 text-gray-800';
                              $statusIcon = 'circle';
                      }
                      ?>
                      <span class="status-badge <?php echo $statusClass; ?>">
                        <i class="fas fa-<?php echo $statusIcon; ?>"></i>
                        <?php echo htmlspecialchars($status); ?>
                      </span>
                    </td>
                    <td>
                      <div class="flex items-center space-x-2 space-x-reverse">
                        <a href="view-order.php?id=<?php echo $order['id'] ?? 0; ?>" class="p-2 text-primary hover:bg-primary/10 rounded-full transition-all" title="عرض التفاصيل">
                          <i class="fas fa-eye"></i>
                        </a>
                        <?php if ($isAdmin || $order['status'] === 'قيد المعالجة'): ?>
                        <a href="edit-order.php?id=<?php echo $order['id'] ?? 0; ?>" class="p-2 text-info hover:bg-info/10 rounded-full transition-all" title="تعديل الطلب">
                          <i class="fas fa-edit"></i>
                        </a>
                        <?php endif; ?>
                        <a href="#" class="p-2 text-gray-400 hover:bg-gray-100 rounded-full transition-all" title="المزيد من الخيارات">
                          <i class="fas fa-ellipsis-v"></i>
                        </a>
                      </div>
                    </td>
                  </tr>
                  <?php endforeach; ?>
                <?php endif; ?>
              </tbody>
            </table>
          </div>
        </div>

        <?php if ($isAdmin): ?>
        <!-- الرسوم البيانية للمدير -->
        <div class="grid grid-cols-1 md:grid-cols-2 gap-6 mb-8">
          <div class="chart-container animate-fadeIn" style="animation-delay: 0.8s;">
            <h3 class="chart-title">
              <i class="fas fa-chart-pie"></i>
              <span>توزيع حالات الطلبات</span>
            </h3>
            <div class="h-64 flex items-center justify-center">
              <div id="statusChart" class="w-full h-full"></div>
            </div>
          </div>

          <div class="chart-container animate-fadeIn" style="animation-delay: 0.9s;">
            <h3 class="chart-title">
              <i class="fas fa-chart-bar"></i>
              <span>توزيع الطلبات حسب الفروع</span>
            </h3>
            <div class="h-64 flex items-center justify-center">
              <div id="branchesChart" class="w-full h-full"></div>
            </div>
          </div>
        </div>

        <!-- ملخص الأداء -->
        <div class="chart-container mb-8 animate-fadeIn" style="animation-delay: 1s;">
          <h3 class="chart-title">
            <i class="fas fa-tachometer-alt"></i>
            <span>ملخص الأداء</span>
          </h3>

          <div class="grid grid-cols-1 md:grid-cols-3 gap-6 mb-6">
            <div class="bg-gray-50 p-4 rounded-lg">
              <div class="flex items-center justify-between mb-2">
                <h4 class="text-sm font-medium text-gray-600">معدل إتمام الطلبات</h4>
                <span class="text-sm font-bold text-green-600">92%</span>
              </div>
              <div class="progress-bar">
                <div class="progress-bar-fill" style="width: 92%;"></div>
              </div>
            </div>

            <div class="bg-gray-50 p-4 rounded-lg">
              <div class="flex items-center justify-between mb-2">
                <h4 class="text-sm font-medium text-gray-600">معدل رضا العملاء</h4>
                <span class="text-sm font-bold text-green-600">88%</span>
              </div>
              <div class="progress-bar">
                <div class="progress-bar-fill" style="width: 88%; background: linear-gradient(90deg, #10b981 0%, #34d399 100%);"></div>
              </div>
            </div>

            <div class="bg-gray-50 p-4 rounded-lg">
              <div class="flex items-center justify-between mb-2">
                <h4 class="text-sm font-medium text-gray-600">معدل الطلبات الملغاة</h4>
                <span class="text-sm font-bold text-red-600">7%</span>
              </div>
              <div class="progress-bar">
                <div class="progress-bar-fill" style="width: 7%; background: linear-gradient(90deg, #ef4444 0%, #f87171 100%);"></div>
              </div>
            </div>
          </div>

          <div class="flex items-center justify-between text-sm">
            <div class="flex items-center text-gray-500">
              <i class="fas fa-info-circle ml-1 text-primary"></i>
              <span>تم تحديث البيانات في <?php echo date('H:i'); ?></span>
            </div>
            <a href="reports.php" class="text-primary hover:text-primary-dark font-medium flex items-center">
              <span>تقرير مفصل</span>
              <i class="fas fa-arrow-left mr-1"></i>
            </a>
          </div>
        </div>
        <?php endif; ?>

        <!-- أفضل الفروع أداءً -->
        <div class="chart-container mb-8 animate-fadeIn" style="animation-delay: 1.1s;">
          <h3 class="chart-title">
            <i class="fas fa-building"></i>
            <span>أفضل الفروع أداءً</span>
          </h3>
          <div class="overflow-x-auto">
            <table class="w-full border-collapse">
              <thead>
                <tr>
                  <th class="text-right py-3 px-4 bg-gray-50 font-semibold text-gray-700 border-b border-gray-200">الفرع</th>
                  <th class="text-right py-3 px-4 bg-gray-50 font-semibold text-gray-700 border-b border-gray-200">عدد الطلبات</th>
                  <th class="text-right py-3 px-4 bg-gray-50 font-semibold text-gray-700 border-b border-gray-200">الإيرادات</th>
                  <th class="text-right py-3 px-4 bg-gray-50 font-semibold text-gray-700 border-b border-gray-200">النسبة</th>
                </tr>
              </thead>
              <tbody>
                <?php foreach ($topBranches as $branch): ?>
                <tr class="hover:bg-gray-50 transition-colors">
                  <td class="py-3 px-4 border-b border-gray-200">
                    <div class="flex items-center">
                      <div class="w-8 h-8 rounded-full bg-info-100 flex items-center justify-center text-info font-semibold ml-2">
                        <i class="fas fa-store-alt text-sm"></i>
                      </div>
                      <span class="font-medium"><?php echo $branch['name']; ?></span>
                    </div>
                  </td>
                  <td class="py-3 px-4 border-b border-gray-200"><?php echo $branch['orders']; ?> طلب</td>
                  <td class="py-3 px-4 border-b border-gray-200"><?php echo number_format($branch['revenue'], 2); ?> ريال</td>
                  <td class="py-3 px-4 border-b border-gray-200">
                    <div class="flex items-center">
                      <div class="w-full bg-gray-200 rounded-full h-2.5 ml-2">
                        <div class="bg-info h-2.5 rounded-full" style="width: <?php echo ($totalOrders > 0) ? ($branch['orders'] / $totalOrders) * 100 : 0; ?>%"></div>
                      </div>
                      <span><?php echo ($totalOrders > 0) ? number_format(($branch['orders'] / $totalOrders) * 100, 1) : '0.0'; ?>%</span>
                    </div>
                  </td>
                </tr>
                <?php endforeach; ?>
              </tbody>
            </table>
          </div>
        </div>

        <!-- أفضل الموظفين أداءً -->
        <div class="chart-container mb-8 animate-fadeIn" style="animation-delay: 1.2s;">
          <h3 class="chart-title">
            <i class="fas fa-user-tie"></i>
            <span>أفضل الموظفين أداءً</span>
          </h3>
          <div class="overflow-x-auto">
            <table class="w-full border-collapse">
              <thead>
                <tr>
                  <th class="text-right py-3 px-4 bg-gray-50 font-semibold text-gray-700 border-b border-gray-200">الموظف</th>
                  <th class="text-right py-3 px-4 bg-gray-50 font-semibold text-gray-700 border-b border-gray-200">عدد الطلبات</th>
                  <th class="text-right py-3 px-4 bg-gray-50 font-semibold text-gray-700 border-b border-gray-200">الإيرادات</th>
                  <th class="text-right py-3 px-4 bg-gray-50 font-semibold text-gray-700 border-b border-gray-200">النسبة</th>
                </tr>
              </thead>
              <tbody>
                <?php foreach ($topEmployees as $employee): ?>
                <tr class="hover:bg-gray-50 transition-colors">
                  <td class="py-3 px-4 border-b border-gray-200">
                    <div class="flex items-center">
                      <div class="w-8 h-8 rounded-full bg-primary-100 flex items-center justify-center text-primary font-semibold ml-2">
                        <?php echo mb_substr($employee['name'], 0, 1, 'UTF-8'); ?>
                      </div>
                      <span class="font-medium"><?php echo $employee['name']; ?></span>
                    </div>
                  </td>
                  <td class="py-3 px-4 border-b border-gray-200"><?php echo $employee['orders']; ?> طلب</td>
                  <td class="py-3 px-4 border-b border-gray-200"><?php echo number_format($employee['revenue'], 2); ?> ريال</td>
                  <td class="py-3 px-4 border-b border-gray-200">
                    <div class="flex items-center">
                      <div class="w-full bg-gray-200 rounded-full h-2.5 ml-2">
                        <div class="bg-primary h-2.5 rounded-full" style="width: <?php echo ($totalOrders > 0) ? ($employee['orders'] / $totalOrders) * 100 : 0; ?>%"></div>
                      </div>
                      <span><?php echo ($totalOrders > 0) ? number_format(($employee['orders'] / $totalOrders) * 100, 1) : '0.0'; ?>%</span>
                    </div>
                  </td>
                </tr>
                <?php endforeach; ?>
              </tbody>
            </table>
          </div>
        </div>

        <!-- ملخص النشاط الأخير -->
        <div class="chart-container mb-8 animate-fadeIn" style="animation-delay: 1.2s;">
          <h3 class="chart-title">
            <i class="fas fa-history"></i>
            <span>النشاط الأخير</span>
          </h3>

          <div class="space-y-4">
            <div class="flex items-start">
              <div class="w-10 h-10 rounded-full bg-blue-100 flex items-center justify-center text-blue-600 ml-4">
                <i class="fas fa-plus"></i>
              </div>
              <div class="flex-1">
                <p class="font-medium">تم إنشاء طلب جديد <a href="#" class="text-primary hover:underline">ORD-001</a></p>
                <p class="text-sm text-gray-500 mt-1">منذ 30 دقيقة</p>
              </div>
            </div>

            <div class="flex items-start">
              <div class="w-10 h-10 rounded-full bg-green-100 flex items-center justify-center text-green-600 ml-4">
                <i class="fas fa-check"></i>
              </div>
              <div class="flex-1">
                <p class="font-medium">تم تسليم طلب <a href="#" class="text-primary hover:underline">ORD-002</a> بنجاح</p>
                <p class="text-sm text-gray-500 mt-1">منذ ساعتين</p>
              </div>
            </div>

            <div class="flex items-start">
              <div class="w-10 h-10 rounded-full bg-purple-100 flex items-center justify-center text-purple-600 ml-4">
                <i class="fas fa-user"></i>
              </div>
              <div class="flex-1">
                <p class="font-medium">تم تسجيل مستخدم جديد في النظام</p>
                <p class="text-sm text-gray-500 mt-1">منذ 3 ساعات</p>
              </div>
            </div>
          </div>

          <div class="mt-4 pt-4 border-t border-gray-200">
            <a href="#" class="text-primary hover:text-primary-dark text-sm font-medium flex items-center justify-center">
              <span>عرض جميع الأنشطة</span>
              <i class="fas fa-arrow-left mr-1"></i>
            </a>
          </div>
        </div>
      </div>
    </main>
  </div>

  <!-- طبقة التراكب للأجهزة المحمولة -->
  <div id="mobileOverlay" class="fixed inset-0 bg-black bg-opacity-50 z-40 hidden"></div>

  <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
  <script>
    document.addEventListener('DOMContentLoaded', function() {
      // تهيئة القائمة الجانبية للأجهزة المحمولة
      const toggleSidebar = document.getElementById('toggleSidebar');
      const sidebar = document.querySelector('aside');
      const overlay = document.getElementById('mobileOverlay');

      if (toggleSidebar && sidebar && overlay) {
        toggleSidebar.addEventListener('click', function() {
          sidebar.classList.toggle('hidden');
          sidebar.classList.toggle('active');
          overlay.classList.toggle('hidden');
          document.body.classList.toggle('overflow-hidden');
        });

        overlay.addEventListener('click', function() {
          sidebar.classList.add('hidden');
          sidebar.classList.remove('active');
          overlay.classList.add('hidden');
          document.body.classList.remove('overflow-hidden');
        });
      }

      // تفعيل القائمة المنسدلة
      const dropdownToggle = document.getElementById('reportsDropdown');
      const dropdownMenu = document.getElementById('reportsMenu');

      if (dropdownToggle && dropdownMenu) {
        dropdownToggle.addEventListener('click', function(e) {
          e.preventDefault();
          this.classList.toggle('active');
          dropdownMenu.classList.toggle('active');
        });
      }

      // إضافة تأثير النقر على الأزرار
      const buttons = document.querySelectorAll('button, a.btn');
      buttons.forEach(button => {
        button.addEventListener('click', function() {
          this.classList.add('scale-95');
          setTimeout(() => {
            this.classList.remove('scale-95');
          }, 100);
        });
      });

      <?php if ($isAdmin): ?>
      // إنشاء الرسوم البيانية للمدير
      initCharts();
      <?php endif; ?>

      // تهيئة تأثيرات التمرير
      initScrollEffects();
    });

    // تهيئة تأثيرات التمرير
    function initScrollEffects() {
      const animatedElements = document.querySelectorAll('.animate-fadeIn');

      const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
          if (entry.isIntersecting) {
            entry.target.style.animationPlayState = 'running';
            observer.unobserve(entry.target);
          }
        });
      }, { threshold: 0.1 });

      animatedElements.forEach(element => {
        element.style.animationPlayState = 'paused';
        observer.observe(element);
      });
    }

    <?php if ($isAdmin): ?>
    // تهيئة الرسوم البيانية
    function initCharts() {
      // تكوين الخط العربي للرسوم البيانية
      Chart.defaults.font.family = 'Tajawal, sans-serif';

      // رسم بياني لحالات الطلبات
      const statusCtx = document.getElementById('statusChart');
      if (statusCtx) {
        new Chart(statusCtx, {
          type: 'doughnut',
          data: {
            labels: ['قيد المعالجة', 'قيد التوصيل', 'تم التسليم', 'ملغى'],
            datasets: [{
              data: [
                <?php echo $pendingOrders ?: 0; ?>,
                <?php echo $inTransitOrders ?: 0; ?>,
                <?php echo $deliveredOrders ?: 0; ?>,
                <?php echo $cancelledOrders ?: 0; ?>
              ],
              backgroundColor: [
                'rgba(245, 158, 11, 0.8)',
                'rgba(59, 130, 246, 0.8)',
                'rgba(16, 185, 129, 0.8)',
                'rgba(239, 68, 68, 0.8)'
              ],
              borderWidth: 0,
              hoverOffset: 15
            }]
          },
          options: {
            responsive: true,
            maintainAspectRatio: false,
            cutout: '70%',
            plugins: {
              legend: {
                position: 'right',
                labels: {
                  usePointStyle: true,
                  padding: 15,
                  font: {
                    size: 12
                  }
                }
              },
              tooltip: {
                backgroundColor: 'rgba(17, 24, 39, 0.9)',
                padding: 12,
                bodyFont: {
                  size: 14
                },
                titleFont: {
                  size: 16
                },
                boxPadding: 6
              }
            },
            animation: {
              animateScale: true,
              animateRotate: true,
              duration: 1500
            }
          }
        });
      }

      // رسم بياني للفروع
      const branchesCtx = document.getElementById('branchesChart');
      if (branchesCtx) {
        const gradient = branchesCtx.getContext('2d').createLinearGradient(0, 0, 0, 400);
        gradient.addColorStop(0, 'rgba(79, 70, 229, 0.9)');
        gradient.addColorStop(1, 'rgba(129, 140, 248, 0.6)');

        new Chart(branchesCtx, {
          type: 'bar',
          data: {
            labels: [<?php
              // استخراج أسماء الفروع من مصفوفة $topBranches
              $branchNames = array_map(function($branch) {
                return "'" . addslashes($branch['name']) . "'";
              }, $topBranches);
              echo implode(', ', $branchNames);
            ?>],
            datasets: [{
              label: 'عدد الطلبات',
              data: [<?php
                // استخراج عدد الطلبات من مصفوفة $topBranches
                $branchOrders = array_map(function($branch) {
                  return $branch['orders'];
                }, $topBranches);
                echo implode(', ', $branchOrders);
              ?>],
              backgroundColor: gradient,
              borderRadius: 8,
              borderSkipped: false,
              barPercentage: 0.6,
              categoryPercentage: 0.7
            }]
          },
          options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
              legend: {
                display: false
              },
              tooltip: {
                backgroundColor: 'rgba(17, 24, 39, 0.9)',
                padding: 12,
                bodyFont: {
                  size: 14
                },
                titleFont: {
                  size: 16
                },
                boxPadding: 6
              }
            },
            scales: {
              y: {
                beginAtZero: true,
                grid: {
                  display: true,
                  color: 'rgba(226, 232, 240, 0.5)'
                },
                border: {
                  display: false
                },
                ticks: {
                  padding: 10,
                  color: '#64748b'
                }
              },
              x: {
                grid: {
                  display: false
                },
                border: {
                  display: false
                },
                ticks: {
                  padding: 10,
                  color: '#64748b'
                }
              }
            },
            animation: {
              duration: 2000,
              easing: 'easeOutQuart'
            }
          }
        });
      }
    }
    <?php endif; ?>
  </script>
</body>

</html>