<?php
/**
 * إدارة المستخدمين
 * تعرض هذه الصفحة قائمة المستخدمين وتتيح إدارتهم
 *
 * @version 1.0
 * @author فريق التطوير
 */

// بدء الجلسة
session_start();

// تضمين ملف التكوين
require_once 'config/config.php';

// تضمين ملف الاتصال بقاعدة البيانات
require_once 'config/db_connection.php';

// تضمين ملفات إدارة الصلاحيات
require_once 'includes/permissions_manager.php';

// تفعيل عرض الأخطاء للتصحيح
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// التحقق من حالة تسجيل الدخول
if (!isset($_SESSION['logged_in']) || $_SESSION['logged_in'] !== true) {
    // إعادة توجيه المستخدم إلى صفحة تسجيل الدخول
    header('Location: auth/login.php');
    exit;
}

// الحصول على معلومات المستخدم
$user = [
    'id' => $_SESSION['user_id'] ?? 0,
    'username' => $_SESSION['username'] ?? '',
    'full_name' => $_SESSION['full_name'] ?? 'المستخدم',
    'email' => $_SESSION['email'] ?? '',
    'role' => $_SESSION['role'] ?? 'user',
    'branch_id' => $_SESSION['branch_id'] ?? null
];

// تحديد ما إذا كان المستخدم مديرًا
$isAdmin = (isset($_SESSION['role']) && ($_SESSION['role'] === 'admin' || $_SESSION['role'] === 'manager'));

// التحقق من صلاحيات المستخدم للوصول إلى صفحة إدارة المستخدمين
$hasPermission = $isAdmin;

// التحقق من وجود دالة checkUserPermission
if (!$hasPermission && function_exists('checkUserPermission')) {
    $hasPermission = checkUserPermission('view_users');
}

if (!$hasPermission) {
    // تسجيل محاولة الوصول غير المصرح به
    error_log("محاولة وصول غير مصرح به: المستخدم {$_SESSION['username']} حاول الوصول إلى صفحة إدارة المستخدمين");

    // إعادة توجيه المستخدم إلى لوحة التحكم
    header('Location: unified-dashboard.php');
    exit;
}

// معالجة الإجراءات (إضافة/تعديل/حذف)
$message = '';
$messageType = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        // التحقق من رمز CSRF
        if (!isset($_POST['csrf_token']) || !verifyCSRFToken($_POST['csrf_token'])) {
            $message = 'خطأ في التحقق من الأمان. يرجى المحاولة مرة أخرى.';
            $messageType = 'error';
        } else {
            switch ($_POST['action']) {
                case 'add':
                    // إضافة مستخدم جديد
                    $username = sanitizeInput($_POST['username'] ?? '');
                    $fullName = sanitizeInput($_POST['full_name'] ?? '');
                    $email = sanitizeInput($_POST['email'] ?? '');
                    $password = $_POST['password'] ?? '';
                    $role = sanitizeInput($_POST['role'] ?? 'employee');
                    $branchId = (int)($_POST['branch'] ?? 0);
                    $status = sanitizeInput($_POST['status'] ?? 'pending');

                    // التحقق من البيانات
                    if (empty($username) || empty($fullName) || empty($email) || empty($password)) {
                        $message = 'يرجى ملء جميع الحقول المطلوبة';
                        $messageType = 'error';
                    } else {
                        try {
                            // التحقق من عدم وجود اسم مستخدم أو بريد إلكتروني مكرر
                            $stmt = $pdo->prepare("SELECT COUNT(*) FROM users WHERE username = ? OR email = ?");
                            $stmt->execute([$username, $email]);
                            $count = $stmt->fetchColumn();

                            if ($count > 0) {
                                $message = 'اسم المستخدم أو البريد الإلكتروني مستخدم بالفعل';
                                $messageType = 'error';
                            } else {
                                // تشفير كلمة المرور
                                $hashedPassword = password_hash($password, PASSWORD_DEFAULT);

                                // بدء المعاملة
                                $pdo->beginTransaction();

                                try {
                                    // إضافة المستخدم الجديد
                                    $stmt = $pdo->prepare("
                                        INSERT INTO users (username, full_name, email, password, role, branch_id, status, registration_date)
                                        VALUES (?, ?, ?, ?, ?, ?, ?, NOW())
                                    ");
                                    $stmt->execute([$username, $fullName, $email, $hashedPassword, $role, $branchId ?: null, $status]);

                                    // الحصول على معرف المستخدم الجديد
                                    $newUserId = $pdo->lastInsertId();

                                    // إضافة الصلاحيات الإضافية إذا تم تحديدها
                                    if (isset($_POST['permissions']) && is_array($_POST['permissions']) && !empty($_POST['permissions'])) {
                                        // التحقق من وجود جدول صلاحيات المستخدمين
                                        $stmt = $pdo->query("SHOW TABLES LIKE 'user_permissions'");
                                        if ($stmt->rowCount() > 0) {
                                            // إضافة الصلاحيات للمستخدم
                                            foreach ($_POST['permissions'] as $permissionId) {
                                                $stmt = $pdo->prepare("
                                                    INSERT INTO user_permissions (user_id, permission_id)
                                                    VALUES (?, ?)
                                                    ON DUPLICATE KEY UPDATE permission_id = VALUES(permission_id)
                                                ");
                                                $stmt->execute([$newUserId, $permissionId]);
                                            }
                                        } else {
                                            // تسجيل رسالة خطأ إذا لم يكن الجدول موجودًا
                                            error_log("جدول صلاحيات المستخدمين غير موجود. لا يمكن إضافة الصلاحيات للمستخدم.");
                                        }
                                    }

                                    // تأكيد المعاملة
                                    $pdo->commit();

                                    $message = 'تم إضافة المستخدم بنجاح';
                                    $messageType = 'success';
                                } catch (PDOException $e) {
                                    // التراجع عن المعاملة في حالة حدوث خطأ
                                    $pdo->rollBack();
                                    throw $e;
                                }
                            }
                        } catch (PDOException $e) {
                            $message = 'حدث خطأ أثناء إضافة المستخدم: ' . $e->getMessage();
                            $messageType = 'error';
                        }
                    }
                    break;

                case 'edit':
                    // تعديل مستخدم
                    $userId = (int)($_POST['user_id'] ?? 0);
                    $username = sanitizeInput($_POST['username'] ?? '');
                    $fullName = sanitizeInput($_POST['full_name'] ?? '');
                    $email = sanitizeInput($_POST['email'] ?? '');
                    $role = sanitizeInput($_POST['role'] ?? 'employee');
                    $branchId = (int)($_POST['branch'] ?? 0);
                    $status = sanitizeInput($_POST['status'] ?? 'pending');

                    // التحقق من البيانات
                    if (empty($userId) || empty($username) || empty($fullName) || empty($email)) {
                        $message = 'يرجى ملء جميع الحقول المطلوبة';
                        $messageType = 'error';
                    } else {
                        try {
                            // التحقق من عدم وجود اسم مستخدم أو بريد إلكتروني مكرر (باستثناء المستخدم الحالي)
                            $stmt = $pdo->prepare("SELECT COUNT(*) FROM users WHERE (username = ? OR email = ?) AND id != ?");
                            $stmt->execute([$username, $email, $userId]);
                            $count = $stmt->fetchColumn();

                            if ($count > 0) {
                                $message = 'اسم المستخدم أو البريد الإلكتروني مستخدم بالفعل';
                                $messageType = 'error';
                            } else {
                                // بدء المعاملة
                                $pdo->beginTransaction();

                                try {
                                    // تحديث بيانات المستخدم
                                    $stmt = $pdo->prepare("
                                        UPDATE users
                                        SET username = ?, full_name = ?, email = ?, role = ?, branch_id = ?, status = ?
                                        WHERE id = ?
                                    ");
                                    $stmt->execute([$username, $fullName, $email, $role, $branchId ?: null, $status, $userId]);

                                    // إضافة الصلاحيات الإضافية إذا تم تحديدها
                                    if (isset($_POST['permissions']) && is_array($_POST['permissions'])) {
                                        // التحقق من وجود جدول صلاحيات المستخدمين
                                        $stmt = $pdo->query("SHOW TABLES LIKE 'user_permissions'");
                                        if ($stmt->rowCount() > 0) {
                                            // حذف الصلاحيات الحالية للمستخدم
                                            $stmt = $pdo->prepare("DELETE FROM user_permissions WHERE user_id = ?");
                                            $stmt->execute([$userId]);

                                            // إضافة الصلاحيات الجديدة
                                            foreach ($_POST['permissions'] as $permissionId) {
                                                $stmt = $pdo->prepare("
                                                    INSERT INTO user_permissions (user_id, permission_id)
                                                    VALUES (?, ?)
                                                ");
                                                $stmt->execute([$userId, $permissionId]);
                                            }
                                        } else {
                                            // تسجيل رسالة خطأ إذا لم يكن الجدول موجودًا
                                            error_log("جدول صلاحيات المستخدمين غير موجود. لا يمكن تحديث صلاحيات المستخدم.");
                                        }
                                    }

                                    // تأكيد المعاملة
                                    $pdo->commit();

                                    $message = 'تم تعديل بيانات المستخدم بنجاح';
                                    $messageType = 'success';
                                } catch (PDOException $e) {
                                    // التراجع عن المعاملة في حالة حدوث خطأ
                                    $pdo->rollBack();
                                    throw $e;
                                }
                            }
                        } catch (PDOException $e) {
                            $message = 'حدث خطأ أثناء تعديل المستخدم: ' . $e->getMessage();
                            $messageType = 'error';
                        }
                    }
                    break;

                case 'reset_password':
                    // إعادة تعيين كلمة المرور
                    $userId = (int)($_POST['user_id'] ?? 0);
                    $newPassword = $_POST['new_password'] ?? '';
                    $confirmPassword = $_POST['confirm_password'] ?? '';

                    // التحقق من البيانات
                    if (empty($userId) || empty($newPassword) || empty($confirmPassword)) {
                        $message = 'يرجى ملء جميع الحقول المطلوبة';
                        $messageType = 'error';
                    } elseif ($newPassword !== $confirmPassword) {
                        $message = 'كلمات المرور غير متطابقة';
                        $messageType = 'error';
                    } else {
                        try {
                            // تشفير كلمة المرور الجديدة
                            $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);

                            // تحديث كلمة المرور
                            $stmt = $pdo->prepare("UPDATE users SET password = ? WHERE id = ?");
                            $stmt->execute([$hashedPassword, $userId]);

                            $message = 'تم إعادة تعيين كلمة المرور بنجاح';
                            $messageType = 'success';
                        } catch (PDOException $e) {
                            $message = 'حدث خطأ أثناء إعادة تعيين كلمة المرور: ' . $e->getMessage();
                            $messageType = 'error';
                        }
                    }
                    break;

                case 'delete':
                    // حذف مستخدم
                    $userId = (int)($_POST['user_id'] ?? 0);

                    if (empty($userId)) {
                        $message = 'معرف المستخدم غير صالح';
                        $messageType = 'error';
                    } else {
                        try {
                            // التحقق من عدم حذف المستخدم الحالي
                            if ($userId == $_SESSION['user_id']) {
                                $message = 'لا يمكن حذف المستخدم الحالي';
                                $messageType = 'error';
                            } else {
                                // حذف المستخدم
                                $stmt = $pdo->prepare("DELETE FROM users WHERE id = ?");
                                $stmt->execute([$userId]);

                                $message = 'تم حذف المستخدم بنجاح';
                                $messageType = 'success';
                            }
                        } catch (PDOException $e) {
                            $message = 'حدث خطأ أثناء حذف المستخدم: ' . $e->getMessage();
                            $messageType = 'error';
                        }
                    }
                    break;

                default:
                    $message = 'إجراء غير معروف';
                    $messageType = 'error';
            }
        }
    }
}

// جلب بيانات المستخدمين من قاعدة البيانات
try {
    $stmt = $pdo->query("
        SELECT u.*, b.name as branch_name
        FROM users u
        LEFT JOIN branches b ON u.branch_id = b.id
        ORDER BY u.id
    ");
    $users = $stmt->fetchAll();
} catch (PDOException $e) {
    $message = 'حدث خطأ أثناء جلب بيانات المستخدمين: ' . $e->getMessage();
    $messageType = 'error';
    $users = [];
}

// جلب بيانات الفروع من قاعدة البيانات
try {
    $stmt = $pdo->query("SELECT id, name FROM branches ORDER BY name");
    $branches = $stmt->fetchAll();
} catch (PDOException $e) {
    $message = 'حدث خطأ أثناء جلب بيانات الفروع: ' . $e->getMessage();
    $messageType = 'error';
    $branches = [];
}

// بيانات الأدوار الافتراضية
$defaultRoles = [
    ['id' => 1, 'name' => 'admin', 'display_name' => 'مدير النظام'],
    ['id' => 2, 'name' => 'manager', 'display_name' => 'مدير'],
    ['id' => 3, 'name' => 'employee', 'display_name' => 'موظف'],
    ['id' => 4, 'name' => 'client', 'display_name' => 'عميل']
];

// جلب بيانات الأدوار من قاعدة البيانات
try {
    // التحقق من وجود دالة getAllRoles أو getSystemRoles
    if (function_exists('getAllRoles')) {
        $roles = getAllRoles();
        if (empty($roles)) {
            $roles = $defaultRoles;
        }
    } else if (function_exists('getSystemRoles')) {
        // استخدام دالة getSystemRoles من ملف permissions_manager.php
        $roles = getSystemRoles();
        if (empty($roles)) {
            $roles = $defaultRoles;
        }
    } else {
        // إذا لم تكن الدوال موجودة، استخدم القيم الافتراضية
        $roles = $defaultRoles;
    }
} catch (Exception $e) {
    $message = 'حدث خطأ أثناء جلب بيانات الأدوار: ' . $e->getMessage();
    $messageType = 'error';
    // استخدام القيم الافتراضية في حالة حدوث خطأ
    $roles = $defaultRoles;
}

// جلب بيانات الصلاحيات من قاعدة البيانات
try {
    // التحقق من وجود دالة getAllPermissions أو getSystemPermissions
    if (function_exists('getAllPermissions')) {
        $permissions = getAllPermissions();
    } else if (function_exists('getSystemPermissions')) {
        // استخدام دالة getSystemPermissions من ملف permissions_manager.php
        $permissions = getSystemPermissions();
    } else {
        // إذا لم تكن الدوال موجودة، استخدم مصفوفة فارغة
        $permissions = [];
    }
} catch (Exception $e) {
    $message = 'حدث خطأ أثناء جلب بيانات الصلاحيات: ' . $e->getMessage();
    $messageType = 'error';
    $permissions = [];
}

// تعيين مسار الشعار
$company_logo = 'assets/images/backgrounds/background.png';

// تنسيق التاريخ بالعربية
function formatDateArabic($date) {
    $timestamp = strtotime($date);
    $day = date('d', $timestamp);
    $month = date('m', $timestamp);
    $year = date('Y', $timestamp);
    $time = date('H:i', $timestamp);

    $months = [
        '01' => 'يناير',
        '02' => 'فبراير',
        '03' => 'مارس',
        '04' => 'أبريل',
        '05' => 'مايو',
        '06' => 'يونيو',
        '07' => 'يوليو',
        '08' => 'أغسطس',
        '09' => 'سبتمبر',
        '10' => 'أكتوبر',
        '11' => 'نوفمبر',
        '12' => 'ديسمبر'
    ];

    return $day . ' ' . $months[$month] . ' ' . $year . ' - ' . $time;
}

// تحويل اسم الدور إلى الاسم المعروض
function getRoleDisplayName($roleName) {
    global $roles;

    // البحث عن الدور في مصفوفة الأدوار
    foreach ($roles as $role) {
        if ($role['name'] === $roleName) {
            return $role['display_name'];
        }
    }

    // إذا لم يتم العثور على الدور، استخدم القيم الافتراضية
    switch ($roleName) {
        case 'admin':
            return 'مدير النظام';
        case 'manager':
            return 'مدير';
        case 'employee':
            return 'موظف';
        case 'client':
            return 'عميل';
        default:
            return $roleName;
    }
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">

<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>نظام طلبات الشحن | إدارة المستخدمين</title>
  <script src="https://cdn.tailwindcss.com"></script>
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
  <link rel="stylesheet" href="assets/css/unified-design.css">
  <link rel="stylesheet" href="assets/css/premium-user-form.css">
  <style>
    /* تنسيق الجدول */
    .data-table {
      width: 100%;
      border-collapse: separate;
      border-spacing: 0;
    }

    .data-table th {
      background-color: var(--primary);
      color: white;
      font-weight: 500;
      text-align: right;
      padding: 0.75rem 1rem;
    }

    .data-table th:first-child {
      border-top-right-radius: var(--border-radius);
    }

    .data-table th:last-child {
      border-top-left-radius: var(--border-radius);
    }

    .data-table td {
      padding: 0.75rem 1rem;
      border-bottom: 1px solid var(--gray-200);
    }

    .data-table tbody tr:hover {
      background-color: var(--gray-50);
    }

    .data-table tbody tr:last-child td {
      border-bottom: none;
    }

    .data-table tbody tr:last-child td:first-child {
      border-bottom-right-radius: var(--border-radius);
    }

    .data-table tbody tr:last-child td:last-child {
      border-bottom-left-radius: var(--border-radius);
    }

    /* تنسيق الحالة */
    .status-badge {
      padding: 0.25rem 0.75rem;
      border-radius: 9999px;
      font-size: 0.75rem;
      font-weight: 600;
      display: inline-flex;
      align-items: center;
      gap: 0.25rem;
    }

    .status-active {
      background-color: rgba(16, 185, 129, 0.1);
      color: var(--success);
    }

    .status-inactive {
      background-color: rgba(239, 68, 68, 0.1);
      color: var(--danger);
    }

    /* تنسيق الأدوار */
    .role-badge {
      padding: 0.25rem 0.75rem;
      border-radius: 9999px;
      font-size: 0.75rem;
      font-weight: 600;
      display: inline-flex;
      align-items: center;
      gap: 0.25rem;
    }

    .role-admin {
      background-color: rgba(79, 70, 229, 0.1);
      color: var(--primary);
    }

    .role-manager {
      background-color: rgba(245, 158, 11, 0.1);
      color: var(--warning);
    }

    .role-user {
      background-color: rgba(59, 130, 246, 0.1);
      color: var(--info);
    }

    /* تنسيق النموذج */
    .modal {
      position: fixed;
      top: 0;
      left: 0;
      right: 0;
      bottom: 0;
      background-color: rgba(0, 0, 0, 0.5);
      display: flex;
      align-items: center;
      justify-content: center;
      z-index: 50;
      opacity: 0;
      pointer-events: none;
      transition: var(--transition);
    }

    .modal.active {
      opacity: 1;
      pointer-events: auto;
    }

    .modal-content {
      background-color: white;
      border-radius: var(--border-radius);
      box-shadow: var(--shadow-lg);
      width: 100%;
      max-width: 500px;
      max-height: 90vh;
      overflow-y: auto;
      transform: translateY(20px);
      transition: var(--transition);
    }

    .modal.active .modal-content {
      transform: translateY(0);
    }

    .form-group {
      margin-bottom: 1rem;
    }

    .form-label {
      display: block;
      margin-bottom: 0.5rem;
      font-weight: 500;
      color: var(--gray-700);
    }

    .form-input {
      width: 100%;
      padding: 0.5rem 0.75rem;
      border: 1px solid var(--gray-300);
      border-radius: var(--border-radius);
      transition: var(--transition);
    }

    .form-input:focus {
      border-color: var(--primary);
      box-shadow: 0 0 0 3px rgba(79, 70, 229, 0.2);
      outline: none;
    }

    .form-select {
      width: 100%;
      padding: 0.5rem 0.75rem;
      border: 1px solid var(--gray-300);
      border-radius: var(--border-radius);
      transition: var(--transition);
      background-color: white;
    }

    .form-select:focus {
      border-color: var(--primary);
      box-shadow: 0 0 0 3px rgba(79, 70, 229, 0.2);
      outline: none;
    }

    /* تأثيرات الحركة */
    @keyframes fadeIn {
      from {
        opacity: 0;
        transform: translateY(10px);
      }
      to {
        opacity: 1;
        transform: translateY(0);
      }
    }

    .animate-fadeIn {
      animation: fadeIn 0.5s ease-out forwards;
    }
  </style>
</head>

<body>
  <!-- طبقة التراكب للأجهزة المحمولة -->
  <div id="overlay" class="overlay"></div>

  <div class="flex min-h-screen">
    <!-- القائمة الجانبية -->
    <aside class="w-64 hidden md:block">
      <div class="p-6">
        <div class="logo-container flex items-center justify-center mb-8 p-2">
          <img src="<?php echo $company_logo; ?>" alt="شعار النظام" class="h-40 w-auto object-contain logo-animation">
        </div>

        <div class="user-info mb-8">
          <div class="flex items-center space-x-4 space-x-reverse mb-3">
            <div class="user-avatar">
              <?php echo mb_substr($user['full_name'], 0, 1, 'UTF-8'); ?>
            </div>
            <div>
              <h3 class="text-white font-semibold"><?php echo htmlspecialchars($user['full_name']); ?></h3>
              <p class="text-indigo-200 text-sm"><?php echo $isAdmin ? 'مدير النظام' : 'مستخدم'; ?></p>
            </div>
          </div>
          <div class="bg-white/10 h-0.5 rounded-full mb-3"></div>
        </div>

        <nav>
          <ul class="space-y-3">
            <!-- لوحة التحكم -->
            <li>
              <a href="unified-dashboard.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white">
                <i class="fas fa-tachometer-alt ml-3 text-lg"></i>
                <span>لوحة التحكم</span>
              </a>
            </li>

            <!-- إنشاء طلب جديد -->
            <li>
              <a href="create-order-unified.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white">
                <i class="fas fa-plus-circle ml-3 text-lg"></i>
                <span>إنشاء طلب جديد</span>
              </a>
            </li>

            <!-- قائمة الطلبات -->
            <li>
              <a href="orders.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white">
                <i class="fas fa-box ml-3 text-lg"></i>
                <span>قائمة الطلبات</span>
              </a>
            </li>

            <?php if ($isAdmin): ?>
            <!-- قائمة التقارير المنسدلة -->
            <li>
              <a href="#" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white dropdown-toggle" id="reportsDropdown">
                <i class="fas fa-chart-bar ml-3 text-lg"></i>
                <span>التقارير</span>
              </a>
              <ul class="dropdown-menu" id="reportsMenu">
                <li>
                  <a href="weekly-reports.php" class="flex items-center p-2 rounded-lg text-white/80 hover:text-white">
                    <i class="fas fa-calendar-week ml-2 text-sm"></i>
                    <span>التقارير الأسبوعية</span>
                  </a>
                </li>
                <li>
                  <a href="monthly-reports.php" class="flex items-center p-2 rounded-lg text-white/80 hover:text-white">
                    <i class="fas fa-calendar-alt ml-2 text-sm"></i>
                    <span>التقارير الشهرية</span>
                  </a>
                </li>
              </ul>
            </li>

            <!-- إدارة المستخدمين -->
            <li>
              <a href="users.php" class="flex items-center p-3 rounded-lg text-white active">
                <i class="fas fa-users ml-3 text-lg"></i>
                <span>إدارة المستخدمين</span>
              </a>
            </li>

            <!-- الفروع -->
            <li>
              <a href="branches.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white">
                <i class="fas fa-code-branch ml-3 text-lg"></i>
                <span>الفروع</span>
              </a>
            </li>

            <!-- إعدادات النظام -->
            <li>
              <a href="settings.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white">
                <i class="fas fa-cog ml-3 text-lg"></i>
                <span>إعدادات النظام</span>
              </a>
            </li>
            <?php endif; ?>
          </ul>

          <div class="pt-6 mt-6 border-t border-white/10">
            <a href="auth/logout.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white">
              <i class="fas fa-sign-out-alt ml-3 text-lg"></i>
              <span>تسجيل الخروج</span>
            </a>
          </div>
        </nav>
      </div>
    </aside>

    <!-- المحتوى الرئيسي -->
    <main class="flex-1 overflow-x-hidden overflow-y-auto bg-gray-50">
      <!-- شريط التنقل العلوي -->
      <header class="bg-white shadow-sm">
        <div class="flex items-center justify-between p-4">
          <!-- زر القائمة للأجهزة المحمولة -->
          <button id="menuBtn" class="md:hidden text-gray-500 hover:text-gray-600 focus:outline-none">
            <i class="fas fa-bars text-xl"></i>
          </button>

          <!-- عنوان الصفحة -->
          <h1 class="text-xl font-semibold text-gray-800">إدارة المستخدمين</h1>

          <!-- أزرار الإجراءات -->
          <div class="flex space-x-2 space-x-reverse">
            <a href="user_permissions.php" class="btn btn-secondary">
              <i class="fas fa-user-shield ml-2"></i>
              <span>صلاحيات المستخدم العادي</span>
            </a>
            <button id="addUserBtn" class="btn btn-primary">
              <i class="fas fa-user-plus ml-2"></i>
              <span>إضافة مستخدم</span>
            </button>
          </div>
        </div>
      </header>

      <!-- محتوى الصفحة -->
      <div class="p-6">
        <?php if (!empty($message)): ?>
        <div class="mb-6 p-4 rounded-lg <?php echo $messageType === 'success' ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'; ?>">
          <div class="flex items-center">
            <i class="fas <?php echo $messageType === 'success' ? 'fa-check-circle' : 'fa-exclamation-circle'; ?> ml-2"></i>
            <span><?php echo $message; ?></span>
          </div>
        </div>
        <?php endif; ?>

        <!-- بطاقة إحصائيات المستخدمين -->
        <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
          <!-- إجمالي المستخدمين -->
          <div class="stat-card p-6 animate-fadeIn" style="animation-delay: 0.1s;">
            <div class="flex items-center justify-between">
              <div class="icon-container">
                <i class="fas fa-users"></i>
              </div>
            </div>
            <div class="stat-value"><?php echo count($users); ?></div>
            <div class="stat-label">إجمالي المستخدمين</div>
          </div>

          <!-- المستخدمين النشطين -->
          <div class="stat-card p-6 animate-fadeIn" style="animation-delay: 0.2s;">
            <div class="flex items-center justify-between">
              <div class="icon-container">
                <i class="fas fa-user-check"></i>
              </div>
            </div>
            <div class="stat-value"><?php echo count(array_filter($users, function($user) { return $user['status'] === 'نشط'; })); ?></div>
            <div class="stat-label">المستخدمين النشطين</div>
          </div>

          <!-- مدراء النظام -->
          <div class="stat-card p-6 animate-fadeIn" style="animation-delay: 0.3s;">
            <div class="flex items-center justify-between">
              <div class="icon-container">
                <i class="fas fa-user-shield"></i>
              </div>
            </div>
            <div class="stat-value"><?php echo count(array_filter($users, function($user) { return $user['role'] === 'admin'; })); ?></div>
            <div class="stat-label">مدراء النظام</div>
          </div>

          <!-- المستخدمين العاديين -->
          <div class="stat-card p-6 animate-fadeIn" style="animation-delay: 0.4s;">
            <div class="flex items-center justify-between">
              <div class="icon-container">
                <i class="fas fa-user"></i>
              </div>
            </div>
            <div class="stat-value"><?php echo count(array_filter($users, function($user) { return $user['role'] === 'user'; })); ?></div>
            <div class="stat-label">المستخدمين العاديين</div>
          </div>
        </div>

        <!-- قائمة المستخدمين -->
        <div class="chart-container animate-fadeIn" style="animation-delay: 0.5s;">
          <h3 class="chart-title">
            <i class="fas fa-users"></i>
            <span>قائمة المستخدمين</span>
          </h3>

          <!-- فلترة المستخدمين -->
          <div class="bg-gray-50 p-4 rounded-lg mb-6">
            <div class="flex flex-wrap items-center justify-between gap-4">
              <!-- فلتر الفروع -->
              <div class="flex items-center">
                <select id="branchFilter" class="form-input">
                  <option value="">كل الفروع</option>
                  <?php foreach ($branches as $branch): ?>
                  <option value="<?php echo $branch['name']; ?>"><?php echo $branch['name']; ?></option>
                  <?php endforeach; ?>
                </select>
              </div>

              <!-- فلتر الدور -->
              <div class="flex items-center">
                <select id="roleFilter" class="form-input">
                  <option value="">كل الأدوار</option>
                  <?php foreach ($roles as $role): ?>
                  <option value="<?php echo $role['name']; ?>"><?php echo $role['display_name']; ?></option>
                  <?php endforeach; ?>
                </select>
              </div>

              <!-- فلتر الحالة -->
              <div class="flex items-center">
                <select id="statusFilter" class="form-input">
                  <option value="">كل الحالات</option>
                  <option value="نشط">نشط</option>
                  <option value="غير نشط">غير نشط</option>
                </select>
              </div>

              <!-- زر التصفية -->
              <button id="filterBtn" class="btn btn-primary btn-sm">
                <i class="fas fa-filter ml-1"></i>
                <span>تصفية</span>
              </button>
            </div>
          </div>

          <!-- جدول المستخدمين -->
          <div class="overflow-x-auto">
            <table class="data-table">
              <thead>
                <tr>
                  <th>المعرف</th>
                  <th>اسم المستخدم</th>
                  <th>الاسم الكامل</th>
                  <th>البريد الإلكتروني</th>
                  <th>الدور</th>
                  <th>الفرع</th>
                  <th>الحالة</th>
                  <th>آخر تسجيل دخول</th>
                  <th>الإجراءات</th>
                </tr>
              </thead>
              <tbody>
                <?php if (empty($users)): ?>
                <tr>
                  <td colspan="9" class="text-center py-8">
                    <div class="flex flex-col items-center justify-center text-gray-500">
                      <i class="fas fa-users text-4xl mb-3 text-gray-300"></i>
                      <p>لا يوجد مستخدمين</p>
                    </div>
                  </td>
                </tr>
                <?php else: ?>
                  <?php foreach ($users as $userItem): ?>
                  <tr>
                    <td class="font-medium"><?php echo $userItem['id']; ?></td>
                    <td><?php echo htmlspecialchars($userItem['username']); ?></td>
                    <td>
                      <div class="flex items-center">
                        <div class="w-8 h-8 rounded-full bg-primary-100 flex items-center justify-center text-primary font-semibold ml-2">
                          <?php echo mb_substr($userItem['full_name'], 0, 1, 'UTF-8'); ?>
                        </div>
                        <span><?php echo htmlspecialchars($userItem['full_name']); ?></span>
                      </div>
                    </td>
                    <td><?php echo htmlspecialchars($userItem['email']); ?></td>
                    <td>
                      <?php
                      $roleClass = '';
                      $roleIcon = '';

                      switch ($userItem['role']) {
                          case 'admin':
                              $roleClass = 'role-admin';
                              $roleIcon = 'user-shield';
                              break;
                          case 'manager':
                              $roleClass = 'role-manager';
                              $roleIcon = 'user-tie';
                              break;
                          case 'employee':
                              $roleClass = 'role-user';
                              $roleIcon = 'user';
                              break;
                          case 'client':
                              $roleClass = 'role-user';
                              $roleIcon = 'user-friends';
                              break;
                          default:
                              $roleClass = '';
                              $roleIcon = 'user';
                      }
                      ?>
                      <span class="role-badge <?php echo $roleClass; ?>">
                        <i class="fas fa-<?php echo $roleIcon; ?> ml-1"></i>
                        <?php echo getRoleDisplayName($userItem['role']); ?>
                      </span>
                    </td>
                    <td><?php echo htmlspecialchars($userItem['branch_name'] ?? 'غير محدد'); ?></td>
                    <td>
                      <?php
                      $statusText = '';
                      $statusClass = '';
                      $statusIcon = '';

                      if (isset($userItem['is_active']) && $userItem['is_active'] == 1) {
                          $statusText = 'نشط';
                          $statusClass = 'status-active';
                          $statusIcon = 'check-circle';
                      } elseif (isset($userItem['status'])) {
                          switch ($userItem['status']) {
                              case 'active':
                                  $statusText = 'نشط';
                                  $statusClass = 'status-active';
                                  $statusIcon = 'check-circle';
                                  break;
                              case 'pending':
                                  $statusText = 'معلق';
                                  $statusClass = 'status-pending';
                                  $statusIcon = 'clock';
                                  break;
                              case 'inactive':
                                  $statusText = 'غير نشط';
                                  $statusClass = 'status-inactive';
                                  $statusIcon = 'times-circle';
                                  break;
                              default:
                                  $statusText = $userItem['status'];
                                  $statusClass = '';
                                  $statusIcon = 'question-circle';
                          }
                      } else {
                          $statusText = 'غير معروف';
                          $statusClass = '';
                          $statusIcon = 'question-circle';
                      }
                      ?>
                      <span class="status-badge <?php echo $statusClass; ?>">
                        <i class="fas fa-<?php echo $statusIcon; ?> ml-1"></i>
                        <?php echo $statusText; ?>
                      </span>
                    </td>
                    <td><?php echo $userItem['last_login'] ? formatDateArabic($userItem['last_login']) : 'لم يسجل الدخول بعد'; ?></td>
                    <td>
                      <div class="flex items-center space-x-2 space-x-reverse">
                        <button class="edit-user-btn p-1 text-info hover:bg-info/10 rounded-full transition-all" title="تعديل المستخدم" data-id="<?php echo $userItem['id']; ?>">
                          <i class="fas fa-edit"></i>
                        </button>
                        <button class="reset-password-btn p-1 text-warning hover:bg-warning/10 rounded-full transition-all" title="إعادة تعيين كلمة المرور" data-id="<?php echo $userItem['id']; ?>">
                          <i class="fas fa-key"></i>
                        </button>
                        <?php if ($userItem['id'] != $_SESSION['user_id']): ?>
                        <button class="delete-user-btn p-1 text-danger hover:bg-danger/10 rounded-full transition-all" title="حذف المستخدم" data-id="<?php echo $userItem['id']; ?>">
                          <i class="fas fa-trash-alt"></i>
                        </button>
                        <?php endif; ?>
                      </div>
                    </td>
                  </tr>
                  <?php endforeach; ?>
                <?php endif; ?>
              </tbody>
            </table>
          </div>
        </div>
      </div>
    </main>
  </div>

  <!-- نموذج إضافة/تعديل مستخدم -->
  <div id="userModal" class="modal">
    <div class="modal-content">
      <div class="p-6">
        <div class="flex items-center justify-between mb-6">
          <h3 class="text-xl font-semibold text-gray-800" id="modalTitle">إضافة مستخدم جديد</h3>
          <button id="closeModal" class="text-gray-400 hover:text-gray-600">
            <i class="fas fa-times text-xl"></i>
          </button>
        </div>

        <form id="userForm" method="post">
          <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">
          <input type="hidden" name="action" id="formAction" value="add">
          <input type="hidden" name="user_id" id="userId" value="">

          <!-- شعار النظام -->
          <div class="text-center mb-6">
            <img src="<?php echo $company_logo; ?>" alt="شعار النظام" class="h-20 mx-auto mb-2 object-contain">
            <div class="text-sm text-gray-500">مجموعة ابراهيم الاحمدي اليمنية</div>
            <div class="h-0.5 w-24 bg-gradient-to-r from-gray-200 via-primary to-gray-200 mx-auto mt-2"></div>
          </div>

          <div class="form-group">
            <label for="username" class="form-label">اسم المستخدم</label>
            <input type="text" id="username" name="username" class="form-input" required>
          </div>

          <div class="form-group">
            <label for="fullName" class="form-label">الاسم الكامل</label>
            <input type="text" id="fullName" name="full_name" class="form-input" required>
          </div>

          <div class="form-group">
            <label for="email" class="form-label">البريد الإلكتروني</label>
            <input type="email" id="email" name="email" class="form-input" required>
          </div>

          <div class="form-group" id="passwordGroup">
            <label for="password" class="form-label">كلمة المرور</label>
            <input type="password" id="password" name="password" class="form-input" required>
          </div>

          <div class="form-group">
            <label for="role" class="form-label">الدور</label>
            <select id="role" name="role" class="form-select" required>
              <?php foreach ($roles as $role): ?>
              <option value="<?php echo $role['name']; ?>"><?php echo $role['display_name']; ?></option>
              <?php endforeach; ?>
            </select>
            <p class="text-xs text-gray-500 mt-1">الدور يحدد صلاحيات المستخدم في النظام</p>
            <div class="mt-2">
              <div class="bg-blue-50 p-2 rounded border border-blue-100 text-xs text-blue-700">
                <i class="fas fa-info-circle ml-1"></i>
                عند اختيار دور "موظف" أو "عميل"، سيتم تحديد صلاحيات المستخدم العادي تلقائياً
              </div>
            </div>
          </div>

          <?php if ($isAdmin && !empty($permissions)): ?>
          <div class="form-group">
            <label class="form-label">الصلاحيات الإضافية</label>

            <!-- صلاحيات المستخدم العادي -->
            <div class="bg-indigo-50 p-4 rounded-lg mb-4 border-2 border-indigo-200">
              <h4 class="text-sm font-bold text-indigo-700 mb-2 flex items-center">
                <i class="fas fa-user-shield ml-2"></i>
                صلاحيات المستخدم العادي
              </h4>
              <p class="text-xs text-gray-600 mb-3">هذه الصلاحيات تمكن المستخدم العادي من إدارة طلباته الخاصة ومتابعتها</p>

              <div class="grid grid-cols-2 gap-3">
                <!-- لوحة التحكم الخاصة بطلباته -->
                <div class="bg-white p-2 rounded border border-indigo-100 hover:shadow-sm transition-all">
                  <div class="flex items-center">
                    <input type="checkbox" id="perm_dashboard_view_own" name="permissions[]" value="dashboard_view_own" class="ml-2 form-checkbox text-indigo-600 h-4 w-4">
                    <label for="perm_dashboard_view_own" class="text-sm font-medium text-gray-700">لوحة التحكم الخاصة بطلباته</label>
                  </div>
                </div>

                <!-- إنشاء طلب خاص به -->
                <div class="bg-white p-2 rounded border border-indigo-100 hover:shadow-sm transition-all">
                  <div class="flex items-center">
                    <input type="checkbox" id="perm_orders_add" name="permissions[]" value="orders_add" class="ml-2 form-checkbox text-indigo-600 h-4 w-4">
                    <label for="perm_orders_add" class="text-sm font-medium text-gray-700">إنشاء طلب خاص به</label>
                  </div>
                </div>

                <!-- إدارة الطلبات الخاصة به -->
                <div class="bg-white p-2 rounded border border-indigo-100 hover:shadow-sm transition-all">
                  <div class="flex items-center">
                    <input type="checkbox" id="perm_orders_manage_own" name="permissions[]" value="orders_manage_own" class="ml-2 form-checkbox text-indigo-600 h-4 w-4">
                    <label for="perm_orders_manage_own" class="text-sm font-medium text-gray-700">إدارة الطلبات الخاصة به</label>
                  </div>
                </div>

                <!-- تعديل الطلبات الخاصة به -->
                <div class="bg-white p-2 rounded border border-indigo-100 hover:shadow-sm transition-all">
                  <div class="flex items-center">
                    <input type="checkbox" id="perm_orders_edit_own" name="permissions[]" value="orders_edit_own" class="ml-2 form-checkbox text-indigo-600 h-4 w-4">
                    <label for="perm_orders_edit_own" class="text-sm font-medium text-gray-700">تعديل الطلبات الخاصة به</label>
                  </div>
                </div>

                <!-- عرض التقارير الأسبوعية الخاصة به -->
                <div class="bg-white p-2 rounded border border-indigo-100 hover:shadow-sm transition-all">
                  <div class="flex items-center">
                    <input type="checkbox" id="perm_weekly_reports_view_own" name="permissions[]" value="weekly_reports_view_own" class="ml-2 form-checkbox text-indigo-600 h-4 w-4">
                    <label for="perm_weekly_reports_view_own" class="text-sm font-medium text-gray-700">عرض التقارير الأسبوعية الخاصة به</label>
                  </div>
                </div>

                <!-- عرض التقارير الشهرية الخاصة به -->
                <div class="bg-white p-2 rounded border border-indigo-100 hover:shadow-sm transition-all">
                  <div class="flex items-center">
                    <input type="checkbox" id="perm_monthly_reports_view_own" name="permissions[]" value="monthly_reports_view_own" class="ml-2 form-checkbox text-indigo-600 h-4 w-4">
                    <label for="perm_monthly_reports_view_own" class="text-sm font-medium text-gray-700">عرض التقارير الشهرية الخاصة به</label>
                  </div>
                </div>

                <!-- صلاحيات الدعم الفني للتواصل مع الإدارة -->
                <div class="bg-white p-2 rounded border border-indigo-100 hover:shadow-sm transition-all">
                  <div class="flex items-center">
                    <input type="checkbox" id="perm_support_contact_admin" name="permissions[]" value="support_contact_admin" class="ml-2 form-checkbox text-indigo-600 h-4 w-4">
                    <label for="perm_support_contact_admin" class="text-sm font-medium text-gray-700">التواصل مع الإدارة (الدعم الفني)</label>
                  </div>
                </div>
              </div>

              <div class="mt-3 flex justify-between items-center">
                <a href="user_permissions.php" class="text-indigo-600 hover:underline text-xs" target="_blank">
                  <i class="fas fa-external-link-alt ml-1"></i>
                  عرض تفاصيل صلاحيات المستخدم العادي
                </a>

                <div class="flex space-x-2 space-x-reverse">
                  <button type="button" id="selectAllRegularPermissions" class="bg-indigo-100 hover:bg-indigo-200 text-indigo-700 px-2 py-1 rounded text-xs transition-colors">
                    <i class="fas fa-check-square ml-1"></i>
                    تحديد الكل
                  </button>
                  <button type="button" id="deselectAllRegularPermissions" class="bg-gray-100 hover:bg-gray-200 text-gray-700 px-2 py-1 rounded text-xs transition-colors">
                    <i class="fas fa-square ml-1"></i>
                    إلغاء التحديد
                  </button>
                </div>
              </div>
            </div>

            <!-- الصلاحيات الإضافية الأخرى -->
            <div class="bg-gray-50 p-3 rounded-lg max-h-40 overflow-y-auto">
              <h4 class="text-sm font-semibold text-gray-700 mb-2">صلاحيات إضافية أخرى</h4>
              <div class="grid grid-cols-2 gap-2">
                <?php
                // تجميع الصلاحيات حسب الفئة
                $groupedPermissions = [];
                foreach ($permissions as $permission) {
                    $category = $permission['category'] ?? 'general';
                    if (!isset($groupedPermissions[$category])) {
                        $groupedPermissions[$category] = [];
                    }
                    $groupedPermissions[$category][] = $permission;
                }

                // عرض الصلاحيات حسب الفئة
                foreach ($groupedPermissions as $category => $categoryPermissions):
                ?>
                  <div class="mb-2">
                    <h4 class="text-sm font-semibold text-gray-700 mb-1"><?php echo ucfirst($category); ?></h4>
                    <?php foreach ($categoryPermissions as $permission): ?>
                    <div class="flex items-center mb-1">
                      <input type="checkbox" id="perm_<?php echo $permission['id']; ?>" name="permissions[]" value="<?php echo $permission['id']; ?>" class="ml-2">
                      <label for="perm_<?php echo $permission['id']; ?>" class="text-xs"><?php echo $permission['display_name']; ?></label>
                    </div>
                    <?php endforeach; ?>
                  </div>
                <?php endforeach; ?>
              </div>
            </div>
            <p class="text-xs text-gray-500 mt-1">يمكنك تعديل صلاحيات المستخدم بشكل مفصل من صفحة إعدادات النظام</p>
          </div>
          <?php endif; ?>

          <div class="form-group">
            <label for="branch" class="form-label">الفرع</label>
            <select id="branch" name="branch" class="form-select">
              <option value="">-- اختر الفرع --</option>
              <?php foreach ($branches as $branch): ?>
              <option value="<?php echo $branch['id']; ?>"><?php echo htmlspecialchars($branch['name']); ?></option>
              <?php endforeach; ?>
            </select>
          </div>

          <div class="form-group">
            <label for="status" class="form-label">الحالة</label>
            <select id="status" name="status" class="form-select" required>
              <option value="active">نشط</option>
              <option value="pending">معلق</option>
              <option value="inactive">غير نشط</option>
            </select>
          </div>

          <!-- تنبيه هام -->
          <div class="bg-yellow-50 border-r-4 border-yellow-400 p-4 mb-6 rounded-lg">
            <div class="flex">
              <div class="flex-shrink-0">
                <i class="fas fa-exclamation-triangle text-yellow-400"></i>
              </div>
              <div class="mr-3">
                <p class="text-sm text-yellow-700">
                  <strong class="font-bold">تنبيه هام:</strong> إضافة مستخدم جديد تمنحه صلاحيات الوصول إلى النظام حسب الدور والصلاحيات المحددة.
                </p>
              </div>
            </div>
          </div>

          <div class="flex justify-end mt-6">
            <button type="button" id="cancelBtn" class="btn btn-secondary ml-4 px-6 py-2 rounded-lg transition-all hover:bg-gray-200">
              <i class="fas fa-times ml-2"></i>
              إلغاء
            </button>
            <button type="submit" class="btn-submit">
              <i class="fas fa-user-plus ml-2"></i>
              حفظ المستخدم
            </button>
          </div>

          <!-- توقيع النظام -->
          <div class="mt-8 pt-4 border-t border-gray-200 text-center text-xs text-gray-500">
            <p>نظام إدارة طلبات الشحن - مجموعة ابراهيم الاحمدي اليمنية</p>
            <p class="mt-1">جميع الحقوق محفوظة &copy; <?php echo date('Y'); ?></p>
          </div>
        </form>
      </div>
    </div>
  </div>

  <!-- نموذج إعادة تعيين كلمة المرور -->
  <div id="resetPasswordModal" class="modal">
    <div class="modal-content">
      <div class="p-6">
        <div class="flex items-center justify-between mb-6">
          <h3 class="text-xl font-semibold text-gray-800">إعادة تعيين كلمة المرور</h3>
          <button id="closeResetModal" class="text-gray-400 hover:text-gray-600">
            <i class="fas fa-times text-xl"></i>
          </button>
        </div>

        <!-- شعار النظام -->
        <div class="text-center mb-6">
          <img src="<?php echo $company_logo; ?>" alt="شعار النظام" class="h-16 mx-auto mb-2 object-contain">
          <div class="text-sm text-gray-500">مجموعة ابراهيم الاحمدي اليمنية</div>
          <div class="h-0.5 w-24 bg-gradient-to-r from-gray-200 via-warning to-gray-200 mx-auto mt-2"></div>
        </div>

        <form id="resetPasswordForm" method="post">
          <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">
          <input type="hidden" name="action" value="reset_password">
          <input type="hidden" name="user_id" id="resetUserId" value="">

          <div class="form-group">
            <label for="newPassword" class="form-label">كلمة المرور الجديدة</label>
            <input type="password" id="newPassword" name="new_password" class="form-input" required>
          </div>

          <div class="form-group">
            <label for="confirmPassword" class="form-label">تأكيد كلمة المرور</label>
            <input type="password" id="confirmPassword" name="confirm_password" class="form-input" required>
          </div>

          <!-- تنبيه أمني -->
          <div class="bg-blue-50 border-r-4 border-blue-400 p-4 mb-6 rounded-lg">
            <div class="flex">
              <div class="flex-shrink-0">
                <i class="fas fa-shield-alt text-blue-500"></i>
              </div>
              <div class="mr-3">
                <p class="text-sm text-blue-700">
                  <strong class="font-bold">تنبيه أمني:</strong> يرجى استخدام كلمة مرور قوية تحتوي على أحرف كبيرة وصغيرة وأرقام ورموز.
                </p>
              </div>
            </div>
          </div>

          <div class="flex justify-end mt-6">
            <button type="button" id="cancelResetBtn" class="btn btn-secondary ml-4 px-6 py-2 rounded-lg transition-all hover:bg-gray-200">
              <i class="fas fa-times ml-2"></i>
              إلغاء
            </button>
            <button type="submit" class="btn-submit" style="background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);">
              <i class="fas fa-key ml-2"></i>
              تعيين كلمة المرور
            </button>
          </div>

          <!-- توقيع النظام -->
          <div class="mt-8 pt-4 border-t border-gray-200 text-center text-xs text-gray-500">
            <p>نظام إدارة طلبات الشحن - مجموعة ابراهيم الاحمدي اليمنية</p>
            <p class="mt-1">جميع الحقوق محفوظة &copy; <?php echo date('Y'); ?></p>
          </div>
        </form>
      </div>
    </div>
  </div>

  <!-- نموذج حذف مستخدم -->
  <div id="deleteModal" class="modal">
    <div class="modal-content">
      <div class="p-6">
        <div class="flex items-center justify-between mb-6">
          <h3 class="text-xl font-semibold text-gray-800">حذف مستخدم</h3>
          <button id="closeDeleteModal" class="text-gray-400 hover:text-gray-600">
            <i class="fas fa-times text-xl"></i>
          </button>
        </div>

        <!-- شعار النظام -->
        <div class="text-center mb-6">
          <img src="<?php echo $company_logo; ?>" alt="شعار النظام" class="h-16 mx-auto mb-2 object-contain">
          <div class="text-sm text-gray-500">مجموعة ابراهيم الاحمدي اليمنية</div>
          <div class="h-0.5 w-24 bg-gradient-to-r from-gray-200 via-danger to-gray-200 mx-auto mt-2"></div>
        </div>

        <div class="bg-red-50 border-r-4 border-red-400 p-4 mb-6 rounded-lg">
          <div class="flex">
            <div class="flex-shrink-0">
              <i class="fas fa-exclamation-circle text-red-500"></i>
            </div>
            <div class="mr-3">
              <p class="text-sm text-red-700">
                <strong class="font-bold">تحذير:</strong> هل أنت متأكد من رغبتك في حذف هذا المستخدم؟ هذا الإجراء لا يمكن التراجع عنه.
              </p>
              <p class="text-sm text-red-700 mt-2">
                سيتم حذف جميع بيانات المستخدم وصلاحياته من النظام بشكل نهائي.
              </p>
            </div>
          </div>
        </div>

        <form id="deleteForm" method="post">
          <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">
          <input type="hidden" name="action" value="delete">
          <input type="hidden" name="user_id" id="deleteUserId" value="">

          <div class="flex justify-end mt-6">
            <button type="button" id="cancelDeleteBtn" class="btn btn-secondary ml-4 px-6 py-2 rounded-lg transition-all hover:bg-gray-200">
              <i class="fas fa-times ml-2"></i>
              إلغاء
            </button>
            <button type="submit" class="btn-submit" style="background: linear-gradient(135deg, #ef4444 0%, #b91c1c 100%);">
              <i class="fas fa-trash-alt ml-2"></i>
              تأكيد الحذف
            </button>
          </div>

          <!-- توقيع النظام -->
          <div class="mt-8 pt-4 border-t border-gray-200 text-center text-xs text-gray-500">
            <p>نظام إدارة طلبات الشحن - مجموعة ابراهيم الاحمدي اليمنية</p>
            <p class="mt-1">جميع الحقوق محفوظة &copy; <?php echo date('Y'); ?></p>
          </div>
        </form>
      </div>
    </div>
  </div>

  <!-- سكريبت القائمة الجانبية للأجهزة المحمولة -->
  <script>
    // تبديل القائمة الجانبية للأجهزة المحمولة
    const menuBtn = document.getElementById('menuBtn');
    const sidebar = document.querySelector('aside');
    const overlay = document.getElementById('overlay');

    if (menuBtn && sidebar && overlay) {
      menuBtn.addEventListener('click', () => {
        sidebar.classList.toggle('hidden');
        sidebar.classList.toggle('fixed');
        sidebar.classList.toggle('inset-0');
        sidebar.classList.toggle('z-40');
        sidebar.classList.toggle('w-64');
        overlay.classList.toggle('hidden');
      });

      overlay.addEventListener('click', () => {
        sidebar.classList.add('hidden');
        sidebar.classList.remove('fixed', 'inset-0', 'z-40', 'w-64');
        overlay.classList.add('hidden');
      });
    }

    // تبديل القوائم المنسدلة
    const dropdownToggles = document.querySelectorAll('.dropdown-toggle');

    dropdownToggles.forEach(toggle => {
      toggle.addEventListener('click', (e) => {
        e.preventDefault();
        const menuId = toggle.getAttribute('id');
        const menu = document.getElementById(menuId.replace('Dropdown', 'Menu'));

        if (menu) {
          menu.classList.toggle('active');
        }
      });
    });

    // فلترة المستخدمين
    const filterBtn = document.getElementById('filterBtn');
    const branchFilter = document.getElementById('branchFilter');
    const roleFilter = document.getElementById('roleFilter');
    const statusFilter = document.getElementById('statusFilter');
    const userRows = document.querySelectorAll('.data-table tbody tr');

    if (filterBtn && branchFilter && roleFilter && statusFilter) {
      filterBtn.addEventListener('click', () => {
        const selectedBranch = branchFilter.value;
        const selectedRole = roleFilter.value;
        const selectedStatus = statusFilter.value;

        userRows.forEach(row => {
          const branch = row.querySelector('td:nth-child(6)').textContent;
          const role = row.querySelector('td:nth-child(5) span').textContent.trim();
          const status = row.querySelector('td:nth-child(7) span').textContent.trim();

          const branchMatch = !selectedBranch || branch === selectedBranch;
          const roleMatch = !selectedRole || role === getRoleDisplayName(selectedRole);
          const statusMatch = !selectedStatus || status === selectedStatus;

          if (branchMatch && roleMatch && statusMatch) {
            row.style.display = '';
          } else {
            row.style.display = 'none';
          }
        });
      });
    }

    // إدارة النماذج
    const userModal = document.getElementById('userModal');
    const resetPasswordModal = document.getElementById('resetPasswordModal');
    const deleteModal = document.getElementById('deleteModal');
    const addUserBtn = document.getElementById('addUserBtn');
    const closeModal = document.getElementById('closeModal');
    const cancelBtn = document.getElementById('cancelBtn');
    const closeResetModal = document.getElementById('closeResetModal');
    const cancelResetBtn = document.getElementById('cancelResetBtn');
    const closeDeleteModal = document.getElementById('closeDeleteModal');
    const cancelDeleteBtn = document.getElementById('cancelDeleteBtn');
    const editUserBtns = document.querySelectorAll('.edit-user-btn');
    const resetPasswordBtns = document.querySelectorAll('.reset-password-btn');
    const deleteUserBtns = document.querySelectorAll('.delete-user-btn');
    const modalTitle = document.getElementById('modalTitle');
    const formAction = document.getElementById('formAction');
    const userId = document.getElementById('userId');
    const resetUserId = document.getElementById('resetUserId');
    const deleteUserId = document.getElementById('deleteUserId');

    // فتح نموذج إضافة مستخدم
    if (addUserBtn && userModal) {
      addUserBtn.addEventListener('click', () => {
        modalTitle.textContent = 'إضافة مستخدم جديد';
        formAction.value = 'add';
        userId.value = '';
        document.getElementById('userForm').reset();

        // إظهار حقل كلمة المرور في وضع الإضافة
        document.getElementById('passwordGroup').style.display = 'block';
        document.getElementById('password').setAttribute('required', 'required');

        userModal.classList.add('active');
      });
    }

    // إغلاق نموذج المستخدم
    if ((closeModal && userModal) || (cancelBtn && userModal)) {
      closeModal.addEventListener('click', () => {
        userModal.classList.remove('active');
      });

      cancelBtn.addEventListener('click', () => {
        userModal.classList.remove('active');
      });
    }

    // إغلاق نموذج إعادة تعيين كلمة المرور
    if ((closeResetModal && resetPasswordModal) || (cancelResetBtn && resetPasswordModal)) {
      closeResetModal.addEventListener('click', () => {
        resetPasswordModal.classList.remove('active');
      });

      cancelResetBtn.addEventListener('click', () => {
        resetPasswordModal.classList.remove('active');
      });
    }

    // إغلاق نموذج حذف المستخدم
    if ((closeDeleteModal && deleteModal) || (cancelDeleteBtn && deleteModal)) {
      closeDeleteModal.addEventListener('click', () => {
        deleteModal.classList.remove('active');
      });

      cancelDeleteBtn.addEventListener('click', () => {
        deleteModal.classList.remove('active');
      });
    }

    // فتح نموذج تعديل مستخدم
    if (editUserBtns.length > 0 && userModal) {
      editUserBtns.forEach(btn => {
        btn.addEventListener('click', () => {
          const id = btn.getAttribute('data-id');
          modalTitle.textContent = 'تعديل بيانات المستخدم';
          formAction.value = 'edit';
          userId.value = id;

          // جلب بيانات المستخدم وملء النموذج
          const userRow = btn.closest('tr');
          const username = userRow.querySelector('td:nth-child(2)').textContent.trim();
          const fullName = userRow.querySelector('td:nth-child(3) span').textContent.trim();
          const email = userRow.querySelector('td:nth-child(4)').textContent.trim();

          // الحصول على الدور من النص المعروض
          const roleText = userRow.querySelector('td:nth-child(5) span').textContent.trim();
          let role = 'employee'; // القيمة الافتراضية

          if (roleText.includes('مدير النظام')) {
            role = 'admin';
          } else if (roleText.includes('مدير')) {
            role = 'manager';
          } else if (roleText.includes('عميل')) {
            role = 'client';
          }

          // الحصول على اسم الفرع
          const branchName = userRow.querySelector('td:nth-child(6)').textContent.trim();

          // الحصول على الحالة
          const statusText = userRow.querySelector('td:nth-child(7) span').textContent.trim();
          let status = 'active'; // القيمة الافتراضية

          if (statusText === 'غير نشط') {
            status = 'inactive';
          } else if (statusText === 'معلق') {
            status = 'pending';
          }

          // ملء النموذج بالبيانات
          document.getElementById('username').value = username;
          document.getElementById('fullName').value = fullName;
          document.getElementById('email').value = email;
          document.getElementById('role').value = role;

          // تحديد الفرع المناسب
          const branchSelect = document.getElementById('branch');
          for (let i = 0; i < branchSelect.options.length; i++) {
            if (branchSelect.options[i].text === branchName || branchName === 'غير محدد' && branchSelect.options[i].value === '') {
              branchSelect.selectedIndex = i;
              break;
            }
          }

          document.getElementById('status').value = status;

          // إخفاء حقل كلمة المرور في وضع التعديل
          document.getElementById('passwordGroup').style.display = 'none';
          document.getElementById('password').removeAttribute('required');

          // جلب صلاحيات المستخدم من الخادم
          fetch('api/get_user_permissions.php?user_id=' + id)
            .then(response => response.json())
            .then(data => {
              if (data.success && data.permissions) {
                // تحديد الصلاحيات في النموذج
                const permissionIds = data.permissions.map(p => p.id);
                const checkboxes = document.querySelectorAll('input[name="permissions[]"]');
                checkboxes.forEach(checkbox => {
                  if (permissionIds.includes(parseInt(checkbox.value))) {
                    checkbox.checked = true;
                  } else {
                    checkbox.checked = false;
                  }
                });
              }
            })
            .catch(error => {
              console.error('خطأ في جلب صلاحيات المستخدم:', error);
            });

          userModal.classList.add('active');
        });
      });
    }

    // فتح نموذج إعادة تعيين كلمة المرور
    if (resetPasswordBtns.length > 0 && resetPasswordModal) {
      resetPasswordBtns.forEach(btn => {
        btn.addEventListener('click', () => {
          const id = btn.getAttribute('data-id');
          resetUserId.value = id;
          resetPasswordModal.classList.add('active');
        });
      });
    }

    // فتح نموذج حذف مستخدم
    if (deleteUserBtns.length > 0 && deleteModal) {
      deleteUserBtns.forEach(btn => {
        btn.addEventListener('click', () => {
          const id = btn.getAttribute('data-id');
          deleteUserId.value = id;
          deleteModal.classList.add('active');
        });
      });
    }

    // التحقق من تطابق كلمات المرور
    const resetPasswordForm = document.getElementById('resetPasswordForm');
    const newPassword = document.getElementById('newPassword');
    const confirmPassword = document.getElementById('confirmPassword');

    if (resetPasswordForm) {
      resetPasswordForm.addEventListener('submit', (e) => {
        if (newPassword.value !== confirmPassword.value) {
          e.preventDefault();
          alert('كلمات المرور غير متطابقة');
        }
      });
    }

    // وظيفة مساعدة لتحويل اسم الدور
    function getRoleDisplayName(roleName) {
      switch (roleName) {
        case 'admin':
          return 'مدير النظام';
        case 'manager':
          return 'مدير';
        case 'user':
          return 'مستخدم';
        default:
          return roleName;
      }
    }

    // تعريف مصفوفة صلاحيات المستخدم العادي
    const regularPermissionIds = [
      'perm_dashboard_view_own',
      'perm_orders_add',
      'perm_orders_manage_own',
      'perm_orders_edit_own',
      'perm_weekly_reports_view_own',
      'perm_monthly_reports_view_own',
      'perm_support_contact_admin'
    ];

    // الحصول على عناصر صلاحيات المستخدم العادي
    const getRegularPermissionCheckboxes = () => {
      return regularPermissionIds.map(id => document.getElementById(id)).filter(checkbox => checkbox !== null);
    };

    // تحديد جميع صلاحيات المستخدم العادي
    const selectAllRegularPermissionsFunc = () => {
      getRegularPermissionCheckboxes().forEach(checkbox => {
        checkbox.checked = true;
      });
    };

    // إلغاء تحديد جميع صلاحيات المستخدم العادي
    const deselectAllRegularPermissionsFunc = () => {
      getRegularPermissionCheckboxes().forEach(checkbox => {
        checkbox.checked = false;
      });
    };

    // إدارة أزرار تحديد/إلغاء تحديد صلاحيات المستخدم العادي
    const selectAllRegularPermissions = document.getElementById('selectAllRegularPermissions');
    const deselectAllRegularPermissions = document.getElementById('deselectAllRegularPermissions');

    if (selectAllRegularPermissions && deselectAllRegularPermissions) {
      // تحديد جميع صلاحيات المستخدم العادي
      selectAllRegularPermissions.addEventListener('click', selectAllRegularPermissionsFunc);

      // إلغاء تحديد جميع صلاحيات المستخدم العادي
      deselectAllRegularPermissions.addEventListener('click', deselectAllRegularPermissionsFunc);
    }

    // تحديد صلاحيات المستخدم العادي تلقائياً عند اختيار دور "موظف" أو "عميل"
    const roleSelect = document.getElementById('role');
    if (roleSelect) {
      roleSelect.addEventListener('change', () => {
        const selectedRole = roleSelect.value;

        // إذا كان الدور "موظف" أو "عميل"، حدد صلاحيات المستخدم العادي
        if (selectedRole === 'employee' || selectedRole === 'client') {
          selectAllRegularPermissionsFunc();
        }
      });

      // تحديد صلاحيات المستخدم العادي عند فتح النموذج إذا كان الدور "موظف" أو "عميل"
      if (addUserBtn) {
        addUserBtn.addEventListener('click', () => {
          // تأخير قليل للتأكد من أن النموذج قد تم تحميله بالكامل
          setTimeout(() => {
            const selectedRole = roleSelect.value;
            if (selectedRole === 'employee' || selectedRole === 'client') {
              selectAllRegularPermissionsFunc();
            }
          }, 100);
        });
      }
    }
  </script>
</body>
</html>
