<?php
/**
 * أداة تنظيم المشروع
 * تستخدم هذه الأداة لتنظيم هيكل المجلدات وإزالة الملفات المكررة
 * 
 * طريقة الاستخدام:
 * 1. قم بتشغيل هذا الملف من المتصفح
 * 2. اختر العمليات التي تريد تنفيذها
 * 3. انقر على زر "تنفيذ"
 */

// تفعيل عرض الأخطاء للتصحيح
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// تعيين وقت التنفيذ الأقصى
set_time_limit(300);

// تعيين حد الذاكرة
ini_set('memory_limit', '256M');

// تحديد المسار الأساسي للمشروع
$base_path = realpath(__DIR__ . '/..');

// تحديد هيكل المجلدات المطلوب
$required_folders = [
    'api',
    'assets',
    'assets/css',
    'assets/js',
    'assets/images',
    'assets/images/logo',
    'assets/images/flags',
    'assets/fonts',
    'config',
    'database',
    'includes',
    'pages',
    'templates',
    'utils'
];

// تحديد الملفات المكررة التي يجب حذفها
$duplicate_files = [
    // ملفات الاختبار
    'test.php',
    'test_connection.php',
    'test_db.php',
    'test_db_connection.php',
    'test_db_connection_updated.php',
    'test_login.php',
    'test_order_number.php',
    'test-login.php',
    'test-register.php',
    'simple-register.php',
    'login-test.php',
    'login-test-new.php',
    'register-simple.php',
    'debug_register.php',
    'debug_session.php',
    
    // ملفات التكوين المكررة
    'subdomain_config.php',
    'shipping/config/subdomain_config.php',
    'shipping/config/deployment.php',
    'config/deployment.php',
    'deploy_config.php',
    
    // ملفات قاعدة البيانات المكررة
    'shipping/database/create_orders_table.sql',
    'shipping/database/create_users_table.sql',
    'shipping/database/create_sqlite_tables.sql',
    
    // ملفات تحويل SVG المكررة
    'convert_svg_simple.py',
    'convert_svg_to_png.py',
    'shipping/convert-svg-simple.ps1',
    'shipping/convert-svg-to-png.ps1'
];

// تحديد الملفات التي يجب نقلها
$files_to_move = [
    // نقل ملفات CSS
    ['source' => 'css', 'destination' => 'assets/css', 'pattern' => '*.css'],
    
    // نقل ملفات JavaScript
    ['source' => 'js', 'destination' => 'assets/js', 'pattern' => '*.js'],
    
    // نقل ملفات HTML
    ['source' => '.', 'destination' => 'pages', 'pattern' => '*.html', 'exclude' => ['index.html']],
    
    // نقل ملفات الصور
    ['source' => 'images', 'destination' => 'assets/images', 'pattern' => '*.*'],
    
    // نقل ملفات API
    ['source' => 'api', 'destination' => 'api', 'pattern' => '*.php'],
    
    // نقل ملفات قاعدة البيانات
    ['source' => '.', 'destination' => 'database', 'pattern' => '*.sql'],
    
    // نقل ملفات التكوين
    ['source' => '.', 'destination' => 'config', 'pattern' => '*config*.php', 'exclude' => ['config.php']],
    
    // نقل ملفات الأدوات المساعدة
    ['source' => '.', 'destination' => 'utils', 'pattern' => ['*.ps1', '*.py'], 'exclude' => ['convert-svg-simple.ps1']]
];

// دالة لإنشاء المجلدات المطلوبة
function createRequiredFolders($folders, $base_path) {
    $created = [];
    $errors = [];
    
    foreach ($folders as $folder) {
        $folder_path = $base_path . '/' . $folder;
        
        if (!file_exists($folder_path)) {
            if (mkdir($folder_path, 0755, true)) {
                $created[] = $folder;
            } else {
                $errors[] = $folder;
            }
        }
    }
    
    return [
        'created' => $created,
        'errors' => $errors
    ];
}

// دالة لحذف الملفات المكررة
function removeDuplicateFiles($files, $base_path) {
    $removed = [];
    $errors = [];
    $not_found = [];
    
    foreach ($files as $file) {
        $file_path = $base_path . '/' . $file;
        
        if (file_exists($file_path)) {
            if (unlink($file_path)) {
                $removed[] = $file;
            } else {
                $errors[] = $file;
            }
        } else {
            $not_found[] = $file;
        }
    }
    
    return [
        'removed' => $removed,
        'errors' => $errors,
        'not_found' => $not_found
    ];
}

// دالة لنقل الملفات
function moveFiles($files_to_move, $base_path) {
    $moved = [];
    $errors = [];
    
    foreach ($files_to_move as $move_info) {
        $source_dir = $base_path . '/' . $move_info['source'];
        $dest_dir = $base_path . '/' . $move_info['destination'];
        
        // التأكد من وجود المجلد الوجهة
        if (!file_exists($dest_dir)) {
            mkdir($dest_dir, 0755, true);
        }
        
        // الحصول على قائمة الملفات
        $patterns = isset($move_info['pattern']) ? (is_array($move_info['pattern']) ? $move_info['pattern'] : [$move_info['pattern']]) : ['*'];
        $exclude = isset($move_info['exclude']) ? (is_array($move_info['exclude']) ? $move_info['exclude'] : [$move_info['exclude']]) : [];
        
        foreach ($patterns as $pattern) {
            $files = glob($source_dir . '/' . $pattern);
            
            foreach ($files as $file) {
                $filename = basename($file);
                
                // تخطي الملفات المستثناة
                if (in_array($filename, $exclude)) {
                    continue;
                }
                
                $dest_file = $dest_dir . '/' . $filename;
                
                // نقل الملف
                if (copy($file, $dest_file)) {
                    $moved[] = [
                        'source' => str_replace($base_path . '/', '', $file),
                        'destination' => str_replace($base_path . '/', '', $dest_file)
                    ];
                    
                    // حذف الملف الأصلي
                    unlink($file);
                } else {
                    $errors[] = str_replace($base_path . '/', '', $file);
                }
            }
        }
    }
    
    return [
        'moved' => $moved,
        'errors' => $errors
    ];
}

// معالجة النموذج
$create_folders = isset($_POST['create_folders']) && $_POST['create_folders'] === 'yes';
$remove_duplicates = isset($_POST['remove_duplicates']) && $_POST['remove_duplicates'] === 'yes';
$move_files = isset($_POST['move_files']) && $_POST['move_files'] === 'yes';

// النتائج
$results = [];

// تنفيذ العمليات المطلوبة
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if ($create_folders) {
        $results['folders'] = createRequiredFolders($required_folders, $base_path);
    }
    
    if ($remove_duplicates) {
        $results['duplicates'] = removeDuplicateFiles($duplicate_files, $base_path);
    }
    
    if ($move_files) {
        $results['files'] = moveFiles($files_to_move, $base_path);
    }
}
?>

<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>أداة تنظيم المشروع</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            margin: 0;
            padding: 20px;
            background-color: #f5f5f5;
        }
        .container {
            max-width: 800px;
            margin: 0 auto;
            background-color: #fff;
            padding: 20px;
            border-radius: 5px;
            box-shadow: 0 2px 5px rgba(0, 0, 0, 0.1);
        }
        h1 {
            color: #333;
            text-align: center;
        }
        .form-group {
            margin-bottom: 15px;
        }
        label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
        }
        button {
            background-color: #4CAF50;
            color: white;
            padding: 10px 15px;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-size: 16px;
        }
        button:hover {
            background-color: #45a049;
        }
        .success {
            color: #4CAF50;
            background-color: #e8f5e9;
            padding: 10px;
            border-radius: 4px;
            margin-top: 10px;
        }
        .error {
            color: #f44336;
            background-color: #ffebee;
            padding: 10px;
            border-radius: 4px;
            margin-top: 10px;
        }
        ul {
            margin: 5px 0;
            padding-right: 20px;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>أداة تنظيم المشروع</h1>
        
        <form method="post">
            <div class="form-group">
                <label>
                    <input type="checkbox" name="create_folders" value="yes" <?php echo $create_folders ? 'checked' : ''; ?>>
                    إنشاء المجلدات المطلوبة
                </label>
            </div>
            
            <div class="form-group">
                <label>
                    <input type="checkbox" name="remove_duplicates" value="yes" <?php echo $remove_duplicates ? 'checked' : ''; ?>>
                    إزالة الملفات المكررة
                </label>
            </div>
            
            <div class="form-group">
                <label>
                    <input type="checkbox" name="move_files" value="yes" <?php echo $move_files ? 'checked' : ''; ?>>
                    نقل الملفات إلى المجلدات المناسبة
                </label>
            </div>
            
            <button type="submit">تنفيذ</button>
        </form>
        
        <?php if (!empty($results)): ?>
            <h2>نتائج التنفيذ</h2>
            
            <?php if (isset($results['folders'])): ?>
                <h3>إنشاء المجلدات</h3>
                
                <?php if (!empty($results['folders']['created'])): ?>
                    <div class="success">
                        <p>تم إنشاء المجلدات التالية بنجاح:</p>
                        <ul>
                            <?php foreach ($results['folders']['created'] as $folder): ?>
                                <li><?php echo htmlspecialchars($folder); ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                <?php endif; ?>
                
                <?php if (!empty($results['folders']['errors'])): ?>
                    <div class="error">
                        <p>فشل في إنشاء المجلدات التالية:</p>
                        <ul>
                            <?php foreach ($results['folders']['errors'] as $folder): ?>
                                <li><?php echo htmlspecialchars($folder); ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                <?php endif; ?>
            <?php endif; ?>
            
            <?php if (isset($results['duplicates'])): ?>
                <h3>إزالة الملفات المكررة</h3>
                
                <?php if (!empty($results['duplicates']['removed'])): ?>
                    <div class="success">
                        <p>تم إزالة الملفات التالية بنجاح:</p>
                        <ul>
                            <?php foreach ($results['duplicates']['removed'] as $file): ?>
                                <li><?php echo htmlspecialchars($file); ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                <?php endif; ?>
                
                <?php if (!empty($results['duplicates']['errors'])): ?>
                    <div class="error">
                        <p>فشل في إزالة الملفات التالية:</p>
                        <ul>
                            <?php foreach ($results['duplicates']['errors'] as $file): ?>
                                <li><?php echo htmlspecialchars($file); ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                <?php endif; ?>
                
                <?php if (!empty($results['duplicates']['not_found'])): ?>
                    <div class="error">
                        <p>الملفات التالية غير موجودة:</p>
                        <ul>
                            <?php foreach ($results['duplicates']['not_found'] as $file): ?>
                                <li><?php echo htmlspecialchars($file); ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                <?php endif; ?>
            <?php endif; ?>
            
            <?php if (isset($results['files'])): ?>
                <h3>نقل الملفات</h3>
                
                <?php if (!empty($results['files']['moved'])): ?>
                    <div class="success">
                        <p>تم نقل الملفات التالية بنجاح:</p>
                        <ul>
                            <?php foreach ($results['files']['moved'] as $file): ?>
                                <li><?php echo htmlspecialchars($file['source'] . ' -> ' . $file['destination']); ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                <?php endif; ?>
                
                <?php if (!empty($results['files']['errors'])): ?>
                    <div class="error">
                        <p>فشل في نقل الملفات التالية:</p>
                        <ul>
                            <?php foreach ($results['files']['errors'] as $file): ?>
                                <li><?php echo htmlspecialchars($file); ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                <?php endif; ?>
            <?php endif; ?>
        <?php endif; ?>
    </div>
</body>
</html>
