<?php
/**
 * نظام حماية وتحسين أداء النظام
 * 
 * هذا الملف يحتوي على وظائف لحماية النظام من الانهيار وتحسين أدائه
 * يتم استدعاؤه في بداية كل صفحة من صفحات النظام
 * 
 * @version 1.0
 * @author فريق التطوير
 */

// منع الوصول المباشر للملف
if (!defined('SYSTEM_PROTECTION_LOADED')) {
    define('SYSTEM_PROTECTION_LOADED', true);
} else {
    // تجنب تحميل الملف أكثر من مرة
    return;
}

// تعيين منطقة زمنية افتراضية
date_default_timezone_set('Asia/Riyadh');

// تعيين ترميز افتراضي
ini_set('default_charset', 'UTF-8');

// تعيين حد الذاكرة
ini_set('memory_limit', '256M');

// تعيين وقت التنفيذ الأقصى
set_time_limit(300);

// تعيين مستوى الإبلاغ عن الأخطاء
if (defined('DEVELOPMENT_MODE') && DEVELOPMENT_MODE === true) {
    // وضع التطوير: عرض جميع الأخطاء
    ini_set('display_errors', 1);
    ini_set('display_startup_errors', 1);
    error_reporting(E_ALL);
} else {
    // وضع الإنتاج: إخفاء الأخطاء
    ini_set('display_errors', 0);
    ini_set('display_startup_errors', 0);
    error_reporting(E_ALL & ~E_NOTICE & ~E_DEPRECATED & ~E_STRICT);
    
    // تسجيل الأخطاء في ملف
    ini_set('log_errors', 1);
    ini_set('error_log', __DIR__ . '/../logs/error.log');
}

// التأكد من وجود مجلد السجلات
if (!file_exists(__DIR__ . '/../logs')) {
    mkdir(__DIR__ . '/../logs', 0755, true);
}

// دالة لتسجيل الأخطاء
function logError($message, $severity = 'ERROR') {
    $logFile = __DIR__ . '/../logs/error.log';
    $timestamp = date('Y-m-d H:i:s');
    $logMessage = "[$timestamp] [$severity] $message" . PHP_EOL;
    
    // كتابة السجل في الملف
    file_put_contents($logFile, $logMessage, FILE_APPEND);
}

// دالة لتسجيل النشاط
function logActivity($userId, $action, $details = '') {
    global $conn;
    
    // التحقق من وجود اتصال بقاعدة البيانات
    if (!isset($conn) || !$conn) {
        logError("فشل تسجيل النشاط: لا يوجد اتصال بقاعدة البيانات", 'WARNING');
        return false;
    }
    
    try {
        $stmt = $conn->prepare("
            INSERT INTO activity_log (user_id, action, details, ip_address, user_agent, created_at)
            VALUES (?, ?, ?, ?, ?, NOW())
        ");
        
        $ipAddress = $_SERVER['REMOTE_ADDR'] ?? 'غير معروف';
        $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? 'غير معروف';
        
        $stmt->execute([$userId, $action, $details, $ipAddress, $userAgent]);
        return true;
    } catch (PDOException $e) {
        logError("فشل تسجيل النشاط: " . $e->getMessage());
        return false;
    }
}

// دالة للتحقق من صحة الجلسة
function validateSession() {
    // التحقق من وجود جلسة نشطة
    if (!isset($_SESSION['user_id']) || !isset($_SESSION['logged_in']) || $_SESSION['logged_in'] !== true) {
        return false;
    }
    
    // التحقق من وقت انتهاء الجلسة
    if (isset($_SESSION['expires_at']) && time() > $_SESSION['expires_at']) {
        // انتهت صلاحية الجلسة
        session_unset();
        session_destroy();
        return false;
    }
    
    // تحديث وقت انتهاء الجلسة
    $_SESSION['expires_at'] = time() + (60 * 60 * 2); // ساعتين
    
    return true;
}

// دالة للتحقق من صلاحيات المستخدم
function checkPermission($requiredRole) {
    // التحقق من وجود جلسة صالحة
    if (!validateSession()) {
        return false;
    }
    
    // التحقق من دور المستخدم
    if (!isset($_SESSION['role'])) {
        return false;
    }
    
    // إذا كان المستخدم مديرًا، فلديه جميع الصلاحيات
    if ($_SESSION['role'] === 'admin') {
        return true;
    }
    
    // التحقق من الدور المطلوب
    if (is_array($requiredRole)) {
        return in_array($_SESSION['role'], $requiredRole);
    } else {
        return $_SESSION['role'] === $requiredRole;
    }
}

// دالة للتحقق من صحة اتصال قاعدة البيانات
function checkDatabaseConnection($conn) {
    if (!$conn) {
        logError("فشل الاتصال بقاعدة البيانات", 'CRITICAL');
        return false;
    }
    
    try {
        // محاولة تنفيذ استعلام بسيط
        $stmt = $conn->query("SELECT 1");
        return $stmt !== false;
    } catch (PDOException $e) {
        logError("فشل الاتصال بقاعدة البيانات: " . $e->getMessage(), 'CRITICAL');
        return false;
    }
}

// دالة لتنظيف المدخلات
function sanitizeInput($input) {
    if (is_array($input)) {
        foreach ($input as $key => $value) {
            $input[$key] = sanitizeInput($value);
        }
        return $input;
    }
    
    // تنظيف المدخلات النصية
    $input = trim($input);
    $input = stripslashes($input);
    $input = htmlspecialchars($input, ENT_QUOTES, 'UTF-8');
    
    return $input;
}

// دالة للتحقق من صحة البريد الإلكتروني
function validateEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

// دالة للتحقق من قوة كلمة المرور
function validatePassword($password) {
    // التحقق من طول كلمة المرور
    if (strlen($password) < 8) {
        return false;
    }
    
    // التحقق من وجود حرف كبير
    if (!preg_match('/[A-Z]/', $password)) {
        return false;
    }
    
    // التحقق من وجود حرف صغير
    if (!preg_match('/[a-z]/', $password)) {
        return false;
    }
    
    // التحقق من وجود رقم
    if (!preg_match('/[0-9]/', $password)) {
        return false;
    }
    
    return true;
}

// دالة لإنشاء رمز CSRF
function generateCSRFToken() {
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    
    return $_SESSION['csrf_token'];
}

// دالة للتحقق من صحة رمز CSRF
function validateCSRFToken($token) {
    if (!isset($_SESSION['csrf_token']) || $token !== $_SESSION['csrf_token']) {
        logError("فشل التحقق من رمز CSRF", 'SECURITY');
        return false;
    }
    
    return true;
}

// دالة لإعادة توجيه المستخدم مع رسالة
function redirectWithMessage($url, $message, $type = 'success') {
    $_SESSION['flash_message'] = $message;
    $_SESSION['flash_type'] = $type;
    
    header("Location: $url");
    exit;
}

// دالة لعرض رسالة فلاش
function displayFlashMessage() {
    if (isset($_SESSION['flash_message'])) {
        $message = $_SESSION['flash_message'];
        $type = $_SESSION['flash_type'] ?? 'success';
        
        // حذف الرسالة من الجلسة
        unset($_SESSION['flash_message']);
        unset($_SESSION['flash_type']);
        
        return [
            'message' => $message,
            'type' => $type
        ];
    }
    
    return null;
}

// تنظيف مدخلات POST و GET
$_POST = sanitizeInput($_POST);
$_GET = sanitizeInput($_GET);

// إنشاء رمز CSRF إذا لم يكن موجودًا
if (session_status() === PHP_SESSION_ACTIVE) {
    generateCSRFToken();
}
