<?php
/**
 * أداة كشف الكود غير المستخدم
 * تستخدم هذه الأداة للبحث عن الملفات والأكواد غير المستخدمة في المشروع
 * 
 * @version 1.0
 */

// تفعيل عرض الأخطاء للتصحيح
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// تعيين وقت التنفيذ الأقصى
set_time_limit(300);

// تعيين حد الذاكرة
ini_set('memory_limit', '256M');

// تحديد المسار الأساسي للمشروع
$base_path = realpath(__DIR__ . '/..');

// الملفات التي سيتم استثناؤها من الفحص
$excluded_files = [
    'index.php',
    'index.html',
    'config.php',
    '.htaccess',
    'README.md',
    'utils/unused_code_detector.php',
    'utils/cleanup_duplicates.php',
    'utils/security_enhancer.php',
    'utils/project_organizer.php',
    'utils/unused_files_detector.php',
    'database/setup_cpanel_database.php',
    'database/unified_database_setup.sql',
    'docs/CPANEL_DEPLOYMENT_GUIDE.md',
    'includes/security_functions.php'
];

// المجلدات التي سيتم استثناؤها من الفحص
$excluded_directories = [
    '.git',
    '.github',
    'node_modules',
    'vendor',
    '.vscode',
    'backups',
    'logs',
    'cache'
];

// أنواع الملفات التي سيتم فحصها
$file_types = [
    'php',
    'js',
    'css',
    'html'
];

// دالة للحصول على جميع الملفات في المشروع
function getAllFiles($base_path, $file_types, $excluded_directories) {
    $files = [];
    
    $iterator = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($base_path, RecursiveDirectoryIterator::SKIP_DOTS)
    );
    
    foreach ($iterator as $file) {
        // تجاهل المجلدات المستثناة
        $relative_path = str_replace($base_path . DIRECTORY_SEPARATOR, '', $file->getPathname());
        $is_excluded = false;
        
        foreach ($excluded_directories as $excluded_dir) {
            if (strpos($relative_path, $excluded_dir . DIRECTORY_SEPARATOR) === 0 || $relative_path === $excluded_dir) {
                $is_excluded = true;
                break;
            }
        }
        
        if ($is_excluded) {
            continue;
        }
        
        // تحقق من نوع الملف
        $extension = $file->getExtension();
        if (in_array(strtolower($extension), $file_types)) {
            $files[] = $relative_path;
        }
    }
    
    return $files;
}

// دالة للبحث عن الملفات غير المستخدمة
function findUnusedFiles($base_path, $files, $excluded_files) {
    $used_files = [];
    $unused_files = [];
    
    // فحص كل ملف للبحث عن استدعاءات لملفات أخرى
    foreach ($files as $file) {
        $content = file_get_contents($base_path . DIRECTORY_SEPARATOR . $file);
        
        // البحث عن استدعاءات require و include
        preg_match_all('/(?:require|include)(?:_once)?\s*\(\s*[\'"](.+?)[\'"]\s*\)/', $content, $matches);
        
        if (!empty($matches[1])) {
            foreach ($matches[1] as $match) {
                // تحويل المسار النسبي إلى مسار كامل
                $included_file = realpath($base_path . DIRECTORY_SEPARATOR . dirname($file) . DIRECTORY_SEPARATOR . $match);
                
                if ($included_file) {
                    $included_file = str_replace($base_path . DIRECTORY_SEPARATOR, '', $included_file);
                    $used_files[] = $included_file;
                }
            }
        }
        
        // البحث عن روابط CSS و JavaScript
        preg_match_all('/(?:href|src)\s*=\s*[\'"](.+?)[\'"]/i', $content, $matches);
        
        if (!empty($matches[1])) {
            foreach ($matches[1] as $match) {
                // تجاهل الروابط الخارجية
                if (strpos($match, 'http') === 0 || strpos($match, '//') === 0) {
                    continue;
                }
                
                // تحويل المسار النسبي إلى مسار كامل
                $linked_file = realpath($base_path . DIRECTORY_SEPARATOR . dirname($file) . DIRECTORY_SEPARATOR . $match);
                
                if ($linked_file) {
                    $linked_file = str_replace($base_path . DIRECTORY_SEPARATOR, '', $linked_file);
                    $used_files[] = $linked_file;
                }
            }
        }
    }
    
    // تحديد الملفات غير المستخدمة
    foreach ($files as $file) {
        if (!in_array($file, $used_files) && !in_array($file, $excluded_files)) {
            $unused_files[] = $file;
        }
    }
    
    return $unused_files;
}

// دالة للبحث عن الدوال غير المستخدمة
function findUnusedFunctions($base_path, $files) {
    $functions = [];
    $used_functions = [];
    $unused_functions = [];
    
    // استخراج جميع الدوال المعرفة في المشروع
    foreach ($files as $file) {
        if (pathinfo($file, PATHINFO_EXTENSION) !== 'php') {
            continue;
        }
        
        $content = file_get_contents($base_path . DIRECTORY_SEPARATOR . $file);
        
        // البحث عن تعريفات الدوال
        preg_match_all('/function\s+([a-zA-Z0-9_]+)\s*\(/i', $content, $matches);
        
        if (!empty($matches[1])) {
            foreach ($matches[1] as $function_name) {
                $functions[$function_name][] = $file;
            }
        }
    }
    
    // البحث عن استدعاءات الدوال
    foreach ($files as $file) {
        $content = file_get_contents($base_path . DIRECTORY_SEPARATOR . $file);
        
        foreach (array_keys($functions) as $function_name) {
            // تجاهل الدالة نفسها في ملفها
            if (in_array($file, $functions[$function_name])) {
                continue;
            }
            
            // البحث عن استدعاءات الدالة
            if (preg_match('/\b' . preg_quote($function_name, '/') . '\s*\(/i', $content)) {
                $used_functions[$function_name] = true;
            }
        }
    }
    
    // تحديد الدوال غير المستخدمة
    foreach ($functions as $function_name => $files) {
        if (!isset($used_functions[$function_name])) {
            $unused_functions[$function_name] = $files;
        }
    }
    
    return $unused_functions;
}

// تنفيذ العمليات
$all_files = getAllFiles($base_path, $file_types, $excluded_directories);
$unused_files = findUnusedFiles($base_path, $all_files, $excluded_files);
$unused_functions = findUnusedFunctions($base_path, $all_files);

// عرض النتائج
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>كشف الكود غير المستخدم</title>
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            line-height: 1.6;
            color: #333;
            margin: 0;
            padding: 20px;
            background-color: #f5f5f5;
        }
        .container {
            max-width: 1000px;
            margin: 0 auto;
            background-color: #fff;
            padding: 20px;
            border-radius: 5px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        }
        h1, h2, h3 {
            color: #2c3e50;
        }
        .file-list {
            list-style-type: none;
            padding: 0;
        }
        .file-list li {
            padding: 8px 12px;
            margin-bottom: 5px;
            background-color: #f8f9fa;
            border-left: 3px solid #e74c3c;
            border-radius: 3px;
        }
        .function-list {
            list-style-type: none;
            padding: 0;
        }
        .function-list li {
            padding: 8px 12px;
            margin-bottom: 5px;
            background-color: #f8f9fa;
            border-left: 3px solid #3498db;
            border-radius: 3px;
        }
        .function-files {
            font-size: 0.9em;
            color: #7f8c8d;
            margin-top: 5px;
        }
        .back-link {
            display: inline-block;
            margin-top: 20px;
            padding: 10px 15px;
            background-color: #3498db;
            color: white;
            text-decoration: none;
            border-radius: 4px;
        }
        .back-link:hover {
            background-color: #2980b9;
        }
        .summary {
            background-color: #f0f7fb;
            border-left: 5px solid #3498db;
            padding: 10px 15px;
            margin-bottom: 20px;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>كشف الكود غير المستخدم</h1>
        
        <div class="summary">
            <p><strong>إجمالي الملفات التي تم فحصها:</strong> <?php echo count($all_files); ?></p>
            <p><strong>الملفات غير المستخدمة:</strong> <?php echo count($unused_files); ?></p>
            <p><strong>الدوال غير المستخدمة:</strong> <?php echo count($unused_functions); ?></p>
        </div>
        
        <h2>الملفات غير المستخدمة</h2>
        <?php if (!empty($unused_files)): ?>
            <ul class="file-list">
                <?php foreach ($unused_files as $file): ?>
                    <li><?php echo htmlspecialchars($file); ?></li>
                <?php endforeach; ?>
            </ul>
        <?php else: ?>
            <p>لم يتم العثور على ملفات غير مستخدمة.</p>
        <?php endif; ?>
        
        <h2>الدوال غير المستخدمة</h2>
        <?php if (!empty($unused_functions)): ?>
            <ul class="function-list">
                <?php foreach ($unused_functions as $function_name => $files): ?>
                    <li>
                        <strong><?php echo htmlspecialchars($function_name); ?></strong>
                        <div class="function-files">
                            الملفات:
                            <?php foreach ($files as $file): ?>
                                <span><?php echo htmlspecialchars($file); ?></span>
                                <?php if (end($files) !== $file): ?>, <?php endif; ?>
                            <?php endforeach; ?>
                        </div>
                    </li>
                <?php endforeach; ?>
            </ul>
        <?php else: ?>
            <p>لم يتم العثور على دوال غير مستخدمة.</p>
        <?php endif; ?>
        
        <a href="../index.php" class="back-link">العودة إلى الصفحة الرئيسية</a>
    </div>
</body>
</html>
