<?php
/**
 * أداة كشف الملفات غير المستخدمة
 * تستخدم هذه الأداة للبحث عن الملفات التي لا يتم استدعاؤها في المشروع
 * 
 * طريقة الاستخدام:
 * 1. قم بتشغيل هذا الملف من المتصفح
 * 2. سيتم عرض قائمة بالملفات التي لم يتم العثور على استدعاءات لها
 */

// تفعيل عرض الأخطاء للتصحيح
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// تعيين وقت التنفيذ الأقصى
set_time_limit(300);

// تعيين حد الذاكرة
ini_set('memory_limit', '256M');

// تحديد المسار الأساسي للمشروع
$base_path = realpath(__DIR__ . '/..');

// أنواع الملفات التي سيتم فحصها
$file_types = [
    'php' => ['php'],
    'js' => ['js'],
    'css' => ['css'],
    'html' => ['html', 'htm']
];

// الملفات التي سيتم استثناؤها من الفحص
$excluded_files = [
    'index.php',
    'index.html',
    'config.php',
    '.htaccess',
    'README.md',
    'utils/unused_files_detector.php',
    'utils/project_organizer.php'
];

// المجلدات التي سيتم استثناؤها من الفحص
$excluded_directories = [
    '.git',
    '.github',
    'node_modules',
    'vendor',
    '.vscode'
];

// دالة للحصول على جميع الملفات في المشروع
function getAllFiles($base_path, $file_types, $excluded_directories) {
    $files = [];
    
    $iterator = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($base_path, RecursiveDirectoryIterator::SKIP_DOTS)
    );
    
    foreach ($iterator as $file) {
        // تخطي المجلدات المستثناة
        $relative_path = str_replace($base_path . '/', '', $file->getPathname());
        $is_excluded = false;
        
        foreach ($excluded_directories as $excluded_dir) {
            if (strpos($relative_path, $excluded_dir . '/') === 0 || $relative_path === $excluded_dir) {
                $is_excluded = true;
                break;
            }
        }
        
        if ($is_excluded) {
            continue;
        }
        
        // التحقق من نوع الملف
        $extension = strtolower($file->getExtension());
        $file_type = null;
        
        foreach ($file_types as $type => $extensions) {
            if (in_array($extension, $extensions)) {
                $file_type = $type;
                break;
            }
        }
        
        if ($file_type !== null) {
            $files[] = [
                'path' => $relative_path,
                'type' => $file_type,
                'name' => $file->getFilename()
            ];
        }
    }
    
    return $files;
}

// دالة للبحث عن استدعاءات الملف في المشروع
function findFileReferences($file, $all_files, $base_path) {
    $references = [];
    $file_name = $file['name'];
    $file_path = $file['path'];
    
    // البحث في جميع الملفات
    foreach ($all_files as $search_file) {
        // تخطي الملف نفسه
        if ($search_file['path'] === $file_path) {
            continue;
        }
        
        $search_file_content = file_get_contents($base_path . '/' . $search_file['path']);
        
        // البحث عن اسم الملف
        if (strpos($search_file_content, $file_name) !== false) {
            $references[] = $search_file['path'];
        }
        
        // البحث عن مسار الملف
        if (strpos($search_file_content, $file_path) !== false) {
            if (!in_array($search_file['path'], $references)) {
                $references[] = $search_file['path'];
            }
        }
    }
    
    return $references;
}

// الحصول على جميع الملفات
$all_files = getAllFiles($base_path, $file_types, $excluded_directories);

// البحث عن الملفات غير المستخدمة
$unused_files = [];
$used_files = [];

foreach ($all_files as $file) {
    // تخطي الملفات المستثناة
    if (in_array($file['path'], $excluded_files)) {
        continue;
    }
    
    // البحث عن استدعاءات الملف
    $references = findFileReferences($file, $all_files, $base_path);
    
    if (empty($references)) {
        $unused_files[] = $file;
    } else {
        $used_files[] = [
            'file' => $file,
            'references' => $references
        ];
    }
}

// فرز الملفات غير المستخدمة حسب النوع
$unused_by_type = [];

foreach ($unused_files as $file) {
    $type = $file['type'];
    
    if (!isset($unused_by_type[$type])) {
        $unused_by_type[$type] = [];
    }
    
    $unused_by_type[$type][] = $file;
}
?>

<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>أداة كشف الملفات غير المستخدمة</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            margin: 0;
            padding: 20px;
            background-color: #f5f5f5;
        }
        .container {
            max-width: 800px;
            margin: 0 auto;
            background-color: #fff;
            padding: 20px;
            border-radius: 5px;
            box-shadow: 0 2px 5px rgba(0, 0, 0, 0.1);
        }
        h1, h2, h3 {
            color: #333;
        }
        h1 {
            text-align: center;
        }
        .file-list {
            margin-top: 10px;
        }
        .file-item {
            padding: 5px 10px;
            margin-bottom: 5px;
            background-color: #f9f9f9;
            border-radius: 3px;
        }
        .file-item:hover {
            background-color: #f0f0f0;
        }
        .summary {
            margin-top: 20px;
            padding: 10px;
            background-color: #e8f5e9;
            border-radius: 4px;
        }
        .type-section {
            margin-bottom: 20px;
        }
        .type-title {
            margin-bottom: 10px;
            padding-bottom: 5px;
            border-bottom: 1px solid #ddd;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>أداة كشف الملفات غير المستخدمة</h1>
        
        <div class="summary">
            <p>
                <strong>إجمالي الملفات:</strong> <?php echo count($all_files); ?><br>
                <strong>الملفات المستخدمة:</strong> <?php echo count($used_files); ?><br>
                <strong>الملفات غير المستخدمة:</strong> <?php echo count($unused_files); ?>
            </p>
        </div>
        
        <h2>الملفات غير المستخدمة</h2>
        
        <?php if (empty($unused_files)): ?>
            <p>لم يتم العثور على ملفات غير مستخدمة.</p>
        <?php else: ?>
            <?php foreach ($unused_by_type as $type => $files): ?>
                <div class="type-section">
                    <h3 class="type-title">ملفات <?php echo strtoupper($type); ?> (<?php echo count($files); ?>)</h3>
                    
                    <div class="file-list">
                        <?php foreach ($files as $file): ?>
                            <div class="file-item">
                                <?php echo htmlspecialchars($file['path']); ?>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            <?php endforeach; ?>
        <?php endif; ?>
        
        <h2>ملاحظات هامة</h2>
        <ul>
            <li>قد تكون بعض الملفات المدرجة كغير مستخدمة مطلوبة في الواقع، خاصة إذا كانت تُستدعى ديناميكيًا.</li>
            <li>تحقق دائمًا من الملفات قبل حذفها للتأكد من أنها غير مطلوبة فعلاً.</li>
            <li>الملفات مثل index.php و config.php تم استثناؤها تلقائيًا من القائمة.</li>
        </ul>
    </div>
</body>
</html>
