<?php
/**
 * التقارير الأسبوعية
 * تعرض هذه الصفحة التقارير الأسبوعية للنظام
 *
 * @version 1.0
 * @author فريق التطوير
 */

// بدء الجلسة
session_start();

// تضمين ملف التكوين
require_once 'config/config.php';

// تفعيل عرض الأخطاء للتصحيح
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// التحقق من حالة تسجيل الدخول
if (!isset($_SESSION['logged_in']) || $_SESSION['logged_in'] !== true) {
    // إعادة توجيه المستخدم إلى صفحة تسجيل الدخول
    header('Location: auth/login.php');
    exit;
}

// الحصول على معلومات المستخدم
$user = [
    'id' => $_SESSION['user_id'] ?? 0,
    'username' => $_SESSION['username'] ?? '',
    'full_name' => $_SESSION['full_name'] ?? 'المستخدم',
    'email' => $_SESSION['email'] ?? '',
    'role' => $_SESSION['role'] ?? 'user',
    'branch_id' => $_SESSION['branch_id'] ?? null
];

// تحديد ما إذا كان المستخدم مديرًا
$isAdmin = (isset($_SESSION['role']) && ($_SESSION['role'] === 'admin' || $_SESSION['role'] === 'manager'));

// التحقق من صلاحيات المستخدم للوصول إلى صفحة التقارير
// السماح للمستخدم العادي بالوصول إلى التقارير الخاصة به فقط

// تحديد الأسبوع الحالي والسابق
$currentWeekStart = date('Y-m-d', strtotime('monday this week'));
$currentWeekEnd = date('Y-m-d', strtotime('sunday this week'));
$previousWeekStart = date('Y-m-d', strtotime('monday last week'));
$previousWeekEnd = date('Y-m-d', strtotime('sunday last week'));

// الحصول على الأسبوع المطلوب من الاستعلام
$selectedWeekStart = $_GET['week_start'] ?? $currentWeekStart;
$selectedWeekEnd = date('Y-m-d', strtotime($selectedWeekStart . ' +6 days'));

// الاتصال بقاعدة البيانات
try {
    // الحصول على إحصائيات الأسبوع
    $weeklyStats = getWeeklyStats($conn, $selectedWeekStart, $selectedWeekEnd, $user['id'], $user['role']);

    // الحصول على الطلبات الأسبوعية
    $weeklyOrders = getWeeklyOrders($conn, $selectedWeekStart, $selectedWeekEnd, $user['id'], $user['role']);
} catch (Exception $e) {
    // تسجيل الخطأ
    error_log("خطأ في الحصول على بيانات التقارير الأسبوعية: " . $e->getMessage());

    // إنشاء بيانات افتراضية في حالة حدوث خطأ
    $weeklyStats = [
        'total_orders' => 0,
        'total_revenue' => 0,
        'average_order_value' => 0,
        'completed_orders' => 0,
        'pending_orders' => 0,
        'cancelled_orders' => 0,
        'daily_orders' => [],
        'top_branches' => [],
        'top_employees' => [],
        'order_status' => []
    ];

    $weeklyOrders = [];
}

/**
 * دالة للحصول على إحصائيات الأسبوع
 *
 * @param PDO $conn اتصال قاعدة البيانات
 * @param string $startDate تاريخ بداية الأسبوع
 * @param string $endDate تاريخ نهاية الأسبوع
 * @param int $userId معرف المستخدم
 * @param string $userRole دور المستخدم
 * @return array إحصائيات الأسبوع
 */
function getWeeklyStats($conn, $startDate, $endDate, $userId, $userRole) {
    // استعلام للحصول على إحصائيات الأسبوع
    $sql = "
        SELECT
            COUNT(*) as total_orders,
            SUM(CASE WHEN status = 'تم التسليم' THEN 1 ELSE 0 END) as completed_orders,
            SUM(CASE WHEN status = 'قيد المعالجة' THEN 1 ELSE 0 END) as pending_orders,
            SUM(CASE WHEN status = 'في الطريق' THEN 1 ELSE 0 END) as in_transit_orders,
            SUM(CASE WHEN status = 'ملغى' THEN 1 ELSE 0 END) as cancelled_orders,
            SUM(amount_agreed_with_customer) as total_revenue
        FROM orders
        WHERE order_upload_date BETWEEN ? AND ?
    ";

    // إضافة شرط المستخدم إذا لم يكن مدير
    $params = [$startDate, $endDate];

    if ($userRole !== 'admin' && $userRole !== 'manager') {
        $sql .= " AND user_id = ?";
        $params[] = $userId;
    }

    $stmt = $conn->prepare($sql);
    $stmt->execute($params);
    $stats = $stmt->fetch(PDO::FETCH_ASSOC);

    // حساب متوسط قيمة الطلب
    $stats['average_order_value'] = $stats['total_orders'] > 0 ?
        $stats['total_revenue'] / $stats['total_orders'] : 0;

    // استعلام للحصول على الطلبات اليومية
    $dailyOrdersSql = "
        SELECT
            DATE_FORMAT(order_upload_date, '%W') as day_name,
            COUNT(*) as orders_count,
            SUM(amount_agreed_with_customer) as revenue
        FROM orders
        WHERE order_upload_date BETWEEN ? AND ?
    ";

    // إضافة شرط المستخدم إذا لم يكن مدير
    $dailyOrdersParams = [$startDate, $endDate];

    if ($userRole !== 'admin' && $userRole !== 'manager') {
        $dailyOrdersSql .= " AND user_id = ?";
        $dailyOrdersParams[] = $userId;
    }

    $dailyOrdersSql .= " GROUP BY DATE_FORMAT(order_upload_date, '%W')";

    $dailyOrdersStmt = $conn->prepare($dailyOrdersSql);
    $dailyOrdersStmt->execute($dailyOrdersParams);
    $dailyOrders = $dailyOrdersStmt->fetchAll(PDO::FETCH_ASSOC);

    // استعلام للحصول على أفضل الفروع
    $topBranchesSql = "
        SELECT
            b.name,
            COUNT(*) as orders,
            SUM(o.amount_agreed_with_customer) as revenue
        FROM orders o
        LEFT JOIN branches b ON o.branch_id = b.id
        WHERE o.order_upload_date BETWEEN ? AND ?
    ";

    // إضافة شرط المستخدم إذا لم يكن مدير
    $topBranchesParams = [$startDate, $endDate];

    if ($userRole !== 'admin' && $userRole !== 'manager') {
        $topBranchesSql .= " AND o.user_id = ?";
        $topBranchesParams[] = $userId;
    }

    $topBranchesSql .= " GROUP BY b.name ORDER BY orders DESC LIMIT 4";

    $topBranchesStmt = $conn->prepare($topBranchesSql);
    $topBranchesStmt->execute($topBranchesParams);
    $topBranches = $topBranchesStmt->fetchAll(PDO::FETCH_ASSOC);

    // استعلام للحصول على أفضل الموظفين
    $topEmployeesSql = "
        SELECT
            o.employee_name as name,
            COUNT(*) as orders,
            SUM(o.amount_agreed_with_customer) as revenue
        FROM orders o
        WHERE o.order_upload_date BETWEEN ? AND ? AND o.employee_name IS NOT NULL AND o.employee_name != ''
    ";

    // إضافة شرط المستخدم إذا لم يكن مدير
    $topEmployeesParams = [$startDate, $endDate];

    if ($userRole !== 'admin' && $userRole !== 'manager') {
        $topEmployeesSql .= " AND o.user_id = ?";
        $topEmployeesParams[] = $userId;
    }

    $topEmployeesSql .= " GROUP BY o.employee_name ORDER BY orders DESC LIMIT 4";

    $topEmployeesStmt = $conn->prepare($topEmployeesSql);
    $topEmployeesStmt->execute($topEmployeesParams);
    $topEmployees = $topEmployeesStmt->fetchAll(PDO::FETCH_ASSOC);

    // استعلام للحصول على توزيع حالة الطلبات
    $orderStatusSql = "
        SELECT
            status,
            COUNT(*) as count,
            COUNT(*) * 100.0 / (SELECT COUNT(*) FROM orders WHERE order_upload_date BETWEEN ? AND ?) as percentage
        FROM orders
        WHERE order_upload_date BETWEEN ? AND ?
    ";

    // إضافة شرط المستخدم إذا لم يكن مدير
    $orderStatusParams = array_merge([$startDate, $endDate], [$startDate, $endDate]);

    if ($userRole !== 'admin' && $userRole !== 'manager') {
        $orderStatusSql .= " AND user_id = ?";
        $orderStatusParams[] = $userId;
        // إضافة شرط المستخدم للاستعلام الفرعي أيضًا
        $orderStatusSql = str_replace("WHERE order_upload_date BETWEEN ? AND ?", "WHERE order_upload_date BETWEEN ? AND ? AND user_id = ?", $orderStatusSql);
        $orderStatusParams = array_merge([$startDate, $endDate, $userId], [$startDate, $endDate, $userId]);
    }

    $orderStatusSql .= " GROUP BY status";

    $orderStatusStmt = $conn->prepare($orderStatusSql);
    $orderStatusStmt->execute($orderStatusParams);
    $orderStatus = $orderStatusStmt->fetchAll(PDO::FETCH_ASSOC);

    // الحصول على إحصائيات الأسبوع السابق للمقارنة
    $prevStartDate = date('Y-m-d', strtotime($startDate . ' -7 days'));
    $prevEndDate = date('Y-m-d', strtotime($endDate . ' -7 days'));

    $prevSql = "
        SELECT
            COUNT(*) as total_orders,
            SUM(CASE WHEN status = 'تم التسليم' THEN 1 ELSE 0 END) as completed_orders,
            SUM(amount_agreed_with_customer) as total_revenue
        FROM orders
        WHERE order_upload_date BETWEEN ? AND ?
    ";

    // إضافة شرط المستخدم إذا لم يكن مدير
    $prevParams = [$prevStartDate, $prevEndDate];

    if ($userRole !== 'admin' && $userRole !== 'manager') {
        $prevSql .= " AND user_id = ?";
        $prevParams[] = $userId;
    }

    $prevStmt = $conn->prepare($prevSql);
    $prevStmt->execute($prevParams);
    $prevStats = $prevStmt->fetch(PDO::FETCH_ASSOC);

    // حساب نسب المقارنة مع الأسبوع السابق
    $previousWeekComparison = 0;
    $revenueComparison = 0;
    $averageComparison = 0;
    $completionComparison = 0;

    if ($prevStats['total_orders'] > 0) {
        // نسبة التغير في عدد الطلبات
        $previousWeekComparison = $prevStats['total_orders'] > 0 ?
            (($stats['total_orders'] - $prevStats['total_orders']) / $prevStats['total_orders']) * 100 : 0;

        // نسبة التغير في الإيرادات
        $revenueComparison = $prevStats['total_revenue'] > 0 ?
            (($stats['total_revenue'] - $prevStats['total_revenue']) / $prevStats['total_revenue']) * 100 : 0;

        // نسبة التغير في متوسط قيمة الطلب
        $prevAverage = $prevStats['total_orders'] > 0 ? $prevStats['total_revenue'] / $prevStats['total_orders'] : 0;
        $averageComparison = $prevAverage > 0 ?
            (($stats['average_order_value'] - $prevAverage) / $prevAverage) * 100 : 0;

        // نسبة التغير في معدل إكمال الطلبات
        $prevCompletionRate = $prevStats['total_orders'] > 0 ?
            ($prevStats['completed_orders'] / $prevStats['total_orders']) * 100 : 0;
        $currentCompletionRate = $stats['total_orders'] > 0 ?
            ($stats['completed_orders'] / $stats['total_orders']) * 100 : 0;
        $completionComparison = $prevCompletionRate > 0 ?
            ($currentCompletionRate - $prevCompletionRate) : 0;
    }

    // تجميع البيانات
    return [
        'total_orders' => (int)$stats['total_orders'],
        'total_revenue' => (float)$stats['total_revenue'],
        'average_order_value' => (float)$stats['average_order_value'],
        'completed_orders' => (int)$stats['completed_orders'],
        'pending_orders' => (int)$stats['pending_orders'],
        'cancelled_orders' => (int)$stats['cancelled_orders'],
        'daily_orders' => $dailyOrders,
        'top_branches' => $topBranches,
        'top_employees' => $topEmployees,
        'order_status' => $orderStatus,
        'previous_week_comparison' => $previousWeekComparison,
        'revenue_comparison' => $revenueComparison,
        'average_comparison' => $averageComparison,
        'completion_comparison' => $completionComparison
    ];
}

/**
 * دالة للحصول على الطلبات الأسبوعية
 *
 * @param PDO $conn اتصال قاعدة البيانات
 * @param string $startDate تاريخ بداية الأسبوع
 * @param string $endDate تاريخ نهاية الأسبوع
 * @param int $userId معرف المستخدم
 * @param string $userRole دور المستخدم
 * @return array الطلبات الأسبوعية
 */
function getWeeklyOrders($conn, $startDate, $endDate, $userId, $userRole) {
    // استعلام للحصول على الطلبات الأسبوعية
    $sql = "
        SELECT
            o.id,
            o.order_number,
            o.client_name,
            o.service_type,
            o.goods_type as cargo_type,
            o.goods_name as cargo_name,
            o.delivery_duration as delivery_time,
            o.payment_type,
            o.amount_paid as paid_amount,
            o.remaining_amount,
            o.second_destination,
            o.delivery_area,
            o.order_upload_date as order_date,
            o.scheduled_delivery_date as delivery_date,
            o.status,
            o.employee_name,
            b.name as branch
        FROM orders o
        LEFT JOIN branches b ON o.branch_id = b.id
        WHERE o.order_upload_date BETWEEN ? AND ?
    ";

    // إضافة شرط المستخدم إذا لم يكن مدير
    $params = [$startDate, $endDate];

    if ($userRole !== 'admin' && $userRole !== 'manager') {
        $sql .= " AND o.user_id = ?";
        $params[] = $userId;
    }

    // إضافة شرط الفرع إذا تم تحديده
    if (isset($_GET['branch']) && $_GET['branch'] !== 'all') {
        $sql .= " AND o.branch_id = ?";
        $params[] = $_GET['branch'];
    }

    // إضافة شرط الحالة إذا تم تحديدها
    if (isset($_GET['status']) && $_GET['status'] !== '') {
        $sql .= " AND o.status = ?";
        $params[] = $_GET['status'];
    }

    $sql .= " ORDER BY o.order_upload_date DESC";

    $stmt = $conn->prepare($sql);
    $stmt->execute($params);
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

// تنسيق التاريخ بالعربية
function formatDateArabic($date) {
    $months = [
        'January' => 'يناير',
        'February' => 'فبراير',
        'March' => 'مارس',
        'April' => 'أبريل',
        'May' => 'مايو',
        'June' => 'يونيو',
        'July' => 'يوليو',
        'August' => 'أغسطس',
        'September' => 'سبتمبر',
        'October' => 'أكتوبر',
        'November' => 'نوفمبر',
        'December' => 'ديسمبر'
    ];

    $formattedDate = date('d F Y', strtotime($date));
    foreach ($months as $en => $ar) {
        $formattedDate = str_replace($en, $ar, $formattedDate);
    }
    return $formattedDate;
}

// تنسيق المبالغ المالية
function formatCurrency($amount) {
    return number_format($amount, 2) . ' ريال';
}

// تحديد عنوان التقرير
$reportTitle = 'التقرير الأسبوعي: ' . formatDateArabic($selectedWeekStart) . ' - ' . formatDateArabic($selectedWeekEnd);

// تعيين مسار الشعار
$company_logo = 'assets/images/Company logo.png';
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">

<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>نظام طلبات الشحن | التقارير الأسبوعية</title>
  <script src="https://cdn.tailwindcss.com"></script>
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
  <link rel="stylesheet" href="assets/css/unified-design.css">
  <style>
    /* تحسين البطاقات الإحصائية */
    .stat-card {
      background: white;
      border-radius: var(--border-radius);
      box-shadow: var(--shadow);
      transition: var(--transition);
      position: relative;
      overflow: hidden;
      border: 1px solid var(--gray-200);
    }

    .stat-card::before {
      content: '';
      position: absolute;
      top: 0;
      right: 0;
      width: 100%;
      height: 4px;
      background: linear-gradient(90deg, var(--primary) 0%, var(--primary-light) 100%);
      opacity: 0;
      transition: var(--transition);
    }

    .stat-card:hover {
      transform: translateY(-5px);
      box-shadow: var(--shadow-md);
    }

    .stat-card:hover::before {
      opacity: 1;
    }

    .stat-card .icon-container {
      background: linear-gradient(135deg, var(--primary-light) 0%, var(--primary) 100%);
      color: white;
      border-radius: 50%;
      width: 3rem;
      height: 3rem;
      display: flex;
      align-items: center;
      justify-content: center;
      box-shadow: var(--shadow);
      transition: var(--transition);
    }

    .stat-card:hover .icon-container {
      transform: scale(1.1) rotate(10deg);
    }

    .stat-card .stat-value {
      font-size: 2rem;
      font-weight: 700;
      color: var(--primary-dark);
      margin-top: 0.5rem;
      margin-bottom: 0.25rem;
    }

    .stat-card .stat-label {
      color: var(--gray-600);
      font-weight: 500;
    }

    /* تحسين الرسوم البيانية */
    .chart-container {
      background-color: white;
      border-radius: var(--border-radius);
      box-shadow: var(--shadow);
      padding: 1.5rem;
      border: 1px solid var(--gray-200);
      transition: var(--transition);
    }

    .chart-container:hover {
      box-shadow: var(--shadow-md);
      border-color: var(--primary-light);
    }

    .chart-title {
      font-size: 1.25rem;
      font-weight: 600;
      color: var(--gray-800);
      margin-bottom: 1.5rem;
      display: flex;
      align-items: center;
      gap: 0.5rem;
    }

    .chart-title i {
      color: var(--primary);
    }

    /* تحسين الجداول */
    .data-table {
      width: 100%;
      border-collapse: separate;
      border-spacing: 0;
    }

    .data-table th {
      background-color: var(--primary);
      color: white;
      font-weight: 500;
      text-align: right;
      padding: 0.75rem 1rem;
    }

    .data-table th:first-child {
      border-top-right-radius: var(--border-radius);
    }

    .data-table th:last-child {
      border-top-left-radius: var(--border-radius);
    }

    .data-table td {
      padding: 0.75rem 1rem;
      border-bottom: 1px solid var(--gray-200);
    }

    .data-table tbody tr:hover {
      background-color: var(--gray-50);
    }

    .data-table tbody tr:last-child td {
      border-bottom: none;
    }

    .data-table tbody tr:last-child td:first-child {
      border-bottom-right-radius: var(--border-radius);
    }

    .data-table tbody tr:last-child td:last-child {
      border-bottom-left-radius: var(--border-radius);
    }

    /* تحسين أزرار التنقل بين الأسابيع */
    .week-nav {
      display: flex;
      align-items: center;
      gap: 1rem;
    }

    .week-nav-btn {
      display: flex;
      align-items: center;
      gap: 0.5rem;
      padding: 0.5rem 1rem;
      border-radius: var(--border-radius);
      transition: var(--transition);
      font-weight: 500;
    }

    .week-nav-btn.prev {
      background-color: var(--gray-100);
      color: var(--gray-700);
    }

    .week-nav-btn.next {
      background-color: var(--primary-50);
      color: var(--primary);
    }

    .week-nav-btn:hover {
      transform: translateY(-2px);
    }

    .week-nav-btn.prev:hover {
      background-color: var(--gray-200);
    }

    .week-nav-btn.next:hover {
      background-color: var(--primary-100);
    }

    /* تأثيرات الحركة */
    @keyframes fadeIn {
      from {
        opacity: 0;
        transform: translateY(10px);
      }
      to {
        opacity: 1;
        transform: translateY(0);
      }
    }

    .animate-fadeIn {
      animation: fadeIn 0.5s ease-out forwards;
    }

    /* تأثير الشعار */
    .logo-reveal {
      animation: logoReveal 1.5s ease-out forwards;
      transition: all 0.5s cubic-bezier(0.175, 0.885, 0.32, 1.275);
      border-radius: 8px;
      position: relative;
      z-index: 2;
      max-height: 100px;
      width: auto;
    }

    /* تحسين مظهر القوائم المنسدلة */
    select.form-input {
      appearance: none;
      background-image: url("data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' fill='none' viewBox='0 0 24 24' stroke='%236b7280'%3E%3Cpath stroke-linecap='round' stroke-linejoin='round' stroke-width='2' d='M19 9l-7 7-7-7'%3E%3C/path%3E%3C/svg%3E");
      background-repeat: no-repeat;
      background-position: left 0.75rem center;
      background-size: 1rem;
      padding-left: 2.5rem;
    }

    select.form-input option {
      padding: 0.5rem;
    }

    select.form-input option:checked {
      background-color: var(--primary-50);
      color: var(--primary);
      font-weight: 600;
    }

    select.form-input optgroup {
      font-weight: 600;
      color: var(--primary);
      background-color: var(--gray-50);
    }

    select.form-input option:disabled {
      color: var(--gray-400);
    }

    .logo-reveal:hover {
      transform: scale(1.08) rotate(1deg);
      filter: brightness(1.2) contrast(1.1) drop-shadow(0 10px 20px rgba(0, 0, 0, 0.25));
    }

    @keyframes logoReveal {
      0% {
        opacity: 0;
        transform: scale(0.8);
      }
      50% {
        opacity: 0.5;
        transform: scale(1.05);
      }
      100% {
        opacity: 1;
        transform: scale(1);
      }
    }

    /* تأثير إضافي للشعار */
    aside .logo-container::before {
      content: '';
      position: absolute;
      top: 0;
      left: 0;
      width: 100%;
      height: 100%;
      background: linear-gradient(135deg, rgba(255,255,255,0.3) 0%, rgba(255,255,255,0) 50%, rgba(255,255,255,0.3) 100%);
      transform: translateX(-100%);
      transition: transform 1s ease;
    }

    aside .logo-container:hover::before {
      transform: translateX(100%);
    }

    /* أنماط الطباعة */
    @media print {
      .no-print, aside, header, .top-bar, #toggleSidebar, .user-profile, nav, footer,
      button:not(.print-only), .flex.items-center.gap-2 a:not([href*="pdf"]) {
        display: none !important;
      }

      .print-only {
        display: block !important;
      }

      body {
        font-size: 12pt;
        background: white;
        color: black;
      }

      main {
        width: 100%;
        margin: 0;
        padding: 0;
      }

      .data-table {
        width: 100%;
        border-collapse: collapse;
        page-break-inside: avoid;
      }

      .data-table th, .data-table td {
        border: 1px solid #000;
        padding: 5px;
        text-align: right;
      }

      .data-table th {
        background-color: #f2f2f2 !important;
        color: black !important;
      }

      .chart-container, .stat-card {
        break-inside: avoid;
        page-break-inside: avoid;
        border: 1px solid #ddd;
        margin-bottom: 15px;
      }

      h1, h2, h3 {
        page-break-after: avoid;
      }

      img.logo-reveal {
        display: block;
        margin: 0 auto 20px;
        max-width: 200px;
      }

      .print-header {
        text-align: center;
        margin-bottom: 20px;
      }
    }
  </style>
</head>

<body>
  <!-- طبقة التراكب للأجهزة المحمولة -->
  <div id="overlay" class="overlay"></div>

  <div class="flex min-h-screen">
    <!-- القائمة الجانبية -->
    <aside class="w-64 hidden md:block">
      <div class="p-6">
        <div class="logo-container flex items-center justify-center mb-8 p-2">
          <img src="<?php echo $company_logo; ?>" alt="شعار النظام" class="h-40 w-auto object-contain logo-reveal">
        </div>

        <div class="user-info mb-8">
          <div class="flex items-center space-x-4 space-x-reverse mb-3">
            <div class="user-avatar">
              <?php echo mb_substr($user['full_name'], 0, 1, 'UTF-8'); ?>
            </div>
            <div>
              <h3 class="text-white font-semibold"><?php echo htmlspecialchars($user['full_name']); ?></h3>
              <p class="text-indigo-200 text-sm"><?php echo $isAdmin ? 'مدير النظام' : 'مستخدم'; ?></p>
            </div>
          </div>
          <div class="bg-white/10 h-0.5 rounded-full mb-3"></div>
        </div>

        <nav>
          <ul class="space-y-3">
            <!-- لوحة التحكم -->
            <li>
              <a href="unified-dashboard.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white">
                <i class="fas fa-tachometer-alt ml-3 text-lg"></i>
                <span>لوحة التحكم</span>
              </a>
            </li>

            <!-- إنشاء طلب جديد -->
            <li>
              <a href="create-order-unified.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white">
                <i class="fas fa-plus-circle ml-3 text-lg"></i>
                <span>إنشاء طلب جديد</span>
              </a>
            </li>

            <!-- قائمة الطلبات -->
            <li>
              <a href="orders.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white">
                <i class="fas fa-box ml-3 text-lg"></i>
                <span>قائمة الطلبات</span>
              </a>
            </li>

            <?php if ($isAdmin): ?>
            <!-- قائمة التقارير المنسدلة -->
            <li>
              <a href="#" class="flex items-center p-3 rounded-lg text-white dropdown-toggle active" id="reportsDropdown">
                <i class="fas fa-chart-bar ml-3 text-lg"></i>
                <span>التقارير</span>
              </a>
              <ul class="dropdown-menu active" id="reportsMenu">
                <li>
                  <a href="weekly-reports.php" class="flex items-center p-2 rounded-lg text-white active">
                    <i class="fas fa-calendar-week ml-2 text-sm"></i>
                    <span>التقارير الأسبوعية</span>
                  </a>
                </li>
                <li>
                  <a href="monthly-reports.php" class="flex items-center p-2 rounded-lg text-white/80 hover:text-white">
                    <i class="fas fa-calendar-alt ml-2 text-sm"></i>
                    <span>التقارير الشهرية</span>
                  </a>
                </li>
              </ul>
            </li>

            <!-- إدارة المستخدمين -->
            <li>
              <a href="users.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white">
                <i class="fas fa-users ml-3 text-lg"></i>
                <span>إدارة المستخدمين</span>
              </a>
            </li>

            <!-- الفروع -->
            <li>
              <a href="branches.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white">
                <i class="fas fa-code-branch ml-3 text-lg"></i>
                <span>الفروع</span>
              </a>
            </li>

            <!-- إعدادات النظام -->
            <li>
              <a href="settings.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white">
                <i class="fas fa-cog ml-3 text-lg"></i>
                <span>إعدادات النظام</span>
              </a>
            </li>
            <?php endif; ?>
          </ul>

          <div class="pt-6 mt-6 border-t border-white/10">
            <a href="auth/logout.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white">
              <i class="fas fa-sign-out-alt ml-3 text-lg"></i>
              <span>تسجيل الخروج</span>
            </a>
          </div>
        </nav>
      </div>
    </aside>

    <!-- المحتوى الرئيسي -->
    <main class="flex-1">
      <!-- الشريط العلوي -->
      <header class="top-bar">
        <div class="flex items-center">
          <button id="toggleSidebar" class="md:hidden text-gray-600 hover:text-primary p-2 rounded-lg">
            <i class="fas fa-bars text-xl"></i>
          </button>

          <!-- شعار النظام في الشريط العلوي (يظهر فقط في الأجهزة المحمولة) -->
          <div class="flex items-center md:hidden ml-3">
            <img src="<?php echo $company_logo; ?>" alt="شعار النظام" class="h-16 w-auto object-contain logo-animation">
          </div>

          <div class="relative ml-4 hidden md:block">
            <input type="text" placeholder="البحث..." class="bg-gray-100 rounded-full py-2 px-4 pl-10 w-64 focus:outline-none focus:ring-2 focus:ring-primary/30 focus:bg-white transition-all">
            <i class="fas fa-search absolute right-3 top-3 text-gray-400"></i>
          </div>
        </div>

        <!-- معلومات المستخدم والإشعارات -->
        <div class="flex items-center space-x-4 space-x-reverse">
          <?php if ($isAdmin): ?>
          <div class="relative">
            <button class="relative p-2 rounded-full hover:bg-gray-100 transition-all">
              <i class="fas fa-bell text-gray-600"></i>
              <span class="absolute -top-1 -right-1 bg-red-500 text-white text-xs rounded-full h-5 w-5 flex items-center justify-center animate-pulse">3</span>
            </button>
          </div>
          <?php endif; ?>

          <div class="relative">
            <button class="relative p-2 rounded-full hover:bg-gray-100 transition-all">
              <i class="fas fa-envelope text-gray-600"></i>
              <?php if ($isAdmin): ?>
              <span class="absolute -top-1 -right-1 bg-primary text-white text-xs rounded-full h-5 w-5 flex items-center justify-center">2</span>
              <?php endif; ?>
            </button>
          </div>

          <div class="border-r border-gray-300 h-8 mx-2"></div>

          <div class="user-profile">
            <div class="user-avatar">
              <?php echo mb_substr($user['full_name'], 0, 1, 'UTF-8'); ?>
            </div>
            <div>
              <p class="font-medium text-sm"><?php echo htmlspecialchars($user['full_name']); ?></p>
              <p class="text-xs text-gray-500"><?php echo $isAdmin ? 'مدير النظام' : 'مستخدم'; ?></p>
            </div>
          </div>
        </div>
      </header>

      <!-- عنصر رأس الصفحة للطباعة فقط -->
      <div class="print-only print-header" style="display: none;">
        <img src="<?php echo $company_logo; ?>" alt="شعار الشركة" class="logo-reveal">
        <h1 style="font-size: 24pt; font-weight: bold; margin-bottom: 10px;">مجموعة ابراهيم الاحمدي اليمنية</h1>
        <h2 style="font-size: 18pt; margin-bottom: 5px;"><?php echo $reportTitle; ?></h2>
        <p style="font-size: 12pt; margin-bottom: 5px;">تاريخ التقرير: <?php echo date('Y-m-d'); ?></p>

        <?php if (isset($_GET['branch']) && $_GET['branch'] !== 'all' || isset($_GET['status']) && $_GET['status'] !== ''): ?>
        <div style="font-size: 12pt; margin-bottom: 15px; color: #444;">
          <?php if (isset($_GET['branch']) && $_GET['branch'] !== 'all'):
            $branchName = '';
            foreach ($branches as $branch) {
              if ($branch['id'] == $_GET['branch']) {
                $branchName = $branch['name'];
                break;
              }
            }
          ?>
          <span style="margin-left: 15px;">الفرع: <strong><?php echo $branchName; ?></strong></span>
          <?php endif; ?>

          <?php if (isset($_GET['status']) && $_GET['status'] !== ''): ?>
          <span>الحالة: <strong><?php echo $_GET['status']; ?></strong></span>
          <?php endif; ?>
        </div>
        <?php endif; ?>

        <hr style="border: 1px solid #000; margin-bottom: 20px;">
      </div>

      <!-- محتوى الصفحة -->
      <div class="p-6">
        <!-- رأس الصفحة -->
        <div class="flex flex-col md:flex-row md:items-center md:justify-between mb-8">
          <div class="animate-fadeIn" style="animation-delay: 0.1s;">
            <h1 class="text-3xl font-bold text-gray-800 flex items-center">
              <span class="bg-gradient-to-r from-primary to-primary-light bg-clip-text text-transparent">
                <?php echo $reportTitle; ?>
              </span>
            </h1>
            <p class="text-gray-500 mt-2 flex items-center">
              <i class="fas fa-chart-line ml-2 text-primary"></i>
              <span>تحليل أداء الطلبات والمبيعات خلال الأسبوع</span>
            </p>
          </div>

          <div class="mt-4 md:mt-0 animate-fadeIn flex items-center space-x-2 space-x-reverse" style="animation-delay: 0.2s;">
            <?php
            $prevWeek = date('Y-m-d', strtotime($selectedWeekStart . ' -7 days'));
            $nextWeek = date('Y-m-d', strtotime($selectedWeekStart . ' +7 days'));
            $isCurrentWeek = ($selectedWeekStart === $currentWeekStart);
            ?>

            <div class="border-r border-gray-300 h-8 mx-2"></div>

            <!-- أزرار التصدير -->
            <div class="flex items-center gap-2">
              <a href="?week_start=<?php echo $prevWeek; ?>" class="flex items-center gap-1 px-3 py-2 bg-blue-50 text-blue-600 rounded-lg hover:bg-blue-100 transition-all">
                <i class="fas fa-chevron-right"></i>
                <span>الأسبوع السابق</span>
              </a>

              <?php if (!$isCurrentWeek): ?>
              <a href="?week_start=<?php echo $nextWeek; ?>" class="flex items-center gap-1 px-3 py-2 bg-blue-50 text-blue-600 rounded-lg hover:bg-blue-100 transition-all">
                <span>الأسبوع التالي</span>
                <i class="fas fa-chevron-left"></i>
              </a>
              <?php endif; ?>

              <button class="flex items-center gap-1 px-4 py-2 bg-blue-500 text-white rounded-lg hover:bg-blue-600 transition-all" onclick="printReport()">
                <i class="fas fa-print"></i>
                <span>طباعة</span>
              </button>

              <script>
                function printReport() {
                  // إضافة معلومات الفلترة إلى عنوان الطباعة
                  let printTitle = document.querySelector('.print-header h2');
                  if (printTitle) {
                    let originalTitle = printTitle.innerText;
                    <?php if (isset($_GET['branch']) && $_GET['branch'] !== 'all'):
                      $branchName = '';
                      foreach ($branches as $branch) {
                        if ($branch['id'] == $_GET['branch']) {
                          $branchName = $branch['name'];
                          break;
                        }
                      }
                    ?>
                    originalTitle += ' (فرع: <?php echo $branchName; ?>)';
                    <?php endif; ?>

                    <?php if (isset($_GET['status']) && $_GET['status'] !== ''): ?>
                    originalTitle += ' (الحالة: <?php echo $_GET['status']; ?>)';
                    <?php endif; ?>

                    printTitle.innerText = originalTitle;
                  }

                  window.print();
                }
              </script>

              <a href="export.php?type=excel&report=weekly&week_start=<?php echo $selectedWeekStart; ?><?php echo isset($_GET['branch']) && $_GET['branch'] !== 'all' ? '&branch=' . $_GET['branch'] : ''; ?><?php echo isset($_GET['status']) && $_GET['status'] !== '' ? '&status=' . urlencode($_GET['status']) : ''; ?>" class="flex items-center gap-1 px-4 py-2 bg-green-500 text-white rounded-lg hover:bg-green-600 transition-all">
                <i class="fas fa-file-excel"></i>
                <span>تصدير Excel</span>
              </a>

              <a href="export.php?type=pdf&report=weekly&week_start=<?php echo $selectedWeekStart; ?><?php echo isset($_GET['branch']) && $_GET['branch'] !== 'all' ? '&branch=' . $_GET['branch'] : ''; ?><?php echo isset($_GET['status']) && $_GET['status'] !== '' ? '&status=' . urlencode($_GET['status']) : ''; ?>" class="flex items-center gap-1 px-4 py-2 bg-indigo-500 text-white rounded-lg hover:bg-indigo-600 transition-all">
                <i class="fas fa-file-pdf"></i>
                <span>تصدير PDF</span>
              </a>

              <a href="export.php?type=orders_list&report=weekly&week_start=<?php echo $selectedWeekStart; ?><?php echo isset($_GET['branch']) && $_GET['branch'] !== 'all' ? '&branch=' . $_GET['branch'] : ''; ?><?php echo isset($_GET['status']) && $_GET['status'] !== '' ? '&status=' . urlencode($_GET['status']) : ''; ?>" class="flex items-center gap-1 px-4 py-2 bg-purple-500 text-white rounded-lg hover:bg-purple-600 transition-all">
                <i class="fas fa-list"></i>
                <span>عرض قائمة الطلبات</span>
              </a>
            </div>
          </div>
        </div>

        <!-- البطاقات الإحصائية -->
        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
          <!-- إجمالي الطلبات -->
          <div class="stat-card p-6 animate-fadeIn" style="animation-delay: 0.3s;">
            <div class="flex justify-between items-start">
              <div>
                <p class="text-gray-500 text-sm">إجمالي الطلبات</p>
                <p class="stat-value"><?php echo $weeklyStats['total_orders']; ?></p>
                <p class="stat-label">طلب</p>
              </div>
              <div class="icon-container">
                <i class="fas fa-box text-xl"></i>
              </div>
            </div>
            <div class="mt-4 text-sm text-gray-500">
              <?php if (isset($weeklyStats['previous_week_comparison']) && $weeklyStats['previous_week_comparison'] > 0): ?>
              <span class="text-green-500 font-medium">
                <i class="fas fa-arrow-up"></i> <?php echo number_format($weeklyStats['previous_week_comparison'], 1); ?>%
              </span>
              <?php elseif (isset($weeklyStats['previous_week_comparison']) && $weeklyStats['previous_week_comparison'] < 0): ?>
              <span class="text-red-500 font-medium">
                <i class="fas fa-arrow-down"></i> <?php echo number_format(abs($weeklyStats['previous_week_comparison']), 1); ?>%
              </span>
              <?php else: ?>
              <span class="text-gray-500 font-medium">
                <i class="fas fa-minus"></i> 0%
              </span>
              <?php endif; ?>
              مقارنة بالأسبوع السابق
            </div>
          </div>

          <!-- إجمالي الإيرادات -->
          <div class="stat-card p-6 animate-fadeIn" style="animation-delay: 0.4s;">
            <div class="flex justify-between items-start">
              <div>
                <p class="text-gray-500 text-sm">إجمالي الإيرادات</p>
                <p class="stat-value"><?php echo number_format($weeklyStats['total_revenue'], 0); ?></p>
                <p class="stat-label">ريال</p>
              </div>
              <div class="icon-container" style="background: linear-gradient(135deg, #10b981 0%, #059669 100%);">
                <i class="fas fa-money-bill-wave text-xl"></i>
              </div>
            </div>
            <div class="mt-4 text-sm text-gray-500">
              <?php if (isset($weeklyStats['revenue_comparison']) && $weeklyStats['revenue_comparison'] > 0): ?>
              <span class="text-green-500 font-medium">
                <i class="fas fa-arrow-up"></i> <?php echo number_format($weeklyStats['revenue_comparison'], 1); ?>%
              </span>
              <?php elseif (isset($weeklyStats['revenue_comparison']) && $weeklyStats['revenue_comparison'] < 0): ?>
              <span class="text-red-500 font-medium">
                <i class="fas fa-arrow-down"></i> <?php echo number_format(abs($weeklyStats['revenue_comparison']), 1); ?>%
              </span>
              <?php else: ?>
              <span class="text-gray-500 font-medium">
                <i class="fas fa-minus"></i> 0%
              </span>
              <?php endif; ?>
              مقارنة بالأسبوع السابق
            </div>
          </div>

          <!-- متوسط قيمة الطلب -->
          <div class="stat-card p-6 animate-fadeIn" style="animation-delay: 0.5s;">
            <div class="flex justify-between items-start">
              <div>
                <p class="text-gray-500 text-sm">متوسط قيمة الطلب</p>
                <p class="stat-value"><?php echo number_format($weeklyStats['average_order_value'], 0); ?></p>
                <p class="stat-label">ريال</p>
              </div>
              <div class="icon-container" style="background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);">
                <i class="fas fa-chart-line text-xl"></i>
              </div>
            </div>
            <div class="mt-4 text-sm text-gray-500">
              <?php if (isset($weeklyStats['average_comparison']) && $weeklyStats['average_comparison'] > 0): ?>
              <span class="text-green-500 font-medium">
                <i class="fas fa-arrow-up"></i> <?php echo number_format($weeklyStats['average_comparison'], 1); ?>%
              </span>
              <?php elseif (isset($weeklyStats['average_comparison']) && $weeklyStats['average_comparison'] < 0): ?>
              <span class="text-red-500 font-medium">
                <i class="fas fa-arrow-down"></i> <?php echo number_format(abs($weeklyStats['average_comparison']), 1); ?>%
              </span>
              <?php else: ?>
              <span class="text-gray-500 font-medium">
                <i class="fas fa-minus"></i> 0%
              </span>
              <?php endif; ?>
              مقارنة بالأسبوع السابق
            </div>
          </div>

          <!-- نسبة إكمال الطلبات -->
          <div class="stat-card p-6 animate-fadeIn" style="animation-delay: 0.6s;">
            <div class="flex justify-between items-start">
              <div>
                <p class="text-gray-500 text-sm">نسبة إكمال الطلبات</p>
                <p class="stat-value"><?php echo number_format($weeklyStats['total_orders'] > 0 ? ($weeklyStats['completed_orders'] / $weeklyStats['total_orders']) * 100 : 0, 1); ?>%</p>
                <p class="stat-label">معدل الإكمال</p>
              </div>
              <div class="icon-container" style="background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%);">
                <i class="fas fa-check-circle text-xl"></i>
              </div>
            </div>
            <div class="mt-4 text-sm text-gray-500">
              <?php if (isset($weeklyStats['completion_comparison']) && $weeklyStats['completion_comparison'] > 0): ?>
              <span class="text-green-500 font-medium">
                <i class="fas fa-arrow-up"></i> <?php echo number_format($weeklyStats['completion_comparison'], 1); ?>%
              </span>
              <?php elseif (isset($weeklyStats['completion_comparison']) && $weeklyStats['completion_comparison'] < 0): ?>
              <span class="text-red-500 font-medium">
                <i class="fas fa-arrow-down"></i> <?php echo number_format(abs($weeklyStats['completion_comparison']), 1); ?>%
              </span>
              <?php else: ?>
              <span class="text-gray-500 font-medium">
                <i class="fas fa-minus"></i> 0%
              </span>
              <?php endif; ?>
              مقارنة بالأسبوع السابق
            </div>
          </div>
        </div>

        <!-- قائمة الطلبات الأسبوعية -->
        <div class="chart-container animate-fadeIn" style="animation-delay: 0.9s;">
          <h3 class="chart-title">
            <i class="fas fa-list"></i>
            <span>قائمة الطلبات الأسبوعية</span>
            <div class="mr-auto text-sm text-gray-500">
              <?php echo formatDateArabic($selectedWeekStart); ?> - <?php echo formatDateArabic($selectedWeekEnd); ?>
            </div>
          </h3>

          <!-- فلترة الطلبات -->
          <div class="bg-white p-6 rounded-lg mb-6 shadow-sm border border-gray-200">
            <div class="flex items-center mb-4">
              <i class="fas fa-filter text-primary ml-2 text-lg"></i>
              <h3 class="text-lg font-semibold text-gray-800">فلترة التقارير</h3>
              <?php if (isset($_GET['branch']) && $_GET['branch'] !== 'all' || isset($_GET['status']) && $_GET['status'] !== ''): ?>
              <div class="mr-auto">
                <a href="weekly-reports.php?week_start=<?php echo $selectedWeekStart; ?>" class="text-sm text-gray-500 hover:text-primary flex items-center">
                  <i class="fas fa-times-circle ml-1"></i>
                  <span>إعادة ضبط الفلاتر</span>
                </a>
              </div>
              <?php endif; ?>
            </div>

            <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
              <!-- فلتر الفروع -->
              <div>
                <label for="branchFilter" class="block text-sm font-medium text-gray-700 mb-2">الفرع</label>
                <div class="relative">
                  <select id="branchFilter" name="branch" class="form-input pl-10 w-full">
                    <option value="all" class="font-bold text-primary">كل الفروع</option>
                    <option disabled>──────────</option>
                    <?php
                    // استعلام للحصول على قائمة الفروع مع استبعاد التكرار
                    $branchesSql = "SELECT DISTINCT id, name FROM branches ORDER BY name";
                    $branchesStmt = $conn->prepare($branchesSql);
                    $branchesStmt->execute();
                    $allBranches = $branchesStmt->fetchAll(PDO::FETCH_ASSOC);

                    // تصفية الفروع المكررة باستخدام PHP (في حالة وجود فروع بنفس الاسم ولكن بمعرفات مختلفة)
                    $uniqueBranches = [];
                    $uniqueBranchNames = [];

                    foreach ($allBranches as $branch) {
                        if (!in_array($branch['name'], $uniqueBranchNames)) {
                            $uniqueBranchNames[] = $branch['name'];
                            $uniqueBranches[] = $branch;
                        }
                    }

                    // الفروع المعتمدة فقط
                    $approvedBranchNames = ['المكلا', 'الصين', 'عدن', 'صنعاء', 'شحن'];

                    // تصفية الفروع حسب الفروع المعتمدة فقط
                    $branches = [];
                    foreach ($uniqueBranches as $branch) {
                        // تحقق مما إذا كان اسم الفرع يحتوي على أحد أسماء الفروع المعتمدة
                        $isApproved = false;
                        foreach ($approvedBranchNames as $approvedName) {
                            if (strpos($branch['name'], $approvedName) !== false) {
                                $isApproved = true;
                                break;
                            }
                        }

                        if ($isApproved) {
                            $branches[] = $branch;
                        }
                    }

                    // عدد الفروع المستبعدة
                    $excludedBranchesCount = count($uniqueBranches) - count($branches);

                    $selectedBranch = $_GET['branch'] ?? 'all';

                    // إضافة تصنيف للفروع لتحسين العرض
                    echo "<optgroup label=\"الفروع المتاحة\">";

                    // ترتيب الفروع أبجدياً
                    usort($branches, function($a, $b) {
                        return strcmp($a['name'], $b['name']);
                    });

                    // ترتيب الفروع المعتمدة حسب الترتيب المحدد
                    $orderedBranchNames = ['المكلا', 'صنعاء', 'عدن', 'شحن', 'الصين'];

                    // ترتيب الفروع حسب الترتيب المحدد
                    usort($branches, function($a, $b) use ($orderedBranchNames) {
                        $aIndex = -1;
                        $bIndex = -1;

                        // البحث عن الفرع في قائمة الفروع المرتبة
                        foreach ($orderedBranchNames as $index => $name) {
                            if (strpos($a['name'], $name) !== false) {
                                $aIndex = $index;
                            }
                            if (strpos($b['name'], $name) !== false) {
                                $bIndex = $index;
                            }
                        }

                        // ترتيب حسب الفهرس
                        if ($aIndex !== -1 && $bIndex !== -1) {
                            return $aIndex - $bIndex;
                        } elseif ($aIndex !== -1) {
                            return -1;
                        } elseif ($bIndex !== -1) {
                            return 1;
                        } else {
                            return strcmp($a['name'], $b['name']);
                        }
                    });

                    // إضافة عنوان للفروع المعتمدة
                    echo "<optgroup label=\"الفروع المعتمدة\">";

                    // عرض الفروع المعتمدة
                    foreach ($branches as $branch) {
                        $selected = ($selectedBranch == $branch['id']) ? 'selected' : '';

                        // تحديد أيقونة لكل فرع
                        $icon = '';
                        if (strpos($branch['name'], 'المكلا') !== false) {
                            $icon = '🏢 ';
                        } elseif (strpos($branch['name'], 'صنعاء') !== false) {
                            $icon = '🏙️ ';
                        } elseif (strpos($branch['name'], 'عدن') !== false) {
                            $icon = '🌊 ';
                        } elseif (strpos($branch['name'], 'شحن') !== false) {
                            $icon = '🚚 ';
                        } elseif (strpos($branch['name'], 'الصين') !== false) {
                            $icon = '🇨🇳 ';
                        }

                        echo "<option value=\"{$branch['id']}\" {$selected}>{$icon}{$branch['name']}</option>";
                    }

                    echo "</optgroup>";
                    ?>
                  </select>
                  <div class="absolute inset-y-0 right-0 flex items-center pr-3 pointer-events-none">
                    <i class="fas fa-code-branch text-gray-400"></i>
                  </div>
                </div>
                <p class="mt-1 text-xs text-gray-500">اختر الفرع لفلترة التقارير والتصدير</p>

                <!-- إضافة نص توضيحي حول الفروع المعتمدة -->
                <p class="mt-1 text-xs text-green-600">
                  <i class="fas fa-check-circle ml-1"></i>
                  تم عرض الفروع المعتمدة فقط (<?php echo count($branches); ?> فروع)
                </p>

                <?php if ($excludedBranchesCount > 0): ?>
                <p class="mt-1 text-xs text-gray-500">
                  <i class="fas fa-info-circle ml-1"></i>
                  تم استبعاد <?php echo $excludedBranchesCount; ?> فروع غير معتمدة
                </p>
                <?php endif; ?>
              </div>

              <!-- فلتر الحالة -->
              <div>
                <label for="statusFilter" class="block text-sm font-medium text-gray-700 mb-2">حالة الطلب</label>
                <div class="relative">
                  <select id="statusFilter" name="status" class="form-input pl-10 w-full">
                    <option value="">كل الحالات</option>
                    <?php
                    // قائمة حالات الطلبات
                    $statuses = ['تم التسليم', 'قيد المعالجة', 'في الطريق', 'ملغى'];

                    $selectedStatus = $_GET['status'] ?? '';

                    foreach ($statuses as $status) {
                        $selected = ($selectedStatus == $status) ? 'selected' : '';
                        echo "<option value=\"{$status}\" {$selected}>{$status}</option>";
                    }
                    ?>
                  </select>
                  <div class="absolute inset-y-0 right-0 flex items-center pr-3 pointer-events-none">
                    <i class="fas fa-tasks text-gray-400"></i>
                  </div>
                </div>
                <p class="mt-1 text-xs text-gray-500">اختر حالة الطلبات لفلترة التقارير</p>
              </div>

              <!-- زر التصفية -->
              <div class="flex items-end">
                <button id="filterBtn" class="btn btn-primary w-full py-2.5" onclick="applyFilters()">
                  <i class="fas fa-filter ml-2"></i>
                  <span>تطبيق الفلتر</span>
                </button>
              </div>
            </div>

            <?php if (isset($_GET['branch']) && $_GET['branch'] !== 'all' || isset($_GET['status']) && $_GET['status'] !== ''): ?>
            <div class="mt-4 pt-4 border-t border-gray-200">
              <div class="flex items-center text-sm text-gray-600">
                <i class="fas fa-info-circle ml-2 text-primary"></i>
                <span>الفلاتر النشطة:</span>

                <?php if (isset($_GET['branch']) && $_GET['branch'] !== 'all'):
                  $branchName = '';
                  foreach ($branches as $branch) {
                    if ($branch['id'] == $_GET['branch']) {
                      $branchName = $branch['name'];
                      break;
                    }
                  }
                ?>
                <span class="mr-2 bg-blue-100 text-blue-800 px-2 py-1 rounded-full text-xs">
                  الفرع: <?php echo $branchName; ?>
                </span>
                <?php endif; ?>

                <?php if (isset($_GET['status']) && $_GET['status'] !== ''): ?>
                <span class="mr-2 bg-green-100 text-green-800 px-2 py-1 rounded-full text-xs">
                  الحالة: <?php echo $_GET['status']; ?>
                </span>
                <?php endif; ?>

                <div class="mr-auto text-xs">
                  <span>سيتم تطبيق نفس الفلاتر عند تصدير التقارير</span>
                </div>
              </div>
            </div>
            <?php endif; ?>

            <script>
              function applyFilters() {
                const branch = document.getElementById('branchFilter').value;
                const status = document.getElementById('statusFilter').value;
                const weekStart = '<?php echo $selectedWeekStart; ?>';

                let url = 'weekly-reports.php?week_start=' + weekStart;

                if (branch && branch !== 'all') {
                  url += '&branch=' + branch;
                }

                if (status) {
                  url += '&status=' + encodeURIComponent(status);
                }

                window.location.href = url;
              }
            </script>
          </div>

          <!-- جدول الطلبات -->
          <div class="overflow-x-auto">
            <table class="data-table">
              <thead>
                <tr>
                  <th>رقم الطلب</th>
                  <th>العميل</th>
                  <th>نوع الخدمة</th>
                  <th>الفرع</th>
                  <th>تاريخ الطلب</th>
                  <th>تاريخ التسليم المتوقع</th>
                  <th>المبلغ</th>
                  <th>الموظف المسؤول</th>
                  <th>الحالة</th>
                  <th>الإجراءات</th>
                </tr>
              </thead>
              <tbody>
                <?php if (empty($weeklyOrders)): ?>
                <tr>
                  <td colspan="10" class="text-center py-8">
                    <div class="flex flex-col items-center justify-center text-gray-500">
                      <i class="fas fa-inbox text-4xl mb-3 text-gray-300"></i>
                      <p>لا توجد طلبات في هذا الأسبوع</p>
                    </div>
                  </td>
                </tr>
                <?php else: ?>
                  <?php foreach ($weeklyOrders as $order): ?>
                  <tr>
                    <td class="font-medium"><?php echo $order['order_number']; ?></td>
                    <td><?php echo $order['client_name']; ?></td>
                    <td><?php echo $order['service_type']; ?></td>
                    <td><?php echo $order['branch'] ?? 'غير محدد'; ?></td>
                    <td><?php echo formatDateArabic($order['order_date']); ?></td>
                    <td><?php echo formatDateArabic($order['delivery_date']); ?></td>
                    <td><?php echo number_format(($order['paid_amount'] ?? 0) + ($order['remaining_amount'] ?? 0), 2); ?> ريال</td>
                    <td><?php echo $order['employee_name'] ? $order['employee_name'] : 'غير محدد'; ?></td>
                    <td>
                      <?php
                      $statusClass = '';
                      $statusIcon = '';

                      switch ($order['status']) {
                          case 'تم التسليم':
                              $statusClass = 'bg-green-100 text-green-800';
                              $statusIcon = 'check-circle';
                              break;
                          case 'قيد المعالجة':
                              $statusClass = 'bg-yellow-100 text-yellow-800';
                              $statusIcon = 'clock';
                              break;
                          case 'ملغى':
                              $statusClass = 'bg-red-100 text-red-800';
                              $statusIcon = 'times-circle';
                              break;
                          default:
                              $statusClass = 'bg-gray-100 text-gray-800';
                              $statusIcon = 'circle';
                      }
                      ?>
                      <span class="px-2 py-1 rounded-full text-xs font-medium <?php echo $statusClass; ?>">
                        <i class="fas fa-<?php echo $statusIcon; ?> ml-1"></i>
                        <?php echo $order['status']; ?>
                      </span>
                    </td>
                    <td>
                      <div class="flex items-center space-x-2 space-x-reverse">
                        <a href="orders/view-order.php?id=<?php echo $order['id']; ?>" class="p-1 text-primary hover:bg-primary/10 rounded-full transition-all" title="عرض التفاصيل">
                          <i class="fas fa-eye"></i>
                        </a>
                        <a href="orders/edit-order.php?id=<?php echo $order['id']; ?>" class="p-1 text-info hover:bg-info/10 rounded-full transition-all" title="تعديل الطلب">
                          <i class="fas fa-edit"></i>
                        </a>
                        <a href="#" class="p-1 text-gray-400 hover:bg-gray-100 rounded-full transition-all" title="المزيد من الخيارات">
                          <i class="fas fa-ellipsis-v"></i>
                        </a>
                      </div>
                    </td>
                  </tr>
                  <?php endforeach; ?>
                <?php endif; ?>
              </tbody>
            </table>
          </div>

          <!-- تذييل الجدول -->
          <div class="mt-4 pt-4 border-t border-gray-200 flex flex-col md:flex-row justify-between items-start md:items-center gap-4">
            <div class="text-sm text-gray-500">
              <div class="flex items-center mb-2">
                <i class="fas fa-info-circle ml-2 text-primary"></i>
                <span>إجمالي الطلبات: <span class="font-medium text-primary"><?php echo count($weeklyOrders); ?></span></span>
              </div>

              <?php if (isset($_GET['branch']) && $_GET['branch'] !== 'all'):
                $branchName = '';
                foreach ($branches as $branch) {
                  if ($branch['id'] == $_GET['branch']) {
                    $branchName = $branch['name'];
                    break;
                  }
                }
              ?>
              <div class="flex items-center text-xs text-gray-500 mb-1">
                <i class="fas fa-filter ml-1 text-blue-500"></i>
                <span>تم فلترة النتائج حسب الفرع: <span class="font-medium"><?php echo $branchName; ?></span></span>
              </div>
              <?php endif; ?>

              <?php if (isset($_GET['status']) && $_GET['status'] !== ''): ?>
              <div class="flex items-center text-xs text-gray-500">
                <i class="fas fa-filter ml-1 text-green-500"></i>
                <span>تم فلترة النتائج حسب الحالة: <span class="font-medium"><?php echo $_GET['status']; ?></span></span>
              </div>
              <?php endif; ?>
            </div>

            <div class="flex flex-wrap items-center gap-4">
              <a href="weekly-reports.php?week_start=<?php echo $currentWeekStart; ?><?php echo isset($_GET['branch']) && $_GET['branch'] !== 'all' ? '&branch=' . $_GET['branch'] : ''; ?><?php echo isset($_GET['status']) && $_GET['status'] !== '' ? '&status=' . urlencode($_GET['status']) : ''; ?>" class="text-primary hover:text-primary-dark text-sm font-medium flex items-center">
                <span>العودة للأسبوع الحالي</span>
                <i class="fas fa-calendar-week mr-1"></i>
              </a>

              <div class="border-r border-gray-300 h-6 mx-1 hidden md:block"></div>

              <a href="orders.php<?php echo isset($_GET['branch']) && $_GET['branch'] !== 'all' ? '?branch=' . $_GET['branch'] : ''; ?><?php echo isset($_GET['status']) && $_GET['status'] !== '' ? (isset($_GET['branch']) && $_GET['branch'] !== 'all' ? '&' : '?') . 'status=' . urlencode($_GET['status']) : ''; ?>" class="text-primary hover:text-primary-dark text-sm font-medium flex items-center">
                <span>عرض جميع الطلبات</span>
                <i class="fas fa-arrow-left mr-1"></i>
              </a>

              <?php if (isset($_GET['branch']) && $_GET['branch'] !== 'all' || isset($_GET['status']) && $_GET['status'] !== ''): ?>
              <div class="border-r border-gray-300 h-6 mx-1 hidden md:block"></div>

              <a href="weekly-reports.php?week_start=<?php echo $selectedWeekStart; ?>" class="text-red-500 hover:text-red-600 text-sm font-medium flex items-center">
                <span>إزالة الفلاتر</span>
                <i class="fas fa-times-circle mr-1"></i>
              </a>
              <?php endif; ?>
            </div>
          </div>
        </div>
      </div>
    </main>
  </div>

  <!-- سكريبت الرسوم البيانية -->
  <script>
    document.addEventListener('DOMContentLoaded', function() {
      // تهيئة القائمة الجانبية للأجهزة المحمولة
      const sidebar = document.querySelector('aside');
      const overlay = document.getElementById('overlay');
      const toggleSidebarBtn = document.getElementById('toggleSidebar');

      // إضافة مستمع حدث لزر تبديل القائمة الجانبية
      if (toggleSidebarBtn) {
        toggleSidebarBtn.addEventListener('click', function() {
          sidebar.classList.toggle('active');
          overlay.classList.toggle('active');
        });
      }

      // إغلاق القائمة الجانبية عند النقر على الخلفية
      if (overlay) {
        overlay.addEventListener('click', function() {
          sidebar.classList.remove('active');
          overlay.classList.remove('active');
        });
      }

      // تهيئة القوائم المنسدلة
      const dropdownToggles = document.querySelectorAll('.dropdown-toggle');

      dropdownToggles.forEach(toggle => {
        toggle.addEventListener('click', function(e) {
          e.preventDefault();

          // الحصول على القائمة المنسدلة المرتبطة
          const dropdownId = this.getAttribute('id');
          const dropdownMenu = document.getElementById(dropdownId.replace('Dropdown', 'Menu'));

          if (dropdownMenu) {
            // تبديل حالة القائمة المنسدلة
            dropdownMenu.classList.toggle('active');
            this.classList.toggle('active');
          }
        });
      });

      // تم حذف الرسوم البيانية
    });
  </script>
</body>

</html>
